# -*- coding: utf-8 -*-
'''
███████╗████████╗██████╗ ██████╗ ██╗  ██╗██╗  ██╗     ██████╗ ██████╗ ███╗   ██╗████████╗██████╗  ██████╗ ██╗     ██╗     ███████╗██████╗ 
██╔════╝╚══██╔══╝██╔══██╗╚════██╗╚██╗██╔╝╚██╗██╔╝    ██╔════╝██╔═══██╗████╗  ██║╚══██╔══╝██╔══██╗██╔═══██╗██║     ██║     ██╔════╝██╔══██╗
█████╗     ██║   ██║  ██║ █████╔╝ ╚███╔╝  ╚███╔╝     ██║     ██║   ██║██╔██╗ ██║   ██║   ██████╔╝██║   ██║██║     ██║     █████╗  ██████╔╝
██╔══╝     ██║   ██║  ██║██╔═══╝  ██╔██╗  ██╔██╗     ██║     ██║   ██║██║╚██╗██║   ██║   ██╔══██╗██║   ██║██║     ██║     ██╔══╝  ██╔══██╗
██║        ██║   ██████╔╝███████╗██╔╝ ██╗██╔╝ ██╗    ╚██████╗╚██████╔╝██║ ╚████║   ██║   ██║  ██║╚██████╔╝███████╗███████╗███████╗██║  ██║
╚═╝        ╚═╝   ╚═════╝ ╚══════╝╚═╝  ╚═╝╚═╝  ╚═╝     ╚═════╝ ╚═════╝ ╚═╝  ╚═══╝   ╚═╝   ╚═╝  ╚═╝ ╚═════╝ ╚══════╝╚══════╝╚══════╝╚═╝  ╚═╝                                                                                                                                              
'''

import ctypes
import os
import time

# Use TI_FTDIx64.dll located in this directory's bin folder:
dirPath = os.path.dirname(os.path.abspath(__file__))
fileName = dirPath + "\\bin\\TI_FTDI_DLL\\x64\\TI_FTDIx64.dll"

# Showing how to use an absolute path for specific TI_FTDIx64.dll version:
# fileName = "C:/TI_FTDI_x64/x64/TI_FTDIx64.dll"

ctypes.windll.kernel32.SetDllDirectoryW(None)

microCtrl = ctypes.cdll.LoadLibrary(fileName)
microCtrl = ctypes.CDLL(fileName)
mapValue = [-1]              # mutable varaible that allows user not to have to explicitly call out the mapping function

def version():
    return 'v1.0.1'

def connected_devices(bool,allDevices):
    '''scan for connected devices and return # found

    Args:
        bool (boolean): print to screen device serial numbers

    Returns:
        int: # of devices found
    '''
    return microCtrl.printConnectedDevices(bool,allDevices)

def create_device_info_list():
    '''Create the internal device info list and return the number of devices

    Raises:
        RuntimeError: raises a runtime error and returns the in from the dll

    Returns:
        list: serial numbers of connected devices
    '''    

    devCount = connected_devices(False,True)
    if devCount:
        get_device_info_detail(0,True)

    return devCount

def get_device_info_detail(index,bool=False):
    deviceInfo = {}

    flags = ctypes.c_ulong()
    type = ctypes.c_ulong()
    devID = ctypes.c_ulong()
    locID = ctypes.c_ulong()
    description = ctypes.create_string_buffer(64)
    serial = ctypes.create_string_buffer(64)    

    val = microCtrl.getDeviceInfo(index, ctypes.byref(flags), ctypes.byref(type), ctypes.byref(devID), ctypes.byref(locID), description, serial, bool)
    deviceInfo.update({'index' : index, 'flags' : flags.value,'type': type.value,'id': devID.value,'location': locID.value,'serial': serial.value,'description': description.value})

    return deviceInfo


def get_num_connected_devices():
    return create_device_info_list()


def print_all_connected_devices():
    num_connected_devices = get_num_connected_devices()
    print(f'There are {num_connected_devices} connected devices.')
    for each in range(num_connected_devices):
        deviceInfo = get_device_info_detail(each)
        print(deviceInfo)
    return


def open_FT(serialNum : str = '',port : str = '',baudRate : int=12000000):
    '''Open the device port and return a handle which will be used for subsequent accesses

    Args:
        serialNum (str): serial number of device with the port A-D. Defaults to ''.
        port (str): port number A-D, if left blank then assumes port is at the end of the serial number
        baudRate (int): maximum baud rate device can run at. Defaults to 12Mbaud

    Raises:
        RuntimeError: returns status error from dll

    Returns:
        pointer: object reference
    '''

    if port:
        serialNum = serialNum + port

    ftDevice = ctypes.c_void_p()

    status = microCtrl.setupFTDIdev(ctypes.byref(ftDevice),serialNum.encode(),baudRate)
    
    if not status:
        return FTDIxxxx(ftDevice)
        # return ftDevice
        
    else:
        raise RuntimeError('Device not found. Returned : ' + str(status))

class FTDIxxxx:

    def __init__(self,device):
        self.device = device


    def map_port(self):
        '''maps the device port to an integer for the dll to keep track of

        Raises:
            RuntimeError: _description_
        '''
        mapValue[0] += 1
        self.mapValue = mapValue[0]

        status = microCtrl.mapHandles(self.device, self.port, self.mapValue)

        if status:
            raise RuntimeError('Handling mapping error. Returned : ' + str(status))        

    def write_bytes(self, mask:int, byte:int):

        if isinstance(byte, list):
            byteLength = len(byte) 
            byte_ctype = (ctypes.c_uint8*byteLength)()
            for i in range(0, byteLength):
                byte_ctype[i] = byte[i]
        else:
            byteLength = 1
            byte_ctype = (ctypes.c_uint8*1)()
            byte_ctype[0] = byte

        microCtrl.writeCustomPattern(self.device, mask, byte_ctype, byteLength)

    def init_SPI(self,sclk:int=0,mosi:int=1,miso:int=2,csb:int=3,addr_bits:int=7,data_bits:int=16,pos_edge:bool=True,rw_bit:bool=True, oe=None, oe_pol=True):
            '''initialize the port as a SPI controller

            Args:
                sclk (int, optional): DUT sclk line. Defaults to 0.
                mosi (int, optional): DUT data input line. Defaults to 1.
                miso (int, optional): DUT data output line. Defaults to 2.
                csb (int, optional): DUT chip select line. Defaults to 3.
                addr_bits (int, optional): number of address bits. Defaults to 7.
                data_bits (int, optional): number of data bits. Defaults to 16.
                pos_edge (bool, optional): positive edge triggered. Defaults to True.
                rw_bit (bool, optional): has a read/write bit. Defaults to True.
                oe (bool, optional): Enable output enable (for 5-wire spi cases). Defaults to None.
                oe_pol (bool optional): set output enable polarity. True = Active Low, False = Active High. Defaults to True. 

            Raises:
                RuntimeError: _description_

            Returns:
                reference: handle to device
            '''
            self.port = ctypes.c_void_p()

            status = microCtrl.initSPIDev(ctypes.byref(self.port),sclk,csb,mosi,miso,addr_bits,data_bits,pos_edge,rw_bit)    

            if isinstance(oe, int):
                if oe <= 7 and oe >= 0:
                    status = microCtrl.setupOE(ctypes.byref(self.port), oe, oe_pol)

            if status:
                raise RuntimeError('SPI initialization error. Returned : ' + str(status))
            else:
                self.map_port()
                return ftdRegCtrl(self.mapValue,self.device,self.port)
  

    def init_I2C(self,scl : int=0, sdaw : int=1, sdar : int=2, dev_addr : int=0x00, addr_bits : int=8, data_bits : int=8):
        '''initalize the device port to act as an I2C controller

        Args:
            scl (int, optional): clock line. Defaults to 0.
            sdaw (int, optional): data write line from controller. Defaults to 1.
            sdar (int, optional): data read line from controller. Defaults to 2.
            dev_addr (int, optional): device address. Defaults to 0x00.
            addr_bits (int, optional): number of address bits. Defaults to 8.
            data_bits (int, optional): number of data bits. Defaults to 8.

        Raises:
            RuntimeError: _description_

        Returns:
            reference: handle to device
        '''
        self.port = ctypes.c_void_p()

        status = microCtrl.initI2CDev(ctypes.byref(self.port),scl,sdaw,sdar,dev_addr,ctypes.c_uint8(int(addr_bits/8)),ctypes.c_uint8(int(data_bits/8)))    

        if status:
            raise RuntimeError('I2C initialization error. Returned : ' + str(status))
        else:
            self.map_port()
            return ftdRegCtrl(self.mapValue,self.device,self.port)
             

class ftdRegCtrl:
    def __init__(self,mapValue,device,port):
        self.mapValue = mapValue
        self.device = device
        self.port = port
        pass

    def close(self):
        '''closes the open FTDI device reference

        Raises:
            RuntimeError: _description_
        '''

        try:
            self.close_I2C()
        except:
            pass

        try:
            self.close_SPI() 
        except:
            pass

        status = microCtrl.closeFTDI(ctypes.byref(self.device))    

        if status:
            raise RuntimeError('FTDI close error. Returned : ' + str(status))   

    def close_SPI(self):
        status = microCtrl.deleteSPIDev(ctypes.byref(self.port))    

        if status:
            raise RuntimeError('SPI close error. Returned : ' + str(status))        

    def close_I2C(self):
        status = microCtrl.deleteI2CDev(ctypes.byref(self.port))    

        if status:
            raise RuntimeError('I2C close error. Returned : ' + str(status))    

    def read(self,*args):
        '''reads value of register at address

        Args:
            can be one of the following:
            \t1) int: single address\n
            \t2) list: of address\n
            \t3) int, int: start address, stop address


        Returns:
            int: value of the register in decimal\n
            OR\n
            list: value of register in decimal
        '''
        
        if len(args) == 1:
            # if it's a single read or a list of addresses
            if type(args[0]) != list:
                # single write with address, value format
                address = args[0]

                # write single value to device
                return microCtrl.read_reg(self.mapValue,address)

            else:
                
                addresses = args[0]
                values = [0xff]*len(addresses)

                if addresses:
                    addressIn = (ctypes.c_uint32 * len(addresses))(*addresses)
                    valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

                    status = microCtrl.read_regs(self.mapValue,addressIn,valuesIn,len(addresses))
                    
                    return [x for x in valuesIn]
        else:
            startAddress = args[0]
            stopAddress = args[1]

            addresses = []
            for address in range(startAddress,stopAddress+1):
                addresses.append(address)

            values = [0xff]*len(addresses)

            if addresses:
                addressIn = (ctypes.c_uint32 * len(addresses))(*addresses)
                valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

                status = microCtrl.read_regs(self.mapValue,addressIn,valuesIn,len(addresses))
                
                return [x for x in valuesIn]            

    def stream_read(self,startAddress : int = 0x00, numRegs : int = 0):
        values = [0x0]*numRegs
        valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

        
        status = microCtrl.multiRegStreamRW(self.device, self.port,int(startAddress),valuesIn,len(values),False)
        return [x for x in valuesIn]

    def stream_write(self,startAddress : int = 0x00, values : list = []):
        valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

        status = microCtrl.multiRegStreamRW(self.device, self.port,startAddress,valuesIn,len(values),True)
        if status:
            raise RuntimeError('Stream write error.  Returned : ' + str(status))
        else:
            return status


    def write(self, *args):
        '''writes single or multiple address/value combinations to DUT in a single stream

        Args:
            input can be any of the following formats but need type int
            \t1) address,value\n
            \t2) [addesses0-X],[values0-X]\n
            \t3) [(address,value),(addressX,valueX)]

        Raises:
            RuntimeError: returns an error if the input is incorrect or dll has error
            
        '''
        if type(args[0]) == int:
            # single write with address, value format
            address = args[0]
            value = args[1]

            # write single value to device
            status = microCtrl.write_reg(self.mapValue,address,value)

            if status:
                raise RuntimeError('Register write error.  Returned : ' + str(status))
            else:
                return 0
        else:
            addresses = []
            values = []
            if type(args[0]) == list and len(args) == 1:
                # format is a list of tuples
                for each in args[0]:
                    addresses.append(each[0])
                    values.append(each[1])
            elif type(args[0]) == list and len(args) == 2:
                # format is a list of addresses followed by a list of values
                addresses = args[0]
                values = args[1]
            else:
                return 0
            
            if addresses:
                addressIn = (ctypes.c_uint32 * len(addresses))(*addresses)
                valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

                status = microCtrl.write_regs(self.mapValue,addressIn,valuesIn,len(addresses))

                if status:
                    raise RuntimeError('Register write error.  Returned : ' + str(status))
                else:
                    return 0
            else:
                raise RuntimeError('Input value(s) are not in the correct format')



# ******************************************************************************************************************
# EEPROM operations 
# ******************************************************************************************************************

    def convert_to_integer_array(self,data):
        '''Converts data to an integer array.

        Args:
            data (str/char_array/int): Can be any of the following: string, character array, single integer

        Returns:
            list: Data converted as unicode representation in integer list.
        '''
        if type(data)==int:
            convertedData=[data] # convert to list of int
        if type(data)==list or type(data)==str: # convert list of characters or string
            if type(data[0])!=int: # if not already integer array:
                # convert to array of integers by taking ord()
                tempList=[]
                for each in data:
                    tempList.append(ord(each))
                convertedData = tempList
            else:
                convertedData = data # do not change array, it is already an integer array
        return convertedData 

    def write_memory(self, startIndex : int=0, data : str = ''):
        '''Write a user-specified selection of registers into this 8K EEPROM.

        Args:
            startIndex (int): Beginning location of the data to written.
            data (str): String of data to be written into EEPROM.
        '''
        intList = self.convert_to_integer_array(data)
        # old method writing single regsiters at a time
        for index in range(0,len(intList)):
            self.write(startIndex+index,intList[index])

        # numBytes = len(intList) 

        # # must use uint32
        # if numBytes>1:
        #     num_chunks = numBytes/32-(numBytes%32)/32 # essential the same as np.floor(numBytes/32)
        #     for idx in range(0,int(num_chunks)): # bulk 32 byte reads in this loop:
        #         values = intList[idx*32:idx*32+32]
        #         mem_allocation=(ctypes.c_uint32*32)(*values)
        #         self.streamWrite(startIndex+(32*idx),mem_allocation)
        # else:
        #     self.write(startIndex,intList[0])
        # return

    def reset_memory(self,startIndex : int=0, numBytes : int = 0, value : int=0xFF):
        '''Sequentially reset the EEPROM memory with a user-specified character/value.

        Args:
            startIndex (int): Beginning location of data to overwrite.
            numBytes (int): Number of EEPROM registers to overwrite.
            value (int, optional): User specified value to reset into the EEPROM. Defaults to 0xFF (Unicode representation = "ÿ").
        '''
        # dataRange = ''.join([chr(value)]*numBytes)
        # self.writeMemory(startIndex,dataRange)

        # old method
        for index in range(0,numBytes):
            self.write_memory(startIndex+index, chr(value)) # chr() is used to converter an integer or hexadecimal value into its unicode representation
        return    

    def read_memory(self, startIndex : int=0,numBytes : int=0):
        '''Read a user-specified selection of registers from this 8K EEPROM.

        Args:
            startIndex (int): Beginning location of the data to read.
            numBytes (int): Number of EEPROM registers to read.

        Returns:
            list: Sequential data in EEPROM from user-specified register as integer list.
        '''

        self.read(0x0) # to avoid the bug of first readback after FTDI connection being -2^31.
        # OLD
        # memDump = [] # Create an empty list
        # for index in range(startIndex, startIndex+numBytes):
        #     memDump.append(self.read(index))
        # return memDump

        memory_dump = [] # Create an empty list
        if numBytes>1:
            num_chunks = numBytes/32-(numBytes%32)/32 # essential the same as np.floor(numBytes/32)
            for idx in range(0,int(num_chunks)): # bulk 32 byte reads in this loop:
                data = self.stream_read(startIndex+(32*idx),32)
                for j in data:
                    memory_dump.append(j)
            num_remaining_bytes = int(numBytes%32)
            if num_remaining_bytes > 0: # if there are any incomplete 32 byte reads needed:
                data = self.stream_read(startIndex+(32*num_chunks),num_remaining_bytes)
                for j in data:
                    memory_dump.append(j)
        else:
            for index in range(startIndex, startIndex+numBytes):
                memory_dump.append(self.read(index))
        return memory_dump        

    def read_all_memory(self):
        '''Read every register in this 8K EEPROM.

        Returns:
            list: Data in EEPROM from register 0x0000 through 0x8191 as integer list.
        '''
        self.read(0x0) # to avoid the bug of first readback after FTDI connection being -2^31.
        memDump = [] # Create empty list
        for index in range(0, 2**13): # 0 -> 8191
            memDump.append(self.read(index)) 
        return memDump

    def read_board_details(self):
        '''Read the memory programmed into EEPROM by contract manufacturer (SVTronics/Krypton/other).

        Returns:
            list: Sequential data in EEPROM of length equal to the sum of {val(0x0)*1000 + val(0x1)*100 + val(0x2)*10 + val(0x3)}
        '''
        lengthOfData = self.read_EEPROM_length() # find out how many bytes of data to read from EEPROM
        return self.read_memory(startIndex=0x4,numBytes=lengthOfData)

    def write_next_available_memory(self,data : str='',resetPadding : int=10):
        '''Writes data at next available register in EEPROM without erasing previous data. Updates the first four data length registers (0x0-0x03)

        Args:
            data (str): String of data to be written into EEPROM.
            resetPadding (int, optional): Amount of extra registers to reset in EEPROM at end of data before writing with new data. Defaults to 10.
        '''
        lengthOfData = self.read_EEPROM_length() # find out how many bytes of data are currently programmed in EEPROM
        newDataIndex = lengthOfData+4 # account for first 4 bytes
        self.reset_memory(newDataIndex, len(data)+resetPadding)
        self.write_memory(newDataIndex, data)
        self.update_data_length(lastIndex=newDataIndex+len(data))
        return

    def add_key_value_pairs(self, newDict : dict = {}):
        '''This function will add the dictionary's key/value pairs to EEPROM at next available memory without over-writing any existing data.

        Args:
            newDict (dict): Dictionary containing additional key/value pairs to write to EEPROM.
        '''
        dataList=[]
        for EVM_param in newDict.keys(): 
            # Add the parameter name/field:
            for charValue in EVM_param:
                dataList.append(ord(charValue))
            # Add additional characters
            dataList.append(ord(':')) # Add ":"
            # Add the parameter value:
            for charValue in newDict[EVM_param]:
                dataList.append(ord(charValue))
            # Add newline character for parsing
            dataList.append(92) # Add backslash (decimal 92)
            dataList.append(ord('n')) # Add "n"
        self.write_next_available_memory(self.convert_to_string(dataList))
        return

    def to_digits(self,value : int = 0):
        '''Takes any number and breaks down into thousandths, hundredths, tens, and ones places.

        Args:
            val (int): Integer

        Returns:
            int: thousandths
            int: hundredths
            int: tens
            int: ones
        '''
        try:
            thousandths = int(str(value)[0])
            hundredths = int(str(value)[1])
            tens = int(str(value)[2])
            ones = int(str(value)[3])
        except IndexError:
            try:
                thousandths = 0
                hundredths = int(str(value)[0])
                tens = int(str(value)[1])
                ones = int(str(value)[2])
            except IndexError:
                try:
                    thousandths = 0
                    hundredths = 0
                    tens = int(str(value)[0])
                    ones = int(str(value)[1])
                except IndexError:
                    try:
                        thousandths = 0
                        hundredths = 0
                        tens = 0
                        ones = int(str(value)[0])
                    except:
                        pass
        return thousandths, hundredths, tens, ones

    def update_data_length(self,lastIndex):
        '''Updates the first 4 registers to reflect new data length.

        Args:
            lastIndex (int): New position of last register of value in EEPROM.
        '''
        thousandths,hundredths,tens,ones = self.to_digits(lastIndex-4)
        self.write_memory(0x0000,chr(thousandths+48))
        self.write_memory(0x0001,chr(hundredths+48))
        self.write_memory(0x0002,chr(tens+48))
        self.write_memory(0x0003,chr(ones+48))
        return        


    def convert_to_string(self,data : list = []):
        '''Converts integer array into it's unicode string.

        Args:
            data (list): A list of integers.

        Returns:
            str: Representation of the input data list as single unicode string.
        '''
        string = ""
        for index in data:
            string += chr(index)
        return string


    def embed_length_into_data(self,data : list = []):
        '''Insert the length of the data list at the front of the data list. Length will be the first 4 elements of returned data.

        Args:
            data (list): A list of integers.

        Returns:
            list: A list of integers with length embedded at first 4 elements. All existing elements are shifted by 4 indexes.
        '''
        thousandths,hundredths,tens,ones = self.to_digits(len(data))
        # + 48 turns this pythonic integer into the unicode value.
        # This is to retain singular data type into or out of the EEPROM as unicode rather than mixing unicode/integer data types.
        data.insert(0,ones+48)
        data.insert(0,tens+48)
        data.insert(0,hundredths+48)
        data.insert(0,thousandths+48)
        return data


    def read_EEPROM_length(self):
        '''Read the first 4 registers of EEPROM memory and return length as integer.

        Returns:
            int: Length of EEPROM memory.
        '''
        # I know this can be done better by using only 2 registers and binary format, however I want to stick with unicode representation for data as entire rest of memory is like this...
        self.read(0x0) # to avoid the bug of first readback after FTDI connection being -2^31.
        lengthOfData = chr(self.read(0x0))
        lengthOfData += chr(self.read(0x1))
        lengthOfData += chr(self.read(0x2))
        lengthOfData += chr(self.read(0x3))
        lengthOfData = int(lengthOfData) # Convert concatenated string into an integer
        
        return lengthOfData


    def mfr_programming_function(self,evmParamsDict : dict = {},resetPadding : int = 0):
            '''Write the user-specified dictionary into EEPROM. Dictionary data will start at EEPROM index 0x0004 a

            Args:
                evmParamsDict (dict): Dictionary of EVM specific parameters to be programmed into EEPROM.
                resetPadding (int, optional): Amount of extra registers to reset in EEPROM at end of data before writing with new data. Defaults to 0.
            '''
            # Building single string to write into EEPROM:
            dataList = []
            for EVM_param in evmParamsDict.keys(): 
                # Add the parameter name/field:
                for char_val in EVM_param:
                    dataList.append(ord(char_val))
                # Add additional characters
                dataList.append(ord(':')) # Add ":"
                # Add the parameter value:
                for char_val in evmParamsDict[EVM_param]:
                    dataList.append(ord(char_val))
                # Add newline character for parsing
                dataList.append(92) # Add backslash (decimal 92)
                dataList.append(ord('n')) # Add "n"
            dataList = self.embed_length_into_data(dataList) # inserts the length into the first 4 registers
            self.reset_memory(0x0000,len(dataList)+resetPadding)
            self.write_memory(0x0000,self.convert_to_string(dataList))
            return

    def read_data_EEPROM(self):
        """reads the entire memory space of the eeprom

        Returns:
            str: string of data with \n delimiter
        """
        data = self.convert_to_string(self.read_board_details())
        return data


class evmPowerMonitor:
    def __init__(self):
        pass



if __name__ == '__main__':

    print_all_connected_devices()
    