/* ============================================================================
 * Copyright (c) 2008-2012 Texas Instruments Incorporated.  
 * Except for those rights granted to you in your license from TI, all rights 
 * reserved.
 * 
 * Software License Agreement
 * Texas Instruments (TI) is supplying this software for use solely and 
 * exclusively on TI devices. The software is owned by TI and/or its suppliers, 
 * and is protected under applicable patent and copyright laws.  You may not 
 * combine this software with any open-source software if such combination would 
 * cause this software to become subject to any of the license terms applicable 
 * to such open source software.
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
 * NO WARRANTIES APPLY TO THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY. 
 * EXAMPLES OF EXCLUDED WARRANTIES ARE IMPLIED WARRANTIES OF MERCHANTABILITY 
 * AND FITNESS FOR A PARTICULAR PURPOSE AND WARRANTIES OF NON-INFRINGEMENT, 
 * BUT ALL OTHER WARRANTY EXCLUSIONS ALSO APPLY. FURTHERMORE, TI SHALL NOT, 
 * UNDER ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, CONSEQUENTIAL 
 * OR PUNITIVE DAMAGES, FOR ANY REASON WHATSOEVER.
 * ============================================================================
 */

/** @file csl_uhpi.c
 *
 *  @brief UHPI functional layer API header file
 *
 *  Path: \(CSLPATH)\src
 */

/* ============================================================================
 * Revision History
 * ================
 * 6-Dec-2008 Created
 * ============================================================================
 */

#include "csl_uhpi.h"
#include "csl_uhpiAux.h"

/** ============================================================================
 *   @n@b UHPI_open
 *
 *   @b Description
 *   @n Open the handle to the UHPI CSL module
 *
 *   @b Arguments
 *   @verbatim
            instance      UHPI Hardware instance number
            UhpiObj		  UHPI CSL object
            CSL_Status	  CSL status
     @endverbatim
  *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Returned for success
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameter
 							CSL_ESYS_BADHANDLE - Bad handle passed
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Initializes UHPI module
 *
 *   @b Modifies
 *   @n UHPI Object structure
 *
 *   @b Example
 *   @verbatim
            CSL_UhpiHandle hUhpi;
            hUhpi = UHPI_open(CSL_UhpiInsId	instance,
									 CSL_UhpiObj		*UhpiObj,
		  						     CSL_Status 		*status)
     @endverbatim
 *  ============================================================================
 */
CSL_UhpiHandle UHPI_open(CSL_UhpiInsId	instance,
					CSL_UhpiObj		*UhpiObj,
		  			CSL_Status 		*status)
{
	CSL_UhpiHandle hUhpi;

	*status = CSL_SOK;
	hUhpi = NULL;

	if(NULL == UhpiObj)
	{
		*status = CSL_ESYS_BADHANDLE;
		return (hUhpi);
	}

 	if((instance < CSL_UHPI_INST_0) || (instance >= CSL_UHPI_INST_INVALID))
	{
		*status = CSL_ESYS_INVPARAMS;
		return (hUhpi);
	}

  	hUhpi = UhpiObj;

	switch (instance)
	{
		case CSL_UHPI_INST_0:
			hUhpi->insId	 	= CSL_UHPI_INST_0;
			hUhpi->Regs			= CSL_UHPI_REGS;
 			hUhpi->sysRegs 		= CSL_SYSCTRL_REGS;
    			break;
		default:
			*status = CSL_ESYS_INVPARAMS;
	}

	/* EBSR setting */
	CSL_FINST(hUhpi->sysRegs->EBSR, SYS_EBSR_PPMODE, MODE0);

	/* Reset HPI*/
	//CSL_SYSCTRL_REGS->PSRCR = 0x000A;
	//CSL_FINST(CSL_SYSCTRL_REGS->PRCR, SYS_PRCR_PG4_RST, RST);

	/* Clock gating setting - enable clock */
	//CSL_FINST(hUhpi->sysRegs->PCGCR2, SYS_PCGCR2_HPICG, ACTIVE);


	return (hUhpi);
}

/** ============================================================================
 *   @n@b UHPI_Config
 *
 *   @b Description
 *   @n Configures the UHPI CSL module
 *
 *   @b Arguments
 *   @verbatim
            instanceNum      UHPI Hardware instance number
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Returned for success
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Initializes UHPI module
 *
 *   @b Modifies
 *   @n UHPI Object structure
 *
 *   @b Example
 *   @verbatim
            CSL_Status    status;
            status = UHPI_config(hUhpi,&Uhpisetup);
     @endverbatim
 *  ============================================================================
 */
CSL_Status UHPI_Config(CSL_UhpiHandle hUhpi, CSL_UhpiSetup *UhpiSetup)
{
	CSL_Status         status;
	volatile Uint16    looper;

	status = CSL_SOK;

	if( NULL == hUhpi )
	{
		return CSL_ESYS_BADHANDLE;
	}
	if( NULL == UhpiSetup)
	{
	 	return CSL_ESYS_INVPARAMS;
    }

	/* Configuration to UHPI module will start here */

	/* Enable the free operation even in emulation suspend mode(can be enhanced to have control over this) */
	//CSL_FINST(hUhpi->Regs->PWREMU_MGMT, UHPI_PWREMU_MGMT_FREE, ENABLE);
	//CSL_FINST(hUhpi->Regs->PWREMU_MGMT, UHPI_PWREMU_MGMT_SOFT, DISABLE);


	switch(UhpiSetup->op_mode) //Only one mode is selected in Phoenix
	{
	 case CSL_UHPI_8BIT_MUXED_DUAL_HALFWORD_CYCLES:
	 	  return CSL_ESYS_INVPARAMS;

	 case CSL_UHPI_16BIT_MUXED_SINGLE_FULLWORD_CYCLES:
		  //CSL_FINST(hUhpi->sysRegs->HPICR, SYSCTRL_HPICR_NMUX, DISABLE);
		  break;

 	 case CSL_UHPI_16BIT_MUXED_DUAL_HALFWORD_CYCLES:
          return CSL_ESYS_INVPARAMS;

	 case CSL_UHPI_16BIT_NONMUXED:
          return CSL_ESYS_INVPARAMS;

	 case CSL_UHPI_32BIT_NONMUXED:
		  return CSL_ESYS_INVPARAMS;

	 case CSL_UHPI_32BIT_MUXED:
		  return CSL_ESYS_INVPARAMS;

	 default:
	 	  return CSL_ESYS_INVPARAMS;
	}

	switch(UhpiSetup->gpio_intr)
	{
	 case CSL_UHPI_GPIO_INTERRUPT_ENABLE:
	 	CSL_FINST(hUhpi->Regs->GPIO_ENL, UHPI_GPIO_ENL_GPIO_EN2, ENABLE); //Inversion bit ignored in CSL
		CSL_FINST(hUhpi->Regs->GPINT_CTRLL, UHPI_GPINT_CTRLL_GPINT_EN2, ENABLE);
	    break;
	 case CSL_UHPI_GPIO_INTERRUPT_DISABLE:
	 	CSL_FINST(hUhpi->Regs->GPIO_ENL, UHPI_GPIO_ENL_GPIO_EN2, DISABLE); //Inversion bit ignored in CSL
		CSL_FINST(hUhpi->Regs->GPINT_CTRLL, UHPI_GPINT_CTRLL_GPINT_EN2, DISABLE);
	 	break;
	 default: return CSL_ESYS_INVPARAMS;
	}


	return(status);
}

/** ============================================================================
 *   @n@b UHPI_Close
 *
 *   @b Description
 *   @n Close the handle to UHPI and stops the clock to UHPI
 *
 *   @b Arguments
 *   @verbatim
            None
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Returned for success
 *							CSL_ESYS_BADHANDLE - Bad handle passed
 *
 *   <b> Pre Condition </b>
 *   @n  UHPI_open should be called successfully.
 *   @n  None
 *
 *   <b> Post Condition </b>
 *
 *
 *   @b Example
 *   @verbatim
            Status = UHPI_Close(hUhpi);
			if( Status != CSL_SOK )
			{
			 printf("UHPI_Close is failed\r\n");
			}
     @endverbatim
 *  ============================================================================
 */
CSL_Status UHPI_Close( CSL_UhpiHandle hUhpi )
{

	if( NULL == hUhpi )
	{
		return CSL_ESYS_BADHANDLE;
	}

	/* Reset HPI*/
//	hUhpi->sysRegs->PSRCR = 0x000A;
//	CSL_FINST(hUhpi->sysRegs->PRCR, SYS_PRCR_PG4_RST, RST);

	/* Clock gating setting - Disable clock */
//	CSL_FINST(hUhpi->sysRegs->PCGCR2, SYS_PCGCR2_HPICG, DISABLED);


	/* Close the handle */
	hUhpi = NULL;

	return(CSL_SOK);
}
