/**
 *  \file   SPI_v1.c
 *
 *  \brief  MCSPI IP V1 specific driver implementation.
 *
 *   This file contains the driver APIs for SPI.
 */

/*
 * Copyright (c) 2014-2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#undef SPI_DMA_ENABLE

//#define SPI_DMA_ENABLE
#include <string.h>
#include <stdint.h>
#include <stdbool.h>
#include <ti/drv/spi/MCSPI.h>
#include <ti/drv/spi/soc/SPI_v1.h>
#include <ti/csl/src/ip/mcspi/V0/mcspi.h>
#include <ti/csl/src/ip/mcspi/V0/hw_mcspi.h>
#include <ti/drv/spi/src/SPI_osal.h>
#ifdef SPI_DMA_ENABLE
#include <ti/sdo/edma3/drv/edma3_drv.h>
#endif

#ifdef __GNUC__
#pragma GCC diagnostic ignored "-Wunused-local-typedefs"
#endif

/* This macro generates compilier error if postulate is false, so
 * allows 0 overhead compile time size check.  This "works" when
 * the expression contains sizeof() which otherwise doesn't work
 * with preprocessor */
#define SPI_COMPILE_TIME_SIZE_CHECK(postulate)                          \
   do {                                                                 \
       typedef struct {                                                 \
         uint8_t NegativeSizeIfPostulateFalse[((int)(postulate))*2 - 1];\
       } PostulateCheck;                                              \
   }                                                                    \
   while (0)

#define SPI_PARAMS_SIZE               (32U)

#define MCSPI_TX_RX_FIFO_OFFSET       (4U)

#ifdef SPI_DMA_ENABLE
/** Flag to indicate that Reception EDMA callback has occured             */
#define MCSPI_RX_EDMA_CALLBACK_OCCURED    (0x2U)

/** Flag to indicate that Transmission EDMA callback has occured          */
#define MCSPI_TX_EDMA_CALLBACK_OCCURED    (0x1U)

/** SPI 8 bit character length                                            */
#define MCSPI_EDMA_8_BIT_CHAR_LEN         (8U)

/** CNT flag to indicate in EDMA Transfer                                 */
#define MCSPI_ACNT_FLAG                   (2U)

/** Destination flag to indicate in EDMA Transfer                         */
#define MCSPI_DEST_INDEX_FLAG             (1U)

/** Interrupt enable bit in OPT register for edma                         */
#define MCSPI_OPT_TCINTEN_SHIFT           (0x00000014U)

#define MCSPI_MAXLINKCNT                  (2U)
/**<  Maximum number of EDMA jobs linked at a time (Must be 2).           */

/**<  Transmit EDMA channel event queue number                            */
#define MCSPI_TXEVENTQUE                  (0U)

/**<  Receive EDMA channel event queue number                             */
#define MCSPI_RXEVENTQUE                  (1U)

/**<  Generic invalidate status                                           */
#define MCSPI_STATUS_INVALID              (0xFFFF)

/**<  Generic validate status                                             */
#define MCSPI_STATUS_VALID                (0x1)

#define MCSPI_OPT_SYNCDIM_SHIFT           (0x00000002U)
/**<  Sync Type AB set bit of OPT register for edma                       */

#define MCSPI_OPT_TCC_SHIFT               (0x0000000CU)
/**<  TCC set bit in OPT register for edma                                */

#define MCSPI_OPT_TCC_MASK                (0x0003F000U)
/**<  Tcc mask in OPT register for edma                                   */

#define MCSPI_OPT_TCCMOD_SHIFT            (0x0000000BU)
/**< Transfer completion selection bit                                    */

#define MCSPI_CNT_MAX_VAL                 (0xFFFFU)
/**<  Max possible value of aCnt, bCnt and cCnt                           */

#define Mcspi_SWI_PRIORITY                (0x01)
/**< Priority of the swi thread which handles the interrupts              */

#define MCSPI_EDMA3CC_OPT_FIFO_WIDTH      (0xFFFFF8FFU)
/**< Set FIFO Width for edma transfer                                     */

#define MCSPI_EDMA3CC_OPT_SAM_CONST_MODE  (0x00000001U)
/**< Set SAM in Constant Addressing Mode                                  */

#define MCSPI_EDMA3CC_OPT_DAM_CONST_MODE  (0x00000002U)
/**< Set DAM in Constant Addressing Mode                                  */

#define MCSPI_EDMA3CC_OPT_SAM_INCR_MODE   (0xFFFFFFFEU)
/**< Set SAM in Increment Mode                                            */

#define MCSPI_EDMA3CC_OPT_DAM_INCR_MODE   (0xFFFFFFFDU)
/**< Set DAM in Increment Mode                                            */

#define MCSPI_EDMA3CC_OPT_SAM_DAM_INCR_MODE (0xFFFFFFFCU)
/**< Set DAM in Increment Mode                                            */

#define MCSPI_EDMA3CC_OPT_SYNC_AB         (0x00000004U)
/**< It is AB-synchronized                                                */

#define MCSPI_EDMA3CC_OPT_SYNC_MASK_VALUE (0xFFFFFFFFU)
/**< Mask Value for Transfer Synchronization                              */

#define MCSPI_EDMA3CC_PARAM_LINK_ADDRESS  (0xFFFFU)
/**< Set link Address                                                     */

#define MCSPI_EDMA3CC_PARAM_LINK_ADDR_MASK_VALUE (0x0000FFFFU)
/**< link Address Mask Value                                              */

#define MCSPI_EDMA3CC_PARAM_ACNT          (1U)
/**< aCnt Value                                                           */

#define MCSPI_EDMA3CC_OPT_STATIC_SHIFT    (0x00000003U)
/**< STATIC Token                                                         */

#define MCSPI_EDMA3CC_SRC_BINDEX          (1U)
/**< Src BIndex Value                                                     */

#define MCSPI_EDMA3CC_DST_BINDEX          (1U)
/**< Dst BIndex Value                                                     */

#define MCSPI_EDMA3CC_OPT_SYNC_A          (0x01U)
/**< It is A-synchronized                                                 */

#define MCSPI_EDMA3CC_OPT_SYNCDIM_SHIFT   (3U)
/**< Transfer synchronization dimension Shift Value                       */

#define MCSPI_EDMA3CC_COUNT_VALUE         (0xFFFFU)
/**< Count Value                                                          */

#define MCSPI_MAX_PWRM_EVENTS             (PWRM_INVALIDEVENT)
/**< Max number of PWRM events for which the mcasp will register          */

#define MCSPI_MAX_PWRM_CONSTRAINTS        (PWRM_DISALLOWEDSLEEPSTATE_MASK)
/**< Max number of constraints for which a moudule can register           */

#define MCSPI_MAXLINKCNT            (2U)
#endif

/* SPI functions */
static void          SPI_close_v1(SPI_Handle handle);
static void          SPI_init_v1(SPI_Handle handle);
static SPI_Handle    SPI_open_v1(SPI_Handle handle, const SPI_Params *params);
static bool          SPI_transfer_v1(SPI_Handle handle,
                                     SPI_Transaction *transaction);
static void          SPI_transferCancel_v1(SPI_Handle handle);
static int32_t       SPI_control_v1(SPI_Handle handle, uint32_t cmd, const void *arg);

/* MCSPI functions */
static void          MCSPI_close_v1(MCSPI_Handle mcHandle);
static void          MCSPI_init_v1(MCSPI_Handle mcHandle);
static MCSPI_Handle  MCSPI_open_v1(MCSPI_Handle mcHandle, const MCSPI_Params *params);
static bool          MCSPI_transfer_v1(MCSPI_Handle mcHandle,
                                       SPI_Transaction *transaction);
static void          MCSPI_transferCancel_v1(MCSPI_Handle mcHandle);
static void          MCSPI_transferCallback_v1(MCSPI_Handle mcHandle,
                                               SPI_Transaction *transaction);
static int32_t       MCSPI_control_v1(MCSPI_Handle mcHandle, uint32_t cmd, const void *arg);

static MCSPI_Handle  MCSPI_get_handle(SPI_Handle handle);
static void          MCSPI_get_params(const SPI_Params *params, MCSPI_Params *mcParams);


#ifdef SPI_DMA_ENABLE
/* DMA functions */
static void MCSPI_DMA_rxIsrHandler_v1(uint32_t tcc,
                                      EDMA3_RM_TccStatus status,
                                      void* appData);
static void MCSPI_DMA_txIsrHandler_v1(uint32_t tcc,
                                      EDMA3_RM_TccStatus status,
                                      void* appData);
static void MCSPI_DMA_doNothing_v1(uint32_t tcc,
                                   EDMA3_RM_TccStatus edmaStatus,
                                   void* appData);
static uint32_t MCSPI_configDMA_v1(MCSPI_Handle mcHandle);
static void MCSPI_DMA_localEdmaUpdateParams_v1(MCSPI_Handle mcHandle,
                                               const SPI_Transaction *transaction,
                                               EDMA3_DRV_PaRAMRegs *rxParamSet,
                                               EDMA3_DRV_PaRAMRegs *txParamSet);
static void MCSPI_DMA_ChIntrStatClear_v1(MCSPI_Handle mcHandle);
static void MCSPI_DMA_localCompleteIOedmaCbk_v1(MCSPI_Handle mcHandle);
static void MCSPI_DMA_UpdateError_v1(MCSPI_Handle mcHandle, uint32_t intStatus);
#endif

/* SPI function table for SPI_v1 implementation */
const SPI_FxnTable SPI_FxnTable_v1 = {
    &SPI_close_v1,
    &SPI_control_v1,
    &SPI_init_v1,
    &SPI_open_v1,
    &SPI_transfer_v1,
    &SPI_transferCancel_v1,
    NULL
};

/* MCSPI function table for MCSPI_v1 implementation */
const MCSPI_FxnTable MCSPI_FxnTable_v1 =
{
    &MCSPI_close_v1,
    &MCSPI_control_v1,
    &MCSPI_init_v1,
    &MCSPI_open_v1,
    &MCSPI_transfer_v1,
    &MCSPI_transferCancel_v1
};

/*
 *  ======== MCSPI_get_handle ========
 *  @pre    Function assumes that the handle is not NULL
 */
static MCSPI_Handle MCSPI_get_handle(SPI_Handle handle)
{
    MCSPI_Handle          mcHandle = NULL;
    SPI_v1_HWAttrs const *hwAttrs;
    uint32_t              chNum;
    uint32_t              i;

    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chNum   = hwAttrs->chNum;

    for (i = 0; i < MCSPI_MAX_NUM_INSTANCES; i++)
    {
        if(MCSPI_config[i][chNum].handle == handle)
        {
            mcHandle = (MCSPI_Handle)&(MCSPI_config[i][chNum]);
            break;
        }
    }

    return (mcHandle);
}

/*
 *  ======== SPI_close_v1 ========
 *  @pre    Function assumes that the handle is not NULL
 */
static void SPI_close_v1(SPI_Handle handle)
{
    SPI_v1_Object   *object;

    /* Input parameter validation */
    OSAL_Assert(handle == NULL);

    /* Get the pointer to the object */
    object = (SPI_v1_Object*)handle->object;

    MCSPI_close_v1(object->mcHandle);

    object->mcHandle = NULL;
}

/*
 *  ======== SPI_init_v1 ========
 *  @pre    Function assumes that the handle is not NULL
 */
static void SPI_init_v1(SPI_Handle handle)
{
    MCSPI_Handle     mcHandle;
    SPI_v1_Object   *object;

    /* Input parameter validation */
    OSAL_Assert(handle == NULL);

    object = (SPI_v1_Object*)handle->object;

    /* Get multi-channel handle */
    mcHandle = MCSPI_get_handle(handle);
    if (mcHandle != NULL)
    {
        MCSPI_init_v1(mcHandle);
    }

    /* initialize object varibles */
    object->hwi                 = NULL;
    object->mutex               = NULL;
    object->transferComplete    = NULL;
    object->transferCallbackFxn = NULL;
    object->chOpenedCnt         = 0;
}

/*
 *  ======== SPI_open_v1 ========
 *  @pre    Function assumes that the handle is not NULL
 */
static SPI_Handle SPI_open_v1(SPI_Handle handle, const SPI_Params *params)
{
    MCSPI_Handle          mcHandle;
    MCSPI_Params          mcParams;
    SPI_v1_Object        *object;

    /* Input parameter validation */
    OSAL_Assert(handle == NULL);

    /* Get the pointer to the object */
    object = (SPI_v1_Object*)handle->object;

    /* Get multi-channel handle */
    mcHandle = MCSPI_get_handle(handle);
    if (mcHandle != NULL)
    {
        if (params != NULL)
        {
            /* covert SPI_Params to MCSPI_Params */
            MCSPI_get_params(params, &mcParams);

            /* save the transferCallbackFxn in SPI_Params */
            object->transferCallbackFxn = params->transferCallbackFxn;
        }
        mcHandle = MCSPI_open_v1(mcHandle, &mcParams);
        if (mcHandle != NULL)
        {
            object->mcHandle = mcHandle;
        }
        else
        {
            object->mcHandle = NULL;
        }
    }

    return (handle);
}

/*
 *  ======== SPI_transfer_v1 ========
 *  @pre    Function assumes that handle and transaction is not NULL
 */
static bool SPI_transfer_v1(SPI_Handle handle, SPI_Transaction *transaction)
{
    MCSPI_Handle   mcHandle;
    bool           ret_val = false;

    /* Input parameter validation */
    OSAL_Assert(!((handle != NULL) && (transaction != NULL)));

    /* Get multi-channel handle */
    mcHandle = MCSPI_get_handle(handle);
    if (mcHandle != NULL)
    {
        ret_val = MCSPI_transfer_v1(mcHandle, transaction);
    }

    return (ret_val);
}


/*
 *  ======== SPI_transferCancel_v1 ========
 */
static void SPI_transferCancel_v1(SPI_Handle handle)
{
    MCSPI_Handle   mcHandle;

    /* Input parameter validation */
    OSAL_Assert(handle == NULL);

    /* Get multi-channel handle */
    mcHandle = MCSPI_get_handle(handle);
    if (mcHandle != NULL)
    {
        MCSPI_transferCancel_v1(mcHandle);
    }

    return;
}

/*
 *  ======== SPI_control_v1 ========
 */
static int32_t SPI_control_v1(SPI_Handle handle, uint32_t cmd, const void *arg)
{
    MCSPI_Handle   mcHandle;
    int32_t        ret_val = SPI_STATUS_ERROR;

    /* Input parameter validation */
    OSAL_Assert(handle == NULL);

    /* Get multi-channel handle */
    mcHandle = MCSPI_get_handle(handle);
    if (mcHandle != NULL)
    {
        ret_val = MCSPI_control_v1(mcHandle, cmd, arg);
    }

    return ret_val;
}

/*
 *  ======== MCSPI_get_params ========
 *  @pre    Function assumes that params and mcParams are not NULL
 */
static void MCSPI_get_params(const SPI_Params *params, MCSPI_Params *mcParams)
{
    /* Compilation error check the SPI_Params size */
    //SPI_COMPILE_TIME_SIZE_CHECK(sizeof(SPI_Params) == SPI_PARAMS_SIZE);

    mcParams->transferMode        = params->transferMode;
    mcParams->transferTimeout     = params->transferTimeout;
    mcParams->transferCallbackFxn = NULL;
    mcParams->mode                = params->mode;
    mcParams->bitRate             = params->bitRate;
    mcParams->dataSize            = params->dataSize;
    mcParams->frameFormat         = params->frameFormat;
    mcParams->custom              = params->custom;

    return;
}

/*
 *  ======== MCSPI_close_v1 ========
 *  @pre    Function assumes that the handle is not NULL
 */
static void MCSPI_close_v1(MCSPI_Handle mcHandle)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get SPI handle and channel */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;

    /* Get the pointer to the instance object and channel object */
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);

    MCSPI_init_v1(mcHandle);

    object->chOpenedCnt--;
    if (object->chOpenedCnt == 0)
    {
        if(object->transferComplete != NULL)
        {
            /* Destruct the semaphore */
            SPI_osalDeleteBlockingLock(object->transferComplete);
            object->transferComplete = NULL;
        }

        /* All the channels closed */
        if(object->hwi != NULL)
        {
            /* Destruct the Hwi */
            SPI_osalHardwareIntDestruct(object->hwi,hwAttrs->eventId);
            object->hwi = NULL;
        }

        if(object->mutex != NULL)
        {
            /* Destruct the instance lock */
            SPI_osalDeleteBlockingLock(object->mutex);
            object->mutex = NULL;
        }

        object->mcHandle = NULL;
        object->transferCallbackFxn = NULL;
    }

#ifdef SPI_DMA_ENABLE
    if (hwAttrs->dmaMode == true)
    {
        /* Unlink and free the channels */
        EDMA3_DRV_unlinkChannel(hwAttrs->edmaHandle, object->edmaLinkChId);
        EDMA3_DRV_freeChannel(hwAttrs->edmaHandle, object->edmaLinkChId);

        EDMA3_DRV_freeChannel(hwAttrs->edmaHandle, hwAttrs->rxDmaEventNumber);
        EDMA3_DRV_freeChannel(hwAttrs->edmaHandle, hwAttrs->txDmaEventNumber);
    }
#endif

    chObj->isOpen = (bool)false;
}
/* Get recieved data from the RX FIFO based on the data size */

/*
 *  ======== MCSPI_receiveData_v1 ========
 *  
 *  Get recieved data words from the RX FIFO based on the data size
 *
 *  if dataSize <= 8 bits, read 8-bit word from RX FIFO,
 *  if 8 bits < dataSize <= 16 bits, read 16-bit word from RX FIFO,
 *  if 16 bits < dataSize <= 32 bits, read 32-bit word from RX FIFO,
 */
static void *MCSPI_receiveData_v1 (uint32_t baseAddr, uint32_t dataSize, void *dataBuf, uint32_t chNum);  /*for misra warnings*/
static void *MCSPI_receiveData_v1 (uint32_t baseAddr, uint32_t dataSize, void *dataBuf, uint32_t chNum)
{
    void *dataPtr;

    if (dataSize <= 8U)
    {
        *(uint8_t *)dataBuf = (uint8_t)McSPIReceiveData(baseAddr, chNum);
        dataPtr = (void *)(((uint32_t)dataBuf) + 1);
    }
    else if (dataSize <= 16U)
    {
        *(uint16_t *)dataBuf = (uint16_t)McSPIReceiveData(baseAddr, chNum);
        dataPtr = (void *)(((uint32_t)dataBuf) + 2);
    }
    else
    {
        *(uint32_t *)dataBuf = (uint32_t)McSPIReceiveData(baseAddr, chNum);
        dataPtr = (void *)(((uint32_t)dataBuf) + 4);
    }

    return (dataPtr);
}

/*
 *  ======== MCSPI_transmitData_v1 ========
 *  
 *  Put data to the TX FIFO based on the data size (in # of bits)
 *
 *  if dataSize <= 8 bits, write 8-bit word to TX FIFO,
 *  if 8 bits < dataSize <= 16 bits, write 16-bit word to TX FIFO,
 *  if 16 bits < dataSize <= 32 bits, write 32-bit word to TX FIFO,
 */
void *MCSPI_transmitData_v1 (uint32_t baseAddr, uint32_t dataSize, void *dataBuf, uint32_t chNum);  /*for misra warnings*/
void *MCSPI_transmitData_v1 (uint32_t baseAddr, uint32_t dataSize, void *dataBuf, uint32_t chNum)
{
    void *dataPtr;

    if (dataSize <= 8)
    {
        McSPITransmitData(baseAddr, (uint32_t)(*(uint8_t *)dataBuf), chNum);
        dataPtr = (void *)(((uint32_t)dataBuf) + 1);
    }
    else if (dataSize <= 16)
    {
        McSPITransmitData(baseAddr, (uint32_t)(*(uint16_t *)dataBuf), chNum);
        dataPtr = (void *)(((uint32_t)dataBuf) + 2);
    }
    else
    {
        McSPITransmitData(baseAddr, *(uint32_t *)dataBuf, chNum);
        dataPtr = (void *)(((uint32_t)dataBuf) + 4);
    }

    return (dataPtr);
}

/*
 *  ======== MCSPI_xferSetup_v1 ========
 *  This functions configures the transfer mode (RX/TX, RX only or TX only)
 *  and setup the TX/RX FIFO trigger level when FIFO mode is enabled
 */
static void MCSPI_xferSetup_v1(MCSPI_Handle mcHandle, SPI_Transaction *transaction);  /*for misra warnings*/
static void MCSPI_xferSetup_v1(MCSPI_Handle mcHandle, SPI_Transaction *transaction)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_ChnCfg const  *chnCfg;
    uint32_t              transBytes;
    uint32_t              txFifo;
    uint32_t              rxFifo;
    uint32_t              countIndex;
    uint32_t              wordCount;

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chnCfg  = &(hwAttrs->chnCfg[chNum]);

    if (chnCfg->trMode == MCSPI_RX_ONLY_MODE)
    {
        chObj->writeBufIdx = NULL;
        chObj->writeCountIdx = 0;
    }
    else
    {
        chObj->writeBufIdx = transaction->txBuf;
        if (chObj->writeBufIdx != NULL)
        {
            chObj->writeCountIdx = transaction->count;
        }
        else
        {
            chObj->writeCountIdx = 0;
        }
    }

    if (chnCfg->trMode == MCSPI_TX_ONLY_MODE)
    {
        chObj->readBufIdx = NULL;
        chObj->readCountIdx = 0;
    }
    else
    {
        chObj->readBufIdx = transaction->rxBuf;
        if (chObj->readBufIdx != NULL)
        {
            chObj->readCountIdx = transaction->count;
        }
        else
        {
            chObj->readCountIdx = 0;
        }
    }

    /* Enable FIFO's dependent on which mode of operation is chosen */
    if (chnCfg->trMode == MCSPI_TX_RX_MODE)
    {
        if ((hwAttrs->txTrigLvl == 0) || (hwAttrs->rxTrigLvl == 0))
        {
            txFifo = MCSPI_TX_FIFO_DISABLE;
            rxFifo = MCSPI_RX_FIFO_DISABLE;
            object->fifoSize = 0;
        }
        else
        {
            txFifo = MCSPI_TX_FIFO_ENABLE;
            rxFifo = MCSPI_RX_FIFO_ENABLE;
            object->fifoSize = MCSPI_RX_TX_FIFO_SIZE;
        }
    }
    else if (chnCfg->trMode == MCSPI_TX_ONLY_MODE)
    {
        /* TX_ONLY Mode */
        if (hwAttrs->txTrigLvl == 0)
        {
            txFifo = MCSPI_TX_FIFO_DISABLE;
            object->fifoSize = 0;
        }
        else
        {
            txFifo = MCSPI_TX_FIFO_ENABLE;
            object->fifoSize = MCSPI_FULL_FIFO_SIZE;
        }
        rxFifo = MCSPI_RX_FIFO_DISABLE;
    }
    else
    {
        /* RX_ONLY Mode */
        txFifo = MCSPI_TX_FIFO_DISABLE;
        if (hwAttrs->rxTrigLvl == 0)
        {
            rxFifo = MCSPI_RX_FIFO_DISABLE;
            object->fifoSize = 0;
        }
        else
        {
            rxFifo = MCSPI_RX_FIFO_ENABLE;
            object->fifoSize = MCSPI_FULL_FIFO_SIZE;
        }
    }
    McSPITxFIFOConfig(hwAttrs->baseAddr, txFifo, chNum);
    McSPIRxFIFOConfig(hwAttrs->baseAddr, rxFifo, chNum);

    if (object->fifoSize > 0)
    {
        /* FIFO mode is enabled */

        /* Set FIFO XFER levels */
        transBytes = transaction->count << chObj->wordLenShift;

        if (transBytes <= object->fifoSize)
        {
            /* Transaction fits entirely in FIFO */
            object->rxTrigLvl = transBytes;
            object->txTrigLvl = transBytes;
        }
        else
        {
            /*
             * Transaction count is more than FIFO size
             */
            object->rxTrigLvl = hwAttrs->rxTrigLvl;
            object->txTrigLvl = hwAttrs->txTrigLvl;

#ifdef SPI_DMA_ENABLE
            if (hwAttrs->dmaMode == true)
            {
                /*
                 * In DMA mode, update the transaction count to be
                 * multiple of FIFO trigger level size
                 */
                if (chnCfg->trMode == MCSPI_RX_ONLY_MODE)
                {
                    transBytes = (transBytes / object->rxTrigLvl) * object->rxTrigLvl;
                }
                else
                {
                    transBytes = (transBytes / object->txTrigLvl) * object->txTrigLvl;
                }
                transaction->count = transBytes >> chObj->wordLenShift;
            }
            else
#endif
            {
                /*
                 * In non-DMA mode, if the RX trigger level equals to the FIFO size,
                 * set the RX trigger level less than the FIFO size, so that TX FIFO
                 * will not under run.
                 */
                if ((chnCfg->trMode == MCSPI_TX_RX_MODE) &&
                    (object->rxTrigLvl == object->fifoSize))
                {
                    object->rxTrigLvl -= MCSPI_TX_RX_FIFO_OFFSET;
                }
            }
        }
        McSPIFIFOTrigLvlSet(hwAttrs->baseAddr, object->rxTrigLvl,
                            object->txTrigLvl, chnCfg->trMode);
    }

    /* 
     * convert trigger level (in bytes) to word count 
     * based on the word length
     */
    if (object->fifoSize > 0)
    {
#ifdef SPI_DMA_ENABLE
        if (hwAttrs->dmaMode == true)
        {
            wordCount = object->txTrigLvl >> chObj->wordLenShift;
        }
        else
#endif
        {
            if (transBytes > MCSPI_RX_TX_FIFO_SIZE)
            {
                wordCount = MCSPI_RX_TX_FIFO_SIZE >> chObj->wordLenShift;
            }
            else
            {
                wordCount = transBytes >> chObj->wordLenShift;
            }
        }
    }
    else
    {
        wordCount = 1U;
    }
    /* Set number of words to be transmitted */
    McSPIWordCountSet(hwAttrs->baseAddr, transaction->count);

    if ((SPI_SLAVE == chObj->spiParams.mode) && (chObj->writeCountIdx != 0))
    {
#ifdef SPI_DMA_ENABLE
        if (hwAttrs->dmaMode == false)
        {
            for (countIndex = 0; countIndex < wordCount; countIndex++)
            {
                chObj->writeBufIdx = MCSPI_transmitData_v1 (hwAttrs->baseAddr,
                                                            chObj->spiParams.dataSize,
                                                            chObj->writeBufIdx,
                                                            chNum);
                chObj->writeCountIdx--;
            }
        }
#else
        for (countIndex = 0; countIndex < wordCount; countIndex++)
        {
            chObj->writeBufIdx = MCSPI_transmitData_v1 (hwAttrs->baseAddr,
                                                        chObj->spiParams.dataSize,
                                                        chObj->writeBufIdx,
                                                        chNum);
            chObj->writeCountIdx--;
        }
#endif
    }
}

/*
 *  ======== MCSPI_primeTransfer_v1 ========
 *  This functions configures the transmit and receive channels for a given
 *  MCSPI_Handle and SPI_Transaction.
 *
 *  Slave driver needs to pre-fill the TX trigger level size data to TX FIFO first before
 *  the channel is enabled. If the transfer size is less than the FIFO size (32 bytes), the
 *  TX/RX trigger level is set to the transfer size. If the transfer size is greater than
 *  or equal to the FIFO size, the TX/RX trigger level is set to the FIFO size. The pre-filling
 *  is required to ensure the TX data is available for master to receive when the channel is
 *  enabled.
 *
 *  In polling mode, master driver continuously writes 1 byte TX data to TX FIFO followed
 *  by reading 1 byte RX data from the RX FIFO until all the data are transfered. slave driver
 *  continuously reads 1 byte data from the RX FIFO followed by writing 1 byte TX data to TX
 *  FIFO until all the data are transfered.
 *
 *  In interrupt mode, master driver enables TX empty, RX full and end of word count interrupts.
 *  slave driver enabls RX full interrupt. The TX/RX data are handled in the ISR.
 *
 *  @pre    Function assumes that the handle and transaction is not NULL
 */
static void MCSPI_primeTransfer_v1(MCSPI_Handle mcHandle, SPI_Transaction *transaction);  /* for misra warnings */
static void MCSPI_primeTransfer_v1(MCSPI_Handle mcHandle, SPI_Transaction *transaction)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_ChnCfg const  *chnCfg;
    uint32_t              channelStatus = 0;
    uint32_t              intStatus;

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chnCfg  = &(hwAttrs->chnCfg[chNum]);

    /* setup transfer mode and FIFO trigger levels */
    MCSPI_xferSetup_v1(mcHandle, transaction);

    /* Enable the McSPI channel for communication */
    McSPIChannelEnable(hwAttrs->baseAddr, chNum);

    /* Interrupt Mode */
    if(chObj->operMode != SPI_OPER_MODE_POLLING)
    {
        intStatus =  MCSPI_INT_EOWKE;
        if (chnCfg->trMode == MCSPI_TX_ONLY_MODE)
        {
            intStatus |= MCSPI_INT_TX_EMPTY(chNum);
        }
        else if (chnCfg->trMode == MCSPI_RX_ONLY_MODE)
        {
            intStatus |= MCSPI_INT_RX_FULL(chNum);
        }
        else
        {
            intStatus |= MCSPI_INT_RX_FULL(chNum) | MCSPI_INT_TX_EMPTY(chNum);
        }
        McSPIIntStatusClear(hwAttrs->baseAddr, intStatus);

        if(SPI_MASTER == chObj->spiParams.mode)
        {
            McSPIIntEnable(hwAttrs->baseAddr, intStatus);

            if (hwAttrs->chMode == MCSPI_SINGLE_CH)
            {
                /* Assert un-used chip select (Force SPIEN) */
                McSPICSAssert(hwAttrs->baseAddr, chNum);
            }
        }
        else
        {
            McSPIIntEnable(hwAttrs->baseAddr, MCSPI_INT_RX_FULL(chNum));
        }
    }
    /* Polling mode */
    else
    {
        if ((SPI_MASTER == chObj->spiParams.mode) &&
            (hwAttrs->chMode == MCSPI_SINGLE_CH))
        {
            /* SPIEN line is forced to low state.*/
            McSPICSAssert(hwAttrs->baseAddr, chNum);
        }

        /* Polling mode transfer */
        while ((chObj->readCountIdx != 0) || (chObj->writeCountIdx != 0))
        {
            channelStatus = McSPIChannelStatusGet(hwAttrs->baseAddr, chNum);
            if ((SPI_MASTER == chObj->spiParams.mode) && (chObj->writeCountIdx != 0))
            {
                while (0U == (channelStatus & CSL_MCSPI_CH0STAT_TXS_MASK))
                {
                    channelStatus = 0;
                    channelStatus = McSPIChannelStatusGet(hwAttrs->baseAddr, chNum);
                }
                chObj->writeBufIdx = MCSPI_transmitData_v1 (hwAttrs->baseAddr,
                                                            chObj->spiParams.dataSize,
                                                            chObj->writeBufIdx,
                                                            chNum);
                chObj->writeCountIdx--;
            }

            if (chObj->readCountIdx != 0)
            {
                while (0U == (channelStatus & CSL_MCSPI_CH0STAT_RXS_MASK))
                {
                    channelStatus = 0;
                    channelStatus = McSPIChannelStatusGet(hwAttrs->baseAddr, chNum);
                }
                chObj->readBufIdx = MCSPI_receiveData_v1(hwAttrs->baseAddr,
                                                         chObj->spiParams.dataSize,
                                                         chObj->readBufIdx,
                                                         chNum);
                chObj->readCountIdx--;
            }

            if ((SPI_SLAVE == chObj->spiParams.mode) && (chObj->writeCountIdx != 0))
            {
                chObj->writeBufIdx = MCSPI_transmitData_v1 (hwAttrs->baseAddr,
                                                            chObj->spiParams.dataSize,
                                                            chObj->writeBufIdx,
                                                            chNum);

                chObj->writeCountIdx--;
            }
        }

        if ((SPI_MASTER == chObj->spiParams.mode) &&
            (hwAttrs->chMode == MCSPI_SINGLE_CH))
        {
            /* Force SPIEN line to the inactive state.*/
            McSPICSDeAssert(hwAttrs->baseAddr, chNum);
        }

        /* Disable the McSPI channel.*/
        McSPIChannelDisable(hwAttrs->baseAddr, chNum);

        chObj->transaction = NULL;
    }
}

#ifdef SPI_DMA_ENABLE
/*
 *  ======== MCSPI_DMA_primeTransfer_v1 ========
 *  This functions configures the transmit and receive channels for a given
 *  MCSPI_Handle and SPI_Transaction
 *
 *  @pre    Function assumes that the handle and transaction is not NULL
 */
static void MCSPI_DMA_primeTransfer_v1(MCSPI_Handle     mcHandle,
                                       SPI_Transaction *transaction);  /*for misra warnings*/
static void MCSPI_DMA_primeTransfer_v1(MCSPI_Handle     mcHandle,
                                       SPI_Transaction *transaction)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;
    EDMA3_DRV_PaRAMRegs   rxParamSet = {0};
    EDMA3_DRV_PaRAMRegs   txParamSet = {0};
    EDMA3_DRV_PaRAMRegs   dummyParamSet = {0};

    /* Input parameter validation */
    OSAL_Assert(!((mcHandle != NULL) && (transaction != NULL)));

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object  = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;

    /* setup transfer mode and FIFO trigger levels */
    MCSPI_xferSetup_v1(mcHandle, transaction);

    /* receive parameter set */
    EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                        hwAttrs->rxDmaEventNumber,
                        &rxParamSet);

    /* transmit parameter set */
    EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                        hwAttrs->txDmaEventNumber,
                        &txParamSet);

    /* Update param sets based on the transaction parameters */
    MCSPI_DMA_localEdmaUpdateParams_v1(mcHandle,
                                       transaction,
                                       &rxParamSet,
                                       &txParamSet);

    /* Dummy paramSet Configuration */
    EDMA3_DRV_getPaRAM(hwAttrs->edmaHandle,
                       EDMA3_DRV_LINK_CHANNEL,
                       &dummyParamSet);

    dummyParamSet.aCnt     = (uint16_t)MCSPI_EDMA3CC_PARAM_ACNT;
    dummyParamSet.linkAddr = MCSPI_EDMA3CC_PARAM_LINK_ADDRESS;
    dummyParamSet.opt     &= ~((uint32_t)0x01U << MCSPI_EDMA3CC_OPT_STATIC_SHIFT);

    /* Now, write the PaRAM Set. */
    EDMA3_DRV_setPaRAM(hwAttrs->edmaHandle,
                       EDMA3_DRV_LINK_CHANNEL,
                       &dummyParamSet);

    /* Write Rx param set */
    EDMA3_DRV_setPaRAM((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                        hwAttrs->rxDmaEventNumber,
                        &rxParamSet);

    /* Write Tx param set */
    EDMA3_DRV_setPaRAM((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                       hwAttrs->txDmaEventNumber,
                       &txParamSet);

    /* Link the Dummy paramset */
    EDMA3_DRV_linkChannel(hwAttrs->edmaHandle,
                          hwAttrs->txDmaEventNumber,
                          EDMA3_DRV_LINK_CHANNEL);

    /* Program the RX side */
    EDMA3_DRV_enableTransfer(hwAttrs->edmaHandle,
                             hwAttrs->rxDmaEventNumber,
                             EDMA3_DRV_TRIG_MODE_EVENT);

    /* Program the TX side */
    EDMA3_DRV_enableTransfer(hwAttrs->edmaHandle,
                             hwAttrs->txDmaEventNumber,
                             EDMA3_DRV_TRIG_MODE_EVENT);

    /* Enable McSPI DMA for transaction
     * MCSPI_CH(chNum)CONF -> DMAR bit for receive DMA
     */
    McSPIDMAEnable((uint32_t)(hwAttrs->baseAddr),
                   ((uint32_t)MCSPI_DMA_RX_EVENT |
                    (uint32_t)MCSPI_DMA_TX_EVENT),
                   chNum);

    /* Enable the channel */
    McSPIChannelEnable((uint32_t)(hwAttrs->baseAddr), chNum);

    if ((SPI_MASTER == chObj->spiParams.mode) &&
        (hwAttrs->chMode == MCSPI_SINGLE_CH))
    {
        /* Assert chip select signal */
        McSPICSAssert((uint32_t)(hwAttrs->baseAddr), chNum);
    }

}
#endif

/*
 *  ======== MCSPI_v1_hwiFxn ========
 *  ISR for the MCSPI in master and slave mode
 *
 *  For both TX and RX enabled mode, the following interrupts are enabled:
 *
 *   TX empty interrupt (master only)
 *   RX full interrupt
 *   End of word count interrupt (master only)
 *
 *  When TX empty interrupt happens, ISR will continuously write TX data
 *  to TX FIFO until the FIFO is full or all the TX data is sent out.
 *
 *  When RX full interrupt happens, ISR will continuously read 1 byte
 *  RX data from RX FIFO followed by writing 1 byte TX data to TX FIFO
 *  until the RX FIFO is empty. When all the bytes are transfered or
 *  end of word count interrupt happens (master only), ISR
 *  will disable all the interrupts and call back to the application
 *  in callback mode or release the semaphore in blocking mode.
 *
 *  If the remaining RX count is less than the RX FIFO size,
 *  ISR will set the RX trigger level to the remaing RX count.
 *
 *  To prevent TX FIFO underrun, RX FIFO trigger level is set less
 *  than the max TX FIFO size
 */
static void MCSPI_v1_hwiFxn (uintptr_t arg); /* for misra warnings */
static void MCSPI_v1_hwiFxn (uintptr_t arg)
{
    MCSPI_Handle          mcHandle;
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_ChnCfg const  *chnCfg;
    uint32_t              intCode = 0;
    bool                  xferDone = false;
    bool                  trigLvlChg = false;
    uint32_t              rdBytes;
    uint32_t              wrBytes;
    uint32_t              chnStatus;

    /* Input parameter validation */
    OSAL_Assert(NULL == (void *)arg);

    /* Get the pointer to the object and hwAttrs */
    mcHandle = (MCSPI_Handle)arg;
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chnCfg  = &(hwAttrs->chnCfg[chNum]);

    intCode = McSPIIntStatusGet(hwAttrs->baseAddr);

    /*
     * Fill the TX FIFO if an TX-empty interrupt has occurred.
     */
    if (MCSPI_INT_TX_EMPTY(chNum) == (intCode & MCSPI_INT_TX_EMPTY(chNum)))
    {
        while ((chObj->writeCountIdx != 0U) &&
               ((McSPIChannelStatusGet(hwAttrs->baseAddr, chNum) & CSL_MCSPI_CH0STAT_TXFFF_MASK) == 0U)) 
        {

            /* Fill the TX FIFO until full */
            chObj->writeBufIdx = MCSPI_transmitData_v1 (hwAttrs->baseAddr,
                                                        chObj->spiParams.dataSize,
                                                        chObj->writeBufIdx,
                                                        chNum);
            chObj->writeCountIdx--;

            if ((chObj->readCountIdx != 0) &&
                ((McSPIChannelStatusGet(hwAttrs->baseAddr, chNum) & CSL_MCSPI_CH0STAT_RXFFE_MASK) == 0U))
            {
                /* Read from the RX FIFO until empty */
                chObj->readBufIdx = MCSPI_receiveData_v1(hwAttrs->baseAddr,
                                                         chObj->spiParams.dataSize,
                                                         chObj->readBufIdx,
                                                         chNum);
                chObj->readCountIdx--;
            }
        }

        McSPIIntStatusClear(hwAttrs->baseAddr, MCSPI_INT_TX_EMPTY(chNum));
        intCode = intCode & ~MCSPI_INT_TX_EMPTY(chNum);
    }

    
    /* RX FIFO is full, empty the FIFO to receive more data if necessary */
    if (MCSPI_INT_RX_FULL(chNum) == (intCode & MCSPI_INT_RX_FULL(chNum)))
    {
        while (chObj->readCountIdx != 0U)
        {
            chnStatus = McSPIChannelStatusGet(hwAttrs->baseAddr, chNum);
            if (((chnStatus & CSL_MCSPI_CH0STAT_RXFFE_MASK) == 0U) ||
                ((chnStatus & CSL_MCSPI_CH0STAT_RXS_MASK) == 1U))
            {
                /* Read from the FIFO until no data left */
                chObj->readBufIdx = MCSPI_receiveData_v1(hwAttrs->baseAddr,
                                                         chObj->spiParams.dataSize,
                                                         chObj->readBufIdx,
                                                         chNum);
                chObj->readCountIdx--;
            }
            else
            {
                break;
            }

            if ((chObj->writeCountIdx != 0) &&
                ((McSPIChannelStatusGet(hwAttrs->baseAddr, chNum) & CSL_MCSPI_CH0STAT_TXFFF_MASK) == 0U))
            {
                /* if TX FIFO not full, write 1 byte to TX FIFO */
                chObj->writeBufIdx = MCSPI_transmitData_v1 (hwAttrs->baseAddr,
                                                            chObj->spiParams.dataSize,
                                                            chObj->writeBufIdx,
                                                            chNum);

                chObj->writeCountIdx--;
            }

            if (chObj->readCountIdx == 0)
            {
                xferDone = true;
                break;
            }
        }

        McSPIIntStatusClear(hwAttrs->baseAddr, MCSPI_INT_RX_FULL(chNum));
        intCode = intCode & ~MCSPI_INT_RX_FULL(chNum);
    }

    /* Update the TX trigger level */
    if (chObj->writeCountIdx != 0)
    {
        wrBytes = chObj->writeCountIdx << chObj->wordLenShift;
        if (wrBytes < object->fifoSize)
        {
            object->txTrigLvl = wrBytes;
            trigLvlChg = true;
        }
    }

    /* Update the RX trigger level */
    if (chObj->readCountIdx != 0)
    {
        rdBytes = chObj->readCountIdx << chObj->wordLenShift;
        if (rdBytes < object->fifoSize)
        {
            object->rxTrigLvl = rdBytes;
            trigLvlChg = true;
        }
    }

    if (trigLvlChg == true)
    {
        McSPIFIFOTrigLvlSet(hwAttrs->baseAddr, object->rxTrigLvl,
                            object->txTrigLvl, chnCfg->trMode);
    }

    if ((MCSPI_INT_EOWKE == (intCode & MCSPI_INT_EOWKE)) || (xferDone == true))
    {
        /* Transaction Done */
        McSPIIntDisable(hwAttrs->baseAddr,
                        MCSPI_INT_RX_FULL(chNum) | MCSPI_INT_EOWKE);
        McSPIIntStatusClear(hwAttrs->baseAddr, MCSPI_INT_EOWKE);
        McSPIChannelDisable(hwAttrs->baseAddr, chNum);
        if ((SPI_MASTER == chObj->spiParams.mode) &&
            (hwAttrs->chMode == MCSPI_SINGLE_CH))
        {
            McSPICSDeAssert(hwAttrs->baseAddr, chNum);
        }
        chObj->transaction->status=SPI_TRANSFER_COMPLETED;
        MCSPI_transferCallback_v1((MCSPI_Handle)arg, chObj->transaction);
        intCode = intCode & ~MCSPI_INT_EOWKE;
    }

    if (intCode)
    {
        McSPIIntStatusClear(hwAttrs->baseAddr, intCode);
    }
}

/*
 *  ======== MCSPI_init_v1 ========
 *  @pre    Function assumes that the handle is not NULL
 */
static void MCSPI_init_v1(MCSPI_Handle mcHandle)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);

    /* reset channel object */
    memset(chObj, 0, sizeof(SPI_v1_chObject));
    chObj->isOpen = (bool)false;
}

/*
 *  ======== MCSPI_open_v1 ========
 *  @pre    Function assumes that the mcHandle is not NULL
 */
static MCSPI_Handle MCSPI_open_v1(MCSPI_Handle        mcHandle,
                                  const MCSPI_Params *params)
{
    SPI_Handle            handle;
    SemaphoreP_Params     semParams;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_ChnCfg const  *chnCfg;
    OsalRegisterIntrParams_t interruptRegParams;
    uint32_t              ret_flag = 0u;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chnCfg  = &(hwAttrs->chnCfg[chNum]);

    /* Determine if the device index was already opened */
    if(chObj->isOpen == true) {
        mcHandle = NULL;
    }
    else
    {
        /* Mark the channel handle as being used */
        chObj->isOpen = (bool)true;

        /* Store the SPI parameters */
        if (params == NULL)
        {
            /* No params passed in, so use the defaults */
            MCSPI_Params_init(&(chObj->spiParams));
            params = &(chObj->spiParams);
        }
        else
        {
            chObj->spiParams = *params;
        }

        OSAL_Assert(!((params->dataSize >= 4) && (params->dataSize <= 32)));

        if (params->dataSize <= 8)
        {
            chObj->wordLenShift = 0;
        }
        else if (params->dataSize <= 16)
        {
            chObj->wordLenShift = 1;
        }
        else
        {
            chObj->wordLenShift = 2;
        }

        /* Store the current mode. Extract operating mode from hwAttrs and params */
        if(SPI_MODE_BLOCKING == params->transferMode)
        {
            if (true == hwAttrs->enableIntr)
            {
                chObj->operMode = SPI_OPER_MODE_BLOCKING;
            }
#ifdef SPI_DMA_ENABLE
            else if (true == hwAttrs->dmaMode)
            {
                chObj->operMode = SPI_OPER_MODE_BLOCKING;
            }
#endif
            else
            {
                chObj->operMode = SPI_OPER_MODE_POLLING;
            }
        }
        else
        {
            chObj->operMode = SPI_OPER_MODE_CALLBACK;
        }

        if ((chObj->operMode != SPI_OPER_MODE_POLLING) &&
            (object->hwi == NULL))
        {
            /* register interrrupt when the 1st
               channel of the instance is opened */
            Osal_RegisterInterrupt_initParams(&interruptRegParams);             

            interruptRegParams.corepacConfig.name=NULL;
            interruptRegParams.corepacConfig.priority = 0x20;
            interruptRegParams.corepacConfig.corepacEventNum = hwAttrs->eventId;
            interruptRegParams.corepacConfig.intVecNum=hwAttrs->intNum; /* Host Interrupt vector */
            interruptRegParams.corepacConfig.isrRoutine  = (void (*)(uintptr_t))(&MCSPI_v1_hwiFxn);
            interruptRegParams.corepacConfig.arg         = (uintptr_t)mcHandle;

            SPI_osalRegisterInterrupt(&interruptRegParams,&(object->hwi));

            if(object->hwi == NULL)
            {
                MCSPI_close_v1(mcHandle);
                ret_flag = 1u;
                mcHandle = NULL;
            }
        }

        if(ret_flag == 0u)
        {
            /*
             * Construct thread safe handles for this SPI peripheral
             * Semaphore to provide exclusive access to the SPI peripheral
             */
            if (object->mutex == NULL)
            {
                SPI_osalSemParamsInit(&semParams);
                semParams.mode = SemaphoreP_Mode_BINARY;
                object->mutex = SPI_osalCreateBlockingLock(1U, &semParams);
            }

            if (chObj->operMode == SPI_OPER_MODE_BLOCKING)
            {
                /*
                 * Construct a semaphore to block task execution for the duration of the
                 * SPI transfer
                 */
                if (object->transferComplete == NULL)
                {
                    SPI_osalSemParamsInit(&semParams);
                    semParams.mode = SemaphoreP_Mode_BINARY;
                    object->transferComplete = SPI_osalCreateBlockingLock(0U, &semParams);
                }
            }
            if (chObj->operMode == SPI_OPER_MODE_CALLBACK)
            {
                /* Check to see if a callback function was defined for async mode */
                OSAL_Assert((params->transferCallbackFxn == NULL) &&
                            (object->transferCallbackFxn == NULL));
            }

            chObj->transaction = NULL;

            /* Extract clock mode from the frame format */
            switch(params->frameFormat)
            {
                case SPI_POL0_PHA0:
                    chObj->clockMode = MCSPI_CLK_MODE_0;
                    break;

                case SPI_POL0_PHA1:
                    chObj->clockMode = MCSPI_CLK_MODE_1;
                    break;

                case SPI_POL1_PHA0:
                    chObj->clockMode = MCSPI_CLK_MODE_2;
                    break;

                case SPI_POL1_PHA1:
                    chObj->clockMode = MCSPI_CLK_MODE_3;
                    break;

                default:
                	chObj->clockMode = MCSPI_CLK_MODE_2;
                    break;
            }

            if (object->chOpenedCnt == 0)
            {
#ifdef SPI_DMA_ENABLE
                if (hwAttrs->dmaMode == true)
                {
                    /* DMA Configuration */
                    MCSPI_configDMA_v1(mcHandle);

                    McSPIDMADisable(hwAttrs->baseAddr,
                                    ((uint32_t) MCSPI_DMA_RX_EVENT | (uint32_t) MCSPI_DMA_TX_EVENT),
                                    chNum);
                }
#endif
                /* Reset SPI Peripheral */
                McSPIReset(hwAttrs->baseAddr);

                MCSPISysConfigSetup(hwAttrs->baseAddr, MCSPI_CLOCKS_OCP_ON_FUNC_ON,
                                    MCSPI_SIDLEMODE_NO, MCSPI_WAKEUP_DISABLE,
                                    MCSPI_AUTOIDLE_OFF);

                if (hwAttrs->chMode == MCSPI_SINGLE_CH)
                {
                    /* Configure 3 pin or 4 pin mode */
                    if(SPI_PINMODE_3_PIN == hwAttrs->pinMode)
                    {
                        /* Disable chip select pin.*/
                        McSPICSDisable(hwAttrs->baseAddr);
                    }
                    else
                    {
                        /* Enable chip select pin.*/
                        McSPICSEnable(hwAttrs->baseAddr);
                    }
                }
            }

            if ((object->chOpenedCnt != 0) &&
                ((params->mode != SPI_MASTER) || (object->mode != SPI_MASTER)))
            {

                /*
                 * McSPI supports multiple master channels or single slave channel per instance:
                 *
                 * 1. if an instance has already had a channle opened in SPI_MASTER mode,
                 * the new channel being opened should have the same SPI mode as the previous
                 * channel opened.
                 *
                 * 2. if an instance has already had a channle opened in SPI_SLAVE mode,
                 * no new channel can be opened.
                 */
                ret_flag = 1u;
                mcHandle = NULL;
            }

            if((SPI_MASTER == params->mode) && (ret_flag == 0))
            {
                if (object->chOpenedCnt == 0)
                {
                    /*
                     * first channel opened in this instance,
                     * enable SPI Master
                     */
                    McSPIMasterModeEnable(hwAttrs->baseAddr);
                    object->mode = SPI_MASTER;
                }

                /* Configure the peripheral as single channel SPI Master */
                McSPIMasterModeConfig(hwAttrs->baseAddr,
                                      hwAttrs->chMode,
                                      chnCfg->trMode,
                                      chnCfg->dataLineCommMode,
                                      chNum);
                /* Clock configuration */
                McSPIClkConfig(hwAttrs->baseAddr,
                               hwAttrs->inputClkFreq,
                               params->bitRate,
                               chNum,
                               chObj->clockMode);

                /* configure initial SPI delay only in signle master mode */
                if ((hwAttrs->chMode == MCSPI_SINGLE_CH) && (object->chOpenedCnt == 0))
                {
                    McSPIInitDelayConfig(hwAttrs->baseAddr, hwAttrs->initDelay);
                }
            }
            else
            {
                if (object->chOpenedCnt == 0)
                {
                    /*
                     * first channel opened in this instance,
                     * enable SPI Slave
                     */
                    McSPISlaveModeEnable(hwAttrs->baseAddr);
                    object->mode = SPI_SLAVE;
                }

#ifdef SPI_DMA_ENABLE
                if (hwAttrs->dmaMode == true)
                {
                    MCSPIMultiChModeEnable(hwAttrs->baseAddr);
                }
                else
#endif
                {
                    MCSPISingleChModeEnable(hwAttrs->baseAddr);
                }

                /* Clock configuration */
                McSPIClkConfig(hwAttrs->baseAddr,
                               hwAttrs->inputClkFreq,
                               params->bitRate,
                               chNum,
                               chObj->clockMode);
            }

            /* Set word length for corresponding channel */
            McSPIWordLengthSet(hwAttrs->baseAddr,
                               MCSPI_WORD_LENGTH(params->dataSize),
                               chNum);

            /* Set polarity of SPIEN to low.*/
            McSPICSPolarityConfig(hwAttrs->baseAddr,
                                  chnCfg->csPolarity,
                                  chNum);
        }

        if(SPI_MASTER == params->mode)
        {
            /* Configure chip-select time control */
            McSPICSTimeControlSet(hwAttrs->baseAddr, chnCfg->tcs, chNum);
        }

        /* increment the channel opened count */
        object->chOpenedCnt++;
    }

    return (mcHandle);
}

/*
 *  ======== MCSPI_transfer_v1 ========
 *  @pre    Function assumes that mcHandle and transaction is not NULL
 */
static bool MCSPI_transfer_v1(MCSPI_Handle mcHandle, SPI_Transaction *transaction)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
#ifdef SPI_DMA_ENABLE
    SPI_v1_HWAttrs const *hwAttrs;
#endif
    uint32_t              chNum;
    bool                  ret_val = false;

    /* Input parameter validation */
    OSAL_Assert(!((mcHandle != NULL) && (transaction != NULL)));

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
#ifdef SPI_DMA_ENABLE
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
#endif

    /* Acquire the lock for the SPI transfer on this instance */
    SPI_osalPendLock(object->mutex, SemaphoreP_WAIT_FOREVER);

    if (transaction->count != 0)
    {
        transaction->status = SPI_TRANSFER_STARTED;

        /* Check if a transfer is in progress */
        if (chObj->transaction)
        {
            transaction->status = SPI_TRANSFER_CANCELED;

            /* transfer is in progress */
            ret_val = (bool)false;
        }
        else
        {
            /* Save the pointer to the transaction */
            chObj->transaction = transaction;

#ifdef SPI_DMA_ENABLE
            if (hwAttrs->dmaMode == true)
            {
                MCSPI_DMA_primeTransfer_v1(mcHandle, transaction);
            }
            else
#endif
            {
                MCSPI_primeTransfer_v1(mcHandle, transaction);
            }

            if (chObj->operMode == SPI_OPER_MODE_BLOCKING)
            {
                SPI_osalPendLock(object->transferComplete, SemaphoreP_WAIT_FOREVER);
            }

            transaction->status = SPI_TRANSFER_COMPLETED;

            ret_val = (bool)true;
        }
    }
    else
    {
        transaction->status=SPI_TRANSFER_CANCELED;
    }

    /* Release the lock for the SPI transfer on this instance */
    SPI_osalPostLock(object->mutex);

    return (ret_val);
}

/*
 *  ======== MCSPI_transferCallback_v1 ========
 *  Callback function for when the SPI is in SPI_MODE_BLOCKING
 *
 *  @pre    Function assumes that the handle is not NULL
 */
static void MCSPI_transferCallback_v1(MCSPI_Handle     mcHandle,
                                      SPI_Transaction *transaction)
{
    SPI_Handle         handle;
    SPI_v1_Object     *object;
    SPI_v1_chObject   *chObj;
    uint32_t           chNum;

    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the channel object */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);

    if (chObj->operMode == SPI_OPER_MODE_CALLBACK)
    {
        if (object->transferCallbackFxn != NULL)
        {
            /* Single channel mode callback */
            object->transferCallbackFxn(handle, transaction);
        }
        else
        {
            /* multi channel mode callback */
            chObj->spiParams.transferCallbackFxn(mcHandle, transaction);
        }
    }
    else
    {
        SPI_osalPostLock(object->transferComplete);
    }

    chObj->transaction = NULL;
}

/*
 *  ======== MCSPI_transferCancel_v1 ========
 */
static void MCSPI_transferCancel_v1(MCSPI_Handle mcHndle)
{
    /* Input parameter validation */
    OSAL_Assert(mcHndle == NULL);

    return;
}

/*
 *  ======== MCSPI_control_v1 ========
 */
static int32_t MCSPI_control_v1(MCSPI_Handle mcHandle, uint32_t cmd, const void *arg)
{
    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    return SPI_STATUS_UNDEFINEDCMD;
}

#ifdef SPI_DMA_ENABLE
/*
 *  ======== MCSPI_configDMA_v1 ========
 */
static uint32_t MCSPI_configDMA_v1(MCSPI_Handle mcHandle)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_HWAttrs const *hwAttrs;
    uint32_t              tcc = 0;
    uint32_t              result = EDMA3_DRV_SOK;
    static uint32_t       dummyParamSetAddr = 0;
    EDMA3_RM_EventQueue   queueNum    = 0;
    int32_t               status;
    uint32_t              count;
    uint32_t              edmaChanNum = 0;
    uint32_t              reqTcc = EDMA3_DRV_TCC_ANY;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    object  = (SPI_v1_Object*)handle->object;
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;

    /* RX Section Configuration */

    /* For receive use EDMA Queue 1 and for transmit use EDMA queue 0 */
    queueNum = (EDMA3_RM_EventQueue) MCSPI_RXEVENTQUE;

    status = EDMA3_DRV_requestChannel((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                      (uint32_t *)&(hwAttrs->rxDmaEventNumber),
                                      (uint32_t *)&(hwAttrs->rxDmaEventNumber),
                                      queueNum,
                                      &MCSPI_DMA_rxIsrHandler_v1,
                                      (void *)mcHandle);

    /* Acquire the  PaRAM entries used for EDMA transfers linking */
    for (count = 0; count < MCSPI_MAXLINKCNT; count++)
    {
        /* For requesting for a PaRam set */
        edmaChanNum = EDMA3_DRV_LINK_CHANNEL;
        reqTcc      = EDMA3_DRV_TCC_ANY;

        status = EDMA3_DRV_requestChannel((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                                          &edmaChanNum,
                                          &reqTcc,
                                          queueNum,
                                          &MCSPI_DMA_rxIsrHandler_v1,
                                          NULL);

        if (EDMA3_DRV_SOK == status)
        {
            status = EDMA3_DRV_getPaRAMPhyAddr(
                (EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                edmaChanNum,
                &object->edmaLinkChPhyAddrRx[count]);
        }
    }

    /* TX Section Configuration */

    queueNum = (EDMA3_RM_EventQueue)(0U);

    status = EDMA3_DRV_requestChannel((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                                      (uint32_t *)&(hwAttrs->txDmaEventNumber),
                                      (uint32_t *)&(hwAttrs->txDmaEventNumber),
                                      queueNum,
                                      &MCSPI_DMA_txIsrHandler_v1,
                                      (void *)mcHandle);

    /* Acquire the  PaRAM entries used for EDMA transfers linking */
    tcc    = EDMA3_DRV_LINK_CHANNEL;
    reqTcc = EDMA3_DRV_TCC_ANY;

    status = EDMA3_DRV_requestChannel((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                                      &tcc,
                                      &reqTcc,
                                      queueNum,
                                      &MCSPI_DMA_doNothing_v1,
                                      NULL);

    status = EDMA3_DRV_getPaRAMPhyAddr((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                                       tcc,
                                       &dummyParamSetAddr);


    /* Acquire the  PaRAM entries used for EDMA transfers linking */
    for (count = 0; count < MCSPI_MAXLINKCNT; count++)
    {
        /* For requesting for a PaRam set */
        edmaChanNum = EDMA3_DRV_LINK_CHANNEL;
        reqTcc      = EDMA3_DRV_TCC_ANY;

        status = EDMA3_DRV_requestChannel(
                    (EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                    &edmaChanNum,
                    &reqTcc,
                    queueNum,
                    &MCSPI_DMA_txIsrHandler_v1,
                    NULL);

        status = EDMA3_DRV_getPaRAMPhyAddr(
                    (EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                    edmaChanNum,
                    &object->edmaLinkChPhyAddrTx[count]);
    }

    return result;
}

/*
 *  ======== MCSPI_DMA_rxIsrHandler_v1 ========
 */
static void MCSPI_DMA_rxIsrHandler_v1(uint32_t           tcc,
                                      EDMA3_RM_TccStatus status,
                                      void*              appData)
{
    MCSPI_Handle          mcHandle;
    uint32_t              chNum;
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_ChnCfg const  *chnCfg;
    EDMA3_DRV_PaRAMRegs   paramSet = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    uint32_t              bytesRemain;

    /* Get the pointer to the object and hwAttrs */
    mcHandle = (MCSPI_Handle)appData;
    chNum    = mcHandle->chnNum;
    handle   = mcHandle->handle;
    object   = (SPI_v1_Object*)handle->object;
    chObj    = &(object->chObject[chNum]);
    hwAttrs  = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chnCfg   = &(hwAttrs->chnCfg[chNum]);

    if (EDMA3_RM_XFER_COMPLETE != status)
    {
        /*Error condition  */
        /* Ensure to clear the error bits of EDMA channel */
        EDMA3_DRV_clearErrorBits((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                                 hwAttrs->rxDmaEventNumber);

        /* Get the PaRAM set for default parameters */
        EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                            hwAttrs->rxDmaEventNumber,
                            &paramSet);

        /* calculate the amount of bytes remaining */
        bytesRemain = (paramSet.aCnt * paramSet.bCnt * paramSet.cCnt);

        /* Update the transaction status and word count transfered */
        chObj->transaction->status = SPI_TRANSFER_FAILED;
        chObj->transaction->count -= (bytesRemain >> chObj->wordLenShift);
    }

    /* EDMA Rx is done disable the Rx DMA channel */
    EDMA3_DRV_disableTransfer(hwAttrs->edmaHandle,
                              tcc,
                              EDMA3_DRV_TRIG_MODE_EVENT);

    /* clear the error bits in the EDMA(as this is the last packet)   */
    EDMA3_DRV_clearErrorBits((EDMA3_DRV_Handle)(hwAttrs->edmaHandle),
                             (uint32_t)tcc);

    if ((MCSPI_TX_EDMA_CALLBACK_OCCURED == object->edmaCbCheck) ||
        (chnCfg->trMode == MCSPI_RX_ONLY_MODE))
    {
        /* Now Both Tx and Rx EDMA Callbacks have happened */
        uint32_t intStatus = 0x00;

        object->edmaCbCheck = 0x0;

        intStatus = McSPIChannelStatusGet((uint32_t)hwAttrs->baseAddr,
                                          chNum);

        /* Clear error status */
        MCSPI_DMA_UpdateError_v1(mcHandle, intStatus);

        /* Call the completion function */
        MCSPI_DMA_localCompleteIOedmaCbk_v1(mcHandle);
    }
    else
    {
        if (EDMA3_RM_XFER_COMPLETE == status)
        {
            object->edmaCbCheck = MCSPI_RX_EDMA_CALLBACK_OCCURED;
        }
    }
}

/*
 *  ======== MCSPI_DMA_txIsrHandler_v1 ========
 */
static void MCSPI_DMA_txIsrHandler_v1(uint32_t           tcc,
                                      EDMA3_RM_TccStatus status,
                                      void*              appData)
{
    MCSPI_Handle          mcHandle;
    uint32_t              chNum;
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_ChnCfg const  *chnCfg;
    EDMA3_DRV_PaRAMRegs   paramSet = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    uint32_t              bytesRemain;

    /* Get the pointer to the object and hwAttrs */
    mcHandle = (MCSPI_Handle)appData;
    chNum    = mcHandle->chnNum;
    handle   = mcHandle->handle;
    object   = (SPI_v1_Object*)handle->object;
    chObj    = &(object->chObject[chNum]);
    hwAttrs  = (SPI_v1_HWAttrs*)handle->hwAttrs;
    chnCfg   = &(hwAttrs->chnCfg[chNum]);

    if (EDMA3_RM_XFER_COMPLETE != status)
    {
        /* Ensure to clear the error bits of EDMA channel */
        EDMA3_DRV_clearErrorBits((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                                 hwAttrs->txDmaEventNumber);

        /* Get the PaRAM set for default parameters */
        EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                            hwAttrs->txDmaEventNumber,
                            &paramSet);

        /* calculate the amount of bytes remaining */
        bytesRemain = (paramSet.aCnt * paramSet.bCnt * paramSet.cCnt);

        /* Update the transaction status and word count transfered */
        chObj->transaction->status = SPI_TRANSFER_FAILED;
        chObj->transaction->count -= (bytesRemain >> chObj->wordLenShift);
    }

    /* EDMA Rx is done disable the Rx DMA channel */
    EDMA3_DRV_disableTransfer(hwAttrs->edmaHandle,
                              tcc,
                              EDMA3_DRV_TRIG_MODE_EVENT);

    /* clear the error bits in the EDMA */
    EDMA3_DRV_clearErrorBits((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                             (uint32_t) tcc);

    if ((MCSPI_RX_EDMA_CALLBACK_OCCURED == object->edmaCbCheck) ||
        (chnCfg->trMode == MCSPI_TX_ONLY_MODE))
    {
        uint32_t intStatus = 0x00;

        object->edmaCbCheck = 0x0;

        intStatus = McSPIChannelStatusGet((uint32_t) hwAttrs->baseAddr,
                                          chNum);

        /* Clear error status */
        MCSPI_DMA_UpdateError_v1(mcHandle, intStatus);

        /* Call the completion function */
        MCSPI_DMA_localCompleteIOedmaCbk_v1(mcHandle);
    }
    else
    {
        if (EDMA3_RM_XFER_COMPLETE == status)
        {
            object->edmaCbCheck = MCSPI_TX_EDMA_CALLBACK_OCCURED;
        }
    }
}

/*
 *  ======== MCSPI_DMA_doNothing_v1 ========
 */
static void MCSPI_DMA_doNothing_v1(uint32_t           tcc,
                                   EDMA3_RM_TccStatus edmaStatus,
                                   void*              appData)
{
    /* No implementation required */
}

/*
 *  ======== MCSPI_DMA_localEdmaUpdateParams_v1 ========
 *  Function to update Rx and Tx param set values.
 */
static void MCSPI_DMA_localEdmaUpdateParams_v1(MCSPI_Handle           mcHandle,
                                               const SPI_Transaction *transaction,
                                               EDMA3_DRV_PaRAMRegs   *rxParamSet,
                                               EDMA3_DRV_PaRAMRegs   *txParamSet)
{
    SPI_Handle            handle;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;
    object  = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);

    /* Receive param set configuration */

    /* Source address */
    rxParamSet->srcAddr =  hwAttrs->baseAddr + MCSPI_CHRX(chNum);

    /* destAddr is address of memory location named buffer.*/
    rxParamSet->destAddr = (uint32_t) transaction->rxBuf;

    if (chObj->readCountIdx != 0)
    {
        /* aCnt holds the number of bytes in an array.*/
        rxParamSet->aCnt = (uint16_t)(1 << chObj->wordLenShift);

        /* bCnt holds the number of such arrays to be transferred.*/
        rxParamSet->bCnt = (uint16_t)(object->rxTrigLvl >> chObj->wordLenShift);

        /* cCnt holds the number of frames of aCnt*bBcnt bytes to be transferred.*/
        rxParamSet->cCnt = (uint16_t)(transaction->count / rxParamSet->bCnt);

    }
    else
    {
        /* No transfer on RX channel */
        rxParamSet->aCnt = (uint16_t)0;
        rxParamSet->bCnt = (uint16_t)0;
        rxParamSet->cCnt = (uint16_t)0;

    }

    /* The srcBidx should not be incremented since it is a h/w register.*/
    rxParamSet->srcBIdx = 0;

    if(NULL != transaction->rxBuf)
    {
        /* The destBidx should be incremented for every byte.*/
        rxParamSet->destBIdx = rxParamSet->aCnt;
    }
    else
    {
        /* The destBidx should not be incremented.*/
        rxParamSet->destBIdx = 0;
    }

    /* A sync Transfer Mode. */
    /* srCIdx and destCIdx set to zero since ASYNC Mode is used.*/
    rxParamSet->srcCIdx = 0;
    rxParamSet->destCIdx = object->rxTrigLvl;

    /* Linking transfers in EDMA3 are not used.*/
    rxParamSet->linkAddr = 0xFFFF;

    rxParamSet->bCntReload = rxParamSet->bCnt;

    /* By default SAM and DAM bits are set to INCR mode                  */
    rxParamSet->opt &= 0xFFFFFFFCU;

    if (NULL == transaction->rxBuf)
    {
        /* Since the EDMA is using the driver internal buffer, so no need to
         * increment the buffer. INCR mode is disabled
         */
        rxParamSet->opt |= MCSPI_EDMA3CC_OPT_DAM_CONST_MODE;
    }

    /* FIFO width is 8 bit                                               */
    rxParamSet->opt &= 0xFFFFF8FFU;

    /* EDMA3_DRV_SYNC_AB                                                 */
    rxParamSet->opt &= MCSPI_EDMA3CC_OPT_SYNC_MASK_VALUE;
    rxParamSet->opt |= MCSPI_EDMA3CC_OPT_SYNC_AB;

    /* EDMA3_DRV_OPT_FIELD_TCINTEN                                       */
    rxParamSet->opt |= 1U << MCSPI_OPT_TCINTEN_SHIFT;

    /* update the transfer completion code                               */
    rxParamSet->opt &= (~MCSPI_OPT_TCC_MASK);
    rxParamSet->opt |= ((hwAttrs->rxDmaEventNumber) <<
                        MCSPI_OPT_TCC_SHIFT);

    /* Transmit param set configuration */

    /* srcAddr holds address of memory location buffer. */
    txParamSet->srcAddr = (uint32_t) transaction->txBuf;

    /* destAddr holds address of McSPI_TX register. */
    txParamSet->destAddr = (uint32_t) hwAttrs->baseAddr +
                           MCSPI_CHTX(chNum);

    if (chObj->writeCountIdx != 0)
    {
        /* aCnt holds the number of bytes in an array. */
        txParamSet->aCnt = (uint16_t)(1 << chObj->wordLenShift);

        /* bCnt holds the number of such arrays to be transferred. */
        txParamSet->bCnt = (uint16_t)(object->txTrigLvl >> chObj->wordLenShift);

        /* cCnt holds the number of frames of aCnt*bBcnt bytes to be transferred. */
        txParamSet->cCnt = (uint16_t)(transaction->count / txParamSet->bCnt);
    }
    else
    {
        /* No transfer on RX channel */
        txParamSet->aCnt = (uint16_t)0;
        txParamSet->bCnt = (uint16_t)0;
        txParamSet->cCnt = (uint16_t)0;
    }

    /*
    ** The srcBidx should be incremented by aCnt number of bytes since the
    ** source used here is memory.
    */
    txParamSet->srcBIdx = txParamSet->aCnt;
    txParamSet->destBIdx = 0;

    /* Async Transfer Mode is set in OPT.*/
    /* srCIdx and destCIdx set to zero since ASYNC Mode is used. */
    txParamSet->srcCIdx = object->txTrigLvl;
    txParamSet->destCIdx = 0;

    /* Linking transfers in EDMA3 are not used. */
    txParamSet->linkAddr = MCSPI_EDMA3CC_PARAM_LINK_ADDRESS;
    txParamSet->bCntReload = txParamSet->bCnt;

    /* By default SAM and DAM bits are set to INCR mode                  */
    txParamSet->opt &= MCSPI_EDMA3CC_OPT_SAM_DAM_INCR_MODE;

    if (NULL == transaction->txBuf)
    {
        /* Since the EDMA is using the driver internal buffer, so no need to
         * increment the buffer. INCR mode is disabled.
         */
        txParamSet->opt |= MCSPI_EDMA3CC_OPT_SAM_CONST_MODE;
    }

    /* FIFO width is 8 bit                                               */
    txParamSet->opt &= MCSPI_EDMA3CC_OPT_FIFO_WIDTH;

    /* EDMA3_DRV_SYNC_AB                                                 */
    txParamSet->opt &= MCSPI_EDMA3CC_OPT_SYNC_MASK_VALUE;
    txParamSet->opt |= MCSPI_EDMA3CC_OPT_SYNC_AB;

    /* EDMA3_DRV_OPT_FIELD_TCINTEN                                       */
    txParamSet->opt |= (1U << MCSPI_OPT_TCINTEN_SHIFT);

    /* update the transfer completion code                               */
    txParamSet->opt &= (~MCSPI_OPT_TCC_MASK);
    txParamSet->opt |= ((hwAttrs->txDmaEventNumber) <<
                        MCSPI_OPT_TCC_SHIFT);
}


/*
 *  ======== MCSPI_DMA_localCompleteIOedmaCbk_v1 ========
 */
static void MCSPI_DMA_localCompleteIOedmaCbk_v1(MCSPI_Handle mcHandle)
{
    SPI_Handle            handle;
    SPI_v1_Object        *object;
    SPI_v1_chObject      *chObj;
    SPI_v1_HWAttrs const *hwAttrs;
    uint32_t              chNum;

    /* Get the pointer to the object and hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    object  = (SPI_v1_Object*)handle->object;
    chObj   = &(object->chObject[chNum]);
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;

    /* disable all the spi transfers */
    McSPIChannelDisable((uint32_t) hwAttrs->baseAddr, chNum);

    /* Clear all the interrupts here, only related to the perticular
     * channel
     */
    MCSPI_DMA_ChIntrStatClear_v1(mcHandle);

    if ((SPI_MASTER == chObj->spiParams.mode) &&
        (hwAttrs->chMode == MCSPI_SINGLE_CH))
    {
        /* Deassert chip select signal */
        McSPICSDeAssert(hwAttrs->baseAddr, chNum);
    }

    EDMA3_DRV_disableTransfer(hwAttrs->edmaHandle,
                              hwAttrs->txDmaEventNumber,
                              EDMA3_DRV_TRIG_MODE_EVENT);

    EDMA3_DRV_disableTransfer(hwAttrs->edmaHandle,
                              hwAttrs->rxDmaEventNumber,
                              EDMA3_DRV_TRIG_MODE_EVENT);

    /* Call the transfer completion callback function */
    chObj->transaction->status = SPI_TRANSFER_COMPLETED;
    MCSPI_transferCallback_v1(mcHandle, chObj->transaction);
}

/**
 *  \brief  function is used clear the interrupt
 */
static void MCSPI_DMA_ChIntrStatClear_v1(MCSPI_Handle mcHandle)
{
    SPI_Handle            handle;
    SPI_v1_HWAttrs const *hwAttrs;
    uint32_t              chNum;
    uint32_t              clrIntr = (uint32_t) 0;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;

    switch (chNum)
    {
        case MCSPI_CHANNEL_0:
                /* clear all channel '0' interrupts */
                clrIntr = (uint32_t) 0;

                clrIntr = (CSL_MCSPI_IRQSTATUS_TX0_EMPTY_MASK |
                               CSL_MCSPI_IRQSTATUS_TX0_UNDERFLOW_MASK |
                               CSL_MCSPI_IRQSTATUS_RX0_FULL_MASK);

                McSPIIntStatusClear((uint32_t) hwAttrs->baseAddr, clrIntr);

            break;

        case MCSPI_CHANNEL_1:
                /* Enable all channel '1' interrupts */
                clrIntr = (uint32_t) 0;
                clrIntr = (CSL_MCSPI_IRQSTATUS_TX1_EMPTY_MASK |
                           CSL_MCSPI_IRQSTATUS_TX1_UNDERFLOW_MASK |
                           CSL_MCSPI_IRQSTATUS_RX1_FULL_MASK);

                McSPIIntStatusClear((uint32_t) hwAttrs->baseAddr, clrIntr);

            break;

        case MCSPI_CHANNEL_2:
                /* Enable all channel '2' interrupts */
                clrIntr = (uint32_t) 0;
                clrIntr = (CSL_MCSPI_IRQSTATUS_TX2_EMPTY_MASK |
                           CSL_MCSPI_IRQSTATUS_TX2_UNDERFLOW_MASK |
                           CSL_MCSPI_IRQSTATUS_RX2_FULL_MASK);

                McSPIIntStatusClear((uint32_t) hwAttrs->baseAddr, clrIntr);

            break;

        case MCSPI_CHANNEL_3:
                /* Enable all channel '3' interrupts */
                clrIntr = (uint32_t) 0;
                clrIntr = (CSL_MCSPI_IRQSTATUS_TX3_EMPTY_MASK |
                           CSL_MCSPI_IRQSTATUS_TX3_UNDERFLOW_MASK |
                           CSL_MCSPI_IRQSTATUS_RX3_FULL_MASK);

                McSPIIntStatusClear((uint32_t) hwAttrs->baseAddr, clrIntr);

            break;

        default:
            break;
    }
}

/**
 *  \brief  function is used clear the interrupt
 */
static void MCSPI_DMA_UpdateError_v1(MCSPI_Handle mcHandle, uint32_t intStatus)
{
    SPI_Handle            handle;
    uint32_t              chNum;
    SPI_v1_HWAttrs const *hwAttrs;

    /* Input parameter validation */
    OSAL_Assert(mcHandle == NULL);

    /* Get the pointer to the hwAttrs */
    handle  = mcHandle->handle;
    chNum   = mcHandle->chnNum;
    hwAttrs = (SPI_v1_HWAttrs*)handle->hwAttrs;

    /* check Timeout interrupt                                           */
    switch (chNum)
    {
        case MCSPI_CHANNEL_0:

            if ((uint32_t) CSL_MCSPI_IRQSTATUS_TX0_UNDERFLOW_MASK ==
                (intStatus & (uint32_t) CSL_MCSPI_IRQSTATUS_TX0_UNDERFLOW_MASK))
            {
                McSPIIntStatusClear(
                    (uint32_t) hwAttrs->baseAddr,
                    (uint32_t) CSL_MCSPI_IRQSTATUS_TX0_UNDERFLOW_MASK);
            }
            break;

        case MCSPI_CHANNEL_1:

            if ((uint32_t) CSL_MCSPI_IRQSTATUS_TX1_UNDERFLOW_MASK ==
                (intStatus & (uint32_t) CSL_MCSPI_IRQSTATUS_TX1_UNDERFLOW_MASK))
            {
                McSPIIntStatusClear(
                    (uint32_t) hwAttrs->baseAddr,
                    (uint32_t) CSL_MCSPI_IRQSTATUS_TX1_UNDERFLOW_MASK);
            }
            break;

        case MCSPI_CHANNEL_2:
            if ((uint32_t) CSL_MCSPI_IRQSTATUS_TX2_UNDERFLOW_MASK ==
                (intStatus & (uint32_t) CSL_MCSPI_IRQSTATUS_TX2_UNDERFLOW_MASK))
            {
                McSPIIntStatusClear(
                    (uint32_t) hwAttrs->baseAddr,
                    (uint32_t) CSL_MCSPI_IRQSTATUS_TX2_UNDERFLOW_MASK);
            }
            break;

        case MCSPI_CHANNEL_3:

            if ((uint32_t) CSL_MCSPI_IRQSTATUS_TX3_UNDERFLOW_MASK ==
                (intStatus & (uint32_t) CSL_MCSPI_IRQSTATUS_TX3_UNDERFLOW_MASK))
            {
                McSPIIntStatusClear(
                    (uint32_t) hwAttrs->baseAddr,
                    (uint32_t) CSL_MCSPI_IRQSTATUS_TX3_UNDERFLOW_MASK);
            }
            break;

        default:
            break;
    }
}
#endif
