/********************************************************************************************
 * FILE PURPOSE: A very small boot test program
 ********************************************************************************************
 * FILE NAME: testsmall.c
 *
 * DESCRIPTION: A very simply GEM + ARM boot test program
 *
 * Required compile definitions:
 *      NGEMS   - number of gem cores
 *      NARMS   - number of arm cores
 *      MAXGEMS - max number of gems supported (also 1st arm device number)
 *      MAXARMS - max number of arms supported
 *
 * Values from the linker command file:
 *      c_int00
 ********************************************************************************************/
#ifdef _TMS320C6X
#include "tiboot.h"
 #define FAR far
#else
 #define FAR
#endif

#pragma DATA_SECTION (results, ".results")
unsigned int results;

extern FAR unsigned int c_int00;

#pragma DATA_SECTION(arm_entry, ".arm_entry")
unsigned int arm_entry;


/* The linker command file maps this to the ipc registers */
#pragma DATA_SECTION (ipcgr, ".ipcgr")
volatile unsigned int ipcgr[MAXGEMS + MAXARMS];

#pragma DATA_SECTION (ipcar, ".ipcar")
volatile unsigned int ipcar[MAXGEMS + MAXARMS];


/* The gem boot magic addresses. This is a local address */
#pragma DATA_SECTION (gemBootMagic, ".gem_boot_magic")
unsigned int gemBootMagic;

/* The arm boot magic addresses. */
#pragma DATA_SECTION (armBootMagic, ".arm_boot_magic")
unsigned int armBootMagic[MAXARMS];

#ifdef _TMS320C6X
 extern volatile cregister unsigned int DNUM;
#else
 unsigned int armNum (void);
#endif

#if 1
#ifdef _TMS320C6X
/* ------------------------------------------------------------------------
 * DDR3 configuration. This const section must be loaded first so that 
 *    the code loaded to DDR3 will work
 * ------------------------------------------------------------------------ */
#pragma  DATA_SECTION (emif4Cfg, ".emif4Cfg")
const BOOT_EMIF4V_TBL_T  emif4Cfg =  {

    /* Config select msw */
    BOOT_EMIF4V_ENABLE_MSW_pllCtl          | \
    BOOT_EMIF4V_ENABLE_MSW_sdRamTiming1    | \
    BOOT_EMIF4V_ENABLE_MSW_sdRamTiming2    | \
    BOOT_EMIF4V_ENABLE_MSW_sdRamTiming3    | \
    BOOT_EMIF4V_ENABLE_MSW_sdRamTiming4    | \
    BOOT_EMIF4V_ENABLE_MSW_sdRamRefreshCtl | \
    BOOT_EMIF4V_ENABLE_MSW_zqConfig        | \
    BOOT_EMIF4V_ENABLE_MSW_sdRamConfig,

    /* Config select slsw */
    (1 << BOOT_EMIF4V_DWCPUB_PTR0)         | \
    (1 << BOOT_EMIF4V_DWCPUB_PTR1)         | \
    (1 << BOOT_EMIF4V_DWCPUB_PTR3)         | \
    (1 << BOOT_EMIF4V_DWCPUB_PTR4)         | \
    (1 << BOOT_EMIF4V_DWCPUB_MR0)          | \
    (1 << BOOT_EMIF4V_DWCPUB_MR1)          | \
    (1 << BOOT_EMIF4V_DWCPUB_MR2)          | \
    (1 << BOOT_EMIF4V_DWCPUB_DTCR)         | \
    (1 << BOOT_EMIF4V_DWCPUB_PGCR1)        | \
    (1 << BOOT_EMIF4V_DWCPUB_PGCR2)        | \
    (1 << BOOT_EMIF4V_DWCPUB_PIR)          | \
    (1 << BOOT_EMIF4V_DWCPUB_DCR)          | \
    (1 << BOOT_EMIF4V_DWCPUB_DTPR0)        | \
    (1 << BOOT_EMIF4V_DWCPUB_DTPR1)        | \
    (1 << BOOT_EMIF4V_DWCPUB_DTPR2)        | \
    (1 << BOOT_EMIF4V_DWCPUB_PLLCR)        | \
    (1 << BOOT_EMIF4V_DWCPUB_ZQ0CR1)       | \
    (1 << BOOT_EMIF4V_DWCPUB_ZQ1CR1)       | \
    (1 << BOOT_EMIF4V_DWCPUB_ZQ2CR1),
    

    0,      /* Config select lsw  */
    
    1,      /* pllPrediv  */
    20,     /* pllMult    */
    6,      /* pllPostDiv */

    0x62008c6a,  /* sdRamConfig */
    0x00001457,  /* sdRamRefreshCtl  */
    0x125c8044,  /* sdRamTiming1 */
    0x00001d29,  /* sdRamTiming2 */
    0x32cdff43,  /* sdRamTiming3 */
    0x543f0adf,  /* sdRamTiming4 */

    0,           /* powerManageCtl, dont care */
    0,           /* vbusmCfg */
    0,           /* vbusmCfgVal1 */
    0,           /* vbusmCfgVal2 */
    0,           /* iODFTTestLogic, dont care */
    0,           /* performCountCfg, dont care */
    0,           /* performCountMstRegSel, dont care */
    0x70073200,  /* zqConfig */

    0,           /* priClassSvceMap, dont care */
    0,           /* mstId2ClsSvce1Map, dont care */
    0,           /* mstId2ClsSvce2Map, dont care */
    0,           /* eccCtl, dont care */
    0,           /* eccRange1, dont care */
    0,           /* eccRange2, dont care */
    0,           /* rdWrtExcThresh, dont care */

    /* Phy registers */
    0x0000ff81,  /* PIR */
    0,           /* PGC0 */
    0x0280c485,  /* PGCR1 */
    0x00f065b8,  /* PGCR2 */
    0,           /* PGSR0 */
    0,           /* PGSR1 */
    0x42c21590,  /* PTR0 */
    0xd05612c0,  /* PTR1 */
    0,           /* PTR2 */
    0x0b4515c2,  /* PTR3 */
    0x0a6e08b4,  /* PTR4 */
    (1 <<10) | (1 << 27) | (1 << 29) | 0xb, /* DCR */
    0x8558aa55,  /* DTPR0 */
    0x12857280,  /* DTPR1 */
    0x5002c200,  /* DTPR2 */
    0x00001a60,  /* MR0 */
    0x00000006,  /* MR1 */
    0x00000010,  /* MR2 */
    0x730035c7,  /* DTCR */
    0,           /* DX0GCR */
    0,           /* DX1GCR */
    0,           /* DX2GCR */
    0,           /* DX3GCR */
    0,           /* DX4GCR */
    0,           /* DX5GCR */
    0,           /* DX6GCR */
    0,           /* DX7GCR */
    0,           /* DX8GCR */
    0x0005c000,  /* PLLCR  */
    0x0000005d,  /* ZQ0CR1 */
    0x0000005b,  /* ZQ1CR1 */
    0x0000005b,  /* ZQ2CR1 */


    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};
#endif
#endif

#ifdef _TMS320C6X
#pragma CODE_SECTION (ddrFinish, ".ddr_code")
void ddrFinish(void) {
	for(;;);
}
#endif

/* The expected boot complete pattern */
unsigned int exptBootComplete;

/* Determine the device number */
unsigned int devNum (void)
{
  #ifdef _TMS320C6X

    return (DNUM);

  #else

    return (armNum() + MAXGEMS);

  #endif

}

unsigned int *formGemBootMagic (int core)
{
    unsigned int v;

    v = (unsigned int)&gemBootMagic;
    v = v | (1 << 28) | (core << 24);

    return ((unsigned int *)v);

}



void testDone (void)
{
#ifdef _TMS320C6X
    ddrFinish();
#endif
    for (;;);

}
    
unsigned int formArmBootAddr (unsigned int raw)
{
  #ifdef SWAP_ARM_MAGIC
   unsigned int v  =  (((raw >> 24) & 0xff) <<  0)  |
                      (((raw >> 16) & 0xff) <<  8)  |
                      (((raw >>  8) & 0xff) << 16)  |
                      (((raw >>  0) & 0xff) << 24)  ;

    return (v);
  #else
    return (raw);
  #endif

}

#define TETRIS_PSC_STATE_ON 0
int romtTetrisPsc (unsigned int core, unsigned int state);
int romtMonitorFunction (void (*fcn)(), ...);


#ifdef _TMS320C6X
int romtTetrisPsc (unsigned int core, unsigned int state) { return (0); }
int romtMonitorFunction (void (*fcn)(), ...) { return (0); }
#endif

volatile unsigned int groupEnableReadBack;


/* This function called through the monitor function call to execute from a secure state */
void armPowerUp (void (*fcn)(), int core)
{
    romtTetrisPsc (core, TETRIS_PSC_STATE_ON);
}

void main (void)
{
    int i;
    int dnum;

    unsigned int *p;

    dnum = devNum();

    /* The master core initializes the results and wakes the other cores */
    if (dnum == 0)  {

        
        exptBootComplete = ((1 << NGEMS) - 1) |  (((1 << NARMS) - 1) << MAXGEMS);

        results = 0x11111111;

        /* Clear the IPC and wake up all processors */
        for (i = 0; i < MAXGEMS + MAXARMS; i++)
            ipcar[i] = 0xfffffff0;


        /* Write the boot magic address */
        for (i = 1; i < NGEMS; i++)  {
            p = formGemBootMagic (i);
            *p = (unsigned int) &c_int00;
        }

        /* Write the boot address to all arm cores */
        for (i = 0; i < NARMS; i++) 
            armBootMagic[i] = formArmBootAddr ((unsigned int) &arm_entry);


        /* Bit 31 in the IPC is used to tell the sim that the core should be run.
         * This should be done in the interrupt controller, but that isn't working */
        /* And now wake up the cores */
        for (i = 0; i < NGEMS; i++)
            ipcgr[i] = (1 << 31) | 1;

        for (i = 0; i < NARMS; i++)
            ipcgr[i + MAXGEMS] = (1 << 31) | 1;


        /* Poll for each core to indicate it has run */
        for (i = 1; i < NGEMS; i++)
            while ((ipcgr[i] & 0x7fffffff)== 0);


        for (i = 0; i < NARMS; i++)
            while ((ipcgr[i + MAXGEMS] & 0x7fffffff) == 0);

        /* Populate the test result */
        results = 0xaaaaaaaa;

    }  else  {

        /* Arm core 0 must wake up the other cores */
        if (dnum == MAXGEMS)  {

            for (i = 1; i < NARMS; i++)
                romtMonitorFunction (armPowerUp, i);
        }
            

        /* secondary cores just poke a non-zero value into their ipc */
        /* The sim has a problem with byte swapping to these registers. It doesnt seem 
         * to recognize the endiannes of the ARM correctly. As the code is written
         * here that wont matter, but it will look strange if you see the values in
         * the ipcgr registers */
        ipcgr[dnum] = 0x10;

    }


    testDone ();

}




    





