/* Create a boot table format CCS file from an elf file 
 *
 *  usage: btbl470 inputfile [-out outputfile] [-h]
 *
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>


FILE *fin;
FILE *fout;

char *finname  = NULL;
char *foutname = NULL;


int parseit (int ac, char *av[])
{
    int i;

    for (i = 1; i < ac;  /* empty */ )  {

        if (!strcmp (av[i], "-h"))  {
            fprintf (stderr, "usage: %s inputfile [-out outputfile] [-h]\n", av[0]);
            return (-1);
        }

        if (!strcmp (av[i], "-out"))  {
            foutname = av[i+1];
            i = i + 2;
            continue;
        }

        if (finname != NULL)  {
            fprintf (stderr, "%s: unknown arg %s\n", av[0], av[i]);
            return (-1);
        }

        finname = av[i++];

    }


    if (finname == NULL)  {
        fprintf (stderr, "%s: no input file specified\n", av[0]);
        return (-1);
    }

    return (0);

}
            

#define MAX_SECTS   500

unsigned int numSects = 0;

typedef struct mysect_s
{
    unsigned int addr;
    unsigned int sizeBytes;
    unsigned char *data;

} mysect_t;

mysect_t mysect[MAX_SECTS];



int main (int argc, char *argv[])
{
    unsigned int entry;
    int nlines;
    int i, j;

    if (parseit (argc, argv))
        return (-1);


    fin = fopen (finname, "rb");
    if (fin == NULL)  {
        fprintf (stderr, "%s: Failed to open input file %s\n", argv[0], finname);
        return (-1);
    }
    

    DLOAD_load (fin, argc, argv, &entry);

    fclose (fin);


    /* Create the output file, in ccs format */
    if (foutname != NULL)  {
        fout = fopen (foutname, "w");
        if (fout == NULL)  {
            fprintf (stderr, "%s: Failed to open output file %s\n", argv[0], foutname);
            return (-1);
        }
    } else {
        fout = stdout;
    }

    /* 2 lines for each section, one for the entry point, one for the trailing 0 length */
    nlines = (numSects * 2) + 1 + 1;

    for (i = 0; i < numSects; i++)
        nlines += ((mysect[i].sizeBytes + 3) >> 2);


    /* write the header and the entry point */
    fprintf (fout, "1651 1 %08x 1 %x\n", entry, nlines);
    fprintf (fout, "0x%08x\n", entry);

    

    /* Write the section data */
    for (i = 0; i < numSects; i++)  {

        fprintf (fout, "0x%08x\n", mysect[i].sizeBytes);
        fprintf (fout, "0x%08x\n", mysect[i].addr);

        for (j = 0; j < mysect[i].sizeBytes; j += 4)  {
            fprintf (fout, "0x%02x%02x%02x%02x\n", mysect[i].data[j+3], mysect[i].data[j+2],
                                                   mysect[i].data[j+1], mysect[i].data[j+0]);

        }

        free (mysect[i].data);

    }

    /* Write the trailing length of 0 */
    fprintf (fout, "0x00000000\n");

    if (foutname != NULL)
        fclose (fout);


    return (0);
    

}

/* This function is called from the parser when there is data to write */
void callOut (unsigned int dest, unsigned int size, void *data)
{
    if (numSects >= MAX_SECTS)  {
        fprintf (stderr, "Error: Max number of sections exceeded\n");
        exit (-1);
    }

    /* round the size up to a multiple of 4 bytes */
    mysect[numSects].addr      = dest;
    mysect[numSects].sizeBytes = (size + 3) & ~3;
    mysect[numSects].data      = malloc (mysect[numSects].sizeBytes * sizeof (unsigned char));
    memcpy (mysect[numSects].data, data, size);

    numSects += 1;

}
    






