/*
* dload.c
*
* Core Dynamic Loader Reference Implementation
*
* This implementation of the core dynamic loader is platform independent,
* but it is object file format dependent.  In particular, this
* implementation supports ELF object file format.
*
* Copyright (C) 2009 Texas Instruments Incorporated - http://www.ti.com/
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <limits.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ewrap.h"
#include "elf32.h"
#include "dload.h"
#include "dload_api.h"


/*---------------------------------------------------------------------------*/
/* Identify target supported by this implementation of the core loader.      */
/*---------------------------------------------------------------------------*/
int DLOAD_TARGET_MACHINE = DLOAD_DEFAULT_TARGET_MACHINE;

/*---------------------------------------------------------------------------*/
/* Counter for generating unique IDs for file handles.                       */
/*   NOTE: File handle is assigned sequencially but is never reclaimed       */
/*         when the modules are unloaded. It is conceivable that a loader    */
/*         running for a long time and loading and unloading modules could   */
/*         wrap-around. The loader generates error in this case.             */
/*---------------------------------------------------------------------------*/
static int32_t file_handle = 1;

/*---------------------------------------------------------------------------*/
/* Support for profiling performance of dynamic loader core.                 */
/*---------------------------------------------------------------------------*/
#if LOADER_DEBUG
static clock_t cycle0 = 0;
static clock_t cycle_end = 0;
#define profile_start_clock() (cycle0 = clock())
#define profile_stop_clock()  (cycle_end = clock())
#define profile_cycle_count() (cycle_end - cycle0)
#endif


/*****************************************************************************/
/* load_object()                                                             */
/*                                                                           */
/*    Finish the process of loading an object file.                          */
/*                                                                           */
/*****************************************************************************/
static int load_object(LOADER_FILE_DESC *fd, DLIMP_Dynamic_Module *dyn_module)
{
   /*------------------------------------------------------------------------*/
   /* With the dynamic loader already running on the target, we are able to  */
   /* relocate directly into target memory, so there is nothing more to be   */
   /* done (at least in the bare-metal dynamic linking ABI model).           */
   /*------------------------------------------------------------------------*/
   return 1;
}

#if 1
/*****************************************************************************/
/* initialize_loaded_module()                                                */
/*                                                                           */
/*    Initialize DLIMP_Loaded_Module internal data object associated with a  */
/*    dynamic module.  This function will also set up a queue of             */
/*    DLIMP_Loaded_Segment(s) associated with the loaded module.             */
/*    This function is called as we are getting ready to actually load the   */
/*    object file contents into target memory.  Each segment will get a      */
/*    target memory request that it can use to ask the client for target     */
/*    memory space.  This function will also assign a file handle to the     */
/*    loaded module.                                                         */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* In applications that use the DSBT model, this function will also need to  */
/* negotiate the module's DSBT index with the client.                        */
/*                                                                           */
/*****************************************************************************/
static void initialize_loaded_module(DLIMP_Dynamic_Module *dyn_module)
{
   int i;

   /*------------------------------------------------------------------------*/
   /* Allocate a DLIMP_Loaded_Module data structure for the specified ELF    */
   /* file and assign a file handle for it (bumping the file handle counter  */
   /* as we go).                                                             */
   /*------------------------------------------------------------------------*/
   DLIMP_Loaded_Module *loaded_module =
          dyn_module->loaded_module = DLIF_malloc(sizeof(DLIMP_Loaded_Module));

#if LOADER_DEBUG || LOADER_PROFILE
   /*------------------------------------------------------------------------*/
   /* Start clock on initialization of loaded module object.                 */
   /*------------------------------------------------------------------------*/
   if (debugging_on || profiling_on)
   {
      printf("Starting initialize_loaded_module() ...\n");
      if (profiling_on) profile_start_clock();
   }
#endif

   if (dyn_module->name != NULL)  {
     loaded_module->name = DLIF_malloc(strlen(dyn_module->name) + 1);
     strcpy(loaded_module->name, dyn_module->name);
   }

   loaded_module->file_handle = file_handle++;
   loaded_module->direct_dependent_only = dyn_module->direct_dependent_only;
   loaded_module->use_count = 1;

   /*------------------------------------------------------------------------*/
   /* In case we wrapped around the file handle, return error.               */
   /*------------------------------------------------------------------------*/
   if (file_handle == 0)
      DLIF_error(DLET_MISC, "DLOAD File handle overflowed.\n");

   /*------------------------------------------------------------------------*/
   /* Initially the loaded module does not have access to its global         */
   /* symbols.  These need to be copied from the dynamic module (see call    */
   /* to DLSYM_copy_globals() below).                                        */
   /*                                                                        */
   /* THESE INITIALIZATIONS SHOULD BE MOVED TO AN INIT ROUTINE FOR THE       */
   /* LOADED MODULE                                                          */
   /*------------------------------------------------------------------------*/
   loaded_module->gsymtab = NULL;
   loaded_module->gstrtab = NULL;
   loaded_module->gsymnum = loaded_module->gstrsz = 0;

   /*------------------------------------------------------------------------*/
   /* Initialize the Array_List of dependencies.                             */
   /*------------------------------------------------------------------------*/
   AL_initialize(&(loaded_module->dependencies), sizeof(int), 1);

   if (dyn_module->symtab)
      DLSYM_copy_globals(dyn_module);
     
   /*------------------------------------------------------------------------*/
   /* Initialize the module loaded segments Array_List.                      */
   /*------------------------------------------------------------------------*/
   AL_initialize(&(loaded_module->loaded_segments),
                 sizeof(DLIMP_Loaded_Segment), dyn_module->phnum);

   /*------------------------------------------------------------------------*/
   /* Spin thru segment headers and process each load segment encountered.   */
   /*------------------------------------------------------------------------*/
   for (i = 0; i < dyn_module->phnum; i++)
      if (dyn_module->phdr[i].p_type == PT_LOAD)
      {
         /*------------------------------------------------------------------*/
         /* Note that this is parallel to and does not supplant the ELF      */
         /* phdr tables.                                                     */
         /*------------------------------------------------------------------*/
         DLIMP_Loaded_Segment seg;
         seg.obj_desc = DLIF_malloc(sizeof(struct DLOAD_MEMORY_SEGMENT));
         seg.phdr.p_vaddr = dyn_module->phdr[i].p_vaddr;
         seg.phdr.p_offset = dyn_module->phdr[i].p_offset;
         seg.obj_desc->target_page = 0; /*not used*/
         seg.modified = 0;
         seg.phdr.p_filesz = seg.obj_desc->objsz_in_bytes 
	                   = dyn_module->phdr[i].p_filesz;
         seg.phdr.p_memsz = seg.obj_desc->memsz_in_bytes 
	                  = dyn_module->phdr[i].p_memsz;
         seg.phdr.p_align = dyn_module->phdr[i].p_align;
         seg.phdr.p_flags = dyn_module->phdr[i].p_flags;
         seg.reloc_offset = 0;
         AL_append(&(loaded_module->loaded_segments), &seg);
      }

   /*------------------------------------------------------------------------*/
   /* Initialize the DSO termination information for this module.            */
   /* It will be copied over from the enclosing dyn_module object when       */
   /* placement is completed and dyn_module's local copy of the dynamic      */
   /* table is updated.                                                      */
   /*------------------------------------------------------------------------*/
   loaded_module->fini_array = NULL;
   loaded_module->fini_arraysz = 0;
   loaded_module->fini = NULL;

#if LOADER_DEBUG || LOADER_PROFILE
   if (debugging_on || profiling_on)
   {
      printf("Finished initialize_loaded_module()\n");
      if (profiling_on)
      {
         profile_stop_clock();
         printf("Took %d cycles.\n", profile_cycle_count());
      }
   }
#endif

}
#endif

/*****************************************************************************/
/* load_static_segment()                                                     */
/*                                                                           */
/*    The core dynamic loader requires that a statically linked executable   */
/*    be placed in target memory at the location that was determined during  */
/*    the static link that created the executable.  Failure to get the       */
/*    required target memory where the static executable is to be loaded     */
/*    will cause the dynamic loader to emit an error and abort the load.     */
/*                                                                           */
/*****************************************************************************/
static BOOL load_static_segment(LOADER_FILE_DESC *fd, 
                                DLIMP_Dynamic_Module *dyn_module)
{
   int i;
   DLIMP_Loaded_Segment* seg = (DLIMP_Loaded_Segment*) 
                              (dyn_module->loaded_module->loaded_segments.buf);

   /*------------------------------------------------------------------------*/
   /* For each segment in the loaded module, build up a target memory        */
   /* request for the segment, get rights to target memory where we want     */
   /* to load the segment from the client, then get the client to write the  */
   /* segment contents out to target memory to the appropriate address.      */
   /*------------------------------------------------------------------------*/
   for (i = 0; i < dyn_module->loaded_module->loaded_segments.size; i++)
   {
      struct DLOAD_MEMORY_REQUEST targ_req;
      seg[i].obj_desc->target_page = 0;
      targ_req.flags = 0;

      /*---------------------------------------------------------------------*/
      /* This is a static executable.  DLIF_allocate should give us the      */
      /* address we ask for or fail.                                         */
      /*---------------------------------------------------------------------*/
      if (seg[i].phdr.p_flags & PF_X) targ_req.flags |= DLOAD_SF_executable;

      targ_req.align = seg[i].phdr.p_align;
      seg[i].obj_desc->target_address = (TARGET_ADDRESS)seg[i].phdr.p_vaddr;
      targ_req.flags &= ~DLOAD_SF_relocatable;
      targ_req.fp = fd;
      targ_req.segment = seg[i].obj_desc;
      targ_req.offset = seg[i].phdr.p_offset;
      targ_req.flip_endian = dyn_module->wrong_endian;

      /*---------------------------------------------------------------------*/
      /* Ask the client side of the dynamic loader to allocate target memory */
      /* for this segment to be loaded into.                                 */
      /*---------------------------------------------------------------------*/
      if (!DLIF_allocate(&targ_req)) return FALSE;

      /*---------------------------------------------------------------------*/
      /* If there is any initialized data in the segment, we'll first write  */
      /* it into a host writable buffer (DLIF_copy()) and then flush it to   */
      /* target memory.                                                      */
      /*---------------------------------------------------------------------*/
      if (seg[i].phdr.p_filesz)
      {
         DLIF_copy(&targ_req);
         DLIF_write(&targ_req);
      }
   }

   return TRUE;
}



#if 1
/*****************************************************************************/
/* delete_DLIMP_Loaded_Module()                                              */
/*                                                                           */
/*    Free host memory associated with a DLIMP_Loaded_Module data structure  */
/*    and all of the DLIMP_Loaded_Segment objects that are associated with   */
/*    it.                                                                    */
/*                                                                           */
/*****************************************************************************/
static void delete_DLIMP_Loaded_Module(DLIMP_Loaded_Module **pplm)
{
    DLIMP_Loaded_Module *loaded_module = *pplm;
    DLIMP_Loaded_Segment *segments = (DLIMP_Loaded_Segment*)
                                          (loaded_module->loaded_segments.buf);

    /*-----------------------------------------------------------------------*/
    /* Spin through the segments attached to this loaded module, freeing up  */
    /* any target memory that was allocated by the client for the segment.   */
    /*-----------------------------------------------------------------------*/
    int i;
    for (i = 0; i < loaded_module->loaded_segments.size; i++)
    {
        if (!DLIF_release(segments[i].obj_desc))
            DLIF_error(DLET_MISC, "Failed call to DLIF_release!\n");;
        DLIF_free(segments[i].obj_desc);
    }

    /*----------------------------------------------------------------------*/
    /* Hacky way of indicating that the base image is no longer available.  */
    /* WHHHHAAAAAAATTT!?!?!?!?!?!                                           */
    /*----------------------------------------------------------------------*/
#if 0
    if (loaded_module->file_handle == DLIMP_application_handle)
        DLIMP_application_handle = 0;
#endif

    /*-----------------------------------------------------------------------*/
    /* Free host heap memory that was allocated for the internal loaded      */
    /* module data structure members.                                        */
    /*-----------------------------------------------------------------------*/
    if (loaded_module->name)    DLIF_free(loaded_module->name);
    if (loaded_module->gsymtab) DLIF_free(loaded_module->gsymtab);
    loaded_module->gsymnum = 0;
    if (loaded_module->gstrtab) DLIF_free(loaded_module->gstrtab);
    loaded_module->gstrsz = 0;
    AL_destroy(&(loaded_module->loaded_segments));
    AL_destroy(&(loaded_module->dependencies));

    /*-----------------------------------------------------------------------*/
    /* Finally, free the host memory for the loaded module object, then NULL */
    /* the pointer that was passed in.                                       */
    /*-----------------------------------------------------------------------*/
    DLIF_free(loaded_module);
    *pplm = NULL;
}

#endif

/*****************************************************************************/
/* new_DLIMP_Dynamic_Module()                                                */
/*                                                                           */
/*   Allocate a dynamic module data structure from host memory and           */
/*   initialize its members to their default values.                         */
/*                                                                           */
/*****************************************************************************/
static DLIMP_Dynamic_Module *new_DLIMP_Dynamic_Module(LOADER_FILE_DESC *fd)
{
    /*-----------------------------------------------------------------------*/
    /* Allocate space for dynamic module data structure from host memory.    */
    /*-----------------------------------------------------------------------*/
    DLIMP_Dynamic_Module *dyn_module = 
             (DLIMP_Dynamic_Module *)DLIF_malloc(sizeof(DLIMP_Dynamic_Module));

    if (!dyn_module)
        return NULL;

    /*-----------------------------------------------------------------------*/
    /* Initialize data members of the new dynamic module data structure.     */
    /*-----------------------------------------------------------------------*/
    dyn_module->name = NULL;
    dyn_module->fd = fd;
    dyn_module->phdr = NULL;
    dyn_module->phnum = 0;
    dyn_module->strtab = NULL;
    dyn_module->strsz = 0;
    dyn_module->dyntab = NULL;
    dyn_module->symtab = NULL;
    dyn_module->symnum = 0;
    dyn_module->gsymtab_offset = 0;
    dyn_module->gstrtab_offset = 0;
    dyn_module->c_args = NULL;
    dyn_module->argc = 0;
    dyn_module->argv = NULL;
    dyn_module->loaded_module = NULL;
    dyn_module->wrong_endian = 0;
    dyn_module->direct_dependent_only = TRUE;
    dyn_module->relocatable = FALSE;
    dyn_module->relocate_entry_point = TRUE;

    dyn_module->dsbt_size = 0;
    dyn_module->dsbt_index = DSBT_INDEX_INVALID;
    dyn_module->dsbt_base_tagidx = -1;

    dyn_module->preinit_array_idx = -1;
    dyn_module->preinit_arraysz = 0;
    dyn_module->init_idx = -1;
    dyn_module->init_array_idx = -1;
    dyn_module->init_arraysz = 0;

    return dyn_module;
}

/*****************************************************************************/
/* detach_loaded_module()                                                    */
/*                                                                           */
/*    Detach loaded module data structure from given dynamic module.  When   */
/*    an object file has been successfully loaded, the loader core will      */
/*    detach the loaded module data structure from the dynamic module data   */
/*    structure because the loaded module must continue to persist until is  */
/*    is actually unloaded from target memory.  If there is a problem with   */
/*    the load, then the host memory associated with the loaded module will  */
/*    be released as part of the destruction of the dynamic module.          */
/*                                                                           */
/*****************************************************************************/
static DLIMP_Loaded_Module *detach_loaded_module(DLIMP_Dynamic_Module *dyn_module)
{
    if (dyn_module && dyn_module->loaded_module)
    {
        DLIMP_Loaded_Module *loaded_module = dyn_module->loaded_module;
        dyn_module->loaded_module = NULL;
        return loaded_module;
    }

    return NULL;
}


#if 1
/*****************************************************************************/
/* delete_DLIMP_Dynamic_Module()                                             */
/*                                                                           */
/*    Remove local copies of the string table, symbol table, program header  */
/*    table, and dynamic table.                                              */
/*                                                                           */
/*****************************************************************************/
static void delete_DLIMP_Dynamic_Module(DLIMP_Dynamic_Module **ppdm)
{
   DLIMP_Dynamic_Module *dyn_module = NULL;
   
   if (!ppdm || (*ppdm == NULL))
   {
      DLIF_error(DLET_MISC, 
                 "Internal Error: invalid argument to dynamic module "
		 "destructor function; aborting loader\n");
      exit(1);
   }
   
   dyn_module = *ppdm;
   if (dyn_module->name)     DLIF_free(dyn_module->name);
   if (dyn_module->strtab)   DLIF_free(dyn_module->strtab);
   if (dyn_module->symtab)   DLIF_free(dyn_module->symtab);
   if (dyn_module->phdr)     DLIF_free(dyn_module->phdr);
   if (dyn_module->dyntab)   DLIF_free(dyn_module->dyntab);

   /*------------------------------------------------------------------------*/
   /* If we left the loaded module attached to the dynamic module, then      */
   /* something must have gone wrong with the load.  Remove the loaded       */
   /* module from the queue of loaded modules, if it is there.  Then free    */
   /* the host memory allocated to the loaded module and its segments.       */
   /*------------------------------------------------------------------------*/
   if (dyn_module->loaded_module != NULL)
      delete_DLIMP_Loaded_Module(&(dyn_module->loaded_module));

   /*------------------------------------------------------------------------*/
   /* Finally, free the host memory for this dynamic module object and NULL  */
   /* the pointer to the object.                                             */
   /*------------------------------------------------------------------------*/
   DLIF_free(dyn_module);
   *ppdm = NULL;
}

#endif

/*****************************************************************************/
/* file_header_magic_number_is_valid()                                       */
/*                                                                           */
/*    Given an object file header, check the magic number to ensure that it  */
/*    is an object file format that we recognize.  This implementation of    */
/*    the dynamic loader core will handle ELF object file format.            */
/*                                                                           */
/*****************************************************************************/
static BOOL file_header_magic_number_is_valid(struct Elf32_Ehdr* header)
{
   /*------------------------------------------------------------------------*/
   /* Check for correct ELF magic numbers in file header.                    */
   /*------------------------------------------------------------------------*/
   if (!header->e_ident[EI_MAG0] == ELFMAG0 ||
       !header->e_ident[EI_MAG1] == ELFMAG1 ||
       !header->e_ident[EI_MAG2] == ELFMAG2 ||
       !header->e_ident[EI_MAG3] == ELFMAG3)
   {
      DLIF_error(DLET_FILE, "Invalid ELF magic number.\n");
      return FALSE;
   }

   return TRUE;
}

/*****************************************************************************/
/* file_header_machine_is_valid()                                            */
/*                                                                           */
/*    Check if the machine specified in the file header is supported by the  */
/*    loader.  If the loader was compiled with support for all targets,      */
/*    the machine will be initially set to EM_NONE.  Once a module has been  */
/*    loaded, all remaining modules must have the same machine value.        */
/*****************************************************************************/
static BOOL file_header_machine_is_valid(Elf32_Half e_machine)
{
   if (DLOAD_TARGET_MACHINE == EM_NONE)
      DLOAD_TARGET_MACHINE = e_machine;

   if (e_machine != DLOAD_TARGET_MACHINE)
      return FALSE;

   return TRUE;
}

/*****************************************************************************/
/* is_valid_elf_object_file()                                                */
/*                                                                           */
/*    Check file size against anticipated end location of string table,      */
/*    symbol table, program header tables, etc.  If we anything untoward,    */
/*    then we declare that the ELF file is corrupt and the load is aborted.  */
/*                                                                           */
/*****************************************************************************/
static BOOL is_valid_elf_object_file(LOADER_FILE_DESC *fd, 
                                     DLIMP_Dynamic_Module *dyn_module)
{
   uint32_t fsz;
   int i;

   /*------------------------------------------------------------------------*/
   /* Get file size.                                                         */
   /*------------------------------------------------------------------------*/
   DLIF_fseek(fd, 0, LOADER_SEEK_END);
   fsz = DLIF_ftell(fd);

   /*------------------------------------------------------------------------*/
   /* Check for invalid table sizes (string table, symbol table, and         */
   /* program header tables).                                                */
   /*------------------------------------------------------------------------*/
   if (!((dyn_module->strsz < fsz) &&
         (dyn_module->symnum < fsz) &&
         (dyn_module->phnum * sizeof(struct Elf32_Phdr)) < fsz))
   {
      DLIF_error(DLET_FILE, "Invalid ELF table bounds.\n");
      return FALSE;
   }

   /*------------------------------------------------------------------------*/
   /* Check for null so_name string in file with dynamic information.        */
   /*------------------------------------------------------------------------*/
   if (dyn_module->dyntab && !strcmp(dyn_module->name, ""))
   {
      DLIF_error(DLET_MISC, "Dynamic file lacks SO_NAME identifier.\n");
      return FALSE;
   }

   /*------------------------------------------------------------------------*/
   /* Check for invalid program header information.                          */
   /*------------------------------------------------------------------------*/
   for (i = 0; i < dyn_module->phnum; i++)
   {
      struct Elf32_Phdr* phdr = dyn_module->phdr + i;

      /*---------------------------------------------------------------------*/
      /* Sanity check for relative sizes of filesz and memsz.                */
      /*---------------------------------------------------------------------*/
      if (!(phdr->p_type != PT_LOAD || phdr->p_filesz <= phdr->p_memsz))
      {
         DLIF_error(DLET_MISC, 
                    "Invalid file or memory size for segment %d.\n", i);
         return FALSE;
      }

      /*---------------------------------------------------------------------*/
      /* Check that segment file offset doesn't go off the end of the file.  */
      /*---------------------------------------------------------------------*/
      if (!(phdr->p_offset + phdr->p_filesz < fsz))
      {
         DLIF_error(DLET_FILE,
                  "File location of segment %d is past the end of file.\n", i);
         return FALSE;
      }
   }

   /*------------------------------------------------------------------------*/
   /* Check that a ET_DYN-type file is relocatable.                          */
   /*------------------------------------------------------------------------*/
   if (dyn_module->fhdr.e_type == ET_DYN && !dyn_module->symtab) return FALSE;

   /*------------------------------------------------------------------------*/
   /* All checks passed.                                                     */
   /*------------------------------------------------------------------------*/
   return TRUE;
}

#if 0
/*****************************************************************************/
/* process_eiosabi()                                                         */
/*                                                                           */
/*   Check the EI_OSABI field to validate it and set any parameters based on */
/*   it.                                                                     */
/*****************************************************************************/
static BOOL process_eiosabi(DLIMP_Dynamic_Module* dyn_module)
{
#if ARM_TARGET
   if (is_arm_module(&dyn_module->fhdr))
      return DLDYN_arm_process_eiosabi(dyn_module);
#endif

#if C60_TARGET
   if (is_c60_module(&dyn_module->fhdr))
      return DLDYN_c60_process_eiosabi(dyn_module);
#endif

   return FALSE;
}
#endif

/*****************************************************************************/
/* dload_file_header()                                                       */
/*                                                                           */
/*    Read ELF file header.  Store critical information in the provided      */
/*    DLIMP_Dynamic_Module record.  Check file header for validity.          */
/*                                                                           */
/*****************************************************************************/
static BOOL dload_file_header(LOADER_FILE_DESC *fd, 
                              DLIMP_Dynamic_Module *dyn_module)
{
   /*------------------------------------------------------------------------*/
   /* Read ELF file header from given input file.                            */
   /*------------------------------------------------------------------------*/
   DLIF_fread(&(dyn_module->fhdr), sizeof(struct Elf32_Ehdr), 1, fd);

   /*------------------------------------------------------------------------*/
   /* Determine target vs. host endian-ness.  Does header data need to be    */
   /* byte swapped?                                                          */
   /*------------------------------------------------------------------------*/
   dyn_module->wrong_endian = 
                     (dyn_module->fhdr.e_ident[EI_DATA] != DLIMP_get_endian());

   /*------------------------------------------------------------------------*/
   /* Swap file header structures, if needed.                                */
   /*------------------------------------------------------------------------*/
   if (dyn_module->wrong_endian)
      DLIMP_change_ehdr_endian(&(dyn_module->fhdr));

   /*------------------------------------------------------------------------*/
   /* Write out magic ELF information for debug purposes.                    */
   /*------------------------------------------------------------------------*/
   printf("ELF: %c%c%c\n", dyn_module->fhdr.e_ident[1],
                           dyn_module->fhdr.e_ident[2],
                           dyn_module->fhdr.e_ident[3]);
   printf("ELF file header entry point: %x\n", dyn_module->fhdr.e_entry);

   /*------------------------------------------------------------------------*/
   /* Verify magic numbers in ELF file header.                               */
   /*------------------------------------------------------------------------*/
   if (!file_header_magic_number_is_valid(&(dyn_module->fhdr)))
   {
      DLIF_error(DLET_FILE, "Invalid ELF file header magic number.\n");
      return 0;
   }

   if (!file_header_machine_is_valid(dyn_module->fhdr.e_machine))
   {
       DLIF_error(DLET_FILE, "Invalid ELF file target machine.\n");
       return 0;
   }

   /*------------------------------------------------------------------------*/
   /* Verify file is an executable or dynamic shared object or library.      */
   /*------------------------------------------------------------------------*/
   if ((dyn_module->fhdr.e_type != ET_EXEC) && 
       (dyn_module->fhdr.e_type != ET_DYN))
   {
      DLIF_error(DLET_FILE, "Invalid ELF file type.\n");
      return FALSE;
   }

#if LOADER_DEBUG || LOADER_PROFILE
   /*------------------------------------------------------------------------*/
   /* Stop profiling clock when file header information has finished         */
   /* loading.  Re-start clock on initialization of symbol table, and        */
   /* dynamic table pointers.                                                */
   /*------------------------------------------------------------------------*/
   if (debugging_on || profiling_on)
   {
      printf("done.\n");
      if (profiling_on)
      {
         profile_stop_clock();
         printf("Took %d cycles.\n", profile_cycle_count());
         profile_start_clock();
      }
   }
#endif

   return TRUE;
}

/*****************************************************************************/
/* dload_program_header_table()                                              */
/*                                                                           */
/*    Make a local copy of the ELF object file's program header table in the */
/*    dynamic module data structure.                                         */
/*                                                                           */
/*****************************************************************************/
static void dload_program_header_table(LOADER_FILE_DESC *fd,
                                       DLIMP_Dynamic_Module *dyn_module)
{
   /*------------------------------------------------------------------------*/
   /* Read the program header tables from the object file.                   */
   /*------------------------------------------------------------------------*/
   struct Elf32_Ehdr *fhdr = &(dyn_module->fhdr);
   dyn_module->phdr = (struct Elf32_Phdr*)
                              (DLIF_malloc(fhdr->e_phnum * fhdr->e_phentsize));
   DLIF_fseek(fd, fhdr->e_phoff, LOADER_SEEK_SET);
   DLIF_fread(dyn_module->phdr, fhdr->e_phentsize, fhdr->e_phnum,fd);
   dyn_module->phnum = fhdr->e_phnum;

   /*------------------------------------------------------------------------*/
   /* Byte swap the program header tables if the target endian-ness is not   */
   /* the same as the host endian-ness.                                      */
   /*------------------------------------------------------------------------*/
   if (dyn_module->wrong_endian)
   {
      int i;
      for (i = 0; i < dyn_module->phnum; i++)
         DLIMP_change_phdr_endian(dyn_module->phdr + i);
   }
}

/*****************************************************************************/
/* dload_headers()                                                           */
/*                                                                           */
/*    Read ELF object file header and program header table information into  */
/*    the given dynamic module data structure.  If the object file contains  */
/*    dynamic information, read in the dynamic tags, dynamic symbol table,   */
/*    and global string table.  Check to make sure that we are not already   */
/*    in the process of loading the module (circular dependencies), then     */
/*    perform some level of sanity checking on the content of the file to    */
/*    provide some assurance that the file is not corrupted.                 */
/*                                                                           */
/*****************************************************************************/
static BOOL dload_headers(LOADER_FILE_DESC *fd,
                          DLIMP_Dynamic_Module *dyn_module)
{
#if LOADER_DEBUG || LOADER_PROFILE
   /*------------------------------------------------------------------------*/
   /* More progress information.  Start timing if profiling is enabled.      */
   /*------------------------------------------------------------------------*/
   if (debugging_on || profiling_on)
   {
      printf("\nReading file headers ...\n");
      if (profiling_on) profile_start_clock();
   }
#endif

   /*------------------------------------------------------------------------*/
   /* Read file header information and check vs. expected ELF object file    */
   /* header content.                                                        */
   /*------------------------------------------------------------------------*/
   if (!dload_file_header(fd, dyn_module))
      return FALSE;

   /*------------------------------------------------------------------------*/
   /* Read program header table information into the dynamic module object.  */
   /*------------------------------------------------------------------------*/
   dload_program_header_table(fd, dyn_module);

   return TRUE;
}

/*****************************************************************************/
/* find_dynamic_segment()                                                    */
/*                                                                           */
/*    Find the dynamic segment in the given ELF object file, if there is     */
/*    one.  If the segment is found, then the segment ID output parameter    */
/*    is set to the index of the dynamic segment in the program header       */
/*    table.  If the dynamic segment is not found, the dynamic module's      */
/*    relocatable flag is set to FALSE, and return FALSE.                    */
/*                                                                           */
/*****************************************************************************/
static BOOL find_dynamic_segment(DLIMP_Dynamic_Module *dyn_module,
                                 Elf32_Word *dyn_seg_idx)
{
   int i;

   /*------------------------------------------------------------------------*/
   /* We should have a valid dynamic module pointer and somewhere to put the */
   /* dynamic segment id, if we find one.  If either of these are missing,   */
   /* we should get an internal error and abort the loader.                  */
   /*------------------------------------------------------------------------*/
   if ((dyn_module == NULL) || (dyn_seg_idx == NULL))
   {
      DLIF_error(DLET_MISC, "Internal error: find_dynamic_segment() needs "
                            "non-NULL arguments.\n");
      exit(1);
   }

   /*------------------------------------------------------------------------*/
   /* Spin through segment program headers to find the dynamic segment.      */
   /*------------------------------------------------------------------------*/
   dyn_module->relocatable = TRUE;
   for (i = 0; i < dyn_module->phnum; i++)
      if (dyn_module->phdr[i].p_type == PT_DYNAMIC)
         { *dyn_seg_idx = i; return TRUE; }

   /*------------------------------------------------------------------------*/
   /* No dynamic segment found, mark the object module as not relocatable    */
   /* and warn the user.                                                     */
   /*------------------------------------------------------------------------*/
   dyn_module->relocatable = FALSE;
   DLIF_warning(DLWT_MISC, "'%s' does not have a dynamic segment; assuming " 
                           "that it is a static executable and it cannot " 
                           "be relocated.\n", dyn_module->name);
   return FALSE;
}


/*****************************************************************************/
/* dload_static_executable()                                                 */
/*                                                                           */
/*    Account for target memory allocated to static executable and wrap up   */
/*    loading.  No relocation is necessary.                                  */
/*                                                                           */
/*****************************************************************************/
static int32_t dload_static_executable(LOADER_FILE_DESC *fd,
                                       DLIMP_Dynamic_Module *dyn_module)
{
   int32_t local_file_handle = 0;

#if LOADER_DEBUG
   if (debugging_on) printf("Starting dload_static_executable() ...\n");
#endif

   /*------------------------------------------------------------------------*/
   /* Set entry point for static executable and attempt to allocate target   */
   /* memory for the static executable.                                      */
   /*------------------------------------------------------------------------*/
   dyn_module->loaded_module->entry_point = dyn_module->fhdr.e_entry;
   if (load_static_segment(fd, dyn_module) && load_object(fd, dyn_module))
   {
      /*---------------------------------------------------------------------*/
      /* If successful, we'll want to detach the loaded module object from   */
      /* the dynamic module object that created it.  Take note of the file   */
      /* handle.                                                             */
      /*---------------------------------------------------------------------*/
      DLIMP_Loaded_Module *loaded_module = detach_loaded_module(dyn_module);
      local_file_handle = loaded_module->file_handle;
   }

   /*------------------------------------------------------------------------*/
   /* Static load failed.  Flag an error.                                    */
   /*------------------------------------------------------------------------*/
   else
      DLIF_error(DLET_MEMORY, 
                 "Failed to allocate target memory for static executable.\n");

   /*------------------------------------------------------------------------*/
   /* Destruct dynamic module object.                                        */
   /*------------------------------------------------------------------------*/
   delete_DLIMP_Dynamic_Module(&dyn_module);

#if LOADER_DEBUG
   if (debugging_on) printf("Finished dload_static_executable()\n");
#endif

   return local_file_handle;
}

/*****************************************************************************/
/* DLOAD_load()                                                              */
/*                                                                           */
/*    Dynamically load the specified file and return a file handle for the   */
/*    loaded file.  If the load fails, this function will return a value of  */
/*    zero (0) for the file handle.                                          */
/*                                                                           */
/*    The core loader must have read access to the file pointed to by fd.    */
/*                                                                           */
/*****************************************************************************/
int32_t DLOAD_load(LOADER_FILE_DESC *fd, int argc, char** argv, int32_t *entry)
{
    DLIMP_Dynamic_Module *dyn_module = new_DLIMP_Dynamic_Module(fd);

    if (!dyn_module)
        return 0;

#if LOADER_DEBUG
   /*------------------------------------------------------------------------*/
   /* Spit out some loader progress information when we begin loading an     */
   /* object.                                                                */
   /*------------------------------------------------------------------------*/
   if (debugging_on) printf("Loading file...\n");
#endif

   /*------------------------------------------------------------------------*/
   /* If no access to a program was provided, there is nothing to do.        */
   /*------------------------------------------------------------------------*/
   if (!fd)
   {
      DLIF_error(DLET_FILE, "Missing file specification.\n");
      return 0;
   }

   /*------------------------------------------------------------------------*/
   /* Read file headers and dynamic information into dynamic module.         */
   /*------------------------------------------------------------------------*/
   if (!dload_headers(fd, dyn_module))
      return 0;

   /*------------------------------------------------------------------------*/
   /* Find the dynamic segment, if there is one, and read dynamic            */
   /* information from the ELF object file into the dynamic module data      */
   /* structure associated with this file.                                   */
   /*------------------------------------------------------------------------*/
   if (!dload_dynamic_segment(fd, dyn_module))
      return 0;

   /*------------------------------------------------------------------------*/
   /* ??? We currently don't have a way of finding the .args section.  So    */
   /*    we are hard-wiring the address of the .args section.  The proposed  */
   /*    solution for this problem is to invent a special segment type or    */
   /*    dynamic tag(s) that identify the location and size of the .args     */
   /*    section for the dynamic loader.                                     */
   /*------------------------------------------------------------------------*/
   /*HACK ---> */dyn_module->c_args = (uint8_t*)(0x02204000); /* <--- HACK*/

   /*------------------------------------------------------------------------*/
   /* Record argc and argv pointers with the dynamic module record.          */
   /*------------------------------------------------------------------------*/
   dyn_module->argc = argc;
   dyn_module->argv = argv;

   /*------------------------------------------------------------------------*/
   /* Perform sanity checking on the read-in ELF file.                       */
   /*------------------------------------------------------------------------*/
   if (!is_valid_elf_object_file(fd, dyn_module))
   {
      DLIF_error(DLET_FILE, "Attempt to load invalid ELF file, '%s'.\n",
                    dyn_module->name);
      return 0;
   }

#if LOADER_DEBUG || LOADER_PROFILE
   /*------------------------------------------------------------------------*/
   /* Stop clock on initialization of ELF file information.  Start clock on  */
   /* initialization of ELF module.                                          */
   /*------------------------------------------------------------------------*/
   if (debugging_on || profiling_on)
   {
      printf("Finished dload_dynamic_segment.\n");
      if (profiling_on)
      {
         profile_stop_clock();
         printf("Took %d cycles.\n", profile_cycle_count());
      }
   }
#endif

   /*------------------------------------------------------------------------*/
   /* Initialize internal ELF module and segment structures.  Sets           */
   /* loaded_module in *dyn_module.  This also deals with assigning a file   */
   /* handle and bumping file handle counter.                                */
   /*------------------------------------------------------------------------*/
   initialize_loaded_module(dyn_module);

   /*------------------------------------------------------------------------*/
   /* Support static loading as special case.                                */
   /*------------------------------------------------------------------------*/
   *entry = 0;
   if (!dyn_module->relocatable)  {
      *entry = dyn_module->fhdr.e_entry;
      return (dload_static_executable(fd, dyn_module));
   }

}

