/*********************************************************************************************
 * Convert boot table format data into gp header format
 *********************************************************************************************
 *
 *  A pretty straitforward conversion, except that the block that contains the entry point
 *  must be moved to the last block
 *
 *   usage: btblToGphdr [-swapData] [-swapHdr] -infile <input> -outfile <output>
 *
 *********************************************************************************************/

#include <stdio.h>
#include "ccsutil.h"
#include "malloc.h"
#include <string.h>

char *inFile = NULL;
char *outFile = NULL;

int swapData = 0;
int swapHdr  = 0;


unsigned int swap (unsigned int x)
{
    unsigned int v;

    v = (((x >> 24) & 0xff) <<  0)  |
        (((x >> 16) & 0xff) <<  8)  |
        (((x >>  8) & 0xff) << 16)  |
        (((x >>  0) & 0xff) << 24)  ;

    return (v);

}

int parseit (int ac, char *av[])
{
    int i;

    if (ac < 5)  {
        fprintf (stderr, "usage: %s [-swapData] [-swapHdr] -infile <input file> -outfile <output file>\n", av[0]);
        return (-1);
        
    }


    for (i = 1; i < ac;  /* empty */ )  {

        if (!strcmp (av[i], "-infile"))  {

            inFile = av[i+1];
            i = i + 2;
            continue;

        }

        if (!strcmp (av[i], "-outfile"))  {

            outFile = av[i+1];
            i = i + 2;
            continue;
        }

        if (!strcmp (av[i], "-swapData"))  {
            swapData = 1;
            i = i + 1;
            continue;
        }

        if (!strcmp (av[i], "-swapHdr"))  {
            swapHdr = 1;
            i = i + 1;
            continue;
        }

        fprintf (stderr, "%s: arg parse error at %s\n", av[0], av[i]);
        return (-1);

    }

    if (inFile == NULL)  {
        fprintf (stderr, "%s: input file not specified\n", av[0]);
        return (-1);
    }

    if (outFile == NULL)  {
        fprintf (stderr, "%s: output file not specified\n", av[1]);
        return (-1);
    }

    return (0);

}



int main (int argc, char *argv[])
{
    unsigned int *inData, *outData;
    unsigned int i, n, addr;
    unsigned int idx, bidx, odx;

    if (parseit (argc, argv))
        return (-1);

    
    inData = openReadCloseCcsFile (inFile, &n, &addr);

    if (inData == NULL)  {
        fprintf (stderr, "%s: error reading input data file %s\n", argv[0], inFile);
        return (-1);
    }

    outData = malloc (n * sizeof (unsigned int));

    if (outData == NULL)  {
        free (inData);
        fprintf (stderr, "%s: malloc failure on %d unsigned ints", argv[0], n);
        return (-1);
    }


    /* Copy the blocks over, but skip the block that has the entry point */
    for (odx = 0, idx = 1; inData[idx] != 0; idx = idx + 2 + (inData[idx] >> 2))  {

        /* Hold on the entry block */
        if ( (inData[0] >= inData[idx + 1]) && (inData[0] < (inData[idx+1] + inData[idx])) )  {

            bidx = idx;
            continue;

        }


        /* Copy the header */
        for (i = 0; i < 2; i++)  {

            if (swapHdr != 0)
                outData[odx++] = swap(inData[idx + i]);
            else
                outData[odx++] = inData[idx + i];

        }

        /* Copy the section to the output file */
        for (i = 0; i < (inData[idx] >> 2); i++)  {
            
            if (swapData != 0) 
                outData[odx++] = swap (inData[idx + i + 2]);
            else
                outData[odx++] = inData[idx + i + 2];

        }

    }


    /* Copy the entry section */


    for (i = 0; i < 2; i++)  {

        if (swapHdr != 0)
            outData[odx++] = swap (inData[bidx + i]);
        else
            outData[odx++] = inData[bidx + i];

    }

    for (i = 0; i < (inData[bidx] >> 2); i++)  {
        
        if (swapData != 0)
            outData[odx++] = swap(inData[bidx + i + 2]);
        else
            outData[odx++] = inData[bidx + i + 2];

    }

    /* The terminating load header */
    outData[odx++] = 0;
    outData[odx++] = 0;


    free (inData);

    openWriteCloseCcsFile (outFile, outData, n, addr);

    free (outData);

    return (0);

}

    

      
 







