/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/*****************************************************************************
 * FILE PURPOSE: Convert a ccs formatted blob into a ccs formatted boot table
 *****************************************************************************
 * DESCRIPTION: The resulting boot table has only a single section. The
 *              entry point and the block address have the same value
 *
 *      usage:  ccsBlobToBtbl [-blobAddr blobAddr] [-entryAddr entryAddr] [-infile inputfile] [-outfile outputfile]
 *****************************************************************************/

#include <stdio.h>
#include <malloc.h>
#include "ccsutil.h"

char *infile  = NULL;
char *outfile = NULL;

unsigned int blobAddr  = 0xffffffff;
unsigned int entryAddr = 0xffffffff;


unsigned int pword (char *c)
{
    unsigned int v;

    if ((c[0] == '0') && (c[1] == 'x'))
        sscanf (&c[2], "%x", &v);
    else
        sscanf (c, "%d", &v);

    return (v);

}

int parseit (int ac, char *av[])
{
    int i;

    if (ac == 1)  {
        fprintf (stderr, "usage: %s [-blobAddr blobAddr] [-entryAddr entryAddr] [-infile inputfile] [-outfile outputfile]\n", av[0]);
        return (-1);
    }

    for (i = 1; i < ac;  )  {

        if (!strcmp (av[i], "-blobAddr"))  {
            blobAddr = pword (av[i+1]);
            i = i + 2;
            continue;
        }

        if (!strcmp (av[i], "-entryAddr"))  {
            entryAddr = pword (av[i+1]);
            i = i + 2;
            continue;
        }

        if (!strcmp (av[i], "-infile"))  {
            infile = av[i+1];
            i = i + 2;
            continue;
        }

        if (!strcmp (av[i], "-outfile"))  {
            outfile = av[i+1];
            i = i + 2;
            continue;
        }

        fprintf (stderr, "%s: unknown option: %s\n", av[0], av[i]);
        return (-1);

    }

    return (0);

}
    


int main (int argc, char *argv[])
{
    unsigned int *d1, *d2, addr;
    int n1, n2;
    FILE *str;
    int  i;


    if (parseit(argc, argv))
        return (-1);

        
    if (infile != NULL)  {

        str = fopen (infile, "r");
        if (str == NULL)  {
            fprintf (stderr, "%s: Failed to open input file %s\n", argv[0], infile);
            return (-1);
        }

    }  else  {

        str = stdin;

    }

    d1 = readCcsFile (str, &n1, &addr);


    if (blobAddr == 0xffffffff)
        blobAddr = addr;

    if (entryAddr == 0xffffffff)
        entryAddr = addr;

    if (infile != NULL)
        fclose (str);

    /* Four extra 32 bit values will be added. Three will be at the top (entry point, block size, block addr)
     * and one at the end (block size) */
    n2 = n1 + 4;

    d2 = malloc (n2 * sizeof(unsigned int));

    if (d2 == NULL)  {

        fprintf (stderr, "%s: Malloc error on %d ints\n", argv[0], n2);
        free (d1);
        return (-1);
    }

    d2[0] = entryAddr;
    d2[1] = n1 * 4;
    d2[2] = blobAddr;

    for (i = 0; i < n1; i++)
        d2[i+3] = d1[i];

    d2[i+3] = 0;

    free (d1);

    if (outfile != NULL)  {

        str = fopen (outfile, "w");
        if (str == NULL)  {
            fprintf (stderr, "%s: Failed to open output file %s\n", argv[0], outfile);
            free (d2);
            return (-1);
        }

    }  else  {

        str = stdout;

    }

    writeCcsFile (str, d2, n2, blobAddr);

    if (outfile != NULL)
        fclose (str);

    free (d2);

    return (0);

}

