/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**************************************************************************
 * FILE PURPOSE: Swap 16 bit values in a 64 bit word
 **************************************************************************
 * FILE NAME: ccsswap1664.c
 *
 * DESCRIPTION: For loading the emif25 memory on the shannon QT, every 64
 *				bit word must be treated as 4 16-bit values, and then
 *				endian swapped.
 *
 *	    original
 *    /-------------------------------------------\
 *    |  word A  |  word  B |  word C  |  word D  |
 *    \-------------------------------------------/
 *
 *	   after translation
 *    /-------------------------------------------\
 *    |  word D  |  word  C |  word B  |  word A  |
 *    \-------------------------------------------/
 *
 *
 *   usage:  ccsswap1664 infile outfile
 *
 ***************************************************************************/

#include <stdio.h>
#include <malloc.h>
#include "ccsutil.h"


void swap1664 (unsigned int *ps)
{
    unsigned int v1, v2;
    unsigned int s1, s2;

    v1 = ps[0];
    v2 = ps[1];

    s1 = (v1 >> 16) & 0xffff;
    s2 = (v1 >>  0) & 0xffff;
    v1 = (s2 << 16) | s1;

    s1 = (v2 >> 16) & 0xffff;
    s2 = (v2 >>  0) & 0xffff;
    v2 = (s2 << 16) | s1;

    ps[0] = v2;
    ps[1] = v1;

}
    



int main (int argc, char *argv[])
{
    FILE *str;
    int  nwords;
    int  i;
    unsigned int *data, addr;

    if (argc != 3)  {
        fprintf (stderr, "usage: %s infile outfile\n", argv[0]);
        return (-1);
    }

    str = fopen (argv[1], "r");
    if (str == NULL)  {
        fprintf (stderr, "%s: Failed to open file %s\n", argv[0], argv[1]);
        return (-1);
    }

    data = readCcsFile (str, &nwords, &addr);
    if (data == NULL)  {
        fprintf (stderr, "%s: Error reading in input file %s\n", argv[0], argv[1]);
        return (-1);
    }

    fclose (str);

    if ((nwords & 1) != 0)  {
        nwords = nwords + 1;
        data = realloc (data, nwords * sizeof (unsigned int));
        if (data == NULL)  {
            fprintf (stderr, "%s: realloc failed to size of %d ints\n", argv[0], nwords);
            return (-1);
        }
    }

    
    for (i = 0; i < nwords; i += 2) 
        swap1664 (&data[i]);


    str = fopen (argv[2], "w");
    if (str == NULL)  {
        fprintf (stderr, "%s: Failed to open output file %s\n", argv[0], argv[2]);
        free (data);
        return (-1);
    }
    

    writeCcsFile (str, data, nwords, addr);

    fclose (str);
    free (data);

    return (0);

}
    




