/*  create a boot table for secure master download
 *
 *  A boot parameter table is created for use in QT development. The master peripherals (PCIe, etc) 
 *  test bench drivers read boot paramter tables, which is why this is created.
 *
 *  The boot parameter table consists of two blocks. First is the download blob which is the
 *  optionally encrypted and signed (advanced and premium secure boot) blob. This is followed
 *  by the signed public key (advanced and premium secure boot) and the start address and
 *  size information for the blob.
 *
 *  usage:  secMasterTable blobFile.ccs -bstart startAddr [-key signedkeyFile.ccs] [-out outfile]
 *
 *  This code requires that the secure host table resides at  0x108efd80, and that 
 *  signed keys are 520 bytes long.
 *
 */

#include <stdio.h>
#include <malloc.h>
#include "ccsutil.h"

#define KEY_DATA_SIZE32  130
#define SEC_HOST_ADDR    0x108efd80

char *blobName = NULL;
char *keyName  = NULL;
char *outName  = NULL;


unsigned int *blobData         = NULL;
int           blobDataSize32;

unsigned int *keyData          = NULL;
int           keyDataSize32;

unsigned int *btblData;
int           btblDataSize32;

unsigned int blobStart         = 0;


void cleanUp (void)
{
    if (blobData != NULL)
        free (blobData);

    if (keyData != NULL)
        free (keyData);

    if (btblData != NULL)
        free (btblData);

}

unsigned int readInt (char *s)
{
    unsigned int ret;

    if ((s[0] == '0') && (s[1] == 'x'))
        sscanf (&s[2], "%x", &ret);
    else
        sscanf (s, "%d", &ret);

    return (ret);

}


int parseit (int ac, char *av[])
{
    int i;

    if (ac == 1)  {
        fprintf (stderr, "usage: %s blobFile.ccs -bstart startAddr [-key signedKeyFile.ccs] [-out outfile]\n", av[0]);
        return (-1);
    }

    for (i = 1; i < ac;  )  {

        if (!strcmp (av[i], "-bstart"))  {
            blobStart = readInt (av[i+1]);
            i = i + 2;
            continue;
        }

        if (!strcmp (av[i], "-key"))  {
            keyName = av[i+1];
            i = i + 2;
            continue;
        }

        if (!strcmp (av[i], "-out"))  {
            outName = av[i+1];
            i = i + 2;
            continue;
        }

        if (blobName == NULL)  {
            blobName = av[i];
            i = i + 1;
            continue;
        }
        
        fprintf (stderr, "%s: Unkown argument: %s\n", av[0], av[i]);
        return (-1);
    }

    if (blobName == NULL)  {
        fprintf (stderr, "%s: Blob input file (ccs format) not specified\n", av[0]);
        return (-1);
    }

    if (blobStart == 0)  {
        fprintf (stderr, "%s: no or invalid blob start address provided\n", av[0]);
        return (-1);
    }

    return (0);

}


int main (int argc, char *argv[])
{
    int i, j;
    unsigned int addr, junk;

    if (parseit (argc, argv))
        return (-1);

    blobData = openReadCloseCcsFile (blobName, &blobDataSize32, &addr);

    if (blobData == NULL)  {
        fprintf (stderr, "%s: Failed to read input file %s\n", argv[0], blobName);
        cleanUp ();
        return (-1);
    }


    /* Read the key or create an 0 key if non is provided */

    if (keyName != NULL)  {

        keyData = openReadCloseCcsFile (keyName, &keyDataSize32, &junk);

        if (keyData == NULL)  {
            fprintf (stderr, "%s: Failed to read key file %s\n", argv[0], keyName);
            cleanUp ();
            return (-1);
        }

        if (keyDataSize32 != KEY_DATA_SIZE32)  {
            fprintf (stderr, "%s: error - key file must have exactly 130 32-bit words\n", argv[0], keyDataSize32);
            cleanUp ();
            return (-1);
        }

    }  else  {

        keyDataSize32 = KEY_DATA_SIZE32;

        keyData = malloc (keyDataSize32 * sizeof (unsigned int));

        if (keyData == NULL)  {
            fprintf (stderr, "%s: malloc failure on %d unsigned ints\n", argv[0], keyDataSize32);
            cleanUp ();
            return (-1);
        }

    }

    /* Create the output file. The boot table is preceeded by 1 32-bit value which is the
     * entry address. This is followed by records of type length, address, data.
     *
     * The total length required (in 32 bit values) is:
     *  1(entry) + 2 (blob length, addres) + blobDataSize32 + 
     *             2 (key length, address) + keyDataSize32  +  1(sechost start address) +
     *                                                         1(sechost size bytes)    + 1(0 length terminator)
     *  where the last two entries are appended to the key data and part of the key
     *  data boot table data (although they could be seperate there is no reason for
     *  them to be
     */
    btblDataSize32 = 1 + 2 + blobDataSize32 + 2 + keyDataSize32 + 1 + 1 + 1;

    btblData = malloc (btblDataSize32 * sizeof (unsigned int));
    
    if (btblData == NULL)  {
        fprintf (stderr, "%s: malloc failed on %d unsigned ints\n", argv[0], btblDataSize32);
        cleanUp ();
        return (-1);
    }

    i = 0;
    /* The entry point */
    btblData[i++] = blobStart;

    /* The blob boot table entry */
    btblData[i++] = blobDataSize32 * 4;
    btblData[i++] = blobStart;

    /* The blob data */
    for (j = 0; j < blobDataSize32; j++)
        btblData[i++] = blobData[j];

    /* The key + blob info table entry */
    btblData[i++] = (keyDataSize32+2) * 4;                /* Add the two secure host words */
    btblData[i++] = SEC_HOST_ADDR;


    /* The key data */
    for (j = 0; j < keyDataSize32; j++)
        btblData[i++] = keyData[j];

    /* The secure host words */
    btblData[i++] = blobStart;
    btblData[i++] = blobDataSize32 * 4;

    /* The terminator word */
    btblData[i++] = 0;

    if (btblDataSize32 != i) 
        fprintf (stderr, "%s warning - generated file size not computed correctly. Expected %d, found %d\n", argv[0], btblDataSize32, i);


    if (outName != NULL)   {

        if (openWriteCloseCcsFile (outName, btblData, btblDataSize32, addr) != 0)  {
            fprintf (stderr, "%s: Error writing output file %s\n", argv[0], outName);
            cleanUp ();
            return (-1);
        }

    }  else  {

        writeCcsFile (stdout, btblData, btblDataSize32, addr);
    }

    cleanUp ();


    return (0);

}
    



    




