/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**************************************************************************************
 * FILE PURPOSE: Generate a secure boot block image
 **************************************************************************************
 * FILE NAME: secbootimage.c
 *
 * DESCRIPTION: A CCS data file is read. The load trailer is appended. The
 *              resulting file is optionally signed, and then optionally
 *              encrypted. The signature (if present) is then appended to the
 *              data block in byte reversed order.
 *
 *          Usage: secbootimage inputfile [-aeskey aeskey(hex)] [-rsakey keyfile]
 *
 *          The output is secimage.ccs
 *
 **************************************************************************************/
#include <stdio.h>
#include <malloc.h>

#include "aes.h"


/***************************************************************************************
 * FUNCTION PURPOSE: Open a CCS data file, read in the data
 ***************************************************************************************
 * DESCRIPTION: The CCS data file read and returned
 ***************************************************************************************/
unsigned char *readCcsFile (char *fname, int *dataLenBytes)
{
  FILE *fin;
  char iline[132];

  unsigned char *data;

  unsigned int v;

  int a, b, c, d, l;
  int i, p;

  unsigned int allocLen;

  fin = fopen (fname, "r");
  if (fin == NULL)  {
    fprintf (stderr, "(%s : %d): Could not open input file %s\n", __FILE__, __LINE__, fname);
    *dataLenBytes = -1;
    return;
  }

  fgets (iline, 131, fin);
  sscanf (iline, "%x %x %x %x %x", &a, &b, &c, &d, &l);

  /* Round the data size up to a multiple of 16 bytes. Without this the signature 
   * will not be correct because openssl pads the data with 0xc, and this program
   * does not do that padding. Note that the length here refers to lines in a CCS
   * file, and each line is 4 bytes. */
  allocLen = (l + 3) & ~3;


  data = malloc ((allocLen*4)*sizeof(unsigned char));
  if (data == NULL)  {
    fprintf (stderr, "(%s : %d): malloc failure\n", __FILE__, __LINE__);
    *dataLenBytes = -1;
    return (NULL);
  }

  for (i = p = 0; i < l; i++)  {
     
     fgets (iline, 131, fin);
     sscanf (&iline[2], "%x", &v);

     data[p++] = (v >> 24) & 0xff;
     data[p++] = (v >> 16) & 0xff;
     data[p++] = (v >>  8) & 0xff;
     data[p++] = (v >>  0) & 0xff;

  }

  for ( ; i < allocLen; i++)  {
    data[p++] = 0;
    data[p++] = 0;
    data[p++] = 0;
    data[p++] = 0;
  }

  fclose (fin);

  *dataLenBytes = p;
  return (data);

}


/************************************************************************************
 * FUNCTION PURPOSE: Convert 4 bits into a hex value
 ************************************************************************************
 * DESCRIPTION: Converts a 4 bit value from binary to ascii
 ************************************************************************************/
unsigned char toHex (unsigned char c)
{
  if (c < 10)
    return (c + '0');

  return (c + 'a' - 10);

}


/***************************************************************************************
 * FUNCTION PURPOSE: Create the initial block header
 ***************************************************************************************
 * DESCRIPTION: The initial block header has the following structure (before encryption)
 *
 *    /-----------------------------\
 *    |      0x70adc0de             |
 *    |    Data length(bytes)       |  This is the number of bytes following this block
 *    |      0x00000000             |
 *    |      0x00000000             |
 *    \-----------------------------/
 *
 *  A binary version is created for encryption and an ascii version is created for
 *  use as the initial vector for the data block.
 *
 ***************************************************************************************/
void headerCreate (int curSize, int sigSize, char *hdr_clr, char *ivs)
{
  unsigned char block[16];
  FILE *fout;
  int   i, j;

  fprintf (stderr, "curSize = %d, sigSize = %d\n", curSize, sigSize);

  /* The current size must be rounded up to a multiple of 16 bytes until cipher text
   * stealing is implemented */
  curSize = (curSize + 15) & ~15;

  block[0] = 0x70;
  block[1] = 0xad;
  block[2] = 0xc0;
  block[3] = 0xde;

  i = curSize + sigSize + 16;   /* The extra 16 is the size of this block */


  block[4] = (i >> 24) & 0xff;
  block[5] = (i >> 16) & 0xff;
  block[6] = (i >>  8) & 0xff;
  block[7] = (i >>  0) & 0xff;

  block[8]  = block[9]  = block[10] = block[11] = 0;
  block[12] = block[13] = block[14] = block[15] = 0;


  fout = fopen (hdr_clr, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open output file %s\n", __FILE__, __LINE__, hdr_clr);
    return;
  }

  fwrite (block, sizeof(unsigned char), 16, fout);
  fclose (fout);


  for (i = j = 0; i < 16; i++)  {
    ivs[j++] = toHex((block[i] >> 4) & 0x0f);
    ivs[j++] = toHex((block[i] >> 0) & 0x0f);
  }

  ivs[j++] = '\0';

}

/***************************************************************************************
 * FILE PURPOSE: Read a file
 ***************************************************************************************
 * DESCRIPTION: The input file is opened and the data read as binary file.
 ***************************************************************************************/
unsigned char *readBinFile (char *fname, int *datalen)
{
  FILE *fin;
  int   len;
  unsigned char *data;

  fin = fopen (fname, "rb");
  if (fin == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open input file %s\n", __FILE__, __LINE__, fname);
    return (NULL);
  }

  fseek (fin, 0, SEEK_END);
  len = ftell (fin);

  data = malloc (len * sizeof (unsigned char));
  if (data == NULL)  {
    fprintf (stderr, "(%s : %d): Malloc failed on %d bytes\n", __FILE__, __LINE__, len);
    return (NULL);
  }

  fseek (fin, 0, SEEK_SET);
  fread (data, sizeof(unsigned char), len, fin);

  fclose (fin);

  *datalen = len;

  return (data);

}

/***************************************************************************************
 * FILE PURPOSE: Write a file
 ***************************************************************************************
 * DESCRIPTION: The output data file is created and the data written into it
 ***************************************************************************************/
void writeBinFile (unsigned char *dat, int len, char *fname)
{
  FILE *fout;

  fout = fopen (fname, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open output file %s\n", __FILE__, __LINE__, fname);
    return;
  }

  fwrite (dat, sizeof(unsigned char), len, fout);

  fclose (fout);

}

/***************************************************************************************
 * FUNCTION PURPOSE: Return the length of a file
 ***************************************************************************************
 * DESCRIPTION: The file is opened, the length is determined, and the file is closed.
 ***************************************************************************************/
int getFileLen(char *fname)
{
  FILE *fin;
  int   len;

  fin = fopen (fname, "rb");
  if (fin == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open input file %s\n", __FILE__, __LINE__, fname);
    return (NULL);
  }

  fseek (fin, 0, SEEK_END);
  len = ftell (fin);

  fclose (fin);

  return (len);

}



/***************************************************************************************
 * FUNCTION PURPOSE: Catenate two files
 ***************************************************************************************
 * DESCRIPTION: Two files are catenated and the result placed into a third file
 ***************************************************************************************/
void filecat (char *f1, char *f2, char *cat)
{
  unsigned char *dat1, *dat2;
  int len1, len2;
  FILE *fout;

  dat1 = readBinFile (f1, &len1);
  dat2 = readBinFile (f2, &len2);

  fout = fopen (cat, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open output file %s\n", __FILE__, __LINE__, cat);
    return;
  }

  fwrite (dat1, sizeof (unsigned char), len1, fout);
  fwrite (dat2, sizeof (unsigned char), len2, fout);

  fclose (fout);

}
 
/***************************************************************************************
 * FUNCTION PURPOSE: Byte reverse an array
 ***************************************************************************************
 * DESCRIPTION: The array is reversed
 ***************************************************************************************/
byteReverse (unsigned char *dat, int len)
{
  int i;
  unsigned char tmp;

  /* This works for both even and odd length arrays */
  for (i = 0; i < len/2; i++)  {
    tmp              = dat[i];
    dat[i]           = dat[len - 1 - i];
    dat[len - 1 - i] = tmp;
  }

}

/***************************************************************************************
 * FUNCTION PURPOSE: Perform a byte swap
 ***************************************************************************************
 * DESCRIPTION: Byte swap data in each 32 bit value
 ***************************************************************************************/
void byteSwap (unsigned char *dat, int len)
{
    int i;
    unsigned char tmp;

    for (i = 0; i < len; i += 4) {

        tmp      = dat[i];
        dat[i]   = dat[i+3];
        dat[i+3] = tmp;

        tmp      = dat[i+1];
        dat[i+1] = dat[i+2];
        dat[i+2] = tmp;

    }

}


/***************************************************************************************
 * FUNCTION PURPOSE: Catenate two arrays
 ***************************************************************************************
 * DESCRIPTION: Two arrays are catenated in a new array
 ***************************************************************************************/
unsigned char *catBin (unsigned char *a1, int l1, unsigned char *a2, int l2, int *l3)
{
  unsigned char *a3;
  int i, p;

  *l3 = l1 + l2;
  a3 = malloc (*l3 * sizeof (unsigned char));
  if (a3 == NULL)  {
    fprintf (stderr, "(%s : %d) : Malloc failed on size of %d\n", __FILE__, __LINE__, *l3);
    return;
  }

  for (i = p = 0; i < l1; i++)
    a3[p++] = a1[i];

  for (i = 0; i < l2; i++)
    a3[p++] = a2[i];

  return (a3);

}

/***************************************************************************************
 * FUNCTION PURPOSE: Create a CCS data file
 ***************************************************************************************
 * DESCRIPTION: The CCS data file is created. The data is packed in big endian moe
 ***************************************************************************************/
void ccsCreate (unsigned char *dat, int len)
{
  FILE *fout;

  int i, p;
  int len4;
  unsigned int v;

  fout = fopen ("secbootimage.ccs", "w");
  if (fout == NULL)  {
    fprintf (stderr, "(%s : %d): Could not open output file secbootimage.ccs\n", __FILE__, __LINE__);
    return;
  }

  fprintf (fout, "1651 1 10000 1 %x\n", (len+3)/4);

  len4 = len / 4;
  
  for (i = p = 0; i < len4; i++)  {

    v = (dat[p++] << 24);
    v = v + (dat[p++] << 16);
    v = v + (dat[p++] << 8);
    v = v + (dat[p++] << 0);

    fprintf (fout, "0x%08x\n", v);

  }

  if ((len & 0x3) != 0)  {
    v = 0;
    for (i = 24 ; p < len; p++, i = i - 8) 
      v = v + (dat[p] << i);
    fprintf (fout, "0x%08x\n", v);
  }

}




/***************************************************************************************
 * FUNCTION PURPOSE: Print the execv command that failed
 ***************************************************************************************
 * DESCRIPTION: The command is sent to stderr
 ***************************************************************************************/
void showString (char *inv, char *args[])
{
  int i;

  fprintf (stderr, "\t The failed command: %s ", inv);

  for (i = 0; args[i] != NULL; i++)
    fprintf (stderr, "%s ", args[i]);

  fprintf (stderr, "\n");

}

char *infile    = NULL;
char *aesString = NULL;
char *rsaFile   = NULL;
unsigned char aesKey[17];

/****************************************************************************************
 * FUNCTION PURPOSE: Parse the input arguments
 ****************************************************************************************
 * DESCRIPTION: The command line arguments are scanned and checked
 ****************************************************************************************/
int parseit (int ac, char *av[])
{
    int i, j;
    char ts[3];

    if (ac == 1)  {
        fprintf (stderr, "usage: %s inputfile [-aeskey aeskey(hex)] [-rsakey keyfile]\n", av[0]);
        return (-1);
    }

    for (i = 1; i < ac; )  {
    
        if (!strcmp (av[i], "-aeskey"))  {
            aesString = av[i+1];
            i = i+2;
         
        }  else if (!strcmp (av[i], "-rsakey"))  {
            rsaFile = av[i+1];
            i = i+2;

        }  else  {
            if (infile != NULL)  {
                fprintf (stderr, "%s: Command line parse error - seemed to find two input file names (%s and %s)\n", 
                         av[0], infile, av[i]);
                return (-1);

            }  else  {
                infile = av[i];
                i = i + 1;
            }

        }

    }

    if (infile == NULL)  {
        fprintf (stderr, "%s: No input file specified\n", av[0]);
        return (-1);
    }

    if (aesKey == NULL)  {
        fprintf (stderr, "%s: An aes key must be specified\n", av[0]);
        return (-1);
    }

    for (i = j = 0; i < 16; i++, j+=2)  {
        ts[0] = aesString[j+0];
        ts[1] = aesString[j+1];
        ts[2] = 0;
        sscanf(ts, "%x", &aesKey[i]);
    }
    

    return (0);

}
        
int main (int argc, char *argv[])
{

  FILE *fout;
  
  char ivs[60];       /* Ascii version of the initial vector for the block AES */

  int i, p; 

  char *hdr_clr = "header_clr.bin";
  char *hdr_enc = "header_enc.bin";

  char *aesh_s[] = { "enc", "-aes-128-ecb", "-in", hdr_clr, "-out", hdr_enc,
                     "-K", NULL, "-iv", "0", "-nopad", NULL }; 
        
  char *blk_clr = "block_clr.bin";
  char *blk_enc = "block_enc.bin";

  char *aesb_s[] = { "enc", "-aes-128-cbc", "-in", blk_clr, "-out", blk_enc,
                     "-K", NULL, "-iv", ivs, "-nopad", "-nosalt", NULL }; 

  char *cat_clr = "hdr_blk_clr.bin";
  char *cat_enc = "hdr_blk_enc.bin";
  char *sig     = "sign.bin";

  char *rsa_s[] = { "dgst", "-sha256", "-binary", "-out", sig, "-sign", NULL, cat_clr, NULL };


  char *rsalen_s[] = { "dgst", "-sha256", "-binary", "-out", sig, "-sign", NULL, NULL, NULL };


  unsigned char *bootDat;
  int bootDatLenBytes;

  unsigned char *encDat = NULL;
  int encDatLen;

  unsigned char *sigDat = NULL;
  int sigDatLen;

  unsigned char *allDat;
  int allDatLen;

  char iline[132];

  int signLen = 0;

  if (parseit (argc, argv) != 0)
    return (-1);

  /* Read in the CCS data file */
  bootDat = readCcsFile (infile, &bootDatLenBytes);
  if (bootDatLenBytes < 0)
    return (-1);


  /* We need to know the size of the signature data before creating the header. Do this
   * with a little cheat. Go ahead and sign the input ccs file, then find out how long
   * it is */
  if (rsaFile != NULL)  {

      if (fork())  {
        wait();  /* wait for the child to terminate */

      }  else  {

        rsalen_s[6] = argv[4];  /* The RSA private key file */
        rsalen_s[7] = argv[1];  /* The input file */

        if (execv ("c:\\openssl\\bin\\openssl", rsalen_s))  {
          fprintf (stderr, "%s: Openssl to sign the data block failed\n", argv[0]);

          showString ("openssl", rsalen_s);
          exit (-1);
        }

      }


    signLen = getFileLen(sig);

  } 


  /* Create the initial header. The binary file in hdr_clr is created. An ascii
   * version is returned in ivs for use as an initialization vector */
  headerCreate (bootDatLenBytes, signLen, hdr_clr, ivs);

  fprintf (stderr, "ivs = %s\n", ivs);


  /* Create a binary file containing the boot data */
  fout = fopen (blk_clr, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "%s: Failed to create output file %s\n", argv[0], blk_clr);
    exit (-1);
  }

  fwrite (bootDat, sizeof(unsigned char), bootDatLenBytes, fout);
  fclose (fout);



  if (aesString != NULL)  {

    AES_KEY mykey;
    unsigned int hdrLen;
    unsigned char *hdr_clr_data;
    unsigned char *hdr_enc_data;
    unsigned char *bootDatEnc;
    unsigned char iv[16*4];


    /* Encrypt the 16 byte header using aes ecb with no iv */
    AES_set_encrypt_key(aesKey, 16*8,  &mykey);

    hdr_clr_data = readBinFile (hdr_clr, &hdrLen);
    hdr_enc_data = malloc(hdrLen * sizeof(unsigned char));

    /* Must byte swap before and after the encoding */
    byteSwap (hdr_clr_data, hdrLen);

    AES_ecb_encrypt(hdr_clr_data, hdr_enc_data, &mykey, AES_ENCRYPT);

    byteSwap (hdr_enc_data, hdrLen);
    /* byteSwap (hdr_clr_data, hdrLen); */   /* Swap back the original for use as the IV */

    writeBinFile (hdr_enc_data, hdrLen, hdr_enc);

    fprintf (stderr, "AES encoding done on header \n");


    /* Encrypt the data block using cbc with the un-encrypted 16 byte header
     * as the IV */
    fprintf (stderr, "AES Key= ");

   for (i = 0; i < 16; i++)
     fprintf (stderr, "%02x", aesKey[i]);
   fprintf (stderr, "\n");
   
   AES_set_encrypt_key(aesKey, 16*8,  &mykey);

   fprintf (stderr, "boot data length before adjustment = %d\n", bootDatLenBytes);
   /* Pad the data out if the length is not a multiple of 16 bytes */
   if ((bootDatLenBytes % 16) != 0)  {
     bootDatLenBytes = (bootDatLenBytes + 15) & ~0xf;
     bootDat = realloc (bootDat, bootDatLenBytes);
   }
   fprintf (stderr, "boot data length after adjustment = %d\n", bootDatLenBytes);

   bootDatEnc = malloc (bootDatLenBytes * sizeof (unsigned char));

  
   fprintf (stderr, "hdr_clr_data = ");
   for (i = 0; i < 16; i++)  {
     fprintf (stderr, "%c", toHex(hdr_clr_data[i] >> 4));
     fprintf (stderr, "%c", toHex(hdr_clr_data[i] & 0xf));
   }

   fprintf (stderr, "\n");

   for (i = 0; i < 64; i++)
     iv[i] = 0;

   for (i = 0; i < 16; i++)
     iv[i] = hdr_clr_data[i];

   byteSwap (bootDat, bootDatLenBytes);
    
   AES_cbc_encrypt2 (bootDat, bootDatEnc, bootDatLenBytes, &mykey, iv, AES_ENCRYPT);

   byteSwap (bootDatEnc, bootDatLenBytes);

  
   writeBinFile (bootDatEnc, bootDatLenBytes, blk_enc);

   fprintf (stderr, "Data encryption done, iv= %s\n", ivs);


  }  else  {

    /* If there is no aes encoding just rename the encoded pointer to point to 
     * the un-encoded name */
    hdr_enc = hdr_clr;
    blk_enc = blk_clr;

  }

    /* Now catenate the two AES encoded files */
    filecat (blk_enc, hdr_enc, cat_enc);


    if (rsaFile != NULL)  {

      /* Catenate the two unencoded files */
      filecat (blk_clr, hdr_clr, cat_clr);


      /* Sign the clear text catenated block */
      if (fork())  {
        wait();  /* wait for the child to terminate */

      }  else  {

        rsa_s[6] = rsaFile;  /* The RSA private key file */
  
        if (execv ("c:\\openssl\\bin\\openssl", rsa_s))  {
          fprintf (stderr, "%s: Openssl to sign the data block failed\n", argv[0]);

          showString ("openssl", rsa_s);
          exit (-1);
        }

      }


      /* Swap the signature bytes, and append these to the AES encrypted data */
      encDat = readBinFile (cat_enc, &encDatLen);
      sigDat = readBinFile (sig, &sigDatLen);

      byteReverse (sigDat, sigDatLen); 

      allDat = catBin (encDat, encDatLen, sigDat, sigDatLen, &allDatLen);


  }  else  {

      /* If there is no rsa then the output is the catenated (optionally) aes encrypted
       * data */
      allDat = readBinFile (cat_enc, &allDatLen);
      
  }



    /* Create the CCS data file */
    ccsCreate (allDat, allDatLen);

    free (allDat);

    if (sigDat != NULL)
        free (sigDat);

    if (encDat != NULL)
        free (encDat);


  return (0);

}

    

   

 

  
