/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/*****************************************************************************************
 * FILE PURPOSE: Convert a boot table to secure loader format
 *****************************************************************************************
 * FILE NAME: seccvt.c
 *
 * DESCRIPTION: Reads an ascii boot table as input. Performs the following operations:
 *
 *              - blocks the data into 112 byte blocks
 *              - pre-pends the decrypt header
 *              - RSA signs the block
 *              - AES encrypts the block
 *              - adds the i2c header
 *              - writes the output as a binary file
 *
 *      usage: seccvt input_file output_file rsa_key_file aes_key_file 
 *
 *****************************************************************************************/

#include <stdio.h>
#include <malloc.h>


/***************************************************************************************
 * Block processing parameters 
 ***************************************************************************************/
unsigned char hdr_btbl[128];   /* Decrypt header plus boot table data */
unsigned char sig_enc[260];    /* I2C header, Decrypt header, boot table data, signature, AES encoded */
unsigned char scratch[128];    /* Some scratch */

/****************************************************************************************
 * FUNCTION PURPOSE: Perform ones complement addition
 ****************************************************************************************
 * DESCRIPTION: Adds two values using ones complemetn addition.
 ****************************************************************************************/
unsigned onesComplementAdd (unsigned value1, unsigned value2)
{
  unsigned result;

  result = (unsigned)value1 + (unsigned)value2;

  result = (result >> 16) + (result & 0xFFFF); /* add in carry   */
  result += (result >> 16);                    /* maybe one more */
  result = (result & 0xffff);
  return (unsigned)result;

} /* end of beth_ones_complement_add() */

/****************************************************************************************
 * FUNCTION PURPOSE: Perform a ones complement checksum on a block
 ****************************************************************************************
 * DESCRIPTION: Computes then puts the ones complement check sum into the 
 *              I2C header at the top of the block.
 ****************************************************************************************/
void blockCheckSum (unsigned char *block, int blockSize)
{
  unsigned checksum = 0;
  unsigned value;
  int i;

  if (blockSize & 0x0001)  {
    fprintf (stderr, "program requires an even blocksize\n");
    exit (-1);
  }

  /* Clear the initial checksum locations */
  block[2] = block[3] = 0;

  for (i = 0; i < blockSize; i += 2) {
    value = (block[i] << 8) | block[i+1];
    checksum = onesComplementAdd (checksum, value);
  }

  /* Put the checksum into the block starting at byte 2. Use big endian */
  checksum = ~checksum;
  block[3] = checksum & 0xff;
  block[2] = (checksum >> 8) & 0xff;

}


/************************************************************************************
 * FUNCTION PURPOSE: Return true if the input value is an ascii hex numeric
 ************************************************************************************
 * DESCRIPTION: Checks if the input value is the ascii representation of 0-9, a-f
 ************************************************************************************/
int asciiByte (unsigned char c)
{
  if ((c >= '0') && (c <= '9'))
    return (1);

  if ((c >= 'A') && (c <= 'F'))
    return (1);

  if ((c >= 'a') && (c <= 'f'))
    return (1);

  return (0);

}


/************************************************************************************
 * FUNCTION PURPOSE: Convert an ascii value to a number
 ************************************************************************************
 * DESCRIPTION: Converts a 4 bit value from ascii to binary
 ************************************************************************************/
int toNum (unsigned char c)
{
  if ((c >= '0') && (c <= '9'))
    return (c - '0');

  if ((c >= 'A') && (c <= 'F'))
    return (c - 'A' + 10);

  return (c - 'a' + 10);

}

/***************************************************************************************
 * FUNCTION PURPOSE: Read a boot table data file
 ***************************************************************************************
 * DESCRIPTION: Reads the ascii boot table data file, converts to binary.
 ***************************************************************************************/
#define SEG_SIZE    32768   /* Segment allocation size */
unsigned char *readBtbl (FILE *fin, unsigned *dataSize)
{
  unsigned char *dat;
  unsigned arraySize;
  unsigned p;
  unsigned char x, y;
  char iline[132];

  arraySize = SEG_SIZE;
  dat = malloc (arraySize * sizeof(unsigned char));
  if (dat == NULL)  {
    fprintf (stderr, "malloc failure on initial allocation\n");
    exit(-1);
  }

  /* Strip the first two lines of the file */
  fgets (iline, 131, fin);
  fgets (iline, 131, fin);

  p = 0;

  for (;;)  {

    /* Search for the first ascii character */
    do  {
      x = fgetc (fin);
      if (x == (unsigned char)EOF)  {
        *dataSize = p;
        return (dat);
      }
    } while (!asciiByte(x));

    /* Read the next character */
    y = fgetc (fin);
    if (y == (unsigned char)EOF)  {
      *dataSize = p;
      return (dat);
    }
    if (asciiByte(y))
      dat[p++] = (toNum(x) << 4) | toNum (y);

    if (p >= arraySize)  {
      arraySize += SEG_SIZE;
      dat = realloc (dat, arraySize);
      if (dat == NULL)  {
        fprintf (stderr, "realloc failed at size %d\n", arraySize);
        exit(-1);
      }
    }

  }

}
      

/**************************************************************************************
 * FUNCTION PURPOSE: Append the decrypt header
 **************************************************************************************
 * DESCRIPTION: The decrypt header is appended in little endian format. The 
 *              size is always 256 bytes, and the random number is always 0
 **************************************************************************************/
void addDecryptHeader (unsigned char hdr[])
{
  /* The first four bytes are the magic number, 0xd590c0de, least significant byte first */
  hdr[0] = 0xde;
  hdr[1] = 0xc0;
  hdr[2] = 0x90;
  hdr[3] = 0xd5;

  /* The next four bytes are the length, which is always 256 bytes */
  hdr[4] = 0x00;
  hdr[5] = 0x01;
  hdr[6] = 0x00;
  hdr[7] = 0x00;

  /* The next word is reserved, and must be 0 */
  hdr[8] = 0x00;
  hdr[9] = 0x00;
  hdr[10] = 0x00;
  hdr[11] = 0x00;

  /* The last word is for a random number, for now is always 0 */
  hdr[12] = 0x00;
  hdr[13] = 0x00;
  hdr[14] = 0x00;
  hdr[15] = 0x00;

}

/**************************************************************************************
 * FUNCTION PURPOSE: Copy up to 112 bytes of data from one array to another
 **************************************************************************************
 * DESCRIPTION: Copies data from array source, starting at index p, to array dest.
 *              The number of elements in source is in in. The index
 *              to be used next is returned.
 **************************************************************************************/
unsigned extractSeg (unsigned char *source, unsigned n, unsigned p, unsigned char *hdr_btbl)
{
  unsigned i;

  for (i = 0; i < 112; i++)  {
    hdr_btbl[i] = source[p++];
    if (p >= n)
      return (p);

  }

  return (p);

}
  


/***************************************************************************************
 * FUNCTION PURPOSE: Perform the main block processing
 ***************************************************************************************
 * DESCRIPTION: Verifies input parameters, reads the input file and initiates processing
 ***************************************************************************************/
int main (int argc, char *argv[])
{
  FILE *fin;      /* Input file  */
  FILE *fout;     /* Output file */
  FILE *ftmp;     /* Temporary file */
  FILE *faes;     /* AES 128 bit key, in ascii format */

  char aesKey[33];   /* Ascii representation of the aes key */

  char *sigc[] = { "dgst", "-sha1", "-binary", "-out", "mtmpsig.bin", "-sign",
                   NULL, "mtmp.bin", NULL };
  char *aesc[] = { "enc", "-aes-128-cbc", "-in", "mtmp.bin", "-out", "mtmpaes.bin",
                   "-K", aesKey, "-iv", "0", "-nopad", NULL };


  unsigned char *data;  /* The raw data */
  unsigned dataSize;    /* Number of bytes in the raw data */
  unsigned p;           /* Processing index                */

  unsigned s;
  int i;



  /* Verify the number of args */
  if (argc != 5)  {
    fprintf (stderr, "usage: %s infile outfile rsa_key_file aes_key_file\n", argv[0]);
    return (-1);
  }

  /* Read the aes key */
  faes = fopen (argv[4], "r");
  if (faes == NULL)  {
    fprintf (stderr, "%s: Could not open %s for reading\n", argv[0], argv[4]);
    return (-1);
  }
  fgets (aesKey, 33, faes);

  aesKey[32] = '\0';
  fclose(faes);


  /* Open the input file, read the data */
  fin = fopen (argv[1], "r");
  if (fin == NULL)  {
    fprintf (stderr, "%s: Could not open file %s for reading\n", argv[0], argv[1]);
    return (-1);
  }

  data = readBtbl (fin, &dataSize);
  fclose (fin);


  /* Open the output file */
  fout = fopen (argv[2], "wb");
  if (fout == NULL)  {
    fprintf (stderr, "%s: Could not open file %s for writing\n", argv[0], argv[2]);
    return (-1);
  }

  /* Segment the text, and process the blocks */
  p = 0;
  while (p < dataSize)  {

    /* Extract up to 112 bytes of data, in all cases padd to get 112 */
    memset (hdr_btbl, 0, 128);
    p = extractSeg (data, dataSize, p, &hdr_btbl[16]);

    /* Prepend the decrypt header */
    addDecryptHeader (hdr_btbl);

    /* Write the output to a temporary file */
    ftmp = fopen ("mtmp.bin", "wb");
    fwrite (hdr_btbl, sizeof(unsigned char), 128, ftmp);
    fclose (ftmp);

    /* Generate the signature */
    if (fork())  {
      /* Wait for the child to terminate */
      wait();

    }  else  {

      sigc[6] = argv[3];
      if (execv ("c:\\openssl\\bin\\openssl", sigc))  {
        fprintf (stderr, "%s: execv failed: %s %s %s %s %s %s %s %s %s %s %s\n", argv[0],
                          "openssl", sigc[0], sigc[1], sigc[2], sigc[3], sigc[4], sigc[5],
                                     sigc[6], sigc[7], sigc[8], sigc[9], sigc[10]);
        exit(-1);
      }

    }

    /* Copy the header to the fill file, leave room for the i2c header, catenate the signature */
    memcpy (&sig_enc[4], hdr_btbl, 128);

    ftmp = fopen ("mtmpsig.bin", "rb");
    if (ftmp == NULL)  {
      fprintf (stderr, "%s: error reading the temp file mtmpsig.bin\n", argv[0]);
      return (-1);
    }

    /* Read the signature into scratch. It must be copied in byte reverse order */
    s = fread (scratch, sizeof(unsigned char), 128, ftmp);
    if (s != 128)  {
      fprintf (stderr, "%s: Did not read 128 bytes from mtmpsig.bin\n", argv[0]);
      return (-1);
    }
    fclose (ftmp);

    /* Copy from scratch to the encode buffer in byte reverse order */
    for (i = 0; i < 128; i++)
      sig_enc[132+i] = scratch[128-i-1];

    /* Write out the secure blocks for encoding */
    ftmp = fopen ("mtmp.bin", "wb");
    fwrite (&sig_enc[4], sizeof(unsigned char), 256, ftmp);
    fclose (ftmp);

    /* Perform an AES encryption over the block */
    if (fork())  {
      /* Wait for the child to terminate */
      wait();

    }  else  {

      if (execv ("c:\\openssl\\bin\\openssl", aesc))  {
        fprintf (stderr, "%s: execv failed on aes\n", argv[0]);
        exit (-1);
      }

    }


    /* Read the encrypted block back into memory */
    ftmp = fopen ("mtmpaes.bin", "rb");
    s = fread (&sig_enc[4], sizeof(unsigned char), 256, ftmp);
    if (s != 256)  {
      fprintf (stderr, "%s: Did not read 256 bytes from mtmpaes.bin\n", argv[0]);
      return (-1);
    }
    fclose (ftmp);

    /* Add the I2C length, and I2C checksum */
    sig_enc[0] = (260 >> 8) & 0x00ff;
    sig_enc[1] = 260 & 0x00ff;

    blockCheckSum (sig_enc, 260);

    /* Write the output */
    fwrite (sig_enc, sizeof(unsigned char), 260, fout);

  }

  fclose (fout);

  return (0);

}




    














