/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**********************************************************************************
 * FILE PURPOSE: Secure support functions
 **********************************************************************************
 * DESCRIPTION: Utilities provided for secure boot support
 **********************************************************************************/
#include <stdio.h>
#include <malloc.h>
#include "secutil.h"

/************************************************************************************
 * FUNCTION PURPOSE: Convert 4 bits into a hex value
 ************************************************************************************
 * DESCRIPTION: Converts a 4 bit value from binary to ascii
 ************************************************************************************/
unsigned char toHex (unsigned char c)
{
  if (c < 10)
    return (c + '0');

  return (c + 'a' - 10);

}

/***************************************************************************************
 * FUNCTION PURPOSE: Create the initial block header
 ***************************************************************************************
 * DESCRIPTION: The initial block header has the following structure (before encryption)
 *
 *    /-----------------------------\
 *    |      0xdec0ad70             |
 *    |    Data length(bytes)       |  This is the number of bytes following this block
 *    |      0x00000000             |
 *    |      0x00000000             |
 *    \-----------------------------/
 *
 *  A binary version is created for encryption and an ascii version is created for
 *  use as the initial vector for the data block.
 *
 ***************************************************************************************/
void headerCreate (int curSize, int sigSize, char *hdr_clr, char *ivs)
{
  unsigned char block[16];
  FILE *fout;
  int   i, j;

  fprintf (stderr, "curSize = %d, sigSize = %d\n", curSize, sigSize);

  /* The current size must be rounded up to a multiple of 16 bytes until cipher text
   * stealing is implemented */
  curSize = (curSize + 15) & ~15;

  block[0] = 0xde;
  block[1] = 0xc0;
  block[2] = 0xad;
  block[3] = 0x70;

  i = curSize + sigSize + 16;   /* The extra 16 is the size of this block */

  fprintf (stderr, "i = %d\n", i);

  block[7] = (i >> 24) & 0xff;
  block[6] = (i >> 16) & 0xff;
  block[5] = (i >>  8) & 0xff;
  block[4] = (i >>  0) & 0xff;

  block[8]  = block[9]  = block[10] = block[11] = 0;
  block[12] = block[13] = block[14] = block[15] = 0;


  fout = fopen (hdr_clr, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open output file %s\n", __FILE__, __LINE__, hdr_clr);
    return;
  }

  fwrite (block, sizeof(unsigned char), 16, fout);
  fclose (fout);


  for (i = j = 0; i < 16; i++)  {
    ivs[j++] = toHex((block[i] >> 4) & 0x0f);
    ivs[j++] = toHex((block[i] >> 0) & 0x0f);
  }

  ivs[j++] = '\0';

}

/***************************************************************************************
 * FUNCTION PURPOSE: Return the length of a file
 ***************************************************************************************
 * DESCRIPTION: The file is opened, the length is determined, and the file is closed.
 ***************************************************************************************/
int getFileLen(char *fname)
{
  FILE *fin;
  int   len;

  fin = fopen (fname, "rb");
  if (fin == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open input file %s\n", __FILE__, __LINE__, fname);
    return (NULL);
  }

  fseek (fin, 0, SEEK_END);
  len = ftell (fin);

  fclose (fin);

  return (len);

}

/***************************************************************************************
 * FUNCTION PURPOSE: Print the execv command that failed
 ***************************************************************************************
 * DESCRIPTION: The command is sent to stderr
 ***************************************************************************************/
void showString (char *inv, char *args[])
{
  int i;

  fprintf (stderr, "\t The failed command: %s ", inv);

  for (i = 0; args[i] != NULL; i++)
    fprintf (stderr, "%s ", args[i]);

  fprintf (stderr, "\n");

}

/***************************************************************************************
 * FILE PURPOSE: Read a file
 ***************************************************************************************
 * DESCRIPTION: The input file is opened and the data read as binary file.
 ***************************************************************************************/
unsigned char *readBinFile (char *fname, int *datalen)
{
  FILE *fin;
  int   len;
  unsigned char *data;

  fin = fopen (fname, "rb");
  if (fin == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open input file %s\n", __FILE__, __LINE__, fname);
    return (NULL);
  }

  fseek (fin, 0, SEEK_END);
  len = ftell (fin);

  data = malloc (len * sizeof (unsigned char));
  if (data == NULL)  {
    fprintf (stderr, "(%s : %d): Malloc failed on %d bytes\n", __FILE__, __LINE__, len);
    return (NULL);
  }

  fseek (fin, 0, SEEK_SET);
  fread (data, sizeof(unsigned char), len, fin);

  fclose (fin);

  *datalen = len;

  return (data);

}

/***************************************************************************************
 * FUNCTION PURPOSE: Catenate two files
 ***************************************************************************************
 * DESCRIPTION: Two files are catenated and the result placed into a third file
 ***************************************************************************************/
void filecat (char *f1, char *f2, char *cat)
{
  unsigned char *dat1, *dat2;
  int len1, len2;
  FILE *fout;

  dat1 = readBinFile (f1, &len1);
  dat2 = readBinFile (f2, &len2);

  fout = fopen (cat, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "(%s : %d): Failed to open output file %s\n", __FILE__, __LINE__, cat);
    return;
  }

  fwrite (dat1, sizeof (unsigned char), len1, fout);
  fwrite (dat2, sizeof (unsigned char), len2, fout);

  fclose (fout);

}


/***************************************************************************************
 * FUNCTION PURPOSE: Byte reverse an array
 ***************************************************************************************
 * DESCRIPTION: The array is reversed
 ***************************************************************************************/
void byteReverse (unsigned char *dat, int len)
{
  int i;
  unsigned char tmp;

  /* This works for both even and odd length arrays */
  for (i = 0; i < len/2; i++)  {
    tmp              = dat[i];
    dat[i]           = dat[len - 1 - i];
    dat[len - 1 - i] = tmp;
  }

}
 
/***************************************************************************************
 * FUNCTION PURPOSE: Catenate two arrays
 ***************************************************************************************
 * DESCRIPTION: Two arrays are catenated in a new array
 ***************************************************************************************/
unsigned char *catBin (unsigned char *a1, int l1, unsigned char *a2, int l2, int *l3)
{
  unsigned char *a3;
  int i, p;

  *l3 = l1 + l2;
  a3 = malloc (*l3 * sizeof (unsigned char));
  if (a3 == NULL)  {
    fprintf (stderr, "(%s : %d) : Malloc failed on size of %d\n", __FILE__, __LINE__, *l3);
    return;
  }

  for (i = p = 0; i < l1; i++)
    a3[p++] = a1[i];

  for (i = 0; i < l2; i++)
    a3[p++] = a2[i];

  return (a3);

}


/******************************************************************************
 * FUNCTION PURPOSE: Encrypt/sign a block
 ******************************************************************************
 * DESCRIPTION: The encryption trailer is appended, the data is signed, the
 *              data is encrypted, and the signature is appended to the 
 *              encrypted data/encyrption trailer
 ******************************************************************************/
unsigned char *bootSignEncrypt (unsigned char *block, int *bsize, char *rsaKeyName, char *aesKey)
{

  FILE *fout;
  
  char ivs[60];       /* Ascii version of the initial vector for the block AES */

  int i, p; 

  char *hdr_clr = "header_clr.bin";
  char *hdr_enc = "header_enc.bin";

  char *aesh_s[] = { "enc", "-aes-128-ecb", "-in", hdr_clr, "-out", hdr_enc,
                     "-K", NULL, "-iv", "0", "-nopad", NULL }; 
        
  char *blk_clr = "block_clr.bin";
  char *blk_enc = "block_enc.bin";

  char *aesb_s[] = { "enc", "-aes-128-cbc", "-in", blk_clr, "-out", blk_enc,
                     "-K", NULL, "-iv", ivs, "-nopad", "-nosalt", NULL }; 

  char *cat_clr = "hdr_blk_clr.bin";
  char *cat_enc = "hdr_blk_enc.bin";
  char *sig     = "sign.bin";

  char *rsa_s[] = { "dgst", "-sha256", "-binary", "-out", sig, "-sign", NULL, cat_clr, NULL };


  char *rsalen_s[] = { "dgst", "-sha256", "-binary", "-out", sig, "-sign", NULL, NULL, NULL };


  unsigned char *bootDat;
  int bootDatLenBytes;

  unsigned char *encDat;
  int encDatLen;

  unsigned char *sigDat;
  int sigDatLen;

  unsigned char *allDat;
  int allDatLen;

  int blockSize;
  int curSize = *bsize;

  int signLen;


  /* We need to know the size of the signature data before creating the header. Do this
   * with a little cheat. Go ahead and sign the private key file itself, then find out how long
   * the resulting signature is */

    if (fork())  {
      wait();  /* wait for the child to terminate */

    }  else  {

      rsalen_s[6] = rsaKeyName;  /* The RSA private key file */
      rsalen_s[7] = rsaKeyName;  /* The input file */

      if (execv ("c:\\openssl\\bin\\openssl", rsalen_s))  {
        fprintf (stderr, "Openssl failed to sign the private key (to determine the sig length)\n");

        showString ("openssl", rsalen_s);
        return (NULL);
      }

    }


  signLen = getFileLen(sig);


  /* Create the initial header. The binary file in hdr_clr is created. An ascii
   * version is returned in ivs for use as an initialization vector */
  headerCreate (curSize, signLen, hdr_clr, ivs);


  /* AES encrypt this header block using a 0 initialization vector */
  if (fork())  {
    wait();  /* wait for the chid to terminate */

  }  else  {

    aesh_s[7] = aesKey;  /* The AES Key */
    if (execv ("c:\\openssl\\bin\\openssl", aesh_s))  {
      fprintf (stderr, "Openssl failed to AES encrypt header block failed\n");

      showString ("openssl", aesh_s);
      return (NULL);
    }
  }


  /* Create a temporary file containing the current block */
  fout = fopen (blk_clr, "wb");
  if (fout == NULL)  {
    fprintf (stderr, "Failed to create output file %s\n", blk_clr);
    return (NULL);
  }

  fwrite (block, sizeof(unsigned char), curSize, fout);
  fclose (fout);


  /* AES encrypt the data */
  if (fork())  {
    wait();  /* wait for the child to terminate */

  }  else  {

    aesb_s[7] = aesKey;  /* The AES Key */

    if (execv ("c:\\openssl\\bin\\openssl", aesb_s))  {
      fprintf (stderr, "Openssl failed to AES encrypt the data block\n");

      showString ("openssl", aesb_s);
      return (NULL);
    }

  }


  /* Now catenate the two AES encoded files */
  filecat (blk_enc, hdr_enc, cat_enc);


  /* Catenate the two unencoded files */
  filecat (blk_clr, hdr_clr, cat_clr);


  /* Sign the clear text catenated block */
  if (fork())  {
    wait();  /* wait for the child to terminate */

  }  else  {

    rsa_s[6] = rsaKeyName;  /* The RSA private key file */

    if (execv ("c:\\openssl\\bin\\openssl", rsa_s))  {
      fprintf (stderr, "Openssl failed to sign the data block\n");

      showString ("openssl", rsa_s);
      return (NULL);
    }

  }

  /* Swap the signature bytes, and append these to the AES encrypted data */
  encDat = readBinFile (cat_enc, &encDatLen);
  sigDat = readBinFile (sig, &sigDatLen);

  byteReverse (sigDat, sigDatLen); 

  /* Form the complete encrptyed/signed block. Return the total size in bsize */
  allDat = catBin (encDat, encDatLen, sigDat, sigDatLen, bsize);


  free (sigDat);
  free (encDat);

  return (allDat);

}








