/* SPDX-License-Identifier: BSD-3-Clause */
/*
 * @file eeprom_cmd.c
 * @copyright Copyright (c) 2022 TQ-Systems GmbH <license@tq-group.com>, D-82229 Seefeld, Germany.
 * @author Michael Bernhardt
 *
 * @date 2022-05-18
 *
 * -----------------------------------------------------------------------------
 * @brief This file contains the implementation of the EEPROM M24C64.
 *
 */
/*******************************************************************************
 * local includes
 ******************************************************************************/

/* runtime */
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
/* project */
#include <drivers/i2c.h>
#include <kernel/dpl/DebugP.h>
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include "ti_board_open_close.h"
#include "utilities.h"
/* own */
#include "eeprom_cmd.h"

/*******************************************************************************
 * local defines
 ******************************************************************************/

#define DECIMAL_BASE       (10U)
#define HEX_BASE           (16U)
#define MAX_INPUT_PARAM    (4U)
#define EEPROM_PAGE_SIZE   (256U)
#define EEPROM_WR_BUF_SIZE (2U + EEPROM_PAGE_SIZE)
#define INDEX_INSTANCE     (0U)
#define INDEX_OFFSET       (1U)
#define INDEX_ACCESS       (2U)
#define INDEX_DATA         (3U)

#define TQ_EEPROM_ADD      (0x50)
#define CUST_EEPROM_ADD    (0x54)

#define MAX_DATA           (1U)

#define MAC_OFFSET        (0x100U)
uint8_t dataBuf[6] = {0};
uint8_t MAC[] = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06};

/*******************************************************************************
 * local macros
 ******************************************************************************/


/*******************************************************************************
 * local typedefs
 ******************************************************************************/

typedef struct
{
    uint8_t instance;
    uint32_t offset;
    uint8_t* data;
    uint8_t length;
} eeprom_t;

/*******************************************************************************
 * local static data
 ******************************************************************************/

static const uint8_t eepromInstance[] = {TQ_EEPROM_ADD, CUST_EEPROM_ADD};

/*******************************************************************************
 * forward declarations
 ******************************************************************************/

static bool eepromRead(eeprom_t* p_eepromData);
static bool eepromWrite(eeprom_t* eepromData);

/*******************************************************************************
 * local static functions
 ******************************************************************************/

/**
 * @brief This function reads data from the EEPROM.
 *
 * @param p_eepromData Pointer to EEPROM data struct.
 * @return success
 */
static bool eepromRead(eeprom_t* p_eepromData)
{
    bool            success        = false;
    int32_t         status         = 0;
    I2C_Handle      i2cHandle      = NULL;
    I2C_Transaction i2cTransaction = {0};
    uint8_t         txBuffer[2]    = {0};

    DebugP_log("[I2C] Read EEPROM instance: %d.\r\n", p_eepromData->instance);

    if ((p_eepromData->instance < ARRAY_SIZE(eepromInstance)) && (p_eepromData->data != NULL))
    {
        txBuffer[0U] = (p_eepromData->offset & 0xFF00U) >> 8U;
        txBuffer[1U] = (p_eepromData->offset & 0x00FFU);

        Drivers_i2cOpen();
        i2cHandle = I2C_getHandle(CONFIG_I2C0);

        /* Set default transaction parameters */
        I2C_Transaction_init(&i2cTransaction);

        /* Override with required transaction parameters */
        i2cTransaction.writeBuf      = txBuffer;
        i2cTransaction.writeCount    = ARRAY_SIZE(txBuffer);
        i2cTransaction.readBuf       = p_eepromData->data;
        i2cTransaction.readCount     = p_eepromData->length;
        i2cTransaction.targetAddress = eepromInstance[p_eepromData->instance];

        status = I2C_transfer(i2cHandle, &i2cTransaction);

        if (status != SystemP_SUCCESS)
        {
            DebugP_logError("[EEPROM] Read failed %d for instance %d !!!\r\n", status, p_eepromData->instance);
            success = false;
        }
        else
        {
            DebugP_log("[EEPROM] Read data 0x%02X.\r\n", p_eepromData->data);
            success = true;
        }

        Board_driversClose();
        Drivers_i2cClose();
    }

    return success;
}

/**
 * @brief This function writes data to the EEPROM.
 *
 * @param p_eepromData Pointer to EEPROM data struct.
 * @return success
 */
static bool eepromWrite(eeprom_t* p_eepromData)
{
    bool            success                      = false;
    int32_t         status                       = 0;
    I2C_Handle      i2cHandle                    = NULL;
    I2C_Transaction i2cTransaction               = {0};
    uint8_t         txBuffer[EEPROM_WR_BUF_SIZE] = {0};


    if ((p_eepromData->instance < ARRAY_SIZE(eepromInstance)) && (p_eepromData->data != NULL))
    {
        txBuffer[0U] = (p_eepromData->offset & 0xFF00U) >> 8U;
        txBuffer[1U] = (p_eepromData->offset & 0x00FFU);

        memcpy(&txBuffer[2U], p_eepromData->data, p_eepromData->length);

        Drivers_i2cOpen();
        i2cHandle = I2C_getHandle(CONFIG_I2C0);

        /* Set default transaction parameters */
        I2C_Transaction_init(&i2cTransaction);

        /* Override with required transaction parameters */
        i2cTransaction.writeBuf     = txBuffer;
        i2cTransaction.writeCount   = 2 + p_eepromData->length;
        i2cTransaction.targetAddress = eepromInstance[p_eepromData->instance];

        status = I2C_transfer(i2cHandle, &i2cTransaction);

        if (status != SystemP_SUCCESS)
        {
            DebugP_logError("[EEPROM] Write failure %d for instance %d !!!\r\n", status, i2cTransaction.targetAddress);
        }
        else
        {
            DebugP_log("[EEPROM] write data 0x%02X.\r\n", *p_eepromData->data);
            success = true;
        }

        Board_driversClose();
        Drivers_i2cClose();
    }

    return success;
}

/*******************************************************************************
 * global functions
 ******************************************************************************/

void setMacAdd(void)
{
    eeprom_t eepromData  = {0};
    eepromData.data   = MAC;
    eepromData.length = ARRAY_SIZE(MAC);
    eepromData.instance = 1;
    eepromData.offset = MAC_OFFSET;

    if( true == eepromWrite(&eepromData))
    {
        printf("success");
    }
    else
    {
        printf("Failed");
    }
}

void readMacAdd(void)
{
    eeprom_t eepromData  = {0};
    eepromData.data   = dataBuf;
    eepromData.length = ARRAY_SIZE(dataBuf);
    eepromData.instance = 1;
    eepromData.offset = MAC_OFFSET;

    if (true == eepromRead(&eepromData))
    {
        printf("MAC: %02X:%02X:%02X:%02X:%02X:%02X", dataBuf[0], dataBuf[1], dataBuf[2], dataBuf[3], dataBuf[4], dataBuf[5]);
    }
    else
    {
      printf("Failed");
    }

}


/*[EOF]************************************************************************/
