/***************************************************************
 * @file       serial_test
 * @brief      对串口进行收发压力测试
 * @author     dengxing
 * @version    v1.0
 * @date       20240626 11:19
 **************************************************************/

#include "./config/uart_config.h"
#include <sys/time.h>

//自发自收才能测延时
//#define TEST_RW_DELAY 1
#define SEND_PKG_SIZE 25000 
//#define SEND_PKG_SIZE 4800000000 //测试48小时左右
#define ERR_MAX_TIMES 1000

enum {
	UART_TYPE_NORMAL = 0,
	UART_TYPE_EXARPCI,
	UART_TYPE_AX9100A,
	UART_TYPE_MAX,
};

struct series_config{
	char *path;
	unsigned char uart_type;
	unsigned int baudrate;
	unsigned long long int pkg_size;
	#ifdef TEST_RW_DELAY
	unsigned char link_port;
	#endif
};

struct series_config s_conf[] = {
	#if 1
	{
		.path = "/dev/ttyS1",
		.uart_type = UART_TYPE_NORMAL,
		.baudrate = 115200,
		.pkg_size = SEND_PKG_SIZE,
		#ifdef TEST_RW_DELAY
		.link_port = 1, //是哪个端口的tx接的本端口的rx
		#endif
	},
	#endif
	#if 1
	{
		.path = "/dev/ttyS2",
		.uart_type = UART_TYPE_NORMAL,
		.baudrate = 460800,
		.pkg_size = SEND_PKG_SIZE,
		#ifdef TEST_RW_DELAY
		.link_port = 0,
		#endif
	},
	#endif
	#if 1
	{
		.path = "/dev/ttyS3",
		.uart_type = UART_TYPE_NORMAL,
		.baudrate = 115200,
		.pkg_size = SEND_PKG_SIZE,
		#ifdef TEST_RW_DELAY
		.link_port = 0,
		#endif
	},
	#endif
	#if 1
	{
		.path = "/dev/ttyS4",
		.uart_type = UART_TYPE_NORMAL,
		.baudrate = 115200,
		.pkg_size = SEND_PKG_SIZE,
		#ifdef TEST_RW_DELAY
		.link_port = 0,
		#endif
	},
	#endif
	#if 1
	{
		.path = "/dev/ttyS5",
		.uart_type = UART_TYPE_NORMAL,
		.baudrate = 115200,
		.pkg_size = SEND_PKG_SIZE,
		#ifdef TEST_RW_DELAY
		.link_port = 0,
		#endif
	},
	#endif
};
#define SERIAL_DEVICE_MAX (sizeof(s_conf)/sizeof(struct series_config))

//#define PRINT_ERR_BUF 1

#ifdef TEST_RW_DELAY
#define RW_DELAY_SIZE 10000
#endif

struct series_info {
	int index;
	int fd;
	pthread_t tid_recv,tid_send;
	struct series_config *conf;
	//测延时
	#ifdef TEST_RW_DELAY
	struct timeval write_time,read_time;
	unsigned long long int sum_rw_time, max_rw_time;
	unsigned long long int packet_rw[RW_DELAY_SIZE];
	#endif
}s_info[SERIAL_DEVICE_MAX];

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#define SEND_STA 	0x55 // 包头
#define SEND_LEN_1	0x07 // 包长1 7字节数据
#define SEND_LEN_2	0x05 // 包长1 5字节数据
#define SEND_CHK_1	0x85 // 校验和1
#define SEND_CHK_2	0x89 // 校验和2
#define SEND_END	0xAA // 包尾
#define SEND_ID_3	0x04
#define SEND_ID_2	0x03
#define SEND_ID_1	0x02
#define SEND_ID_0	0x01

//所有数据包总和字节数不超过这么多
#define PACKET_MAX_SIZE 256
#define PACKET_NUM 2

struct packet_info {
	unsigned char packet[PACKET_MAX_SIZE];
	unsigned int len[PACKET_NUM];
	unsigned int pre_sum[PACKET_NUM]; //前n项和
};

struct packet_info p_info = {

	.packet = {
		//第一个包
		SEND_STA,SEND_ID_3,SEND_ID_2,SEND_ID_1,SEND_ID_0,SEND_LEN_1,0x10,0x12,0x12,0x14,0x14,0x15,0x16,SEND_CHK_1,SEND_END,
		//第二个包
		SEND_STA,SEND_ID_3,SEND_ID_2,SEND_ID_1,SEND_ID_0,SEND_LEN_2,0x17,0x22,0x19,0x20,0x21,SEND_CHK_2,SEND_END,
	},
	.len = {15, 13},
};

static void init_pkg()
{
	int i, sum = 0;
	for(i = 0; i < PACKET_NUM; i++) {
		sum += p_info.len[i];
		p_info.pre_sum[i] = sum;
	}
}

void *receive_thread(void *arg) {
	struct series_info *series = (struct series_info *)arg;
    int fd = series->fd;
    unsigned char rx_buffer[256]={0};
	int i;
	unsigned long long int packect_index = 0;
	unsigned long long int recv_cnt = 0;
	//unsigned long long int need_recv = series->conf->pkg_size/2*(Packet1_len + Packet2_len)+(series->conf->pkg_size%2)*Packet1_len;
	int pk_sum = p_info.pre_sum[PACKET_NUM - 1];
	
	unsigned long long int need_recv = series->conf->pkg_size / PACKET_NUM * pk_sum;
	if(series->conf->pkg_size % PACKET_NUM) {
		need_recv += p_info.pre_sum[(series->conf->pkg_size % PACKET_NUM) - 1];
	}

	time_t get_time = 0;
	time(&get_time); //时间
	
	//printf( "rec start time %ld\n", get_time);
	
	fd_set read_fds;
	FD_ZERO(&read_fds);
	FD_SET(fd, &read_fds);

	int dxtmpflag = 1;
    while (recv_cnt < need_recv) {
		fd_set tmp_fds = read_fds;
		if (select(fd + 1, &tmp_fds, NULL, NULL, NULL) == -1) {
            perror("Select failed");
            pthread_exit(NULL);
        }
		if (!FD_ISSET(fd, &tmp_fds))
			continue;
		
        int ret = read(fd, rx_buffer, sizeof(rx_buffer) - 1);
        if (ret < 0) {
			//continue;
			printf("read error rx_buffer[%lld]\n",recv_cnt);
			pthread_exit(NULL);
        }
		
		//check
		for(i = 0; i < ret && dxtmpflag; i++){
			if(p_info.packet[(recv_cnt+i) % pk_sum] != rx_buffer[i]){
				//dump_debug_msg(fd);
				printf("%d read error timestamp %ld recv_cnt %lld rx_buffer[%d]=0x%02x Packet[%lld] is 0x%x\n", series->index, time(&get_time), recv_cnt, i, rx_buffer[i], (recv_cnt+i) % pk_sum, p_info.packet[(recv_cnt+i) % pk_sum]);
				#ifdef PRINT_ERR_BUF
				printf("rx:");
				for(i = 0; i < ret; i++){
					printf("0x%x, ",rx_buffer[i]);
				}
				printf("\n");
				#endif
				//pthread_exit(NULL);
				dxtmpflag = 0;
			}
			#ifdef TEST_RW_DELAY
			int k = 0;
			for(k = 0; k < PACKET_NUM; k++) {
				//判断是否为某个包的结尾
				if((recv_cnt+i) % pk_sum == p_info.pre_sum[k] - 1)
					break;
			}
			
			if(k == PACKET_NUM)
				continue;

			gettimeofday(&series->read_time, NULL);
			//获取对应tx编号
			int index = series->conf->link_port;

			unsigned long long int rw_time = series->read_time.tv_sec * 1000 * 1000 + series->read_time.tv_usec - s_info[index].packet_rw[packect_index % RW_DELAY_SIZE];
			series->max_rw_time = rw_time > series->max_rw_time ? rw_time : series->max_rw_time;
			series->sum_rw_time += rw_time;
			
			//packet_rw默认为0, 如果写时间戳为1,说明还没写，发生了错误
			if(s_info[index].packet_rw[packect_index % RW_DELAY_SIZE] == 1) {
					printf("waring: packet_rw err, rw_time %lld\n", rw_time);
			}
			//将使用过的重置为1
			s_info[index].packet_rw[packect_index % RW_DELAY_SIZE] = 1;
			packect_index++;

			#endif
			//printf("check ok\n");
		}
		recv_cnt += ret;
    }
	time(&get_time); //时间
	printf("%d rec end time %ld receive %lld\n", series->index, get_time, recv_cnt);
	#ifdef TEST_RW_DELAY
	long long int avg = series->sum_rw_time / packect_index;
	printf("========== TEST_RW_DELAY %d max_rw_time %lld(us) avg_rw_time %lld(us) ==========\n ",series->index ,series->max_rw_time ,avg);
	#endif
    pthread_exit(NULL);
}

void *send_thread(void *arg) {

	struct series_info *series = (struct series_info *)arg;
    int fd = series->fd;
	int sendbuf_flag = 0, curbuflen = 0;
	unsigned char * curbufaddr = NULL;//当前需要发送的包起始地址
	unsigned long long int packet_id = 0;
	int ret = 0;
	
	//记录时间戳
	unsigned long long int start_time = 0;
	time_t get_time = 0;	
	time(&get_time); //时间
	start_time = get_time;
	//printf("send start timea1 %ld dx\n", get_time);

	//#define TEST_DELAY_2 0

	long long int err_sum = 0;
    while (packet_id < series->conf->pkg_size) {

		//设置要发送的包
		sendbuf_flag %= PACKET_NUM;
		
		curbuflen = p_info.len[sendbuf_flag];
		if(sendbuf_flag == 0) {
			curbufaddr = &p_info.packet[0];
		} else {
			curbufaddr = &p_info.packet[p_info.pre_sum[sendbuf_flag - 1]];
		}
		sendbuf_flag ++;

		int sent_cnt=0;
		int err_cnt=0;
		
		#ifdef TEST_RW_DELAY
		gettimeofday(&series->write_time, NULL);
		if(series->packet_rw[packet_id % RW_DELAY_SIZE] > 1) {
			printf("waring: packet_rw some wrong\n");
		}
		series->packet_rw[packet_id % RW_DELAY_SIZE] = series->write_time.tv_sec * 1000 * 1000 + series->write_time.tv_usec;
		#endif

		while(sent_cnt < curbuflen){
			int need_send = curbuflen - sent_cnt;
			ret = write(fd, &curbufaddr[sent_cnt], need_send); 
			if(ret < 0){
				err_cnt++;
				err_sum++;
				//printf("err_cnt %d\n", err_cnt);
				if(err_cnt >= ERR_MAX_TIMES) {
					printf("write error curbufaddr[%d] packet_id %lld send 0x%x need %d\n", sent_cnt, packet_id, curbufaddr[sent_cnt],need_send);
					pthread_exit(NULL);
				}
				usleep(10000);
				continue;
			}
			sent_cnt += ret;
			#if TEST_DELAY_2
			usleep(TEST_DELAY_2);
			#endif
		}
		//usleep(100);
		packet_id ++;
	}

	time(&get_time); //时间
	printf("%d send end time %ld pack %lld error times %lld spend %lld s\n", series->index, get_time, packet_id, err_sum, get_time-start_time);	
	//fclose(Record_fd);
    pthread_exit(NULL);
}

static int open_port(int type, char *device, int baudrate)
{
	switch(type){
		case UART_TYPE_NORMAL:
			return open_serial_port(device, baudrate);
		case UART_TYPE_EXARPCI: 
			return open_exar_pci_port(device);
		case UART_TYPE_AX9100A:
			return open_ax9100a_pci_port(device, baudrate);
		default:
			return open_serial_port(device, baudrate);
	}
}

int main() {

	int i, init_idx, ret = 0;
	
	do {
	
		//打开节点
		for(init_idx = 0; init_idx < SERIAL_DEVICE_MAX; init_idx++) {
		
			s_info[init_idx].fd = open_port(s_conf[init_idx].uart_type, s_conf[init_idx].path, s_conf[init_idx].baudrate);
			s_info[init_idx].index = init_idx;
			s_info[init_idx].conf = &s_conf[init_idx];
			
			if (s_info[init_idx].fd < 0) {
				printf("open err:%s\n", s_conf[init_idx].path);
				ret = s_info[init_idx].fd;
				break;
			}
			printf("%s %d\n", s_conf[init_idx].path, init_idx);
		}
		
		//初始化要发送的packet
		init_pkg();

		//创建接收线程
		for(i = 0; i < SERIAL_DEVICE_MAX; i++) {
		
			ret = pthread_create(&s_info[i].tid_recv, NULL, receive_thread, &s_info[i]);
			if (ret != 0) {
				printf("Failed to create receive thread for serial %d\n", i);
				break;
			}
		}
		
		sleep(3);
		
		//创建发送线程
		for(i = 0; i < SERIAL_DEVICE_MAX; i++) {
			ret = pthread_create(&s_info[i].tid_send, NULL, send_thread, &s_info[i]);
			if (ret != 0) {
				perror("Failed to create send thread for serial 1");
				break;
			}
		}

		// 等待线程结束
		for(i = 0; i < SERIAL_DEVICE_MAX; i++) {
			pthread_join(s_info[i].tid_send, NULL);
			pthread_join(s_info[i].tid_recv, NULL);
			
		}
	}while(0);

	for(i = 0; i < init_idx; i++) {
		close(s_info[i].fd);
	}
	
    return ret;
}