/*
 *
 * Copyright (c) 2020 Texas Instruments Incorporated
 *
 * All rights reserved not granted herein.
 *
 * Limited License.
 *
 * Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
 * license under copyrights and patents it now or hereafter owns or controls to make,
 * have made, use, import, offer to sell and sell ("Utilize") this software subject to the
 * terms herein.  With respect to the foregoing patent license, such license is granted
 * solely to the extent that any such patent is necessary to Utilize the software alone.
 * The patent license shall not apply to any combinations which include this software,
 * other than combinations with devices manufactured by or for TI ("TI Devices").
 * No hardware patent is licensed hereunder.
 *
 * Redistributions must preserve existing copyright notices and reproduce this license
 * (including the above copyright notice and the disclaimer and (if applicable) source
 * code license limitations below) in the documentation and/or other materials provided
 * with the distribution
 *
 * Redistribution and use in binary form, without modification, are permitted provided
 * that the following conditions are met:
 *
 * *       No reverse engineering, decompilation, or disassembly of this software is
 * permitted with respect to any software provided in binary form.
 *
 * *       any redistribution and use are licensed by TI for use only with TI Devices.
 *
 * *       Nothing shall obligate TI to provide you with source code for the software
 * licensed and provided to you in object code.
 *
 * If software source code is provided to you, modification and redistribution of the
 * source code are permitted provided that the following conditions are met:
 *
 * *       any redistribution and use of the source code, including any resulting derivative
 * works, are licensed by TI for use only with TI Devices.
 *
 * *       any redistribution and use of any object code compiled from the source code
 * and any resulting derivative works, are licensed by TI for use only with TI Devices.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of its suppliers
 *
 * may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * DISCLAIMER.
 *
 * THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <utils/draw2d/include/draw2d.h>
#include <utils/perf_stats/include/app_perf_stats.h>
#include <utils/console_io/include/app_get.h>
#include <utils/grpx/include/app_grpx.h>
#include <VX/vx_khr_pipelining.h>

#ifdef APP_TIDL_TRACE_DUMP
#include <tivx_utils_tidl_trace.h>
#endif
#include "app_common.h"
#include <TI/tivx.h>
#include <TI/j7.h>
#include <utils/iss/include/app_iss.h>
/* #define WRITE_INTERMEDIATE_OUTPUTS */

#ifndef x86_64
#define APP_ENABLE_PIPELINE_FLOW
#endif

static char *get_test_file_path()
{
    char *tivxPlatformGetEnv(char *env_var);

#if defined(SYSBIOS)
    return tivxPlatformGetEnv("VX_TEST_DATA_PATH");
#else
    return getenv("VX_TEST_DATA_PATH");
#endif
}

static char * ct_alloc_mem (uint32_t size)
{
    return (appMemAlloc(APP_MEM_HEAP_DDR, size, 0));
}

static void ct_free_mem(char *buf, uint32_t size)
{
    appMemFree(APP_MEM_HEAP_DDR, buf, size);
}

static int32_t ct_read_raw_image(tivx_raw_image image, const char* fileName, uint16_t file_byte_pack, uint16_t downshift_bits)
{
    int32_t status = 0;
    FILE* f = 0;
    size_t sz;
    char* buf = 0;
    char file[APP_MAX_FILE_PATH];
    char failsafe_test_data_path[3] = "./";
    char * test_data_path = get_test_file_path();
    struct stat s;

    if (!fileName)
    {
        printf("Image file name not specified\n");
        return -1;
    }

    if(NULL == test_data_path)
    {
        printf("Test data path is NULL. Defaulting to current folder \n");
        test_data_path = failsafe_test_data_path;
    }

    if (stat(test_data_path, &s))
    {
        printf("Test data path %s does not exist. Defaulting to current folder \n", test_data_path);
        test_data_path = failsafe_test_data_path;
    }

    sz = snprintf(file, APP_MAX_FILE_PATH, "%s/%s", test_data_path, fileName);

    f = fopen(file, "rb");
    if (!f)
    {
        printf("Cannot open File %s\n", file);
        return -1;
    }

    fseek(f, 0, SEEK_END);
    sz = ftell(f);
    if( sz > 0 )
    {
        buf = (char*)ct_alloc_mem(sz);
        if (NULL != buf)
        {
            fseek(f, 0, SEEK_SET);
            if( fread(buf, 1, sz, f) == sz )
            {
                vx_uint32 width, height;
                vx_imagepatch_addressing_t image_addr;
                vx_rectangle_t rect;
                vx_map_id map_id;
                void *data_ptr;
                vx_uint32 num_bytes = 1;
                tivx_raw_image_format_t format[3];

                tivxQueryRawImage(image, TIVX_RAW_IMAGE_WIDTH, &width, sizeof(vx_uint32));
                tivxQueryRawImage(image, TIVX_RAW_IMAGE_HEIGHT, &height, sizeof(vx_uint32));
                tivxQueryRawImage(image, TIVX_RAW_IMAGE_FORMAT, &format, sizeof(format));

                if( format[0].pixel_container == TIVX_RAW_IMAGE_16_BIT )
                {
                    num_bytes = 2;
                }
                else if( format[0].pixel_container == TIVX_RAW_IMAGE_8_BIT )
                {
                    num_bytes = 1;
                }
                else if( format[0].pixel_container == TIVX_RAW_IMAGE_P12_BIT )
                {
                    num_bytes = 0;
                }

                rect.start_x = 0;
                rect.start_y = 0;
                rect.end_x = width;
                rect.end_y = height;

                tivxMapRawImagePatch(image,
                                     &rect,
                                     0,
                                     &map_id,
                                     &image_addr,
                                     &data_ptr,
                                     VX_WRITE_ONLY,
                                     VX_MEMORY_TYPE_HOST,
                                     TIVX_RAW_IMAGE_PIXEL_BUFFER
                );

                if((file_byte_pack == num_bytes) && (downshift_bits == 0))
                {
                    int i;
                    uint8_t *dst = data_ptr;
                    uint8_t *src = (uint8_t*)buf;
                    for(i = 0; i < height; i++)
                    {
                        memcpy((void*)&dst[image_addr.stride_y*i], (void*)&src[width*num_bytes*i], width*num_bytes);
                    }
                }
                else if((file_byte_pack == 2) && (num_bytes == 2))
                {
                    int i, j;
                    uint16_t *dst = data_ptr;
                    uint16_t *src = (uint16_t*)buf;
                    for(j = 0; j < height; j++)
                    {
                        for(i = 0; i < width; i++)
                        {
                            dst[i] = src[i] >> downshift_bits;
                        }
                        dst += image_addr.stride_y/2;
                        src += width;
                    }
                }
                else if((file_byte_pack == 2) && (num_bytes == 1))
                {
                    int i, j;
                    uint8_t *dst = data_ptr;
                    uint16_t *src = (uint16_t*)buf;
                    for(j = 0; j < height; j++)
                    {
                        for(i = 0; i < width; i++)
                        {
                            dst[i] = src[i];
                        }
                        dst += image_addr.stride_y;
                        src += width;
                    }
                }
                else
                {
                    if(file_byte_pack != num_bytes)
                    {
                        printf("ct_read_raw_image: size mismatch!!\n");
                        fclose(f);
                        return -1;
                    }
                }
                tivxUnmapRawImagePatch(image, map_id);
            }
        }
    }

    ct_free_mem(buf, sz);
    fclose(f);

    return (status);
}
vx_int32 write_output_image_fp(FILE * fp, vx_image out_image)
{
    vx_uint32 width, height;
    vx_df_image df;
    vx_imagepatch_addressing_t image_addr;
    vx_rectangle_t rect;
    vx_map_id map_id1, map_id2;
    void *data_ptr1 = NULL;
    void *data_ptr2 = NULL;
    vx_uint32 num_bytes_per_4pixels;
    vx_uint32 num_luma_bytes_written_to_file=0;
    vx_uint32 num_chroma_bytes_written_to_file=0;
    vx_uint32 num_bytes_written_to_file=0;
    vx_uint32 imgaddr_width, imgaddr_height, imgaddr_stride;
    int i;

    vxQueryImage(out_image, VX_IMAGE_WIDTH, &width, sizeof(vx_uint32));
    vxQueryImage(out_image, VX_IMAGE_HEIGHT, &height, sizeof(vx_uint32));
    vxQueryImage(out_image, VX_IMAGE_FORMAT, &df, sizeof(vx_df_image));


    if(VX_DF_IMAGE_NV12 == df)
    {
        num_bytes_per_4pixels = 4;
    }
    else if(TIVX_DF_IMAGE_NV12_P12 == df)
    {
        num_bytes_per_4pixels = 6;
    }
    else
    {
        num_bytes_per_4pixels = 8;
    }

    rect.start_x = 0;
    rect.start_y = 0;
    rect.end_x = width;
    rect.end_y = height;

    vxMapImagePatch(out_image,
                    &rect,
                    0,
                    &map_id1,
                    &image_addr,
                    &data_ptr1,
                    VX_WRITE_ONLY,
                    VX_MEMORY_TYPE_HOST,
                    VX_NOGAP_X
    );

    if(!data_ptr1)
    {
        printf("data_ptr1 is NULL \n");
        return -1;
    }

    imgaddr_width  = image_addr.dim_x;
    imgaddr_height = image_addr.dim_y;
    imgaddr_stride = image_addr.stride_y;
    printf("imgaddr_width = %d \n", imgaddr_width);
    printf("imgaddr_height = %d \n", imgaddr_height);
    printf("imgaddr_stride = %d \n", imgaddr_stride);
    printf("width = %d \n", width);
    printf("height = %d \n", height);

    num_luma_bytes_written_to_file = 0;

    for(i=0;i<height;i++)
    {
        num_luma_bytes_written_to_file  += fwrite(data_ptr1, 1, width*num_bytes_per_4pixels/4, fp);
        data_ptr1 += imgaddr_stride;
    }
    vxUnmapImagePatch(out_image, map_id1);

    fflush(fp);


    if(VX_DF_IMAGE_NV12 == df || TIVX_DF_IMAGE_NV12_P12 == df)
    {
        vxMapImagePatch(out_image,
                        &rect,
                        1,
                        &map_id2,
                        &image_addr,
                        &data_ptr2,
                        VX_WRITE_ONLY,
                        VX_MEMORY_TYPE_HOST,
                        VX_NOGAP_X
        );

        if(!data_ptr2)
        {
            printf("data_ptr2 is NULL \n");
            return -1;
        }

        imgaddr_width  = image_addr.dim_x;
        imgaddr_height = image_addr.dim_y;
        imgaddr_stride = image_addr.stride_y;

        num_chroma_bytes_written_to_file = 0;
        for(i=0;i<imgaddr_height/2;i++)
        {
            num_chroma_bytes_written_to_file  += fwrite(data_ptr2, 1, imgaddr_width*num_bytes_per_4pixels/4, fp);
            data_ptr2 += imgaddr_stride;
        }

        fflush(fp);

        vxUnmapImagePatch(out_image, map_id2);
    }

    num_bytes_written_to_file = num_luma_bytes_written_to_file + num_chroma_bytes_written_to_file;
    printf("Written %d bytes \n", num_bytes_written_to_file);

    return num_bytes_written_to_file;
}

vx_int32 write_output_image_nv12_8bit(char * file_name, vx_image out_nv12)
{
    FILE * fp = fopen(file_name, "wb");
    if(!fp)
    {
        APP_PRINTF("Unable to open file %s\n", file_name);
        return -1;
    }
    vx_uint32 len1 = write_output_image_fp(fp, out_nv12);
    fclose(fp);
    APP_PRINTF("%d bytes written to %s\n", len1, file_name);
    return len1;
}
vx_int32 write_h3a_image(char * file_name, vx_user_data_object out_h3a)
{
    FILE * fp = fopen(file_name, "wb");
    tivx_h3a_data_t *h3a_data = NULL;
    vx_map_id h3a_buf_map_id;
    vx_uint8 *h3a_payload = NULL;
    vx_uint32 h3a_size = 0;
    if(!fp)
    {
        APP_PRINTF("Unable to open file %s\n", file_name);
        return -1;
    }

    vxMapUserDataObject(
            out_h3a,
            0,
            sizeof(tivx_h3a_data_t),
            &h3a_buf_map_id,
            (void **)&h3a_data,
            VX_READ_ONLY,
            VX_MEMORY_TYPE_HOST,
            0
    );

    h3a_payload = h3a_data->data;// + 12; /*C model 12-byte header*/
    h3a_size = fwrite(h3a_payload, 1, TIVX_VPAC_VISS_MAX_H3A_STAT_NUMBYTES, fp);
    APP_PRINTF("%d bytes saved from H3A output buffer \n", h3a_size);
    fflush(fp);
    fclose(fp);
    vxUnmapUserDataObject(out_h3a, h3a_buf_map_id);

    return h3a_size;
}
vx_status app_viss(vx_int32 argc, vx_char* argv[])
{
    printf("entering inside viss\n");
    vx_status status = VX_SUCCESS;
    int32_t load,time;
    vx_user_data_object configuration=NULL;
    vx_user_data_object ae_awb_result=NULL;
    tivx_raw_image raw = NULL;
    vx_context context;
    printf("read raw in viss\n");
    vx_image y12 = NULL, uv12_c1 = NULL, y8_r8_c2 = NULL, uv8_g8_c3 = NULL, s8_b8_c4 = NULL;
    vx_distribution histogram = NULL;
    vx_user_data_object h3a_aew_af = NULL;
    /* Dcc objects */
    vx_user_data_object dcc_param_viss = NULL;
    const vx_char dcc_viss_user_data_object_name[] = "dcc_viss";
    vx_size dcc_buff_size = 1;
    vx_map_id dcc_viss_buf_map_id;
    uint8_t * dcc_viss_buf;
    int32_t dcc_status, iter, sleep_iter, load_iter;
    uint32_t sensor_dcc_id;
    uint32_t sensor_dcc_mode;
    char *sensor_name = NULL;
    char *file_name = NULL;
    uint16_t downshift_bits;
    uint64_t start, end;

    tivx_vpac_viss_params_t params;
    tivx_ae_awb_params_t ae_awb_params;

    vx_graph graph = 0;
    vx_node node = 0;
    printf("debug1");

    tivx_raw_image_create_params_t raw_params;

    raw_params.width = 1936;//1280;
    raw_params.height = 1096; //944;//1096;
    raw_params.meta_height_after = 4;
    sensor_dcc_id = 390;
    sensor_name = SENSOR_SONY_IMX390_UB953_D3;
    sensor_dcc_mode = 0;
    file_name = "psdkra/app_single_cam/IMX390_001/input2.raw";
    printf("debug2");
    downshift_bits = 0;
    char yuv_image_fname[100];
    char h3a_image_fname[100];

    raw_params.num_exposures = 1;
    raw_params.line_interleaved = vx_false_e;
    raw_params.format[0].pixel_container = TIVX_RAW_IMAGE_16_BIT;
    raw_params.format[0].msb = 11;
    raw_params.meta_height_before = 0;

    if (vx_true_e == tivxIsTargetEnabled(TIVX_TARGET_VPAC_VISS1))
    {
        vx_uint32 width, height, i;

        /* Create OpenVx Context */
        context = vxCreateContext();
        if (NULL == context)
        {
            printf ("Could not create context \n");
            //  return ;
        }
        printf("debug3\n");
        tivxHwaLoadKernels(context);

        raw = tivxCreateRawImage(context, &raw_params);

        tivxQueryRawImage(raw,
                          TIVX_RAW_IMAGE_WIDTH, &width, sizeof(width));
        tivxQueryRawImage(raw,
                          TIVX_RAW_IMAGE_HEIGHT, &height, sizeof(height));
        printf("debug4\n");

        // Note: image is non-zero but not validated
        y8_r8_c2 = vxCreateImage(context, width, height, VX_DF_IMAGE_NV12);
        printf("heightxwidth:%d\n",width,height);

        /* Create/Configure configuration input structure */
        tivx_vpac_viss_params_init(&params);

        params.sensor_dcc_id = sensor_dcc_id;
        params.ee_mode = TIVX_VPAC_VISS_EE_MODE_OFF;
        params.mux_output0 = 0;
        params.mux_output1 = 0;
        params.mux_output2 = TIVX_VPAC_VISS_MUX2_NV12;
        params.mux_output3 = 0;
        params.mux_output4 = 3;
        params.h3a_in = TIVX_VPAC_VISS_H3A_IN_LSC;
        params.h3a_aewb_af_mode = TIVX_VPAC_VISS_H3A_MODE_AEWB;
        params.chroma_mode = TIVX_VPAC_VISS_CHROMA_MODE_420;
        params.bypass_glbce = 0;
        params.bypass_nsf4 = 0;



        configuration = vxCreateUserDataObject(context, "tivx_vpac_viss_params_t",
                                               sizeof(tivx_vpac_viss_params_t), &params);

        /* Create/Configure ae_awb_params input structure */
        tivx_ae_awb_params_init(&ae_awb_params);

        ae_awb_params.ae_valid = 1;
        ae_awb_params.exposure_time = 16666;
        ae_awb_params.analog_gain = 1030;
        ae_awb_params.awb_valid = 1;
        ae_awb_params.color_temperature = 3000;
        for (i=0; i<4; i++)
        {
            ae_awb_params.wb_gains[i] = 525;
            ae_awb_params.wb_offsets[i] = 2;
        }

        ae_awb_result = vxCreateUserDataObject(context,
                                               "tivx_ae_awb_params_t", sizeof(tivx_ae_awb_params_t), &ae_awb_params);
        printf("debug5\n");
        /* Creating DCC */
        dcc_buff_size = appIssGetDCCSizeVISS(sensor_name, sensor_dcc_mode);
        if (0 == dcc_buff_size)
        {
            printf ("Incorrect DCC Buf size \n");
            //  return ;
        }

        dcc_param_viss = vxCreateUserDataObject( context, (const vx_char*)&dcc_viss_user_data_object_name,
                                                 dcc_buff_size, NULL);

        vxMapUserDataObject(
                dcc_param_viss,
                0,
                dcc_buff_size,
                &dcc_viss_buf_map_id,
                (void **)&dcc_viss_buf,
                VX_WRITE_ONLY,
                VX_MEMORY_TYPE_HOST,
                0);
        memset(dcc_viss_buf, 0xAB, dcc_buff_size);

        dcc_status = appIssGetDCCBuffVISS(sensor_name, sensor_dcc_mode, dcc_viss_buf, dcc_buff_size);
        if (0 != dcc_status)
        {
            printf ("GetDCCBuf failed\n");
            vxUnmapUserDataObject(dcc_param_viss, dcc_viss_buf_map_id);
            return ;
        }

        vxUnmapUserDataObject(dcc_param_viss, dcc_viss_buf_map_id);
        /* Done w/ DCC */

        /* Creating H3A output */
        h3a_aew_af = vxCreateUserDataObject(context, "tivx_h3a_data_t", sizeof(tivx_h3a_data_t), NULL);

        if(NULL != h3a_aew_af)
        {
            vxMapUserDataObject(h3a_aew_af,
                                0,
                                sizeof(tivx_h3a_data_t),
                                &dcc_viss_buf_map_id,
                                (void **)&dcc_viss_buf,
                                (vx_enum)VX_WRITE_ONLY,
                                (vx_enum)VX_MEMORY_TYPE_HOST,
                                0);

            memset(dcc_viss_buf, 0, sizeof(tivx_h3a_data_t));

            vxUnmapUserDataObject(h3a_aew_af, dcc_viss_buf_map_id);
        }

        graph = vxCreateGraph(context);
        printf("debug6\n");
        node = tivxVpacVissNode(graph,
                                configuration,
                                ae_awb_result,
                                dcc_param_viss,
                                raw, y12,uv12_c1, y8_r8_c2, uv8_g8_c3, s8_b8_c4,
                                h3a_aew_af, histogram);

        vxSetNodeTarget(node, VX_TARGET_STRING, TIVX_TARGET_VPAC_VISS1);

        ct_read_raw_image(raw, file_name, 2, downshift_bits);

        vxVerifyGraph(graph);
        int num_bytes_io = 0;

        start = tivxPlatformGetTimeInUsecs();
        for (iter = 0; iter < 1/*time*/; iter++) {
            for (load_iter = 0; load_iter < (250 * load / 100); load_iter++)
                vxProcessGraph(graph);
            snprintf(yuv_image_fname, 100, "%s.yuv","img_viss");
            printf("YUV file name %s \n", yuv_image_fname);
            num_bytes_io = write_output_image_nv12_8bit(yuv_image_fname, y8_r8_c2);
            if(num_bytes_io < 0)
            {
                printf("Error writing to VISS NV12 file \n");
                //  return 0;
            }
            printf("debug7\n");
            snprintf(h3a_image_fname, 100, "%s.bin","h3a");
            printf("H3A file name %s \n", h3a_image_fname);
            num_bytes_io = write_h3a_image(h3a_image_fname, h3a_aew_af);
            if(num_bytes_io < 0)
            {
                printf("Error writing to H3A file \n");
            }

            for (sleep_iter = 0; sleep_iter < (250 * (100 - load) / 100); sleep_iter++)
                usleep(4000);
        }

        end = tivxPlatformGetTimeInUsecs();
        printf ("Processing time for Resolution %dx%d is Time %d.%dms\n",
                width, height, (uint32_t)(end-start)/1000,(uint32_t)(end-start)%1000);


        vxReleaseNode(&node);
        vxReleaseGraph(&graph);
        vxReleaseImage(&y8_r8_c2);
        tivxReleaseRawImage(&raw);
        vxReleaseUserDataObject(&configuration);
        vxReleaseUserDataObject(&ae_awb_result);

        vxReleaseUserDataObject(&h3a_aew_af);
        vxReleaseUserDataObject(&dcc_param_viss);

        tivxHwaUnLoadKernels(context);

        vxReleaseContext(&context);
    }
}
