/*
 *  ======== main.c ========
 */

#include <xdc/std.h>

#include <xdc/runtime/Error.h>
#include <xdc/runtime/System.h>

#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/hal/Hwi.h>

#include <ti/sysbios/knl/Task.h>
#include <ti/include/UARTStdio.h>
/* Assembler modules lifted from StarterWare  */
#include "include/armv7a/cp15.h"
#include "include/armv7a/am335x/evmAM335x.h"
#include "include/hw/soc_AM335x.h"      /*  Hardware addresses for this Sitara */
#include "include/gpio_v2.h"        /* GPIO directions,  GPIOxxxxx(..)  functions */
#include "edma.h"
#include "ThisEdma.h"

/*  Just used for flashing the lights...  */
#define GPIO_LED0                      21
#define GPIO_LED1                      22
#define GPIO_LED2                      23
#define GPIO_LED3                      24

Void FlashTask(UArg a0, UArg a1);
Void DMATask(UArg a0, UArg a1);


void MMUConfigure(void);
void SetupEdma ();
void InitEdmaChannel(void);
void DumpBuffer(void);

/* Overly large buffers, that  when/if I get it to work, I will try large data */
volatile unsigned char gSrcBuffer[65536*2];
volatile unsigned char gDstBuffer[65536*2];

#define EVT_QUEUE_NUM  2
#define EDMA3_MEMTSTEVT   (12)
/*
 *  ======== main ========
 */
Int main() {
    Task_Handle flashtask;
    Task_Params flashParams;
    Error_Block flashErrorBlock;

    Task_Handle task;
    Error_Block eb;
    unsigned char *p;
    int x;

    MMUConfigure();     /*  Set the MMU protection so we can actually access things */

    UARTStdioInit();        /*  Give me a UART output for print statements is we want */
    UARTprintf("-----------------\nEnter main()\n");

    GPIO1ModuleClkConfig();     /*  Create some  "I'm alive" flashing lights */
    GPIOModuleEnable(SOC_GPIO_1_REGS);
    GPIODirModeSet(SOC_GPIO_1_REGS,GPIO_LED0,GPIO_DIR_OUTPUT);
    GPIODirModeSet(SOC_GPIO_1_REGS, GPIO_LED1,GPIO_DIR_OUTPUT);
    GPIODirModeSet(SOC_GPIO_1_REGS,GPIO_LED2,GPIO_DIR_OUTPUT);
    GPIODirModeSet(SOC_GPIO_1_REGS, GPIO_LED3,GPIO_DIR_OUTPUT);

    p = (unsigned char *) gSrcBuffer;  /*  Now...  Build the original buffer, set every  byte */
    for (x = 0; x < 512; x++) {
        *p = (unsigned char ) x;
        p++;
    }
    UARTprintf("Initialized Buffers SRC: %x  DST: %x \n",gSrcBuffer,gDstBuffer);
    DumpBuffer();

    Error_init(&eb);        /*  Create the task that will call the DMA */
    task = Task_create(DMATask, NULL, &eb);
    if (task == NULL) {
        UARTprintf("Task_create() failed!\n");
        BIOS_exit(0);
    }

    Error_init(&flashErrorBlock);   /*  Create the flasing lights task */
    Task_Params_init(&flashParams);
    flashParams.priority = 6;
    flashtask = Task_create(FlashTask, &flashParams, &flashErrorBlock);
    if (flashtask == NULL) {
        UARTprintf("Network Task_create() failed!\n");
        BIOS_exit(0);
    }
    BIOS_start();    /* does not return */
    return(0);
}


/************************************
 * Looking for ANY interrupt event to see if the DMA
 * tells us anything
 ****************************************/
volatile unsigned int m_complete;
void DmaComplete(unsigned int pValue) {  /* tpcc_int_pend_po0 */
    Hwi_clearInterrupt(12);
    EDMA3ClrIntr(SOC_EDMA30CC_0_REGS,EDMA3_MEMTSTEVT);

//    CP15DCacheFlushBuff((unsigned int) gDstBuffer, 512);  <-- No affect
    m_complete++;
}

void DmaMemprotectErr(unsigned int pValue) {    /* tpcc_mpint_pend_po */
    Hwi_clearInterrupt(13);
}
void DmaError(unsigned int pValue) {    /* tpcc_errint_pend_po */
    Hwi_clearInterrupt(14);
}



Void DMATask(UArg a0, UArg a1) {
    unsigned int status;
    unsigned int lastinterrupt;
    Hwi_Params edmaCompParams;              /* Int 12 */
    Hwi_Params edmaMemprotectErrParams;     /* Int 13 */
    Hwi_Params edmaErrorParams;             /* Int 14 */
    Error_Block eb;

    Hwi_Params_init(&edmaCompParams);
    edmaCompParams.priority = -1;
    edmaCompParams.maskSetting = Hwi_MaskingOption_SELF;
    edmaCompParams.enableInt = TRUE;
    Error_init(&eb);
    Hwi_create(12, DmaComplete, &edmaCompParams, &eb);

    Hwi_Params_init(&edmaMemprotectErrParams);
    edmaMemprotectErrParams.priority = -1;
    edmaMemprotectErrParams.maskSetting = Hwi_MaskingOption_SELF;
    edmaMemprotectErrParams.enableInt = TRUE;
    Error_init(&eb);
    Hwi_create(13, DmaMemprotectErr, &edmaMemprotectErrParams, &eb);

    Hwi_Params_init(&edmaErrorParams);
    edmaErrorParams.priority = -1;
    edmaErrorParams.maskSetting = Hwi_MaskingOption_SELF;
    edmaErrorParams.enableInt = TRUE;
    Error_init(&eb);
    Hwi_create(14, DmaError, &edmaErrorParams, &eb);

    Hwi_enableInterrupt(14);
    Hwi_enableInterrupt(13);
    Hwi_enableInterrupt(12);
    Hwi_enable();
    SetupEdma ();

    lastinterrupt = 0;

    EDMA3SetEvt(SOC_EDMA30CC_0_REGS, EDMA3_MEMTSTEVT);

    Task_sleep(500);
    do {
        /*  Loop on complete of the IPR bits  (base plus 1068 for 1 = completed) */

        status = EDMA3GetIntrStatus(SOC_EDMA30CC_0_REGS);
        if (status != 0) {      /*  ANY int status every show up????   */
            UARTprintf("Int status is %04x\n",status);
            /*   Use ICR  (base + 1070) to clear  */
            EDMA3ClrIntr(SOC_EDMA30CC_0_REGS,EDMA3_MEMTSTEVT);
        }
        if (m_complete != lastinterrupt) {
            EDMA3CCPaRAMEntry param;
            UARTprintf("Int # %d occurred \n",m_complete);
            m_complete = lastinterrupt;
            CP15DCacheFlushBuff((unsigned int) gDstBuffer, 512);
            DumpBuffer();

            EDMA3GetPaRAM(SOC_EDMA30CC_0_REGS,EDMA3_MEMTSTEVT,&param);
            UARTprintf("ACNT: %d BCNT: %d SRC: %x  DST: %x \n",param.aCnt, param.bCnt,
                       param.srcAddr, param.destAddr);
            if (param.bCnt) {  /* Any left?  Keep going */
                UARTprintf(" - Triggering the event again\n" );
                EDMA3SetEvt(SOC_EDMA30CC_0_REGS, EDMA3_MEMTSTEVT);
            }


        }

        Task_sleep(50);
    } while (1);
}

void SetupEdma () {
    EDMA3CCPaRAMEntry paramSet;
    EDMAModuleClkConfig();
    /* This set a 1:1 mapping between the channel number and the PaRAM sets
     * Then sets ALL the registers to quenum, sets the global "regionID" to 0,
     * cleans up all the event flags */
    EDMA3Init(SOC_EDMA30CC_0_REGS, EVT_QUEUE_NUM);

    /*  Enable the EESR for event 12 */
    EDMA3EnableDmaEvt(SOC_EDMA30CC_0_REGS, EDMA3_MEMTSTEVT);

    /*  Enables the channel in the shadow region (is only 0 anyway)
     * yet another mapping of the channel to the queue number ( as did in "EDMA3Init()")
     * Enable the event interrupt   */
    EDMA3RequestChannel(SOC_EDMA30CC_0_REGS, EDMA3_CHANNEL_TYPE_DMA,
                        EDMA3_MEMTSTEVT, EDMA3_MEMTSTEVT, EVT_QUEUE_NUM);

    /*  Now create the PaRAM settings  */
    /* Fill the PaRAM Set with transfer specific information */
    paramSet.srcAddr = (unsigned int) gSrcBuffer;
    paramSet.destAddr = (unsigned int) gDstBuffer;

    paramSet.aCnt = 0x10;  /* 16 bytes */
    paramSet.bCnt = 0x10;      /*  Do it in 5 blocks */
    paramSet.cCnt = 1;      /*  Total of 1 time */

    paramSet.srcBIdx = paramSet.aCnt;
    paramSet.destBIdx = paramSet.aCnt;

    paramSet.srcCIdx = 0;       /*  No jumps of the C index, because we don't */
    paramSet.destCIdx = 0;

    paramSet.linkAddr = 0xFFFF;   /* Address for linking (AutoReloading of a PaRAM Set) A value of 0xFFFF means no linking */
    paramSet.bCntReload = 0; // Only relevant for A-sync transfers (we are doing AB sync) ;
    /* OPT PaRAM Entries. */
    paramSet.opt = 0x00000000u;
    /* Setting the Transfer Complete Code(TCC) to interrupt 12 . */
    paramSet.opt |= ((EDMA3_MEMTSTEVT << EDMA3CC_OPT_TCC_SHIFT) & EDMA3CC_OPT_TCC);
    /* Enabling the Completion Interrupt. */
    paramSet.opt |= EDMA3CC_OPT_TCINTEN;
    /*  When using A only transfers, trigger the Interrupt for each Array */
    paramSet.opt |= EDMA3CC_OPT_ITCINTEN;

    //paramSet.opt |= EDMA3CC_OPT_SYNCDIM;    /*  AB sync - event triggers B count of A transfers */
    /* Now write the PaRAM Set */
    EDMA3SetPaRAM(SOC_EDMA30CC_0_REGS, EDMA3_MEMTSTEVT, &paramSet);
}


/********************************************
 * Code below here is not relevant to the DMA test
 ********************************************/

#define RGN_L2_WBWA                (0x08)
#define SECT_ATTR(NS, nG, S, AP2, TEX, AP, Dom, XN, C, B) \
             (0x02 | (NS << 19) | (nG << 17) | (S <<16) | \
             (AP2 <<15) | (TEX << 12) | (AP << 10) | (Dom << 5)\
             | (XN << 4) | (C << 3) | (B << 2))


#define CACHEABLE_TLB_ATTR   SECT_ATTR(1, 0, 0, 0, 5, 3, 0, 0, 1, 0)
#define NORM_TLB_ATTR        SECT_ATTR(1, 0, 0, 0, 0, 3, 0, 1, 0, 0)

static volatile unsigned int pageTable[4*1024]__attribute__((aligned(16*1024)));
void MMUConfigure(void) {

    unsigned int index;

    for (index = 0; index < 0x1000 /* 4KB */; index++) {
        /* Set the cache-able memory attributes */
        if ((index >= 0x800 && index < 0x900) ||   // DDR
            (index == 0x403) ||                    // OCMC
            (index == 0x402))                      // SRAM
        {
            pageTable[index] = (index << 20) | CACHEABLE_TLB_ATTR;
        }

        /* Set the non-cacheable memory attributes */
        else  {
            pageTable[index] = (index << 20) | NORM_TLB_ATTR;
        }
    }

    /* Invalidate the TLB entries */
    CP15TlbInvalidate();

    CP15DomainAccessClientSet();

    /* Set TTB0 value. We use only TTB0 here (N = 0) */
    CP15Ttb0Set(((unsigned int )pageTable) | RGN_L2_WBWA);

    /* Enables MMU */
    CP15MMUEnable();

    /* Enable Data Cache */
    CP15DCacheEnable();

    /* Disable Instruction Cache*/
    CP15ICacheEnable();
}


/**************************************
 * LED Flash Task - To indicate that the CPU is still alive
**************************************/
unsigned char sequence[6] = { 0x01, 0x02, 0x04, 0x08, 0x04, 0x02};
Void FlashTask(UArg a0, UArg a1) {
    int x = 0;
    UARTprintf("enter Flash Task()\n");
    do {
        GPIOPinWrite(SOC_GPIO_1_REGS,GPIO_LED0,(sequence[x] & 0x01) ? GPIO_PIN_HIGH : GPIO_PIN_LOW );
        GPIOPinWrite(SOC_GPIO_1_REGS,GPIO_LED1,(sequence[x] & 0x02) ? GPIO_PIN_HIGH : GPIO_PIN_LOW );
        GPIOPinWrite(SOC_GPIO_1_REGS,GPIO_LED2,(sequence[x] & 0x04) ? GPIO_PIN_HIGH : GPIO_PIN_LOW );
        GPIOPinWrite(SOC_GPIO_1_REGS,GPIO_LED3,(sequence[x] & 0x08) ? GPIO_PIN_HIGH : GPIO_PIN_LOW );
        x++;
        if (x > 5) { x = 0; }
        Task_sleep(500);
    } while (1);
}


void DumpBuffer() {
    int x;
    UARTprintf("----SRC Buffer-----\n");
    for (x = 0; x < 256; x++) {
        UARTprintf("%02x ", gSrcBuffer[x]);
        if ((x & 0x1f) == 0x1f) UARTprintf("\n");
    }
    if ((x & 0x1f) != 0x1f) UARTprintf("\n");

    UARTprintf("----DEST Buffer-----\n");
    for (x = 0; x < 256; x++) {
        UARTprintf("%02x ", gDstBuffer[x]);
        if ((x & 0x1f) == 0x1f) UARTprintf("\n");
    }
    if ((x & 0x1f) != 0x1f) UARTprintf("\n");
}


