/*
*
* Copyright (c) {2015 - 2017} Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/
#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include <cmath>
#include <stdlib.h>

#include "ti_dl.h"
#include "tidl_import_config.h"

#define ERROR_MSG      10000
#define WARNING_MSG    1

#define MODEL_CHECK_CONV_SUGGESTIONS 0

static bool is16bit = false;

static int32_t convertResultLayerNum2OrgLayerNum(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure, int32_t resultLayerNum)
{
  int32_t i1;

  for(i1 = 0; i1 < orgTIDLNetStructure->numLayers; i1++)
  {
    if(resultTIDLNetSructure->TIDLLayers[resultLayerNum].layerType == orgTIDLNetStructure->TIDLPCLayers[i1].layerType &&
       resultTIDLNetSructure->TIDLLayers[resultLayerNum].numInBufs == orgTIDLNetStructure->TIDLPCLayers[i1].numInBufs &&
       resultTIDLNetSructure->TIDLLayers[resultLayerNum].numOutBufs == orgTIDLNetStructure->TIDLPCLayers[i1].numOutBufs &&
       resultTIDLNetSructure->TIDLLayers[resultLayerNum].numInBufs > 0 && orgTIDLNetStructure->TIDLPCLayers[i1].numOutBufs > 0 &&
       resultTIDLNetSructure->TIDLLayers[resultLayerNum].inData[0].dataId == orgTIDLNetStructure->TIDLPCLayers[i1].inData[0].dataId &&
       resultTIDLNetSructure->TIDLLayers[resultLayerNum].outData[0].dataId == orgTIDLNetStructure->TIDLPCLayers[i1].outData[0].dataId)
    {
      return i1;
    }
  }

  printf("ERROR  : Model Check cannot find corrosponding layers from result network to original network.\n");
  return -1;
}


static int32_t checkConvLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  bool notOptimal_input64align = false;
  bool notOptimal_output64align = false;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];

    if (layer.layerType == TIDL_ConvolutionLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ConvParams_t& params = layer.layerParams.convParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Whitelist (validated)
       */
      int32_t validatedParams[][8] = 
      {
        // -1 means dont care
        // kernelH kernelW strideH strideW dilationH dilationW padH padW
        {        1,      1,      2,      2,       -1,       -1,   0,   0},
        {        1,      1,      1,      1,       -1,       -1,   0,   0},
        {        3,      3,      1,      1,       -1,       -1,   1,   1},
        {        3,      3,      2,      2,       -1,       -1,   0,   0},
        {        3,      3,      2,      2,       -1,       -1,   1,   1},
        {        3,      3,      1,      1,        4,        4,   4,   4}, // Caffe JSegnet
        {        3,      3,      1,      1,        6,        6,   6,   6}, // ONNX JSegnet
        {        3,      3,      1,      1,       12,       12,  12,  12}, // ONNX TIAD JMODNet
        {        3,      3,      1,      1,       18,       18,  18,  18}, // ONNX TIAD JMODNet
        {        5,      5,      2,      2,       -1,       -1,   2,   2}, // JacintNet
        {        7,      7,      2,      2,       -1,       -1,   3,   3}, // ResNet50
      };

      for(i2 = 0; i2 < sizeof(validatedParams)/8/sizeof(int32_t); i2++)
      {
        if((validatedParams[i2][0] == -1 || params.kernelH == validatedParams[i2][0]) && (validatedParams[i2][1] == -1 || params.kernelW == validatedParams[i2][1]))
          validated = true;
        else
          validated = false;

        if((validatedParams[i2][2] == -1 || params.strideH == validatedParams[i2][2]) && (validatedParams[i2][3] == -1 || params.strideW == validatedParams[i2][3]))
          validated = true;
        else
          validated = false;

        if((validatedParams[i2][4] == -1 || params.dilationH == validatedParams[i2][4]) && (validatedParams[i2][5] == -1 || params.dilationW == validatedParams[i2][5]))
          validated = true;
        else
          validated = false;

        if((validatedParams[i2][6] == -1 || params.padH == validatedParams[i2][6]) && (validatedParams[i2][7] == -1 || params.padW == validatedParams[i2][7]))
          validated = true;
        else
          validated = false;

        if(validated)
          break;
      }

      
      /*
       * Blacklist (confirmed not support)
       */

      if((params.kernelH == 1 && params.kernelW == 3) ||
         (params.kernelH == 3 && params.kernelW == 1) ||
         (params.kernelH == 1 && params.kernelW == 5) ||
         (params.kernelH == 5 && params.kernelW == 1))
      {
        warning = true;
        printf("WARNING: [TIDL_ConvolutionLayer] %s kernel size 1x3 or 1x5 is supported by MMALIB, but not validated by TIDL. Be careful.\n", layerPC.name);
      }

      if(params.kernelH == 11 && params.kernelW == 11 &&
         params.strideH == 4 && params.strideW == 4)
      {
        warning = true;
        printf("WARNING: [TIDL_ConvolutionLayer] %s kernel size 11x11 with stride 4 is supported by MMALIB, but not validated by TIDL. Be careful\n", layerPC.name);
      }

      if(params.kernelH > 7  && params.kernelW > 7 &&
         params.strideH == 2 && params.strideW == 2)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_ConvolutionLayer] %s kernel size larger than 7 with stride 2 is not supported !!!\n", layerPC.name);
      }

      if(params.kernelH != 11 && params.kernelW != 11 &&
         params.strideH == 4 && params.strideW == 4)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_ConvolutionLayer] %s stride 4 only support kernel size 11x11 !!!\n", layerPC.name);
      }



      /*
       * Performance Suggestion
       */
      if((params.kernelH * params.kernelW * layer.inData[0].dimValues[1] / params.numGroups + params.enableBias) % 64)
      {
        notOptimal = true;
        notOptimal_input64align = true;
      }
      if(layer.outData[0].dimValues[1] / params.numGroups % 64)
      {
        notOptimal = true;
        notOptimal_output64align = true;
      }
      
      /*
       * Conclusion
       */
      if(!validated && !fatalError)
      {
        printf("WARNING: [TIDL_ConvolutionLayer] %s Paramater is not validated. Please be carefull with this layer!\n", layerPC.name);
        printf("                                 Kernel %dx%d Stride %dx%d dilation %dx%d Pad %dx%d Bias %d \n", 
               params.kernelH,   params.kernelW, 
               params.strideH,   params.strideW, 
               params.dilationH, params.dilationW, 
               params.padH,      params.padW, 
               params.enableBias);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }

#if MODEL_CHECK_CONV_SUGGESTIONS
  if(notOptimal_input64align)
    printf("SUGGESTION: [TIDL_ConvolutionLayer] params.kernelH * params.kernelW * layer.inData[0].dimValues[1] / params.numGroups is not 64 aligned, the performance is not efficient on MMA!\n");
  
  if(notOptimal_output64align)
    printf("SUGGESTION: [TIDL_ConvolutionLayer] layer.outData[0].dimValues[1] / params.numGroups is not 64 aligned, the performance is not efficient on MMA!\n");
#endif

  return msg;
}


static int32_t checkPoolingLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_PoolingLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_PoolingParams_t& params = layer.layerParams.poolParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Whitelist
       */
      if(params.kernelH == 0 && params.kernelW == 0)
      {
        validated = true; // Global pooling
      }

      if(params.kernelH == 2 && params.kernelW == 2 &&
         params.strideH == 2 && params.strideW == 2)
      {
        validated = true;
      }

      if(params.kernelH == 1 && params.kernelW == 1 &&
         ((params.strideH == 2 && params.strideW == 2) || (params.strideH == 1 && params.strideW == 1 )) &&
         params.poolingType == TIDL_MaxPooling)
      {
        validated = true;
      }

      if(params.kernelH == 3 && params.kernelW == 3 &&
         ((params.strideH == 1 && params.strideW == 1) || (params.strideH == 2 && params.strideW == 2)) &&
         params.padH    == 1 && params.padW    == 1 &&
         params.poolingType == TIDL_MaxPooling)
      {
        validated = true;
      }

      if(params.kernelH == 3 && params.kernelW == 3 &&
         params.strideH == 2 && params.strideW == 2 &&
         params.padH    == 0 && params.padW    == 0)
      {
        validated = true;
      }

      if(is16bit && params.poolingType == TIDL_MaxPooling)
      {
        validated = true;
      }


      /*
       * Performance Suggestion
       */
      if(params.kernelH == 4 && params.kernelW == 4)
      {
        notOptimal = true;
        printf("SUGGESTION: [TIDL_PoolingLayer] %s params.kernelH * params.kernelW * layer.inData[0].dimValues[1] / params.numGroups is not 64 aligned, the performance is not efficient on MMA!\n", layerPC.name);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkBatchNormLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_BatchNormLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_BatchNormParams_t& params = layer.layerParams.batchNormParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Performance Suggestion
       */
      if(is16bit)
      {
        notOptimal = true;
        printf("SUGGESTION: [TIDL_BatchNormLayer] %s 16 bits is not optimal in this release.\n", layerPC.name);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkInnerProductLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_InnerProductLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_InnerProductParams_t& params = layer.layerParams.innerProductParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;


      /*
       * Blacklist (confirmed not support)
       */
      if(layer.inData[0].dimValues[0] != 1 ||
         layer.inData[0].dimValues[1] != 1 ||
         layer.inData[0].dimValues[2] != 1)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_InnerProductLayer] %s input shape of inner product must be 1x1x1xN.\n", layerPC.name);
      }


      /*
       * Performance Suggestion
       */
      if(layer.inData[0].dimValues[3] * layer.outData[0].dimValues[3] > 2048 * 2048)
      {
        notOptimal = true;
        printf("SUGGESTION: [TIDL_InnerProductLayer] %s Size larger than 2048 * 2048 is not optimal.\n", layerPC.name);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkDeconvLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_Deconv2DLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ConvParams_t& params = layer.layerParams.convParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Whitelist
       */
      if(params.kernelH == 4 && params.kernelW == 4 &&
         params.strideH == 2 && params.strideW == 2)
      {
        validated = true;
      }

      if(params.kernelH == 2 && params.kernelW == 2 &&
         params.strideH == 2 && params.strideW == 2)
      {
        validated = true;
      }

      /*
       * Blacklist (confirmed not support)
       */
      if(params.kernelH == 8 || params.kernelW == 8)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_Deconv2DLayer] %s kernel 8x8 in deconv is not supported.\n", layerPC.name);
      }

      /*
       * Performance Suggestion
       */
      
      printf("SUGGESTION: [TIDL_Deconv2DLayer] %s Please change to Upsample/Resize if possible. Upsample/Resize will be more efficient.\n", layerPC.name);

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkConcatLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_ConcatLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ConcatParams_t& params = layer.layerParams.concatParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkCropLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  return 0;
}


static int32_t checkDropOutLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_DropOutLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_DropOutLayer] %s should be removed in import process. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkDetectionOutLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_DetectionOutputLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_DetectOutputParams_t& params = layer.layerParams.detectOutParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;


      /*
       * Blacklist (confirmed not support)
       */
      if(layer.numInBufs != 2)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_DetectionOutputLayer] %s Detection out layer must be 2 inputs. If not, it will not work!\n", layerPC.name);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkShuffleChannelLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_ShuffleChannelLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ShuffleLayerParams_t& params = layer.layerParams.shuffleLayerParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Performance Suggestion
       */
      notOptimal = true;
      printf("SUGGESTION: [TIDL_ShuffleChannelLayer] %s ShuffleChannel layer is not optimal in this version.\n", layerPC.name);


      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkResizeLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_ResizeLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ResizeLayerParams_t& params = layer.layerParams.resizeParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Whitelist
       */
      if(params.resizeRatio[2] == 2 || params.resizeRatio[2] == 4)
      {
        validated = true;
      }


      /*
       * Performance Suggestion
       */
      if(!validated)
        printf("SUGGESTION: [TIDL_ResizeLayer] %s Resize kernel other than 2x2/4x4 with nearest/bilinear is not optimal.\n", layerPC.name);

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkPriorBoxLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_PriorBoxLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_PriorBoxLayer] %s should be removed in import process. Please make sure you have detection out layer in the model. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkPermuteLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_PermuteLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_PermuteLayer] %s should be removed in import process. Please make sure you have detection out layer in the model. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkShapeLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_ShapeLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_ShapeLayer] %s should be removed in import process. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkSqueezeLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_SqueezeLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_SqueezeLayer] %s should be removed in import process. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkPadLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_PadLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_PadLayer] %s should be removed in import process. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkTransposeLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_TransposeLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_TransposeLayer] %s should be removed in import process. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkClipLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_ClipLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      
      printf("ERROR  : [TIDL_ClipLayer] %s should be removed in import process. if not, this model will not work!\n", layerPC.name);

      msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkSliceLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_SliceLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ConcatParams_t& params = layer.layerParams.concatParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;


      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkEltwiseLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_EltWiseLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_EltWiseParams_t& params = layer.layerParams.eltWiseParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;


      /*
       * Blacklist (confirmed not support)
       */
      if(layer.numInBufs > 2)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_EltWiseLayer] %s Only support 2 inputs.\n", layerPC.name);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkFlattenLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_FlattenLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;

      /*
       * Performance Suggestion
       */
      if(is16bit)
      {
        notOptimal = true;
        printf("SUGGESTION: [TIDL_FlattenLayer] %s Flatten with 16 bit is not optimal in this version.\n", layerPC.name);
      }


      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkArgmaxLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_ArgMaxLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_ArgMaxParams_t& params = layer.layerParams.argMaxParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;


      /*
       * Blacklist (confirmed not support)
       */
      if(layer.outData[0].dimValues[0] != 1 ||
         layer.outData[0].dimValues[1] != 1)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_ArgMaxLayer] %s Only support axis == 1.\n", layerPC.name);
      }

      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkSoftmaxLayers(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t i1, i2, i3, i4;
  int32_t msg = 0;

  for (i1 = 0; i1 < resultTIDLNetSructure->numLayers; i1++)
  {
    sTIDL_Layer_t& layer = resultTIDLNetSructure->TIDLLayers[i1];
    
    if (layer.layerType == TIDL_SoftMaxLayer)
    {
      int32_t orgLayerNum = convertResultLayerNum2OrgLayerNum(resultTIDLNetSructure, orgTIDLNetStructure, i1);
      sTIDL_LayerPC_t& layerPC = orgTIDLNetStructure->TIDLPCLayers[orgLayerNum];
      sTIDL_SoftMaxParams_t& params = layer.layerParams.softMaxParams;
      bool validated = false;
      bool warning = false;
      bool notOptimal = false;
      bool fatalError = false;


      /*
       * Blacklist (confirmed not support)
       */
      if(layer.inData[0].dimValues[0] != 1 ||
         layer.inData[0].dimValues[1] != 1 ||
         layer.inData[0].dimValues[2] != 1)
      {
        fatalError = true;
        printf("ERROR  : [TIDL_SoftMaxLayer] %s input shape of softmax must be 1x1x1xN.\n", layerPC.name);
      }


      if(warning)
        msg += WARNING_MSG;
      if(fatalError)
        msg += ERROR_MSG;
    }
  }
  
  return msg;
}


static int32_t checkQuantStatsAvailable(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t msg = 0;

  if(resultTIDLNetSructure->isQuantStatsAvailable == 0)
  {
    printf("ERROR  : [TIDL_E_QUANT_STATS_NOT_AVAILABLE] tidl_quant_stats_tool.out fails to collect dynamic range. Please look into quant stats log. This model will get fault on target.\n");
    msg += ERROR_MSG;
  }
  
  return msg;
}


static int32_t checkDataflowInfoAvailable(sTIDL_Network_t * resultTIDLNetSructure, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  int32_t msg = 0;

  if(resultTIDLNetSructure->dataFlowInfo == 0)
  {
    printf("WARNING: [TIDL_E_DATAFLOW_INFO_NULL] ti_cnnperfsim.out fails to allocate memory in MSMC. Please look into perfsim log. This model can only be used on PC emulation, it will get fault on target.\n");
    msg += WARNING_MSG;
  }
  
  return msg;
}
void tidlModelCheck(tidl_import_config * params, sTIDL_OrgNetwork_t * orgTIDLNetStructure)
{
  sTIDL_Network_t * resultNetStructure;
  int32_t msgCount = 0;
  int32_t ret;

  FILE* fp = fopen((char*)params->outputNetFile, "rb");
  if(fp == NULL)
  {
    printf("ERROR  : TIDL Model Check cannot open the result network file %s.\n", params->outputNetFile);
    return ;
  }

  resultNetStructure = (sTIDL_Network_t*)malloc(sizeof(sTIDL_Network_t));

  ret = fread(resultNetStructure, 1, sizeof(sTIDL_Network_t), fp);

  is16bit = params->numFeatureBits == 16;

  if(gParams.debugTraceLevel > 0)
  {
    printf("****************************************************\n");
    printf("**               TIDL Model Checker              **\n");
    printf("****************************************************\n");
  }

  // TIDL_ConvolutionLayer
  msgCount += checkConvLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_PoolingLayer
  msgCount += checkPoolingLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_BatchNormLayer
  msgCount += checkArgmaxLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_BatchNormLayer
  msgCount += checkBatchNormLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_BatchNormLayer
  msgCount += checkClipLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_ConcatLayer
  msgCount += checkConcatLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_CropLayer
  msgCount += checkCropLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_DeconvLayer
  msgCount += checkDeconvLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_DetectionOutLayer
  msgCount += checkDetectionOutLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_DropOutLayer
  msgCount += checkDropOutLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_EltwiseLayer
  msgCount += checkEltwiseLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_FlattenLayer
  msgCount += checkFlattenLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_InnerProductLayer
  msgCount += checkInnerProductLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_PadLayer
  msgCount += checkPadLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_PermuteLayer
  msgCount += checkPermuteLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_PriorBoxLayer
  msgCount += checkPriorBoxLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_ShapeLayer
  msgCount += checkShapeLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_ShapeLayer
  msgCount += checkSqueezeLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_ShuffleChannelLayer
  msgCount += checkShuffleChannelLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_SliceLayer
  msgCount += checkSliceLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_SoftmaxLayer
  msgCount += checkSoftmaxLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_TransposeLayer
  msgCount += checkTransposeLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_UpSampleLayer
  msgCount += checkResizeLayers(resultNetStructure, orgTIDLNetStructure);

  // TIDL_E_QUANT_STATS_NOT_AVAILABLE
  msgCount += checkQuantStatsAvailable(resultNetStructure, orgTIDLNetStructure);

  // TIDL_E_DATAFLOW_INFO_NULL
  msgCount += checkDataflowInfoAvailable(resultNetStructure, orgTIDLNetStructure);


  if(msgCount == 0)
  {
    printf("****************************************************\n");
    printf("**                ALL MODEL CHECK PASSED          **\n");
    printf("****************************************************\n\n");
  }
  else
  {
    printf("****************************************************\n");
    printf("**        %3d WARNINGS        %3d ERRORS          **\n", msgCount % ERROR_MSG, msgCount / ERROR_MSG);
    printf("****************************************************\n");
  }
  
  free(resultNetStructure);
}
