/*
*
* Copyright (c) {2015 - 2017} Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#ifdef _WIN32
#include <asprintf.h>
#endif
#include "perfsim.h"




#include "tidl_import_config.h"

#include "ti_dl.h"

#include <google/protobuf/io/coded_stream.h>
#include <google/protobuf/io/zero_copy_stream_impl.h>
#include <google/protobuf/message.h>
#include <google/protobuf/text_format.h>
using namespace std;
using ::google::protobuf::Message;


#include "tidl_import_common.h"
#include "tidl_import_common_model_check.h"

void onnx_import(tidl_import_config * params);
sTIDL_OrgNetwork_t      orgTIDLNetStructure;
sTIDL_OrgNetwork_t      tempTIDLNetStructure;
sTIDL_Network_t         tIDLNetStructure;
char inDataNames[TIDL_MAX_ALG_IN_BUFS][TILD_MAX_DATA_NAME];
char outDataNames[TIDL_MAX_ALG_OUT_BUFS][TILD_MAX_DATA_NAME];
char outMetaLayerNames[TIDL_MAX_ALG_OUT_BUFS][TILD_MAX_DATA_NAME];
int32_t numNetInData = 0;
int32_t numNetOutData = 0;
char inConfgFilename[500];

const char * TIDL_LayerString[] =
{
"TIDL_DataLayer",
"TIDL_ConvolutionLayer",
"TIDL_PoolingLayer",
"TIDL_ReLULayer",
"TIDL_PReLULayer",
"TIDL_EltWiseLayer",
"TIDL_InnerProductLayer",
"TIDL_SoftMaxLayer",
"TIDL_BatchNormLayer",
"TIDL_BiasLayer",
"TIDL_ScaleLayer",
"TIDL_Deconv2DLayer",
"TIDL_ConcatLayer",
"TIDL_SplitLayer",
"TIDL_SliceLayer",
"TIDL_CropLayer",
"TIDL_FlattenLayer",
"TIDL_DropOutLayer",
"TIDL_ArgMaxLayer",
"TIDL_DetectionOutputLayer",
"TIDL_ShuffleChannelLayer" ,
"TIDL_ResizeLayer" ,
"TIDL_RoiPoolingLayer",
"TIDL_OdPostProcessingLayer",
"TIDL_CustomLayer",
"TIDL_UnsupportedLayer" ,
"TIDL_ConstDataLayer" ,
"TIDL_PriorBoxLayer" ,
"TIDL_PermuteLayer" ,
"TIDL_ReshapeLayer" ,
"TIDL_ShapeLayer" ,
"TIDL_SqueezeLayer" ,
"TIDL_PadLayer" ,
"TIDL_TransposeLayer" ,
"TIDL_CLipLayer" ,
"TIDL_MinimumLayer" ,
};

struct deferredOption {
  char *opt;
  char **args;
};

char* getFileNameFromPath(char* path)
{
  for (int32_t i = strlen(path) - 1; i; i--)
  {
    if ((path[i] == '/') || (path[i] == '\\'))
    {
      return &path[i + 1];
    }
  }
  return path;
}

void getDirFromPath(char* path)
{
  for (int32_t i = strlen(path) - 1; i; i--)
  {
    if ((path[i] == '/') || (path[i] == '\\'))
    {
      path[i] = '\0';
      return;
    }
  }
  path[0] = '\0';
  return;
}
void getDosPath(char* orgpath, char * dosPath)
{
  strcpy(dosPath, orgpath);
  for (int32_t i = 0; i < strlen(dosPath); i++)
  {
    if (dosPath[i] == '/')
    {
      dosPath[i] = '\\';
    }
  }
  return;
}

void getAbsPath(char* path, char * absPath)
{
  char syscmd[500];
  char dosPath[500];
#ifdef _WIN32
  getDosPath(path, dosPath);
  sprintf(syscmd, "dir /b /s %s", dosPath);
#else
  sprintf(syscmd, "readlink -f %s", path);
#endif

#ifdef _WIN32
  FILE * fp = _popen(syscmd,  "r");
#else
  FILE * fp = popen(syscmd,  "r");
#endif

  if (fp == NULL)
  {
    printf("Error while runing command : %s", syscmd);
  }
  fscanf(fp, "%s", absPath);
  fclose(fp);
  return;
}

void setDefaultParams(tidl_import_config * params)
{
  int32_t i;
  params->randParams          = 0;
  params->modelType           = 0; // 0 - caffe, 1- tensorFlow
  params->quantizationStyle   = TIDL_QuantStyleNP2Fixed;
  params->quantRoundAdd       = 50; // 0 - caffe, 1- tensorFlow
  params->numParamBits        = 8;
  params->inFileFormat        = 2; // 0 - Encoded, 1- RAW
  params->numFrames           = -1;
  params->foldBnInConv2D      = 1;
  params->foldPreBnConv2D      = 1;
  params->foldEltWiseInConv2D = 0;
  params->foldMaxPoolInConv2D = 0;
  params->preProcType         = 0;
  params->postProcType        = 0;
  params->postProcDataId      = 0;
  params->numFeatureBits      = 8;
  params->metaArchType        =-1;
  params->debugTraceLevel     = 0;
  params->writeTraceLevel     = 0;
  params->quantRangeUpdateFactor       = 0.1;
  params->compileConstraintsFlag = 1;
  params->executeNetworkCompiler = 1 ;
  params->executeQuantsTool = 1 ;

#ifdef _WIN32
  strcpy((char*)params->tidlStatsTool,"..\\..\\test\\PC_dsp_test_dl_algo.out.exe");
  strcpy((char*)params->perfSimTool,  "..\\..\\utils\\perfsim\\ti_cnnperfsim.out.exe");
  strcpy((char*)params->graphVizTool, "..\\..\\utils\\tidlModelGraphviz\\out\\tidl_graphVisualiser.out.exe");
  strcpy((char*)params->perfSimConfig, "..\\..\\test\\testvecs\\config\\import\\perfsim_base.cfg");
#else
  strcpy((char*)params->tidlStatsTool, "../../test/PC_dsp_test_dl_algo.out");
  strcpy((char*)params->perfSimTool, "../../utils/perfsim/ti_cnnperfsim.out");
  strcpy((char*)params->graphVizTool, "../../utils/tidlModelGraphviz/out/tidl_graphVisualiser.out");
  strcpy((char*)params->perfSimConfig, "../../test/testvecs/config/import/perfsim_base.cfg");
#endif

  strcpy((char*)params->inDataNamesList, "");
  strcpy((char*)params->outDataNamesList, "");
  strcpy((char*)params->fileNameGrpInfo, "");

  for (i = 0; i < TIDL_MAX_ALG_IN_BUFS; i++)
  {
    params->inElementType[i] = TIDL_UnsignedChar;
    params->inQuantFactor[i] =  1.0;
    params->inWidth[i]       = -1;
    params->inHeight[i]      = -1;
    params->inNumChannels[i] = -1;
    params->numRoi[i] = 1;
    params->resizeWidth[i]  = -1;
    params->resizeHeight[i] = -1;
    params->inResizeType[i] = TIDL_inResizeTypeDefault;
    params->inDataFormat[i] = TIDL_inDataFormatRGBPlanar;
    params->reserved[i]     = 0;
    params->inDataNorm[i]   = 0;
  }
}

#define TIDL_CFG_MAX_LINE_SIZE (3000)
int32_t getNumberOfLinesIntheFile(char * fileName)
{
  FILE * fp1;
  int32_t i, lineCnt = 0;
  char line[TIDL_CFG_MAX_LINE_SIZE];

  fp1 = fopen((const char *)fileName, "r");
  if (fp1 == NULL)
  {
    printf("Could not open %s file for reading \n", fileName);
    return 0;
  }
  while (!feof(fp1))
  {
    fgets(line, TIDL_CFG_MAX_LINE_SIZE, fp1);
    lineCnt++;
  }
  return(lineCnt);
}

void tidlQuantStatsTool(tidl_import_config * params)
{
  FILE * fp;
  char sysCommand[500];
  char orgPath[500];
  char absPath[500];
  char fileName[500];
  char dirName[500];

#ifdef _WIN32
  sprintf(sysCommand, "if exist tempDir rmdir /S/Q tempDir");
#else
  sprintf(sysCommand, "[ -d tempDir ] && rm -rf tempDir");
#endif
  system(sysCommand);
  sprintf(sysCommand, "mkdir tempDir");
  system(sysCommand);

  fp = fopen("tempDir/qunat_stats_config.txt", "w+");
  if(fp== NULL)
  {
    printf("Could not open config  file tempDir/qunat_stats_config.txt  \n");
    return;
  }
  fprintf(fp, "inFileFormat    = %d\n",params->inFileFormat);
  fprintf(fp, "numFrames   = %d\n",params->numFrames);
  fprintf(fp, "postProcType   = %d\n",params->postProcType);
  fprintf(fp, "postProcDataId   = %d\n", params->postProcDataId);
  fprintf(fp, "quantRangeUpdateFactor   = %f\n", params->quantRangeUpdateFactor);
  getAbsPath((char *)params->inData, absPath);
  fprintf(fp, "inData   = %s\n",absPath);
  fprintf(fp, "outData   = \"./trace/stats_tool_out.bin\"\n");

  getAbsPath((char *)params->outputNetFile, absPath);
  fprintf(fp, "netBinFile     = %s\n", absPath);

  sprintf(fileName, "%s%d.bin", (char *)params->outputParamsFile, 1);
  getAbsPath(fileName, absPath);
  fprintf(fp, "ioConfigFile        = %s\n", absPath);
  fprintf(fp, "flowCtrl        = 3\n");
  fprintf(fp, "writeTraceLevel        = %d\n", gParams.writeTraceLevel);
  fprintf(fp, "debugTraceLevel        = %d\n", gParams.debugTraceLevel);

  fclose(fp);

  fp = fopen("tempDir/configFilesList.txt", "w+");
  if (fp == NULL)
  {
    printf("Could not open config  file tempDir/configFilesList.txt  \n");
    return;
  }
  strcpy(orgPath, "./tempDir/qunat_stats_config.txt");
  getAbsPath(orgPath, absPath);

  fprintf(fp, "1 %s \n0\n", absPath);
  fclose(fp);


  strcpy(orgPath, "./tempDir/configFilesList.txt");
  getAbsPath(orgPath, absPath);

  strcpy(fileName, getFileNameFromPath((char *)params->tidlStatsTool));

  getAbsPath((char *)params->tidlStatsTool, dirName);
  getDirFromPath(dirName);

#ifdef _WIN32
  sprintf(sysCommand, "cd %s && %s %s", dirName, fileName, absPath);
#else
  sprintf(sysCommand, "cd %s && ./%s %s", dirName, fileName, absPath);
#endif
  if(gParams.debugTraceLevel > 0)
  {
    printf(sysCommand);
  }
  system(sysCommand);

  return;

}

int32_t tidl_getModelSize(uint8_t * fileString)
{
  FILE * fptr;
  int32_t netSize;
  fptr = fopen((const char *)fileString, "rb");
  if (fptr)
  {
    fseek(fptr, 0L, SEEK_END);
    netSize = ftell(fptr);
    fclose(fptr);
    return netSize;
  }
  else
  {
    printf("Could Not Open Files %s\n", fileString);
    return -1;
  }

}

int tidlRunGraphVizTool(tidl_import_config * params)
{
  FILE * fp;
  char sysCommand[500];
  char absPath[500];

  fp = fopen((const char *)params->graphVizTool, "r");
  if (fp == NULL)
  {
    //printf("INFO : Couldn't open graphVizTool file: %s , Skipping Visualization \n", params->graphVizTool);
    return(0);
  }
  fclose(fp);

  getAbsPath((char *)params->graphVizTool, absPath);
  sprintf(sysCommand, "%s %s", absPath, params->outputNetFile);
  if(gParams.debugTraceLevel > 0)
  {
    printf("%s\n", sysCommand);
  }
  system(sysCommand);
}

int tidlWriteTensorNamesToFile(tidl_import_config * params)
{
  FILE  * fp;
  FILE *layerInfoFile = fopen(LAYER_INFO_FILENAME, "w");
  int32_t i;
  int modelSize = tidl_getModelSize(params->outputNetFile);
  sTIDL_Network_t * tidlNet = (sTIDL_Network_t *)malloc(modelSize);
  if (tidlNet == NULL)
  {
    printf("Could not Allocate memory for model read\n");
    return 0;
  }
  fp = fopen((const char *)params->outputNetFile, "rb");
  if (fp)
  {
    fread(tidlNet, 1, modelSize, fp);
    fclose(fp);
  }
  else
  {
    printf("Could not open %s\n", params->outputNetFile);
    return 0;
  }

  for (i = 0; i < tidlNet->numLayers; i++)
  {
    fprintf(layerInfoFile, "%d %d %s \n", i, tidlNet->TIDLLayers[i].outData[0].dataId,
      TIDL_getOutDataName(&orgTIDLNetStructure, tidlNet->TIDLLayers[i].outData[0].dataId));
  }
  free(tidlNet);
  fclose(layerInfoFile);
}

int tidlRunPerfSimTool(tidl_import_config * params)
{
  FILE * fp;
  char sysCommand[500];
  char orgPath[500];
  char absPath[500];
  char fileName[500];
  char dirName[500];
  sPerfSim_t * perSimInfo = (sPerfSim_t *)malloc(sizeof(sPerfSim_t));
  sPerfSim_t * orderedPerSimInfo = (sPerfSim_t *)malloc(sizeof(sPerfSim_t));

  int i, j, k, l, foundInData, numFeatureBytes;
  int32_t currLayersGroupId = 1;

  fp = fopen((const char *)params->perfSimConfig, "r");
  if (fp == NULL)
  {
    printf("INFO : Couldn't open perfSimConfig file: %s , Skipping Performance Simulation \n", params->perfSimConfig);
    return(0);
  }
  fclose(fp);

  fp = fopen((const char *)params->perfSimTool, "r");
  if (fp == NULL)
  {
    printf("Couldn't open perfSimTool file: %s  \n", params->perfSimTool);
    return(0);
  }
  fclose(fp);

#ifdef _WIN32
  sprintf(sysCommand, "if exist tempSimDir rmdir /S/Q tempSimDir");
#else
  sprintf(sysCommand, "[ -d tempSimDir ] && rm -rf tempSimDir");
#endif
  system(sysCommand);
  sprintf(sysCommand, "mkdir tempSimDir");
  system(sysCommand);

  getAbsPath((char *)params->perfSimConfig, absPath);
#ifdef _WIN32
  sprintf(sysCommand, "copy %s tempSimDir\\perf_sim_config.txt", absPath);
#else
  sprintf(sysCommand, "cp %s tempSimDir/perf_sim_config.txt", absPath);
#endif
  if(gParams.debugTraceLevel > 0)
  {
    printf("%s\n", sysCommand);
  }
  system(sysCommand);


  fp = fopen("tempSimDir/perf_sim_config.txt", "a+");
  if (fp == NULL)
  {
    printf("Could not open config  file tempSimDir/qunat_stats_config.txt  \n");
    return 0;
  }
  getAbsPath((char *)params->outputNetFile, absPath);
  fprintf(fp, "FILENAME_NET     = %s\n", absPath);
  fprintf(fp, "FILEFORMAT_NET     = -1\n");
  if(strcmp((char *)params->fileNameGrpInfo,"") != 0)
  {
    getAbsPath((char *)params->fileNameGrpInfo, absPath);
    fprintf(fp, "FILENAME_GRPINFO     = %s\n", absPath);
  }


  fprintf(fp, "OUTPUT_DIR     = %s.perfsim\n", getFileNameFromPath(inConfgFilename));
  if ((params->numParamBits > 8) || (params->numFeatureBits > 8))
  {
	  fprintf(fp, "DATATYPE     =   1\n");
  }
  fclose(fp);

  strcpy(orgPath, "tempSimDir/perf_sim_config.txt");
  getAbsPath(orgPath, absPath);

  strcpy(fileName, getFileNameFromPath((char *)params->perfSimTool));

  getAbsPath((char *)params->perfSimTool, dirName);
  getDirFromPath(dirName);

#ifdef _WIN32
  sprintf(sysCommand, "cd %s && %s %s %d %d 2", dirName, fileName, absPath, params->compileConstraintsFlag, params->reserved[0]);
#else
  sprintf(sysCommand, "cd %s && ./%s %s %d %d 2", dirName, fileName, absPath, params->compileConstraintsFlag, params->reserved[0]);
#endif
  if(gParams.debugTraceLevel > 0)
  {
    printf("%s\n", sysCommand);
  }
  system(sysCommand);

  strcat(dirName, "/perfSimInfo.bin");

  fp = fopen(dirName, "rb");
  if (fp)
  {
    fread(perSimInfo, 1, sizeof(sPerfSim_t), fp);
    fclose(fp);
  }
  else
  {
    printf("Could not open %s\n", dirName);
    return 0;
  }
  memcpy(orderedPerSimInfo, perSimInfo, sizeof(sPerfSim_t));

  int modelSize = tidl_getModelSize(params->outputNetFile);
  sTIDL_Network_t * tidlNet = (sTIDL_Network_t *)malloc(modelSize);
  if(tidlNet == NULL)
  {
    printf("Could not Allocate memory for model read\n");
    return 0;
  }
  fp = fopen((const char *)params->outputNetFile, "rb");
  if (fp)
  {
    fread(tidlNet, 1, modelSize, fp);
    fclose(fp);
  }
  else
  {
    printf("Could not open %s\n", params->outputNetFile);
    return 0;
  }

  numFeatureBytes= (params->numFeatureBits + 7)>>3;

  for (i = 0; i < tidlNet->numLayers; i++)
  {
    //printf("%2d : | %d %d | %d %d |\n", i, tidlNet->TIDLLayers[i].outData[0].padW, perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padC,
    //  tidlNet->TIDLLayers[i].outData[0].padH, perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padR);
    if(tidlNet->TIDLLayers[i].numOutBufs > 0)
    {
      tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_LINE_PITCH] = tidlNet->TIDLLayers[i].outData[0].dimValues[TIDL_DIM_WIDTH] +
        perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padC;
      tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH] = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].bufWidth/numFeatureBytes;
      //tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_ROI_PITCH] = tidlNet->TIDLLayers[i].outData[0].dimValues[TIDL_DIM_NUMCH] * tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH];
      tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_ROI_PITCH] = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].bufSize/numFeatureBytes;
      tidlNet->TIDLLayers[i].outData[0].padW = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padC;
      tidlNet->TIDLLayers[i].outData[0].padH = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padR;
      if((tidlNet->TIDLLayers[i].layerType == TIDL_PoolingLayer) && (tidlNet->TIDLLayers[i].layerParams.poolParams.kernelW == 0))
      {
        tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH] = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].bufWidth = tidlNet->TIDLLayers[i].outData[0].dimValues[TIDL_DIM_WIDTH];
        tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_ROI_PITCH] = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].bufSize = tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH]*tidlNet->TIDLLayers[i].outData[0].dimValues[TIDL_DIM_BATCH];
        tidlNet->TIDLLayers[i].outData[0].padW = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padC = 0;
        tidlNet->TIDLLayers[i].outData[0].padH = perSimInfo->sdataFlowInfo[i].bufInfo[OUT_FEAT_MAP][WRITE].padR = 0;
      }
      else if(tidlNet->TIDLLayers[i].layerType == TIDL_InnerProductLayer)
      {
        tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH] =  tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_ROI_PITCH];
      }

      if(((tidlNet->TIDLLayers[i].layersGroupId != currLayersGroupId) && tidltb_isInDataBuff(tidlNet, tidlNet->TIDLLayers[i].outData[0].dataId,currLayersGroupId)) ||
              tidltb_isOutDataBuff(tidlNet, tidlNet->TIDLLayers[i].outData[0].dataId,currLayersGroupId) )
      {

          int32_t inPadL = tidlNet->TIDLLayers[i].outData[0].padW;
          int32_t inPadT = tidlNet->TIDLLayers[i].outData[0].padH;
          int32_t inPadR = 0;
          int32_t inPadB = 0;
          if(tidlNet->TIDLLayers[i].outData[0].padW)
            inPadB = tidlNet->TIDLLayers[i].outData[0].padH + 1;
          else
            inPadB = tidlNet->TIDLLayers[i].outData[0].padH;

          int32_t tempChannelPicth = tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_LINE_PITCH] *
                                               (tidlNet->TIDLLayers[i].outData[0].dimValues[TIDL_DIM_HEIGHT] + inPadT + inPadB);

          if(tempChannelPicth < tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH])
          {
            int32_t totalHeight = (tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH] +
                                   tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_LINE_PITCH] -1)/ tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_LINE_PITCH];
            tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH] = totalHeight * tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_LINE_PITCH];
          }
          else
          {
            tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH] = tempChannelPicth;
          }

          tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_ROI_PITCH] = tidlNet->TIDLLayers[i].outData[0].pitch[TIDL_CHANNEL_PITCH]*
                                                                    tidlNet->TIDLLayers[i].outData[0].dimValues[TIDL_DIM_NUMCH];


      }
    }

  }

  for (i = 0; i < tidlNet->numLayers; i++)
  {
    for (j = 0; j < tidlNet->TIDLLayers[i].numInBufs; j++)
    {
      foundInData = 0;
      for (k = 0; ((k < tidlNet->numLayers) && (foundInData == 0)); k++)
      {
        for (l = 0; ((l < tidlNet->TIDLLayers[k].numOutBufs) && (foundInData == 0)); l++)
        {
          if (tidlNet->TIDLLayers[i].inData[j].dataId == tidlNet->TIDLLayers[k].outData[l].dataId)
          {
            if ((tidlNet->TIDLLayers[i].layerType == TIDL_PoolingLayer) && (tidlNet->TIDLLayers[i].layerParams.poolParams.kernelW == 0))
            {
              perSimInfo->sdataFlowInfo[k].bufInfo[OUT_FEAT_MAP][WRITE].padCFillZeros = perSimInfo->sdataFlowInfo[k].bufInfo[OUT_FEAT_MAP][WRITE].padC;
            }
            tidlNet->TIDLLayers[i].inData[j].padW = tidlNet->TIDLLayers[k].outData[l].padW;
            tidlNet->TIDLLayers[i].inData[j].padH = tidlNet->TIDLLayers[k].outData[l].padH;
            tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_LINE_PITCH]    = tidlNet->TIDLLayers[k].outData[l].pitch[TIDL_LINE_PITCH];
            tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_CHANNEL_PITCH] = tidlNet->TIDLLayers[k].outData[l].pitch[TIDL_CHANNEL_PITCH];
            tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_ROI_PITCH]     = tidlNet->TIDLLayers[k].outData[l].pitch[TIDL_ROI_PITCH];
            foundInData = 1;
          }
        }
      }
      if (foundInData == 0)
      {
        printf("Could not find Indata for data ID %d \n", tidlNet->TIDLLayers[i].inData[j].dataId);
      }
    }
  }
  int netSize = (char*)(&tidlNet->TIDLLayers[tidlNet->numLayers]) - (char*)(tidlNet);
  sTIDL_Network_t * tempTidlNet = (sTIDL_Network_t *)malloc(netSize);
  memcpy(tempTidlNet, tidlNet, netSize);
  for (i = 0; i < tidlNet->numLayers; i++)
  {
    tempTidlNet->TIDLLayers[i] = tidlNet->TIDLLayers[perSimInfo->layerExecutionOrder[i]];
    orderedPerSimInfo->sdataFlowInfo[i] = perSimInfo->sdataFlowInfo[perSimInfo->layerExecutionOrder[i]];
  }
  memcpy(tidlNet, tempTidlNet, netSize);

  uint32_t alignedModelSize = modelSize;
  sBuffer_t buf;
  buf.ptr = (void*)orderedPerSimInfo;
  buf.bufSize = (sizeof(sDataFlowInfo_t) * tidlNet->numLayers);
  tidlNet->dataFlowInfo = TIDL_alignParamsWrite(NULL, &buf, &alignedModelSize, 1);

  fp = fopen((const char *)params->outputNetFile, "wb+");
  if (fp)
  {
    fwrite(tidlNet, 1, modelSize, fp);
  }
  else
  {
    printf("Could not open %s\n", params->outputNetFile);
    return 0;
  }
  alignedModelSize = modelSize;
  TIDL_writeInfo(tidlNet, &orgTIDLNetStructure, (const char *)params->outputParamsFile, tidlNet->numLayers, 1, orderedPerSimInfo);
  
  TIDL_alignParamsWrite(fp, &buf, &alignedModelSize, 1);
  fclose(fp);
  return 0;

}

int tidlRunQuantStatsTool(void)
{
  FILE * fp;

  if(!gParams.executeQuantsTool)
  {
    return 0;
  }
  fp = fopen((const char *)gParams.inData, "r");
  if (fp == NULL)
  {
    printf("INFO : Couldn't open inData file: %s  , Skipping Range Collection for Quantization \n", gParams.inData);
    return(0);
  }
  fclose(fp);

  fp = fopen((const char *)gParams.tidlStatsTool, "r");
  if (fp == NULL)
  {
    printf("Couldn't open tidlStatsTool file: %s  \n", gParams.tidlStatsTool);
    return(0);
  }
  fclose(fp);

  if (gParams.numFrames == -1)
  {
    if (gParams.inFileFormat == 2)
    {
      gParams.numFrames = getNumberOfLinesIntheFile((char *)gParams.inData);
    }
    else if ((gParams.inFileFormat == 0) || (gParams.inFileFormat == 1) || (gParams.inFileFormat == 3))
    {
      gParams.numFrames = 1;
    }
  }
  if (gParams.numFrames > 0)
  {
    printf("\n~~~~~Running TIDL in PC emulation mode to collect Activations range for each layer~~~~~\n");
    tidlQuantStatsTool(&gParams);
  }

}

/**
----------------------------------------------------------------------------
@ingroup    TIDL_Import
@fn         tidlValidateImportParams
@brief      Function validates input parameters related to tidl import
            sets appropriate error in response to violation from
            expected values.

@param      params : TIDL Create time parameters
@remarks    None
@return     Error related to parameter.
----------------------------------------------------------------------------
*/
int32_t tidlValidateImportParams(tidl_import_config * params)
{

  if(checkMandatoryParams(&gsTokenMap_tidl_import_config[0]) != 0)
  {
      printf("\nMissing one or more mandatory parameters");
      return -1;
  }

  /* randParams can be either 0 or 1*/
  if(params->foldMaxPoolInConv2D == 1)
  {
     params->foldMaxPoolInConv2D = 0;
     printf("\n foldMaxPoolInConv2D is NOT suported in the current release. Disabling Now");
  }
  if(params->foldEltWiseInConv2D == 1)
  {
     params->foldEltWiseInConv2D = 0;
     printf("\n foldEltWiseInConv2D is NOT suported in the current release. Disabling Now");
  }
  if((params->numParamBits > 8) && (params->numFeatureBits <= 8))
  {
    params->numFeatureBits = 16;
  }
  if((params->numParamBits <= 8) && (params->numFeatureBits > 8))
  {
    params->numParamBits = 12;
  }

  if((params->randParams != 0) && (params->randParams != 1))
  {
    printf("\n Invalid randParams setting : set either 0 or 1");
    return -1;
  }
  /* modelType can be either 0 or 1*/
  else if((params->modelType != 0) && (params->modelType != 1) && (params->modelType != 2) && (params->modelType != 3))
  {
    printf("\n Invalid modelType parameter setting : set either 0 or 1 or 2 or 3");
    return -1;
  }
  /* Currently quantizationStyle = 1 is supported */
  /*else if(params->quantizationStyle != 1)
  {
    printf("\n Invalid quantizationStyle parameter setting : set it to 1");
    return -1;
  }*/
  /* quantRoundAdd can be 0 to 100 */
  else if((params->quantRoundAdd < 0) || (params->quantRoundAdd > 100))
  {
    printf("\n Invalid quantRoundAdd parameter setting : set it 0 to 100");
    return -1;
  }
  /* numParamBits can be 4 to 12 */
  else if((params->numParamBits < 4) || (params->numParamBits > 16))
  {
    printf("\n Invalid numParamBits parameter setting : set it 4 to 16");
    return -1;
  }
  /* inFileFormat can be either 0 or 1*/
  else if((params->inFileFormat < 0) && (params->inFileFormat > 2))
  {
    printf("\n Invalid inFileFormat parameter setting");
    return -1;
  }
  /* numFrames can be >0  */
  else if(params->numFrames < -1)
  {
    printf("\n Invalid numFrames parameter setting : set it to >0 ");
    return -1;
  }
  /* foldBnInConv2D can be either 0 or 1*/
  else if((params->foldBnInConv2D != 0) && (params->foldBnInConv2D != 1))
  {
    printf("\n Invalid foldBnInConv2D parameter setting : set either 0 or 1");
    return -1;
  }
  /* preProcType can be 0 to 3 */
  else if((params->preProcType < 0) || (params->preProcType > 3))
  {
    printf("\n Invalid preProcType parameter setting : set it 0 to 3");
    return -1;
  }
  /* inElementType can be either 0 or 1*/
  else if((params->inElementType[0] < 0) && (params->inElementType[0] > 3 ))
  {
    printf("\n Invalid inElementType parameter setting : set either 0 to 3");
    return -1;
  }
  /* inQuantFactor can be >0  */
  else if(params->inQuantFactor[0] <= 0)
  {
    printf("\n Invalid inQuantFactor parameter setting : set it to >0 ");
    return -1;
  }
  /* inWidth can be >0  */
  else if((params->inWidth[0] < -1) || (params->inWidth[0] == 0))
  {
    printf("\n Invalid inWidth parameter setting : set it to >0 ");
    return -1;
  }
  /* inHeight can be >0  */
  else if((params->inHeight[0] < -1) || (params->inHeight[0] == 0))
  {
    printf("\n Invalid inHeight parameter setting : set it to >0 ");
    return -1;
  }
  /* inNumChannels can be 1 to 1024  */
  else if((params->inNumChannels[0] < -1) || (params->inNumChannels[0] == 0) || (params->inNumChannels[0] > 1024))
  {
    printf("\n Invalid inNumChannels parameter setting : set it 1 to 1024 ");
    return -1;
  }
  else if((params->numRoi[0] < -1) || (params->numRoi[0] == 0) || (params->numRoi[0] > 1024))
  {
    printf("\n Invalid numRoi parameter setting : set it 1 to 1024 ");
    return -1;
  }
  else
  {
    return 0;
  }

}

void caffe_import( tidl_import_config * params);
void caffe_import_new( tidl_import_config * params);
void tf_import(tidl_import_config * params);

void tidl_updateNetPicth(sTIDL_Network_t * tidlNet)
{
  int32_t i, j;
  for (i = 0; i < tidlNet->numLayers; i++)
  {
    for (j = 0; j < tidlNet->TIDLLayers[i].numOutBufs; j++)
    {
      tidlNet->TIDLLayers[i].outData[j].pitch[TIDL_LINE_PITCH] = tidlNet->TIDLLayers[i].outData[j].dimValues[TIDL_DIM_WIDTH] +
        tidlNet->TIDLLayers[i].outData[j].padW;
      int32_t isPadW = tidlNet->TIDLLayers[i].outData[j].padW ? 1 : 0;
      tidlNet->TIDLLayers[i].outData[j].pitch[TIDL_CHANNEL_PITCH] = tidlNet->TIDLLayers[i].outData[j].pitch[TIDL_LINE_PITCH] * (tidlNet->TIDLLayers[i].outData[j].dimValues[TIDL_DIM_HEIGHT] + tidlNet->TIDLLayers[i].outData[j].padH * 2 + isPadW);
      tidlNet->TIDLLayers[i].outData[j].pitch[TIDL_ROI_PITCH] = tidlNet->TIDLLayers[i].outData[j].dimValues[TIDL_DIM_NUMCH] * tidlNet->TIDLLayers[i].outData[j].pitch[TIDL_CHANNEL_PITCH];
    }
  }
  for (i = 0; i < tidlNet->numLayers; i++)
  {
    for (j = 0; j < tidlNet->TIDLLayers[i].numInBufs; j++)
    {
      tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_LINE_PITCH] = tidlNet->TIDLLayers[i].inData[j].dimValues[TIDL_DIM_WIDTH] +
        tidlNet->TIDLLayers[i].inData[j].padW;
      int32_t isPadW = tidlNet->TIDLLayers[i].inData[j].padW ? 1 : 0;
      tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_CHANNEL_PITCH] = tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_LINE_PITCH] * (tidlNet->TIDLLayers[i].inData[j].dimValues[TIDL_DIM_HEIGHT] + tidlNet->TIDLLayers[i].inData[j].padH * 2 + isPadW);
      tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_ROI_PITCH] = tidlNet->TIDLLayers[i].inData[j].dimValues[TIDL_DIM_NUMCH] * tidlNet->TIDLLayers[i].inData[j].pitch[TIDL_CHANNEL_PITCH];
    }
  }
}

void printTokenHelpAndExit(char *appname, char *token)
{
  int c = 0;

  printf("* %s --help-%s\n"
         "\n", appname, token);
  while(gsTokenMap_tidl_import_config[c].tokenName) {
    if(!strcmp(token, gsTokenMap_tidl_import_config[c].tokenName)) {
      printf("--%-28s : %s\n"
             "\n", token, gsTokenMap_tidl_import_config[c].shortHelp);
      if(gsTokenMap_tidl_import_config[c].mandatory)
      {
        printf("[Mandatory parameter]\n"
               "\n");
      }
      else
      {
        printf("[Optional parameter]\n"
               "\n");
      }
      if(gsTokenMap_tidl_import_config[c].longHelp)
        printf("%s\n", gsTokenMap_tidl_import_config[c].longHelp);
      else
        printf("%s does not have extended help\n", token);
      exit(0);
    }
    c++;
  }

  /* Should not reach here */
  exit(0);
}

void printHelpAndExit(char *appname)
{
  int c = 0;

  printf("usage:\n");
  printf("* %s -h\n", appname);
  printf("* %s --help\n", appname);
  printf("* %s --help-<parameter>\n", appname);
  printf("* %s <config-file> [OPTIONS]\n", appname);
  printf("\n");

  printf("This tool reads import parameters from the config file.\n"
         "\n"
         "A parameter read from config file can be overridden by adding\n"
         "\"--<parameter> <arg0> [<arg1> ...]\" to the command line options\n"
         "\n"
         "A parameter can also be added (if not provided by config file)\n"
         "by adding \"--<parameter> <arg0> [<arg1> ...]\" to the command\n"
         "line options\n"
         "\n"
         "The list of supported import parameters is given below.\n"
         "You can get details about each parameter by running the application\n"
         "with a --help-<parameter> option.\n"
         "\n"
         "For example, to know more about \"modelType\" parameter, run the\n"
         "application as:\n"
         "%s --help-modelType\n"
         "\n", appname);
  printf("\n"
         "Mandatory import parameters:\n");
  while(gsTokenMap_tidl_import_config[c].tokenName) {
    if(gsTokenMap_tidl_import_config[c].mandatory)
    {
      printf("--%-28s: %s\n", gsTokenMap_tidl_import_config[c].tokenName, gsTokenMap_tidl_import_config[c].shortHelp);
    }
    c++;
  }
  printf("\n");
  printf("Supported optional import parameters:\n");
  c = 0;
  while(gsTokenMap_tidl_import_config[c].tokenName) {
    if(!gsTokenMap_tidl_import_config[c].mandatory)
    {
      printf("--%-28s: %s\n", gsTokenMap_tidl_import_config[c].tokenName, gsTokenMap_tidl_import_config[c].shortHelp);
    }
    c++;
  }

  exit(0);
}

bool isOptionAToken(const char *opt)
{
  int c = 0;
  while(gsTokenMap_tidl_import_config[c].tokenName) {
    if(!strcmp(opt, gsTokenMap_tidl_import_config[c].tokenName))
      return true;
    c++;
  }

  return false;
}

bool isOptionATokenHelp(const char *opt, char **ret_token, bool leadingDashes)
{
  int c = 0;
  const char *helpToken = leadingDashes ? "--help-" : "help-";

  while(gsTokenMap_tidl_import_config[c].tokenName) {
    char *token = gsTokenMap_tidl_import_config[c].tokenName;
    if(strstr(opt, token) &&
        strlen(opt) == strlen(helpToken) + strlen(token) &&
        !strncmp(opt, helpToken, strlen(helpToken))) {
      *ret_token = token;
      return true;
    }
    c++;
  }

  return false;
}

/* Convenience function to add a deferredOption to a NULL-terminated list */
struct deferredOption *addDeferredOption(struct deferredOption *d_opts, struct deferredOption *new_d_opt)
{
  int i = 0;

  while(d_opts[i].opt)
  {
    i++;
  }

  /* old Sentinel */
  i++;

  d_opts = (struct deferredOption *)realloc(d_opts, sizeof(struct deferredOption) * (i + 1));

  /* new sentinel */
  memset(&d_opts[i], 0, sizeof(struct deferredOption));
  /* new option */
  d_opts[i - 1].opt = new_d_opt->opt;
  d_opts[i - 1].args = new_d_opt->args;

  return d_opts;
}

/* Convenience function to add an option argumnet to a NULL-terminated list */
char **addArg(char **args, char *new_arg)
{
  int i = 0;

  while(args[i])
  {
    i++;
  }

  /* old Sentinel */
  i++;

  args = (char **)realloc(args, sizeof(char *) * (i + 1));

  /* new sentinel and arg */
  args[i] = NULL;
  args[i - 1] = new_arg;

  return args;
}

/* Convenience function to add an option to a NULL-terminated list */
struct option *addOption(struct option *lopts, char *opt, int flag)
{
  int i = 0;

  while(lopts[i].name)
  {
    i++;
  }

  /* old Sentinel */
  i++;

  lopts = (struct option *)realloc(lopts, sizeof(struct option) * (i + 1));

  /* new sentinel */
  memset(&lopts[i], 0, sizeof(struct option));
  /* new option */
  lopts[i - 1].name = strdup(opt);
  lopts[i - 1].has_arg = flag;
  lopts[i - 1].flag = NULL;
  lopts[i - 1].val = 0;

  return lopts;
}

int32_t main(int32_t argc, char *argv[])
{
  int32_t c;
  int lindex;
  struct option *lopts;
  struct deferredOption *d_opts;
  char *token = NULL;
  int32_t status = 0;
  int32_t runQuantStep = 0;
  FILE * fp;
  if(argc < 2)
  {
    printHelpAndExit(argv[0]);
  }

  /*Build a list of options that can be parsed */
  lopts = (struct option *)calloc(sizeof(struct option), 1);

  /* Add --help as an option */
  lopts = addOption(lopts, "help", no_argument);

  c = 0;
  while(gsTokenMap_tidl_import_config[c].tokenName)
  {
    char *str;
    lopts = addOption(lopts, gsTokenMap_tidl_import_config[c].tokenName, required_argument);

    asprintf(&str, "help-%s", gsTokenMap_tidl_import_config[c].tokenName);
    lopts = addOption(lopts, str, no_argument);
    free(str);

    c++;
  }

  if(!strcmp(argv[1], "--help"))
    printHelpAndExit(argv[0]);
  else if(!strcmp(argv[1], "-h"))
    printHelpAndExit(argv[0]);
  else if(isOptionATokenHelp((const char *)argv[1], &token, true))
    printTokenHelpAndExit(argv[0], token);

  /* Must be config file if it did not exit */
  fp = fopen(argv[1], "r");
  if(fp== NULL)
  {
    printf("Could not open config  file : %s  \n",argv[1]);
    return(0);
  }
  fclose(fp);
  strcpy(inConfgFilename, argv[1]);

  /*
   * First stage, parse --help, -h and --help-<p>, and build
   * a list of deferred options
   *
   * Do not parse any of the --<p> options yet, as they
   * are supposed to override values later.
   */
  d_opts = (struct deferredOption *)calloc(sizeof(struct deferredOption), 1);
  optind = 2;

  while((c = getopt_long(argc, argv, "h", lopts, &lindex)) != -1)
  {
    const char *lopt = NULL;

    token = NULL;

    switch(c)
    {
      case 0:
        lopt = lopts[lindex].name;

        if(isOptionAToken(lopt)) {
          int k;
          struct deferredOption d_opt;

          d_opt.opt = (char *)lopt;
          d_opt.args = (char **)calloc(sizeof(char *), 1);

          d_opt.args = addArg(d_opt.args, optarg);

          k = 0;
          while(argv[optind + k] && strncmp("-", argv[optind + k], strlen("-"))) {
              d_opt.args = addArg(d_opt.args, argv[optind + k]);
              k++;
          }

          d_opts = addDeferredOption(d_opts, &d_opt);
        }

        else if(isOptionATokenHelp(lopt, &token, false))
          printTokenHelpAndExit(argv[0], token);
        else if(!strcmp(lopt, "help"))
          printHelpAndExit(argv[0]);
        else
          printHelpAndExit(argv[0]);

        break;
      case 'h':
        printHelpAndExit(argv[0]);
      default:
        printHelpAndExit(argv[0]);
    }
  }

  setDefaultParams(&gParams);

  status = readparamfile(argv[1], &gsTokenMap_tidl_import_config[0]) ;

  if(status == -1)
  {
    printf("Parser Failed");
    return -1 ;
  }

  /*
   * Second stage, parse deferred options and override
   */
  c = 0;
  while(d_opts[c].opt)
  {
    status = readparamarg(d_opts[c].opt, &gsTokenMap_tidl_import_config[0], d_opts[c].args);
    if(status == -1)
    {
      printf("Parser Failed");
      return -1 ;
    }
    c++;
  }

  status = tidlValidateImportParams(&gParams);
  if(status == -1)
  {
    printf("\n Validation of Parameters Failed \n");
    return -1 ;
  }

  /*  inputNetFile && inputParamsFile */
  fp = fopen((const char *)gParams.inputNetFile, "r");
  if(fp== NULL)
  {
    printf("Couldn't open inputNetFile file: %s  \n", gParams.inputNetFile);
    return(0);
  }
  fclose(fp);

  if(gParams.modelType == 0)
  {
    fp = fopen((const char *)gParams.inputParamsFile, "r");
    if(fp== NULL)
    {
      printf("Couldn't open inputParamsFile file: %s  \n", gParams.inputParamsFile);
      return(0);
    }
    fclose(fp);
  }
  if((gParams.metaArchType == 1) || (gParams.metaArchType == 2))
  {
    tf_metaArch_import(&gParams);
  }
  else if(gParams.metaArchType == 3)
  {
    tidl_metaArch_import(&gParams);
  }
  else if(gParams.metaArchType != -1)
  {
    printf("Unsupported  metaArchType : %d  \n", gParams.metaArchType);
    return(0);
  }


  if(gParams.modelType == 0)
  {
    caffe_import_new(&gParams);
    //caffe_import(&gParams);
  }
  else if (gParams.modelType == 1)
  {
    tf_import(&gParams);
  }
  else if (gParams.modelType == 2)
  {
    onnx_import(&gParams);
  }
   else if (gParams.modelType == 3)
  {
    tfLite_import(&gParams);
  }
  int32_t tiLayerIndex;
  int32_t layerIndex = orgTIDLNetStructure.numLayers;

  TIDL_convertDeconv2DtoConv(orgTIDLNetStructure, layerIndex);
  tidl_makeDataIdLayerIdSame(&orgTIDLNetStructure, layerIndex);
  /* Quantize Layer Params */
  //TIDL_thresholdParams(orgTIDLNetStructure, layerIndex); //My modification
  TIDL_importQuantLayerParams(orgTIDLNetStructure, layerIndex);
  tiLayerIndex = tidl_copyPCNetToDeviceNet(orgTIDLNetStructure, tIDLNetStructure, layerIndex);

  tidl_addOutDataLayer(tIDLNetStructure, tiLayerIndex);

  TIDL_fillDataBufPadRequirements(&tIDLNetStructure);
  tidl_updateNetPicth(&tIDLNetStructure);
  tIDLNetStructure.isQuantStatsAvailable = 0;
  /* Offsets in the Net needs to be updated before writing the net to file */
  TIDL_writeModel(&tIDLNetStructure, &orgTIDLNetStructure, (const char *)NULL, layerIndex);
  TIDL_writeModel(&tIDLNetStructure, &orgTIDLNetStructure, (const char *)(&gParams)->outputNetFile, layerIndex);
  TIDL_writeInfo(&tIDLNetStructure,  &orgTIDLNetStructure,  (const char *)(&gParams)->outputParamsFile, tIDLNetStructure.numLayers, 1, NULL);

  tidlRunQuantStatsTool();
  if(gParams.executeNetworkCompiler)
  {
    tidlRunPerfSimTool(&gParams);
  }
  tidlWriteTensorNamesToFile(&gParams);
  tidlRunGraphVizTool(&gParams);

  tidlModelCheck(&gParams, &orgTIDLNetStructure);

  return (0);
}
