//
// BeagleBone Black  Bare Metal UDP/TCP/MMC demo
//
// mixed C & assembly 
// demonstrates Ethernet, MMC, MMU, DMA, Cache, IRQ, UART0, RTC and GPIO usage
//
// This is a modded port [TI Starterware] and other contributors 
//
// The BBB will tell you it's MAC address and IP# then it will spew 
// all the packets out UART0 as the lwIP echoserver returns them.
// Ping packets are 14+60 bytes long, you can check their contents.  
// It will also respond to HTTP requests made via a browser to that IP#.  
//
// Plug your BBB into your router and ping from a PC:
// ping 10.0.0.2    (or whatever IP the BBB is assigned)
//
// Hit the boot button to flash a USR LED (interrupt function check)
// Enter ? to display filesystem manipulation functions on the UART console
// USERLEDs will flash with IRQ activity but ethernet is too fast to see 
// 
// Ethernet i/o is handled by ISR.
// Incoming traffic comes via ethernet_input() in etharp.c and *packet_ptr below.  
// Outgoing traffic goes via tcp_write() in tcp_out.c, echod.c and httpd.c   
// See rawapi.txt in /doc for interface info.  Most of the ethernet middleware 
// is in cpswif.c .  Lower code is in eth_hwif.c and eth.s .  
//
// lwIP is a Picasso!  It is quite intricate so it is untouched.  
// The ethernet drivers are pretty good also and required 
// very little tweaking to build.  Changes are marked with "// n!"
//
// The MMC/FAT32 filesystem runs concurrently.  MMC i/o is handled by ISR
// Chan at www.elm-chan.org wrote a great multi-platform micro-filesystem utility.
// The MMC drivers were also well written and are mostly detangled now.  
// The Ethernet drivers are not (yet) detangled but they are not so bad.  
// Also there are many unused routines in the libraries.  
// Removing them will save much memory.  Total image size is only 85k Bytes!  
// 
// Built with GNU/ARM tools :) on platform Win32 :(
// Also builds with GNU tools on (Raspberry) Pi-Top laptop running Linux :) 
// Load/boot via UART, MMC or jtag   see the Makefile  
//
// Boot sequence:  
//   bootloader "MLO" resides on MMC then bootloader loads image "app" 
//   via UART0/xmodem (terminal emulator minicom on Linux or ExtraPutty on Win32)
//   image is loaded to 0x80000000 and execution begins there
//   per Fast External Booting 26.1.6 spruh73n
//   - or -    boot via MMC card
//   rename "MLO_MMC" to "MLO" this will load/run "app" automatically 
//   All these are provided in the zipfile  
//
// Known bugs/TODO:
//   MMC file copy only 1Mbyte/sec - needs further debugging
//   move MLO bootloader to ROM - for dev/debug
//   move image to on-board NAND - for runtime release
//   UART xmodem file transfer
//   implement web page serving from the filesystem
//
// This is somewhat ready to build a tough little server that can be 
// integrated with sensors and actuators for Internet-of-Things.  Or you
// can strip it down to pure TCP/UDP for blockchain, torrent, firewall
// or encrypted communication.  It is totally self-contained and free.  
// It is not so easy at first glance, but it is do-able.  The building-
// blocks are all here.  Collaboration is most welcome... 
// 
// get involved   www.baremetal.tech            d7@hushmail.com
//
// LEGAL NOTICE:  This is abandoned knowledge.  It does not work.  
//                All references to TI and other corporations are 
//                provided for information purposes only.  
//
#include <stdint.h>
#include "bbb_main.h"

#include "locator.h"
#include "echod.h"
#include "httpd.h"
#include "lwiplib.h"
#include "lwipopts.h"
#include "soc_AM335x.h"
#include "beaglebone.h"
#include "hw_control_AM335x.h"

#include "dma.h"
#include "dma_event.h"
#include "ff.h"
#include "mmc_hwif.h"
#include "mmc_api.h"
#include "con_uif.h"

#define MMC_IN_FREQ              96000000 // 96MHz 
#define MMC_INIT_FREQ            400000   // 400kHz 
#define MMC_CARD_DETECT_PINNUM   6
#define EVT_QUEUE_NUM            0    // DMA Event queue number - MMC
#define REGION_NUMBER            0    // DMA Region number - MMC
#define MMC_BLK_SIZE             512
#define MMC_DATA_SIZE            512  // Global data pointer
mmcCardInfo sdCard;       // SD card info struc 
mmcCtrlInfo  ctrlInfo;    // Controller info struc

// DMA callback function array - MMC
static void (*cb_Fxn[DMA_NUM_TCC]) (unsigned int tcc);

// global packet handles to ethernet_input()
uint32_t eth_rx_count;
uint32_t packet_len;
char *packet_ptr;

// Global flags for interrupt handling 
volatile unsigned int sdBlkSize = MMC_BLK_SIZE;
volatile unsigned int callbackOccured = 0; 
volatile unsigned int xferCompFlag = 0; 
volatile unsigned int dataTimeout = 0;
volatile unsigned int cmdCompFlag = 0;
volatile unsigned int cmdTimeout = 0; 
volatile unsigned int errFlag = 0;

unsigned int uart0_irq_old=0;
extern char *CwdBuf;
extern FRESULT FATFsMount(unsigned int driveNum, void *ptr, char *CmdBuf);
extern void CmdLineProcess(char *CmdBuf);
extern int Cmd_help(int argc, char *argv[]);

//
// check mmc0 command status
//
unsigned int mmc0_get_cmd_status(mmcCtrlInfo *ctrl) {
  unsigned int status = 0;

  while ((cmdCompFlag == 0) && (cmdTimeout == 0));  // block here
  if (cmdCompFlag) {
    status = 1;
    cmdCompFlag = 0;
  }
  if (cmdTimeout) {
    status = 0;
    cmdTimeout = 0;
  }
  return status;
}

//
// check mmc0 transfer status
//
unsigned int mmc0_get_xfer_status(mmcCtrlInfo *ctrl) {
  unsigned int status = 0;
  volatile unsigned int timeOut = 0xFFFF;

  while ((xferCompFlag == 0) && (dataTimeout == 0));  // block here
  if (xferCompFlag) {
    status = 1;
    xferCompFlag = 0;
  }
  if (dataTimeout) {
    status = 0;
    dataTimeout = 0;
  }
  if (HWREG(ctrl->memBase + MMC_CMD) & MMC_CMD_DP) {  // poll for callback 
nothing(); //  this is necessary prog hangs  - TODO
    while(callbackOccured == 0 && ((timeOut--) != 0));  // block here
    callbackOccured = 0;
    if(timeOut == 0) status = 0;
  }
  ctrlInfo.dmaEnable = 0;
  return status;
}

//
// callback from DMA Completion Handler - MMC
//
static void callback(unsigned int tccNum) {
    callbackOccured = 1;
    DMADisableTransfer(SOC_DMA0CC_0_REGS, tccNum, DMA_TRIG_MODE_EVENT);
}

//
// DMA config - MMC
//
static void dma_config(void) {
  dma_clk_cfg();
  DMAInit(SOC_DMA0CC_0_REGS, EVT_QUEUE_NUM);
  // Request DMA Channel and TCC for MMC Transmit
  DMARequestChannel(SOC_DMA0CC_0_REGS, DMA_CHANNEL_TYPE_DMA,
                      DMA_CHA_MMC0_TX, DMA_CHA_MMC0_TX, EVT_QUEUE_NUM);
  // Registering Callback Function for TX
  cb_Fxn[DMA_CHA_MMC0_TX] = &callback;
  // Request DMA Channel and TCC for MMC Receive 
  DMARequestChannel(SOC_DMA0CC_0_REGS, DMA_CHANNEL_TYPE_DMA,
                      DMA_CHA_MMC0_RX, DMA_CHA_MMC0_RX, EVT_QUEUE_NUM);
  // Registering Callback Function for RX
  cb_Fxn[DMA_CHA_MMC0_RX] = &callback;
}

//
// setup a DMA receive transfer - MMC
//
void MMCRxDMAsetup(void *ptr, unsigned int blkSize, unsigned int nblks) {
  DMACCPaRAMEntry paramSet;
  paramSet.srcAddr    = ctrlInfo.memBase + MMC_DATA;
  paramSet.destAddr   = (unsigned int)ptr;
  paramSet.srcBIdx    = 0;
  paramSet.srcCIdx    = 0;
  paramSet.destBIdx   = 4;
  paramSet.destCIdx   = (unsigned short)blkSize;
  paramSet.aCnt       = 0x4;
  paramSet.bCnt       = (unsigned short)blkSize/4;
  paramSet.cCnt       = (unsigned short)nblks;
  paramSet.bCntReload = 0x0;
  paramSet.linkAddr   = 0xffff;
  paramSet.opt        = 0;
  paramSet.opt |= ((DMA_CHA_MMC0_RX << DMACC_OPT_TCC_SHIFT) & DMACC_OPT_TCC);
  paramSet.opt |= (1 << DMACC_OPT_TCINTEN_SHIFT);  // Tx complete irq enab
  paramSet.opt |= (1 << 0);  // read FIFO : SRC Constant addr mode
  paramSet.opt |= (2 << 8);  // SRC FIFO width 32 bit 
  paramSet.opt |= (1 << 2);  // AB-Sync mode 
  DMASetPaRAM(SOC_DMA0CC_0_REGS, DMA_CHA_MMC0_RX, &paramSet);
  DMAEnableTransfer(SOC_DMA0CC_0_REGS, DMA_CHA_MMC0_RX, DMA_TRIG_MODE_EVENT);
}

//
// setup a DMA transmit transfer - MMC
//
void MMCTxDMAsetup(void *ptr, unsigned int blkSize, unsigned int blks) {
  DMACCPaRAMEntry paramSet;
  paramSet.srcAddr    = (unsigned int)ptr;
  paramSet.destAddr   = ctrlInfo.memBase + MMC_DATA;
  paramSet.srcBIdx    = 4;
  paramSet.srcCIdx    = blkSize;
  paramSet.destBIdx   = 0;
  paramSet.destCIdx   = 0;
  paramSet.aCnt       = 0x4;
  paramSet.bCnt       = (unsigned short)blkSize/4;
  paramSet.cCnt       = (unsigned short)blks;
  paramSet.bCntReload = 0x0;
  paramSet.linkAddr   = 0xffff;
  paramSet.opt        = 0;
  paramSet.opt |= ((DMA_CHA_MMC0_TX << DMACC_OPT_TCC_SHIFT) & DMACC_OPT_TCC);
  paramSet.opt |= (1 << DMACC_OPT_TCINTEN_SHIFT);  // Tx complete irq enab
  paramSet.opt |= (1 << 1);  // read FIFO : DST constant addr mode 
  paramSet.opt |= (2 << 8);  // DST FIFO width 32 bit 
  paramSet.opt |= (1 << 2);  // AB-Sync mode
  DMASetPaRAM(SOC_DMA0CC_0_REGS, DMA_CHA_MMC0_TX, &paramSet);
  DMAEnableTransfer(SOC_DMA0CC_0_REGS, DMA_CHA_MMC0_TX, DMA_TRIG_MODE_EVENT);
}

//
//  setup a DMA transfer - MMC
//
void MMCXferSetup(mmcCtrlInfo *ctrl, unsigned char rwFlag, void *ptr,
                             unsigned int blkSize, unsigned int nBlks) {
  callbackOccured = 0;
  xferCompFlag = 0;
  if (rwFlag == 1) MMCRxDMAsetup(ptr, blkSize, nBlks);
  else MMCTxDMAsetup(ptr, blkSize, nBlks);

  ctrl->dmaEnable = 1;
  MMCBlkLenSet(ctrl->memBase, blkSize);
}

//
//  DMA channel controller completion IRQ service routine - MMC
//
void DMACompletionISR(void) {
  volatile unsigned int pendingIrqs;
  volatile unsigned int isIPR = 0;
  unsigned int indexl, i=0;

  indexl = 1;
  isIPR = DMAGetIntrStatus(SOC_DMA0CC_0_REGS);
  if(isIPR) {
    while ((i < DMACC_COMPL_HANDLER_RETRY_COUNT)&& (indexl != 0)) {
      indexl = 0;
      pendingIrqs = DMAGetIntrStatus(SOC_DMA0CC_0_REGS);
      while (pendingIrqs) {
        if(pendingIrqs & 1) {
          // if user has not given any callback function
          // while requesting TCC, its TCC specific bit
          // in IPR register will NOT be cleared.
          // here write to ICR to clear corresponding IPR bits 
          DMAClrIntr(SOC_DMA0CC_0_REGS, indexl);
          if (cb_Fxn[indexl] != NULL) {
            (*cb_Fxn[indexl])(indexl);
          }
        }
        ++indexl;
        pendingIrqs >>= 1;
      }
      i++;
    }
  }
}
//
//  DMA channel controller error IRQ service routine - MMC
//
void DMACCErrorISR(void) {
  volatile unsigned int evtqueNum = 0;  // Event Queue Num
  volatile unsigned int pendingIrqs, isIPRH=0, isIPR=0, i=0, index;

  pendingIrqs = 0x0;
  index = 0x1;
  isIPR  = DMAGetIntrStatus(SOC_DMA0CC_0_REGS);
  isIPRH = DMAIntrStatusHighGet(SOC_DMA0CC_0_REGS);
  if((isIPR | isIPRH ) || (DMAQdmaGetErrIntrStatus(SOC_DMA0CC_0_REGS) != 0)
  || (DMAGetCCErrStatus(SOC_DMA0CC_0_REGS) != 0)) {
    // Loop for DMACC_ERR_HANDLER_RETRY_COUNT
    // break when no pending interrupt is found
    while ((i < DMACC_ERR_HANDLER_RETRY_COUNT) && (index != 0)) {
      index = 0;
      if(isIPR) pendingIrqs = DMAGetErrIntrStatus(SOC_DMA0CC_0_REGS);
      else pendingIrqs = DMAErrIntrHighStatusGet(SOC_DMA0CC_0_REGS);
      while (pendingIrqs) {  // Process all pending interrupts
        if(pendingIrqs & 1) {
          // Write to EMCR to clear the corresponding EMR bits 
          // Clear any SER
          if(isIPR) DMAClrMissEvt(SOC_DMA0CC_0_REGS, index);
          else DMAClrMissEvt(SOC_DMA0CC_0_REGS, index + 32);
        }
        ++index;
        pendingIrqs >>= 1;
      }
      index = 0;
      pendingIrqs = DMAQdmaGetErrIntrStatus(SOC_DMA0CC_0_REGS);
      while (pendingIrqs) {  // Process pending interrupts
        if(pendingIrqs & 1) {
          // write to QEMCR to clear corresponding QEMR bits, clear any QSER
          DMAQdmaClrMissEvt(SOC_DMA0CC_0_REGS, index);
        }
        ++index;
        pendingIrqs >>= 1;
      }
      index = 0;
      pendingIrqs = DMAGetCCErrStatus(SOC_DMA0CC_0_REGS);
      if (pendingIrqs != 0) {
        // Process all pending CC error interrupts
        // Queue threshold error for different event queues
        for (evtqueNum = 0; evtqueNum < SOC_DMA_NUM_EVQUE; evtqueNum++) {
          if((pendingIrqs & (1 << evtqueNum)) != 0) {
            // Clear error interrupt
            DMAClrCCErr(SOC_DMA0CC_0_REGS, (1 << evtqueNum));
          }
        }
        // Transfer completion code error 
        if ((pendingIrqs & (1 << DMACC_CCERR_TCCERR_SHIFT)) != 0) {
          DMAClrCCErr(SOC_DMA0CC_0_REGS, (0x01u << DMACC_CCERR_TCCERR_SHIFT));
        }
        ++index;
      }
      i++;
    }
  }
}

//
// Mux out MMC0 and init MMC data structures
//
static void mmc0_config(void) {
  pinmux(CONTROL_CONF_MMC0_DAT3, 0x30); // MUXMODE 0, SLOW SLEW, RX ACTIVE
  pinmux(CONTROL_CONF_MMC0_DAT2, 0x30);
  pinmux(CONTROL_CONF_MMC0_DAT1, 0x30);
  pinmux(CONTROL_CONF_MMC0_DAT0, 0x30);
  pinmux(CONTROL_CONF_MMC0_CLK, 0x30);
  pinmux(CONTROL_CONF_MMC0_CMD, 0x30);
  pinmux(CONTROL_CONF_SPI0_CS1, 0x35); // MUXMODE 5, SLOW SLEW, RX ACTIVE

  ctrlInfo.memBase = SOC_MMC_0_REGS;
  ctrlInfo.intrMask = (MMC_INTR_CMDCOMP | MMC_INTR_CMDTIMEOUT |
                       MMC_INTR_DATATIMEOUT | MMC_INTR_TRNFCOMP);
  ctrlInfo.busWidth = (SD_BUS_WIDTH_1BIT | SD_BUS_WIDTH_4BIT);
  ctrlInfo.highspeed = 1;
  ctrlInfo.ocr = (SD_OCR_VDD_3P0_3P1 | SD_OCR_VDD_3P1_3P2);
  ctrlInfo.card = &sdCard;
  ctrlInfo.ipClk = MMC_IN_FREQ;
  ctrlInfo.opClk = MMC_INIT_FREQ;
  ctrlInfo.cdPinNum = MMC_CARD_DETECT_PINNUM;
  
  sdCard.ctrl = &ctrlInfo;

  callbackOccured = 0;
  xferCompFlag = 0;
  dataTimeout = 0;
  cmdCompFlag = 0;
  cmdTimeout = 0;
}

//
//  mmc0 IRQ service routine
//
void mmc0_isr(void) {
  volatile unsigned int status = 0;

  status = MMCIntrStatusGet(ctrlInfo.memBase, 0xffffffff);
  mmc0_clear_status(status);
  if (status & MMC_STAT_CMDCOMP) cmdCompFlag = 1;
  if (status & MMC_STAT_ERR) {
    errFlag = status & 0xFFFF0000;
    if (status & MMC_STAT_CMDTIMEOUT) cmdTimeout = 1;
    if (status & MMC_STAT_DATATIMEOUT) dataTimeout = 1;
  }
  if (status & MMC_STAT_TRNFCOMP) xferCompFlag = 1;
  if (status & MMC_STAT_CARDINS) {
    FATFsMount(0, &sdCard, &CmdBuf[0]);  // card inserted
    ConsolePrintf("\n\r");               // print current working dir
    CmdLineProcess(&CmdBuf[0]);
  }
  if (status & MMC_STAT_CREM) {  // card removed
    CwdBuf[0] = 0x0;             // wipe current working directory
    callbackOccured = 0;
    xferCompFlag = 0;
    dataTimeout = 0;
    cmdCompFlag = 0;
    cmdTimeout = 0;
    mmc0_init();
    mmc0_irq_enab();
    ConsolePrintf("\n\rPlease insert card... ");
  }
}

//
// main
//
void main(void){
  uint32_t i, ipAddr, packet_count, iblink=0, iled=1, irev=0;
  LWIP_IF lwipIfPort1;

  cache_en();
  mclk_1GHz();
  gpio_init(SOC_GPIO_1_REGS, (0xf << 21));  // enab USR LEDs, pin # 21-24
  uart0_init();
  tim_init();
  rtc_init();
  irq_init();
  mmu_init();
  dma_config();
  mmc0_config();
  mmc0_init();
  mmc0_irq_enab();
  eth_init();

  ipAddr = 0;
  MACAddrGet(0, lwipIfPort1.macArray);  // get MAC address port 1
  ConsolePrintf("\n\rPort 1 MAC:  ");
  for(i = 0; i <= 5; i++) {
    hexprintbyte(lwipIfPort1.macArray[5-i]);
    uart_tx(SOC_UART_0_REGS, 0x20);
  }
  uart_tx(SOC_UART_0_REGS, 0x0d);

  lwipIfPort1.instNum = 0;
  lwipIfPort1.slvPortNum = 1; 
  lwipIfPort1.ipAddr = 0; 
  lwipIfPort1.netMask = 0; 
  lwipIfPort1.gwAddr = 0; 
  lwipIfPort1.ipMode = IPADDR_USE_DHCP; 
  ipAddr = lwIPInit(&lwipIfPort1);
  if(ipAddr) {
    ConsolePrintf("\n\rPort 1 IP Address (ping it): ");
    ConsolePrintf("%d.%d.%d.%d\n\r", (ipAddr & 0xFF), ((ipAddr >> 8) & 0xFF),
                  ((ipAddr >> 16) & 0xFF), ((ipAddr >> 24) & 0xFF));
  } else {
    ConsolePrintf("\n\rPort 1 IP acquisition failed\n\r");
  }

  echo_init();  // init echo server

  httpd_init();  // init http server

  if(!FATFsMount(0, &sdCard, &CmdBuf[0])) ConsolePrintf("Filesystem mounted\n\r");

  CmdBufIdx = 0;            // init command line buffer
  CmdBuf[CmdBufIdx] = '\0';
  ConsolePrintf("Enter ? for help\n\r");
  packet_count = eth_rx_count;
  while(1) {  // loop 4 ever
    //
    // application code here
    // design it as a non-blocking state-machine
    // execution must return back to this loop
    // custom console commands go in file con_uif.c and array CmdTable[]
    // 

    if(packet_count != eth_rx_count) {  // print incoming UDP data
      packet_count = eth_rx_count;
      for(i = 0; i < packet_len; i++) {
        if(!(i % 0x10)) {
          uart_tx(SOC_UART_0_REGS, 0x0d);
          uart_tx(SOC_UART_0_REGS, 0x0a);
        }
        hexprintbyte(*packet_ptr++);
        uart_tx(SOC_UART_0_REGS, 0x20);
      }
      ConsolePrintf("\n\rPKT#%d LEN=%d\n\r", packet_count, packet_len);
      packet_len = 0;
    }

    if(uart0_irq_old != uart0_irq_count) {  // detect console keystroke
      uart0_irq_old = uart0_irq_count & 0x7fffffff;
      if(uart0_rbuf > 0x0) {                // ASCII character received
        char_collect((char)uart0_rbuf);     // console command execution
        uart0_rbuf = 0x0;
      }
    }

    if(iblink == 0x0) gpio_on(SOC_GPIO_1_REGS, iled<<21);  // blink USRLEDs
    if(iblink == 0xffff) {             // blink intensity
      gpio_off(SOC_GPIO_1_REGS, iled<<21);
      if(irev == 0) iled = iled<<1;
      else iled = iled>>1;
      if(iled == 0x1) irev = 0x0;
      if(iled == 0x8) irev = 0x1;
    }
    iblink = (iblink + 1) & 0x3fffff;  // blink period
  }
}

//eof
