/******************************************************************************
 * Copyright (c) 2017 Texas Instruments Incorporated - http://www.ti.com
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

/**
 *  \file evmDRA75x.c
 *
 *  \brief EVM board file for DRA75x boards.
 *
 *  This board library serves mainly to enable the various PDK example and test
 *  applications from CCS, which are loaded via CCS debugging session and assume
 *  certain initializations being performed by the GEL script.  When integrating
 *  applications to be loaded via the SBL, and/or in a production system, this
 *  library should be mostly obsoleted as the configurations present here are to
 *  be taken care of through the SBL code.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/* CSL Chip Functional Layer Header Files */
#include <ti/csl/soc.h>

/* UART Driver Header Files */
#include <ti/drv/uart/UART.h>
#include <ti/drv/uart/UART_stdio.h>

/* I2C Driver Header Files */
#include <ti/drv/i2c/I2C.h>
#include <ti/drv/i2c/soc/I2C_soc.h>

#include "board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */



/* I/O Delay related registers */
#define CFG_IO_DELAY_UNLOCK_KEY          (0x0000AAAA)
#define CFG_IO_DELAY_LOCK_KEY            (0x0000AAAB)

#define CFG_IO_DELAY_ACCESS_PATTERN      (0x00029000)
#define CFG_IO_DELAY_LOCK_MASK           (0x400)

#define CFG_IO_DELAY_LOCK                (*(volatile uint32_t*)(SOC_DELAYLINE_BASE + 0x02C))
#define CFG_RGMII0_TXCTL_OUT             (*(volatile uint32_t*)(SOC_DELAYLINE_BASE + 0x74C))
#define CFG_RGMII0_TXD0_OUT              (*(volatile uint32_t*)(SOC_DELAYLINE_BASE + 0x758))
#define CFG_RGMII0_TXD1_OUT              (*(volatile uint32_t*)(SOC_DELAYLINE_BASE + 0x764))
#define CFG_RGMII0_TXD2_OUT              (*(volatile uint32_t*)(SOC_DELAYLINE_BASE + 0x770))
#define CFG_RGMII0_TXD3_OUT              (*(volatile uint32_t*)(SOC_DELAYLINE_BASE + 0x77C))

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief      Enable additional modules used by the examples.
 *
 *  When integrating with SBL, this function should be obsoleted and instead
 *  the appropriate entries should be added to gModuleEnableTable in
 *  <PDK>/ti/boot/sbl_auto/sbl_utils/src/tda2xx/sbl_utils_tda2xx.c.
 */
Board_STATUS Board_moduleEnable(void);

/**
 *  \brief      Configure pinmux for modules used in the examples.
 *
 *  When integrating with SBL, this function should be obsoleted and instead
 *  the appropriate entries should be added to gPadDelayConfig* tables in
 *  <PDK>/ti/boot/sbl_auto/sbl_utils/src/tda2xx/sbl_utils_tda2xx_iodelay.c.
 *  With Silicon revision 1.1 or earlier, the tables with suffix 1_0 should be
 *  modified.  Otherwise, the tables with suffix 2_0 should be modified
 */
Board_STATUS Board_pinmuxConfig(void);

/**
 *  \brief      Modify PLL settings as required by the examples.
 *
 *  When integrating with SBL, this function should be obsoleted and instead
 *  the appropriate entries should be added to pDpll* tables for the specified
 *  OPP in
 *  <PDK>/ti/boot/sbl_auto/sbl_lib/src/tda2xx/sbl_lib_tda2xx_prcm_dpll.c
 *
 */
Board_STATUS Board_pllConfig(void);

/**
 *  \brief      Configure Gigabit Ethernet interface.
 */
Board_STATUS Board_gmiiConfig(void);

/**
 *  \brief      Configure EVM I/O expanders.
 *
 *  Note that this is EVM-specific and ideally is not required for a production
 *  design that doesn't multiplex board signals.
 */
Board_STATUS Board_ioexpConfig(void);

void delay(uint32_t delay);

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
void DllAbeConfig(uint32_t sampRate)
{
    /* Choose SYS_CLK2 (22.5792 MHZ) as source for ABE_PLL REF CLK */
    HW_WR_FIELD32(SOC_CKGEN_PRM_BASE + CM_CLKSEL_ABE_PLL_SYS, \
                  CM_CLKSEL_ABE_PLL_REF_CLKSEL, \
                  CM_CLKSEL_ABE_PLL_SYS_CLKSEL_SEL_SYS_CLK2);

    /* Reprogram ABE DPLL for 451.584 MHz output on PER_ABE_X1_GFCLK line */

    /* step 1: disable the PLL, if enabled (ex: via GEL) */
    while (HW_RD_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKMODE_DPLL_ABE, \
                         CM_CLKMODE_DPLL_ABE_DPLL_EN) ==
           CM_CLKMODE_DPLL_ABE_DPLL_EN_DPLL_LOCK_MODE)
        HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKMODE_DPLL_ABE, \
                      CM_CLKMODE_DPLL_ABE_DPLL_EN, \
                      CM_CLKMODE_DPLL_ABE_DPLL_EN_DPLL_FR_BYP_MODE);

    switch (sampRate)
    {
        case 44100:
        {
            /* Reprogram ABE DPLL for 22.5792 MHz output on PER_ABE_X1_GFCLK line */
            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKSEL_DPLL_ABE, \
                          CM_CLKSEL_DPLL_ABE_DPLL_MULT, \
                          0x0A);

            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKSEL_DPLL_ABE, \
                          CM_CLKSEL_DPLL_ABE_DPLL_DIV, \
                          0x09);

            /* step 3: Configure output clocks parameters - M2 = 1  M3 = 1 */
            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_DIV_M2_DPLL_ABE, \
                          CM_DIV_M2_DPLL_ABE_DIVHS, \
                          0x1);
            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_DIV_M3_DPLL_ABE, \
                          CM_DIV_M3_DPLL_ABE_DIVHS, \
                          0x1);
            break;
        }
        case 48000:
        {
            /* Reprogram ABE DPLL for 24.576 MHz output on PER_ABE_X1_GFCLK line */
            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKSEL_DPLL_ABE, \
                          CM_CLKSEL_DPLL_ABE_DPLL_MULT, \
                          0xA0);

            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKSEL_DPLL_ABE, \
                          CM_CLKSEL_DPLL_ABE_DPLL_DIV, \
                          0x30);

            /* step 3: Configure output clocks parameters - M2 = 3  M3 = 1 */
            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_DIV_M2_DPLL_ABE, \
                          CM_DIV_M2_DPLL_ABE_DIVHS, \
                          0x3);
            HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_DIV_M3_DPLL_ABE, \
                          CM_DIV_M3_DPLL_ABE_DIVHS, \
                          0x1);
            break;
        }
    }

    /* step 4: Confirm that the PLL has locked */
    while (HW_RD_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKMODE_DPLL_ABE, \
                         CM_CLKMODE_DPLL_ABE_DPLL_EN) !=
           CM_CLKMODE_DPLL_ABE_DPLL_EN_DPLL_LOCK_MODE)
        HW_WR_FIELD32(SOC_CKGEN_CM_CORE_AON_BASE + CM_CLKMODE_DPLL_ABE, \
                      CM_CLKMODE_DPLL_ABE_DPLL_EN, \
                      CM_CLKMODE_DPLL_ABE_DPLL_EN_DPLL_LOCK_MODE);
}

void ModuleMcaspEnable(uint32_t instNum)
{
    switch(instNum)
    {
        case 3:
        {
            /* McASP3 Module Control - Select DPLL_ABE_X1 as McASP3 AUX CLK */
            HW_WR_FIELD32(SOC_L4PER_CM_CORE_BASE + CM_L4PER2_MCASP3_CLKCTRL, \
                          CM_L4PER2_MCASP3_CLKCTRL_MODULEMODE, \
                          CM_L4PER2_MCASP3_CLKCTRL_MODULEMODE_ENABLE);

            while (HW_RD_REG32(SOC_L4PER_CM_CORE_BASE + CM_L4PER2_MCASP3_CLKCTRL) != \
                   CM_L4PER2_MCASP3_CLKCTRL_MODULEMODE_ENABLE);
            break;
        }
    }
}

void PinmuxI2cConfig (uint32_t instNum)
{
    switch(instNum)
    {
        case 2:
        {
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_I2C2_SDA, \
                          CTRL_CORE_PAD_I2C2_SDA_I2C2_SDA_MUXMODE, \
                          CTRL_CORE_PAD_I2C2_SDA_I2C2_SDA_MUXMODE_I2C2_SDA_0);
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_I2C2_SCL, \
                          CTRL_CORE_PAD_I2C2_SCL_I2C2_SCL_MUXMODE, \
                          CTRL_CORE_PAD_I2C2_SCL_I2C2_SCL_MUXMODE_I2C2_SCL_0);
            break;
        }
    }
}

void PinmuxMcaspConfig (uint32_t instNum)
{
    switch(instNum)
    {
        case 3:
        {
            /* XREF_CLK2 selects McASP3 AHCLKX*/
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_XREF_CLK2, \
                          CTRL_CORE_PAD_XREF_CLK2_XREF_CLK2_MUXMODE, \
                          CTRL_CORE_PAD_XREF_CLK2_XREF_CLK2_MUXMODE_MCASP3_AHCLKX_3);
            /* McASP3 ACLKX */
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MCASP3_ACLKX, \
                          CTRL_CORE_PAD_MCASP3_ACLKX_MCASP3_ACLKX_MUXMODE, \
                          CTRL_CORE_PAD_MCASP3_ACLKX_MCASP3_ACLKX_MUXMODE_MCASP3_ACLKX_0);
            /* McASP3 FSX */
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MCASP3_FSX, \
                          CTRL_CORE_PAD_MCASP3_FSX_MCASP3_FSX_MUXMODE, \
                          CTRL_CORE_PAD_MCASP3_FSX_MCASP3_FSX_MUXMODE_MCASP3_FSX_0);
            /* McASP3 AXR0 */
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MCASP3_AXR0, \
                          CTRL_CORE_PAD_MCASP3_AXR0_MCASP3_AXR0_MUXMODE, \
                          CTRL_CORE_PAD_MCASP3_AXR0_MCASP3_AXR0_MUXMODE_MCASP3_AXR0_0);
            /* McASP3 AXR1 */
            HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MCASP3_AXR1, \
                          CTRL_CORE_PAD_MCASP3_AXR1_MCASP3_AXR1_MUXMODE, \
                          CTRL_CORE_PAD_MCASP3_AXR1_MCASP3_AXR1_MUXMODE_MCASP3_AXR1_0);
            break;
        }
    }
}

void PinmuxMcspiConfig (uint32_t instNum)
{
    switch(instNum)
    {
        case 1:
        {
            HW_WR_REG32((SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_SPI1_SCLK), 0x000C0000);
            HW_WR_REG32((SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_SPI1_D1),   0x000C0000);
            HW_WR_REG32((SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_SPI1_D0),   0x000C0000);
            HW_WR_REG32((SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_SPI1_CS0),  0x00060000);
            HW_WR_REG32((SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_SPI1_CS1),  0x00060000);

            break;
        }
    }
}

void PinmuxMmcConfig (uint32_t instNum)
{
    uint32_t regVal = 0U;

    switch(instNum)
    {
        case 1:
        {
            /* MMCSD1 Pad configurations (SD card slot) */
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MMC1_CLK,  0x00040000);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MMC1_CMD,  0x00060000);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MMC1_DAT0, 0x00060000);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MMC1_DAT1, 0x00060000);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MMC1_DAT2, 0x00060000);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MMC1_DAT3, 0x00060000);

            /* i890: MMC1 IOs and PBIAS Must Be Powered-Up before Isolation:
             * Power-up the MMC1 IOs and PBIAS before starting the Isolation Sequence. This can be
             * done by setting the CTRL_CORE_CONTROL_PBIAS[27] SDCARD_BIAS_PWRDNZ
             * and CTRL_CORE_CONTROL_PBIAS[26] SDCARD_IO_PWRDNZ bits to 1.
             */
            /* CTRL_CORE_CONTROL_PBIAS */
            regVal = HW_RD_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS);
            regVal &= ~CTRL_CORE_CONTROL_PBIAS_SDCARD_IO_PWRDNZ_MASK;
            HW_WR_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS, regVal);
            delay(10); /* wait 10 us */

            regVal &= ~CTRL_CORE_CONTROL_PBIAS_SDCARD_BIAS_PWRDNZ_MASK;
            HW_WR_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS, regVal);

            /* Enable SDCARD_BIAS_VMODE */
            regVal |= CTRL_CORE_CONTROL_PBIAS_SDCARD_BIAS_VMODE_MASK; /* 3v */
            HW_WR_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS, regVal);

            regVal = HW_RD_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS);
            regVal |= CTRL_CORE_CONTROL_PBIAS_SDCARD_BIAS_PWRDNZ_MASK;
            HW_WR_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS, regVal);
            delay(150); /* wait 10 us */

            regVal |= CTRL_CORE_CONTROL_PBIAS_SDCARD_IO_PWRDNZ_MASK;
            HW_WR_REG32(SOC_CTRL_MODULE_CORE_CORE_PAD_REGISTERS_BASE + CTRL_CORE_CONTROL_PBIAS, regVal);

            delay(150); /* wait 10 us */
            break;
        }

        case 2:
        {
            /* MMCSD2 Pad configurations (EMMC) */
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A23, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_CS1, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A24, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A25, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A26, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A27, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A19, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A20, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A21, 0x00060001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A22, 0x00060001);
            break;
        }
    }
}

void PinmuxQspiConfig (uint32_t instNum)
{
    switch(instNum)
    {
        case 1:
        {
            /* QSPI Pad configurations */
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A18, 0x00070001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A13, 0x00070001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_CS2, 0x00070001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A17, 0x00070001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A16, 0x00070001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A15, 0x00070001);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_GPMC_A14, 0x00070001);
            break;
        }
    }
}

void PinmuxUartConfig (uint32_t instNum)
{
    switch(instNum)
    {
        case 1:
        {
            /* UART Pad configurations */
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_UART1_RXD, 0x00040000);
            HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_UART1_TXD, 0x00000000);
            break;
        }
    }
}

void PinmuxCpswConfig (void)
{
    uint32_t regVal, delta, coarse, fine;

    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MDIO_MCLK,    0x000B0000);
    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_MDIO_D,       0x000F0000);

    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_RXC,   0x00050000);
    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_RXCTL, 0x00050000);
    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_RXD3,  0x00050000);
    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_RXD2,  0x00050000);
    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_RXD1,  0x00050000);
    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_RXD0,  0x00050000);

    HW_WR_REG32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_TXC,   0x00010000);

    /*
     * Adjust I/O delays on the Tx control and data lines of each MAC port. This is
     * a workaround in order to work properly with the DP83865 PHYs on the EVM. In 3COM
     * RGMII mode this PHY applies it's own internal clock delay, so we essentially need to
     * counteract the DRA75x internal delay, and we do this by delaying the control and
     * data lines. If not using this PHY, you probably don't need to do this stuff!
     */

    /* Global unlock for I/O Delay registers */
    CFG_IO_DELAY_LOCK = CFG_IO_DELAY_UNLOCK_KEY;

    /* Tweaks to RGMII0 Tx Control and Data */
    CFG_RGMII0_TXCTL_OUT = (CFG_IO_DELAY_ACCESS_PATTERN & ~CFG_IO_DELAY_LOCK_MASK);
    HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_TXCTL,
                  CTRL_CORE_PAD_RGMII0_TXCTL_RGMII0_TXCTL_MUXMODE,
                  CTRL_CORE_PAD_RGMII0_TXCTL_RGMII0_TXCTL_MUXMODE_RGMII0_TXCTL_0);
    delta       = (0x3 << 5) + 0x8;     /* Delay value to add to calibrated value */
    regVal    = CFG_RGMII0_TXCTL_OUT & ~0xFFFFFC00;
    coarse      = ((regVal >> 5) & 0x1F) + ((delta >> 5) & 0x1F);
    coarse      = (coarse > 0x1F) ? (0x1F) : (coarse);
    fine        = (regVal & 0x1F) + (delta & 0x1F);
    fine        = (fine > 0x1F) ? (0x1F) : (fine);
    regVal    = CFG_IO_DELAY_ACCESS_PATTERN | CFG_IO_DELAY_LOCK_MASK | ((coarse << 5) | (fine));
    CFG_RGMII0_TXCTL_OUT = regVal;

    CFG_RGMII0_TXD0_OUT = (CFG_IO_DELAY_ACCESS_PATTERN & ~CFG_IO_DELAY_LOCK_MASK);
    HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_TXD0,
                  CTRL_CORE_PAD_RGMII0_TXD0_RGMII0_TXD0_MUXMODE,
                  CTRL_CORE_PAD_RGMII0_TXD0_RGMII0_TXD0_MUXMODE_RGMII0_TXD0_0);
    delta       = (0x3 << 5) + 0x8;     /* Delay value to add to calibrated value */
    regVal    = CFG_RGMII0_TXD0_OUT & ~0xFFFFFC00;
    coarse      = ((regVal >> 5) & 0x1F) + ((delta >> 5) & 0x1F);
    coarse      = (coarse > 0x1F) ? (0x1F) : (coarse);
    fine        = (regVal & 0x1F) + (delta & 0x1F);
    fine        = (fine > 0x1F) ? (0x1F) : (fine);
    regVal    = CFG_IO_DELAY_ACCESS_PATTERN | CFG_IO_DELAY_LOCK_MASK | ((coarse << 5) | (fine));
    CFG_RGMII0_TXD0_OUT = regVal;

    CFG_RGMII0_TXD1_OUT = (CFG_IO_DELAY_ACCESS_PATTERN & ~CFG_IO_DELAY_LOCK_MASK);
    HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_TXD1,
                  CTRL_CORE_PAD_RGMII0_TXD1_RGMII0_TXD1_MUXMODE,
                  CTRL_CORE_PAD_RGMII0_TXD1_RGMII0_TXD1_MUXMODE_RGMII0_TXD1_0);
    delta       = (0x3 << 5) + 0x2;     /* Delay value to add to calibrated value */
    regVal    = CFG_RGMII0_TXD1_OUT & ~0xFFFFFC00;
    coarse      = ((regVal >> 5) & 0x1F) + ((delta >> 5) & 0x1F);
    coarse      = (coarse > 0x1F) ? (0x1F) : (coarse);
    fine        = (regVal & 0x1F) + (delta & 0x1F);
    fine        = (fine > 0x1F) ? (0x1F) : (fine);
    regVal    = CFG_IO_DELAY_ACCESS_PATTERN | CFG_IO_DELAY_LOCK_MASK | ((coarse << 5) | (fine));
    CFG_RGMII0_TXD1_OUT = regVal;

    CFG_RGMII0_TXD2_OUT = (CFG_IO_DELAY_ACCESS_PATTERN & ~CFG_IO_DELAY_LOCK_MASK);
    HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_TXD2,
                  CTRL_CORE_PAD_RGMII0_TXD2_RGMII0_TXD2_MUXMODE,
                  CTRL_CORE_PAD_RGMII0_TXD2_RGMII0_TXD2_MUXMODE_RGMII0_TXD2_0);
    delta       = (0x4 << 5) + 0x0;     /* Delay value to add to calibrated value */
    regVal    = CFG_RGMII0_TXD2_OUT & ~0xFFFFFC00;
    coarse      = ((regVal >> 5) & 0x1F) + ((delta >> 5) & 0x1F);
    coarse      = (coarse > 0x1F) ? (0x1F) : (coarse);
    fine        = (regVal & 0x1F) + (delta & 0x1F);
    fine        = (fine > 0x1F) ? (0x1F) : (fine);
    regVal    = CFG_IO_DELAY_ACCESS_PATTERN | CFG_IO_DELAY_LOCK_MASK | ((coarse << 5) | (fine));
    CFG_RGMII0_TXD2_OUT = regVal;

    CFG_RGMII0_TXD3_OUT = (CFG_IO_DELAY_ACCESS_PATTERN & ~CFG_IO_DELAY_LOCK_MASK);
    HW_WR_FIELD32(SOC_CORE_PAD_IO_REGISTERS_BASE + CTRL_CORE_PAD_RGMII0_TXD3,
                  CTRL_CORE_PAD_RGMII0_TXD3_RGMII0_TXD3_MUXMODE,
                  CTRL_CORE_PAD_RGMII0_TXD3_RGMII0_TXD3_MUXMODE_RGMII0_TXD3_0);
    delta       = (0x4 << 5) + 0x0;     /* Delay value to add to calibrated value */
    regVal    = CFG_RGMII0_TXD3_OUT & ~0xFFFFFC00;
    coarse      = ((regVal >> 5) & 0x1F) + ((delta >> 5) & 0x1F);
    coarse      = (coarse > 0x1F) ? (0x1F) : (coarse);
    fine        = (regVal & 0x1F) + (delta & 0x1F);
    fine        = (fine > 0x1F) ? (0x1F) : (fine);
    regVal    = CFG_IO_DELAY_ACCESS_PATTERN | CFG_IO_DELAY_LOCK_MASK | ((coarse << 5) | (fine));
    CFG_RGMII0_TXD3_OUT = regVal;

    /* Global lock */
    CFG_IO_DELAY_LOCK = CFG_IO_DELAY_LOCK_KEY;
}

Board_STATUS Board_init(Board_initCfg cfg)
{
    Board_STATUS status;

    if (cfg & BOARD_INIT_PINMUX_CONFIG)
    {
        status = Board_pinmuxConfig();
        if (status != BOARD_SOK)
            return status;

        status = Board_ioexpConfig();
        if (status != BOARD_SOK)
            return status;
    }

    if (cfg & BOARD_INIT_PLL)
    {
        status = Board_pllConfig();
        if (status != BOARD_SOK)
            return status;
    }

    if (cfg & BOARD_INIT_MODULE_CLOCK)
    {
        status = Board_moduleEnable();
        if (status != BOARD_SOK)
            return status;
    }

    if (cfg & BOARD_INIT_ETH_PHY)
    {
        status = Board_gmiiConfig();
        if (status != BOARD_SOK)
            return status;
    }

    if (cfg & BOARD_INIT_UART_STDIO)
        UART_stdioInit(BOARD_UART_INSTANCE);

    return BOARD_SOK;
}

Board_STATUS Board_pinmuxConfig(void)
{
    PinmuxCpswConfig();
    PinmuxI2cConfig(2U);
    PinmuxMcaspConfig(3U);
    PinmuxMcspiConfig(1U);
    PinmuxMmcConfig(1U);
    PinmuxMmcConfig(2U);
    PinmuxQspiConfig(1U);
    PinmuxUartConfig(1U);

    /* Configure IRQ crossbar */
#if defined(__ARM_ARCH_7A__)
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_MPU, CSL_XBAR_INST_MPU_IRQ_72, CSL_XBAR_UART1_IRQ);
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_MPU, CSL_XBAR_INST_MPU_IRQ_92, CSL_XBAR_GMAC_SW_IRQ_RX_PULSE);
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_MPU, CSL_XBAR_INST_MPU_IRQ_93, CSL_XBAR_GMAC_SW_IRQ_TX_PULSE);
#elif defined(__TI_ARM_V7M4__)
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_IPU1, CSL_XBAR_INST_IPU1_IRQ_24, CSL_XBAR_UART1_IRQ);
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_IPU1, CSL_XBAR_INST_IPU1_IRQ_60, CSL_XBAR_GMAC_SW_IRQ_RX_PULSE);
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_IPU1, CSL_XBAR_INST_IPU1_IRQ_61, CSL_XBAR_GMAC_SW_IRQ_TX_PULSE);
#elif defined(_TMS320C6X)
    CSL_xbarIrqConfigure(CSL_XBAR_IRQ_CPU_ID_DSP1, CSL_XBAR_INST_DSP1_IRQ_48, CSL_XBAR_UART1_IRQ);
#endif

    return BOARD_SOK;
}

Board_STATUS Board_moduleEnable(void)
{
    ModuleMcaspEnable(3);

    return BOARD_SOK;
}

Board_STATUS Board_pllConfig(void)
{
    DllAbeConfig(BOARD_AUDIO_SAMPLERATE);

    return BOARD_SOK;
}

Board_STATUS Board_gmiiConfig(void)
{
    /* Set GMII selection as RGMII for ports 1 and 2 */
    HW_WR_FIELD32(SOC_CTRL_MODULE_CORE_CORE_REGISTERS_BASE + CTRL_CORE_CONTROL_IO_1,
                  CTRL_CORE_CONTROL_IO_1_GMII1_SEL, 2U);
    HW_WR_FIELD32(SOC_CTRL_MODULE_CORE_CORE_REGISTERS_BASE + CTRL_CORE_CONTROL_IO_1,
                  CTRL_CORE_CONTROL_IO_1_GMII2_SEL, 2U);

    /* Enable GMAC reset isolation */
    HW_WR_FIELD32(SOC_CTRL_MODULE_CORE_CORE_REGISTERS_BASE + CTRL_CORE_CONTROL_IO_2,
                  CTRL_CORE_CONTROL_IO_2_GMAC_RESET_ISOLATION_ENABLE, 0U);
    HW_WR_FIELD32(SOC_CTRL_MODULE_CORE_CORE_REGISTERS_BASE + CTRL_CORE_CONTROL_IO_2,
                  CTRL_CORE_CONTROL_IO_2_GMAC_RESET_ISOLATION_ENABLE, 1U);

    return BOARD_SOK;
}

Board_STATUS Board_ioexpConfig(void)
{
    I2C_Params i2cParams;
    I2C_Handle handle;
    I2C_Transaction i2cTransaction;
    I2C_HwAttrs     i2cCfg;
    bool            enableIntrTmp;
    char rxTxBuf[2] = {0};
    int16_t ret;

    I2C_init();

    I2C_Params_init(&i2cParams);

    /* I2C1 writes to I/O expanders */
    /* Force I2C1 to operate in polling mode, temporarily */
    I2C_socGetInitCfg(0, &i2cCfg);
    enableIntrTmp = i2cCfg.enableIntr;
    i2cCfg.enableIntr = false;
    I2C_socSetInitCfg(0, &i2cCfg);

    /* Open I2C1 instance */
    handle = I2C_open(0, &i2cParams);
    if (handle == NULL) {
        UART_printf("Board_ioexpConfig: failed to open I2C1\n");
        return BOARD_I2C_OPEN_FAIL;
    }

    /* Common transaction parameters */
    I2C_transactionInit(&i2cTransaction);
    i2cTransaction.writeBuf = (uint8_t *)&rxTxBuf[0];
    i2cTransaction.readBuf = (uint8_t *)&rxTxBuf[0];
    i2cTransaction.timeout = 2000U;

    /*
     * Set EXP_ETH0_RSTn = EXP_ETH1_RSTn = 1 (PCF8575's P10 and P11)
     * in order to drive the board's PHYs out of reset
     */
    i2cTransaction.slaveAddress = I2C_ADDR_PCF8575_PHY_NRESET;

    i2cTransaction.writeCount = 0;
    i2cTransaction.readCount = 2;
    ret = I2C_transfer(handle, &i2cTransaction);
    if (ret != I2C_STS_SUCCESS) {
        UART_printf("Board_ioexpConfig: transfer to slave 0x%x failed: %d\n",
                    i2cTransaction.slaveAddress, ret);
    }

    i2cTransaction.writeCount = 2;
    i2cTransaction.readCount = 0;
    rxTxBuf[1] |= 0x03;
    ret = I2C_transfer(handle, &i2cTransaction);
    if (ret != I2C_STS_SUCCESS) {
        UART_printf("Board_ioexpConfig: transfer to slave 0x%x failed: %d\n",
                    i2cTransaction.slaveAddress, ret);
    }

    /*
     * Set SEL_ENET_MUX_S0 = 1 (PCF8575's P4) in order to select
     * the EMAC0 pins in the bus switch.
     * Set MMC_PWR_ON = 1 (PCF8575's P5) in order to provide
     * EVM_3V3_SD supply from the bus switch.
     */
    i2cTransaction.slaveAddress = I2C_ADDR_PCF8575_EMAC_PINS;

    i2cTransaction.writeCount = 0;
    i2cTransaction.readCount = 2;
    ret = I2C_transfer(handle, &i2cTransaction);
    if (ret != I2C_STS_SUCCESS) {
        UART_printf("Board_ioexpConfig: transfer to slave 0x%x failed: %d\n",
                    i2cTransaction.slaveAddress, ret);
    }

    i2cTransaction.writeCount = 2;
    i2cTransaction.readCount = 0;
    rxTxBuf[0] |= 0x30;
    ret = I2C_transfer(handle, &i2cTransaction);
    if (ret != I2C_STS_SUCCESS) {
        UART_printf("Board_ioexpConfig: transfer to slave 0x%x failed: %d\n",
                    i2cTransaction.slaveAddress, ret);
    }

    I2C_close(handle);

    /* Reset mode of I2C1 operation back to driver default */
    i2cCfg.enableIntr = enableIntrTmp;
    I2C_socSetInitCfg(0, &i2cCfg);

    /* I2C2 writes to I/O expanders */
    /* Force I2C2 to operate in polling mode, temporarily */
    I2C_socGetInitCfg(1, &i2cCfg);
    enableIntrTmp = i2cCfg.enableIntr;
    i2cCfg.enableIntr = false;
    I2C_socSetInitCfg(1, &i2cCfg);

    /* Open I2C2 instance */
    handle = I2C_open(1, &i2cParams);
    if (handle == NULL) {
        UART_printf("Board_ioexpConfig: failed to open I2C2\n");
        return BOARD_I2C_OPEN_FAIL;
    }

    /*
     * Set VIN6_SEL_S0 = 0 (PCF8575's P1) in order to select
     * the McASP3 pins in the bus switch
     */
    i2cTransaction.slaveAddress = I2C_ADDR_PCF8575_MCASP_SEL;

    i2cTransaction.writeCount = 0;
    i2cTransaction.readCount = 2;
    ret = I2C_transfer(handle, &i2cTransaction);
    if (ret != I2C_STS_SUCCESS) {
        UART_printf("Board_ioexpConfig: transfer to slave 0x%x failed: %d\n",
                    i2cTransaction.slaveAddress, ret);
    }

    i2cTransaction.writeCount = 2;
    i2cTransaction.readCount = 0;
    rxTxBuf[0] &= ~(0x02);

    ret = I2C_transfer(handle, &i2cTransaction);
    if (ret != I2C_STS_SUCCESS) {
        UART_printf("Board_ioexpConfig: transfer to slave 0x%x failed: %d\n",
                    i2cTransaction.slaveAddress, ret);
    }

    /* Reset mode of I2C2 operation back to driver default */
    i2cCfg.enableIntr = enableIntrTmp;
    I2C_socSetInitCfg(1, &i2cCfg);

    I2C_close(handle);

    return BOARD_SOK;
}

void delay(uint32_t delay)
{
    volatile uint32_t i;
    for (i = 0; i < (1000 * delay); ++i) ;
}

