/**
 *   @file  idkAM571x_ethernet_config.c
 *
 *   @brief
 *      This file contains the ICSS EMAC PHY configurations for idkAM571x board
 *
 *      Board_icssEthConfig: This API configures 10/100 PHYs connected to two
 *      ICSS ports. Several configurations are needed for normal operation of
 *      the ICSS Ethernet PHYs which will be taken care by this function.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2017, Texas Instruments, Inc.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  \par
*/

#include <ti/csl/soc.h>
#include <ti/csl/csl_gpio.h>
#include <ti/csl/csl_mdioAux.h>
#include <ti/csl/soc/am571x/src/cslr_soc.h>
#include <ti/csl/cslr_icss.h>

#include "board_internal.h"
#include "board_cfg.h"

#define BOARD_MAX_ICSS_EMAC_PORTS     4

#define BOARD_SWITCH_DEFAULT_MDIOBUSFREQ      (2200000u)
#define BOARD_SWITCH_DEFAULT_MDIOCLOCKFREQ    (200000000u)

/* Structure holding gpio pin configuration parameters. */
typedef struct gpioPinCfg {
    uint32_t instance;      /**< GPIO instance number. */
    uint32_t pin;           /**< GPIO pin number. */
    uint32_t pinMux;
    uint32_t baseAddr;

} gpioPinCfg_t;


static gpioPinCfg_t phyResetInfo[BOARD_MAX_ICSS_EMAC_PORTS];
static gpioPinCfg_t phyIntInfo[BOARD_MAX_ICSS_EMAC_PORTS];


/**
 * \brief  Local Delay function
 *
 * \return  none
 */
static void Board_delay(uint32_t delay); /* for misra warning */
static void Board_delay(uint32_t delay)
{
    volatile uint32_t del = delay;
    while (del--) {}
}


/**
 * \brief  Resets the ICSS Ethernet PHYs
 *
 * \return  none
 */
static void Board_phyReset(void); /* for misra warning */
static void Board_phyReset(void)
{
    int32_t i =0;
    int32_t startPort = 0;

    for(i=startPort;i<BOARD_MAX_ICSS_EMAC_PORTS-2U;i++)
    {
        /* Do GPIO pin mux for M4 only*/
#ifdef __TI_ARM_V7M4__
        HWREG(CSL_MPU_CORE_PAD_IO_REGISTERS_REGS + phyResetInfo[i].pinMux) = (0x20000 | 0x0E);
#endif
        GPIOModuleEnable(phyResetInfo[i].baseAddr);
        GPIODirModeSet(phyResetInfo[i].baseAddr, phyResetInfo[i].pin, GPIO_DIR_OUTPUT);


        GPIOPinWrite(phyResetInfo[i].baseAddr, phyResetInfo[i].pin, GPIO_PIN_HIGH);
        GPIOPinWrite(phyResetInfo[i].baseAddr, phyResetInfo[i].pin, GPIO_PIN_LOW);
        Board_delay(1000);
        GPIOPinWrite(phyResetInfo[i].baseAddr, phyResetInfo[i].pin, GPIO_PIN_HIGH);
        Board_delay(1000);
    }

}


/**
 * \brief  Configures  ICSS ethernet PHY  Interrupt pins configured as GPIO input in order to 
             disable the TLK105 phy powerdown
 *
 * \return  none
 */
static void Board_phyIntConfig(void); /* for misra warning */
static void Board_phyIntConfig(void)
{
  int32_t i =0;
  int32_t startPort=0;

    for(i=startPort;i<BOARD_MAX_ICSS_EMAC_PORTS;i++)
    {
#ifdef __TI_ARM_V7M4__
        HWREG(CSL_MPU_CORE_PAD_IO_REGISTERS_REGS + phyIntInfo[i].pinMux) = (0x20000 | 0x0E);
#endif
        GPIOModuleEnable(phyIntInfo[i].baseAddr);
        GPIODirModeSet(phyIntInfo[i].baseAddr, phyIntInfo[i].pin, GPIO_DIR_INPUT);
    }
}

/**
 * \brief   PHY setup routine for  ICSS  Ethernet PHYs.
 *     Setting the GPIOs for PHY reset, routing ICSS signals to PHYs,
 *     PHY interrupt lines.
 *
 * \return  none
 */
static void Board_phySetup(void); /* for misra warning */
static void Board_phySetup(void)
{
    //GPIO_PRU1_ETH_RESETn
    phyResetInfo[0].instance=5; //MCASP1_AXR6 - GPIO5_8
    phyResetInfo[0].pin=8;
    phyResetInfo[0].pinMux=  CSL_CONTROL_CORE_PAD_IO_PAD_MCASP1_AXR6;
    phyResetInfo[0].baseAddr=SOC_GPIO5_BASE;

    //GPIO_PRU2_ETH_RESETn
    phyResetInfo[1].instance=5; //MCASP1_AXR7 - GPIO5_9
    phyResetInfo[1].pin=9;
    phyResetInfo[1].pinMux= CSL_CONTROL_CORE_PAD_IO_PAD_MCASP1_AXR7;
    phyResetInfo[1].baseAddr=SOC_GPIO5_BASE;

    //PRU1_ETH0_INTn
     phyIntInfo[0].instance=3; //VIN2A_CLK0 - GPIO3_28 - Input
     phyIntInfo[0].pin=28;
     phyIntInfo[0].pinMux=CSL_CONTROL_CORE_PAD_IO_PAD_VIN2A_CLK0;
     phyIntInfo[0].baseAddr=CSL_MPU_GPIO3_REGS;
        
      //PRU1_ETH1_INTn
     phyIntInfo[1].instance=3; //VIN2A_DE0- GPIO3_29 - Input
     phyIntInfo[1].pin=29;
     phyIntInfo[1].pinMux=CSL_CONTROL_CORE_PAD_IO_PAD_VIN2A_DE0;
     phyIntInfo[1].baseAddr=CSL_MPU_GPIO3_REGS;

     //PRU2_ETH0_INTn
     phyIntInfo[2].instance=3; //VIN2A_FLD0- GPIO3_30 - Input
     phyIntInfo[2].pin=30;
     phyIntInfo[2].pinMux=CSL_CONTROL_CORE_PAD_IO_PAD_VIN2A_FLD0;
     phyIntInfo[2].baseAddr=CSL_MPU_GPIO3_REGS;
     
     //PRU2_ETH1_INTn
     phyIntInfo[3].instance=3; //VIN2A_HSYNC0- GPIO3_31 - Input
     phyIntInfo[3].pin=31;
     phyIntInfo[3].pinMux=CSL_CONTROL_CORE_PAD_IO_PAD_VIN2A_HSYNC0;
     phyIntInfo[3].baseAddr=CSL_MPU_GPIO3_REGS;

    Board_phyReset();

    Board_phyIntConfig();

}


/**
 * \brief   MDIO peripheral initialization. Enables MDIO state machine.
 *
 * \return  none
 */
static void Board_mdioInit(void); /* for misra warning */
static void Board_mdioInit(void)
{
    CSL_MDIO_init(CSL_MPU_PRUSS1_U_MII_MDIO_REGS,
                                BOARD_SWITCH_DEFAULT_MDIOCLOCKFREQ,
                                BOARD_SWITCH_DEFAULT_MDIOBUSFREQ);
    HWREG(CSL_MPU_PRUSS1_U_MII_MDIO_REGS + CSL_ICSSMIIMDIO_CONTROL) = 0x4000009F;/*200/(0x9F+1) = 1.25 MHz*/

    CSL_MDIO_init(CSL_MPU_PRUSS2_U_MII_MDIO_REGS,
                                BOARD_SWITCH_DEFAULT_MDIOCLOCKFREQ,
                                BOARD_SWITCH_DEFAULT_MDIOBUSFREQ);
    HWREG(CSL_MPU_PRUSS2_U_MII_MDIO_REGS + CSL_ICSSMIIMDIO_CONTROL) = 0x4000009F;/*200/(0x9F+1) = 1.25 MHz*/
}

/**
 * \brief  Board specific configurations for ICSS  Ethernet PHYs
 *
 *  This function takes care of making several board level configurations
 *  required for ICSS EMAC PHY as listed below.
 *   - MDIO initialization
 *   - Setting the GPIOs for PHY reset, routing ICSS signals to PHYs,
 *     PHY interrupt lines.
 *   - Resetting the PHYs for proper address latching

 *
 * \return  none
 */
Board_STATUS Board_icssEthConfig(void);
Board_STATUS Board_icssEthConfig(void)
{
    /* MDIO initialization */
     Board_mdioInit();
    /* ICSS EMAC PHY setup */
     Board_phySetup();
    
    return BOARD_SOK;
}

