/*
 * Description:
 *  Opens a SAR connection to an ADC to poll for data
 *  Performs an FFT on the polled data.
 *  Prepares to send that data to the Bluetooth module [Future implementation]
 *
 */

// included libraries
#include "uart.h"
#include "FFT.h"
#include "AIC.h"
#include "common.h"

// defined constants
const unsigned int num_samples = 512;
const char DEBUG = 1;
const char EVM = 1;

// Function declarations
void DoFFT();


// Global variables
char IsExitting = 0;
short FFTBufferX[512];      // FFT buffer for the X axis
short FFTBufferY[512];      // FFT buffer for the Y axis
short FFTBufferZ[512];      // FFT buffer for the Z axis
short FFTBufferX_imag[512]; // FFT buffer for the X axis
short FFTBufferY_imag[512]; // FFT buffer for the Y axis
short FFTBufferZ_imag[512]; // FFT buffer for the Z axis
short SampleBufferX[512];   // Buffer for holding the samples for the X axis
short SampleBufferY[512];   // Buffer for holding the samples for the Y axis
short SampleBufferZ[512];   // Buffer for holding the samples for the Z axis
short SampleBufferA[512];   // Buffer for holding the samples for the A input [NOT USED, but is necessary]

int i;

/**
 * main.c
 */
int main(void)
{
    int status = 0;
    char string[136];

    // Starts up the UART controller
    if(DEBUG)
    {
        //printf("Setting up UART controller...\n");
        status = UART_Initialize();
        if(status != 0)
        {
            return(status);
        }
    }

    // Starts up the AIC Audio codec for the headphone lines
    if(EVM)
    {
        status = AIC3206_Initialize();
        if(status != 0)
            return (status);
    }

    // Begins collecting data
    //printf("Entering iteration routine...");
    while(IsExitting != 255)
    {

        for(i = 0; i < num_samples; i++)
        {
            // Resets all of the data in the buffer to 0
            *(FFTBufferX + i) = 0;
            *(FFTBufferY + i) = 0;
            *(FFTBufferZ + i) = 0;
            *(SampleBufferX + i) = 0;
            *(SampleBufferY + i) = 0;
            *(SampleBufferZ + i) = 0;
            *(SampleBufferA + i) = 0;
        }

        printf("Collecting samples...\n");
        DoFFT(); // Collects num_samples samples
        //printf("Finished FFT collection");

        if(DEBUG)
        {
            // Transmits the FFT data to the UART
            /*
             * Transmits in format $num0,num1,num2,...,numN$
             */
            UART_Write("$");
            //sprintf(*string, "%d", *FFTBuffer);
            //UART_Write(*string);

            for(i = 0; i < num_samples; i++)
            {

                sprintf(string, "%d,%d,%d,%d,%d,%d", *(SampleBufferX + i), *(FFTBufferX + i), *(SampleBufferY + i), *(FFTBufferY + i), *(SampleBufferZ + i), *(FFTBufferZ + i));
                UART_Write(string);
                if(i != (num_samples - 1))
                    UART_Write(",");
            }

            UART_Write("$");
            //IsExitting = 255;
        }
    }

    if(DEBUG)
    {
        // Shuts down the UART controller
        UART_Close();
    }

	return 0;
}

void DoFFT()
{
    //printf("Beginning an FFt data set...");

    // Reads samples
    read_samples(SampleBufferX, SampleBufferY, SampleBufferZ, SampleBufferA, num_samples);

    for(i = 0; i < num_samples; i++)
    {
        *(FFTBufferX + i) = *(SampleBufferX + i);
        *(FFTBufferY + i) = *(SampleBufferY + i);
        *(FFTBufferZ + i) = *(SampleBufferZ + i);
    }

    // Performs FFT
    FFT_PerformTransform(SampleBufferX, FFTBufferX, FFTBufferX_imag, num_samples);
    FFT_PerformTransform(SampleBufferY, FFTBufferY, FFTBufferY_imag, num_samples);
    FFT_PerformTransform(SampleBufferZ, FFTBufferZ, FFTBufferZ_imag, num_samples);
}
