/**
 *  \file   main.c
 *
 *  \brief  Test application main file. The test application demonstrates
 *          raw data write & read on a MMCSD device.
 *
 */

/*
 * Copyright (C) 2014-2017 Texas Instruments Incorporated - http://www.ti.com/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of
 * its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/* XDCtools Header files */
#include <xdc/std.h>
#include <xdc/runtime/Error.h>
#include <xdc/cfg/global.h>
#include <xdc/runtime/System.h>
#include <stdio.h>
#include <ti/sysbios/knl/Task.h>

/* BIOS Header files */
#include <ti/sysbios/BIOS.h>

#include <xdc/runtime/Error.h>

/* TI-RTOS Header files */
#include <ti/csl/cslr_device.h>
#include "MMCSD_log.h"
#include <ti/drv/mmcsd/MMCSD.h>
#include <ti/drv/mmcsd/soc/MMCSD_soc.h>

#include <ti/board/board.h>


#include <ti/drv/gpio/GPIO.h>
#include <ti/drv/gpio/soc/GPIO_soc.h>

#ifdef MEASURE_TIME
#include "profiling.h"
#endif
/**********************************************************************
 ************************** Macros ************************************
 **********************************************************************/

/* Global data pointers */
#define HSMMCSD_DATA_SIZE               512

#define PAGE_OFFSET   0x0
#define MMCSTARTSECTOR  0x300000 //@1.5GB //100

#define DATA_PATTERN_00     0
#define DATA_PATTERN_FF     1
#define SDMMC_DATA_PATTERN_AA 2
#define SDMMC_DATA_PATTERN_55 3
#define SDMMC_DATA_PATTERN_RAMP32 4
#define SDMMC_DATA_PATTERN_INC  8

/* The below definitions define the nature of the test. The unit test perform multiple iterations
 * of TEST_TOTAL_BLOCK_SIZE blocks (of size SECTORSIZE bytes), until a total of
 * TEST_TOTAL_BYTES_TO_TRANSFER bytes are transferred. In this example, they both were set
 * the same (1MB), but the test is configurable to transfer say, a total of 100MB in 5MB chunks.
 * In this case TEST_TOTAL_BLOCK_SIZE=5MB and TEST_TOTAL_BYTES_TO_TRANSFER=100MB.
 */
#define SECTORSIZE    (512)
//#define TEST_TOTAL_BYTES_TO_TRANSFER (4*512*512) /* Total bytes to be transferred in the unit test */
#define TEST_TOTAL_BYTES_TO_TRANSFER (512) /* Total bytes to be transferred in the unit test */
#define MMCTESTSECTORS     (TEST_TOTAL_BYTES_TO_TRANSFER/SECTORSIZE)
// #define TEST_TOTAL_BLOCK_SIZE   (4*512*512) /* Number of bytes read/written at a time */
#define TEST_TOTAL_BLOCK_SIZE   (512) /* Number of bytes read/written at a time */
#define TESTSECTORS (TEST_TOTAL_BLOCK_SIZE/SECTORSIZE)

#define MMCSD_INSTANCE_EMMC    (1U)


/* GPIO pin value definitions */
#define GPIO_PIN_VAL_LOW     (0U)
#define GPIO_PIN_VAL_HIGH    (1U)

#define GPIO_MUX_SEL 4

/* ON Board LED pins which are connected to GPIO pins. */
typedef enum GPIO_PIN {
    GPIO_PIN_EMMC_RST      = 0U,
    GPIO_PIN_COUNT
}GPIO_PIN;

/* GPIO Driver board specific pin configuration structure */
GPIO_PinConfig gpioPinConfigs[] = {
    /* Output pin : EMMC_RST */
    GPIO_DEVICE_CONFIG(GPIO_EMMC_RST_PORT_NUM, GPIO_EMMC_RST_PIN_NUM) | GPIO_CFG_OUTPUT,
};

/* GPIO Driver call back functions */
GPIO_CallbackFxn gpioCallbackFunctions[] = {
    NULL
};


GPIO_v1_Config GPIO_v1_config = {
    gpioPinConfigs,
    gpioCallbackFunctions,
    sizeof(gpioPinConfigs) / sizeof(GPIO_PinConfig),
    sizeof(gpioCallbackFunctions) / sizeof(GPIO_CallbackFxn),
    0,
};

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef CSL_control_core_pad_ioRegs *CSL_padRegsOvly;

/**********************************************************************
 ************************** Internal functions ************************
 **********************************************************************/

static int32_t HSMMCSDReadWriteTest(MMCSD_Handle handle);

static int32_t fillMmcPageData(uint8_t *buf, int32_t length, uint8_t flag,uint32_t *rampBase);

static void EmmcsReset(void);

/* Delay function */
static void delay(unsigned int delayValue);

/**********************************************************************
 ************************** Global Variables **************************
 **********************************************************************/

MMCSD_Handle handle = NULL;
#define DATA_BUF_ALIGN               (256)

uint8_t tx[SECTORSIZE*TESTSECTORS];

__attribute__((aligned(256))) // GCC way of aligning


uint8_t rx[SECTORSIZE*TESTSECTORS];

__attribute__((aligned(256))) // GCC way of aligning


/*
 *  ======== Board specific GPIO init ========
 */
 void board_initGPIO(void){
    
}
/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

/*
 *  ======== test function ========
 */
void mmcsd_test(UArg arg0, UArg arg1){
    MMCSD_Error ret;

    MMCSD_v1_HwAttrs hwAttrsConfig;


    if(MMCSD_socGetInitCfg(MMCSD_INSTANCE_EMMC,&hwAttrsConfig)!=0) {
    	System_printf ("\nUnable to obtain config.Exiting. TEST FAILED.\r\n");

        return;
    }

      if(hwAttrsConfig.enableInterrupt==1) {
    	  System_printf ("\n****** Non-DMA: Interrupt Mode ******\r\n");
      }  else {
    	  System_printf ("\n****** Non-DMA: Polling Mode ******\r\n");
      }
      System_printf("Read/Write test with sector size %d bytes which involves ",SECTORSIZE);
      System_printf("reading/writing %d blocks at a time, with a total transfer size of %d bytes\n",TESTSECTORS,TEST_TOTAL_BYTES_TO_TRANSFER);


      if(MMCSD_socSetInitCfg(MMCSD_INSTANCE_EMMC,&hwAttrsConfig)!=0) {
    	  System_printf ("\nUnable to set config.Exiting. TEST FAILED.\r\n");
         return;
 	}

    GPIO_init();

    EmmcsReset();
    profile_init();

    MMCSD_init();

    if ((ret = MMCSD_open(MMCSD_INSTANCE_EMMC, NULL, &handle))!=MMCSD_OK)    {
    	System_printf ("\nMMCSD Open failed with return code %d\n",ret);
    }

    HSMMCSDReadWriteTest(handle);
}

/*
 *  ======== main ========
 */

int main(void){
	Task_Handle task;
	Error_Block eb;
    /* Call board init functions */
    Board_initCfg boardCfg;

    board_initGPIO();

    boardCfg = BOARD_INIT_PINMUX_CONFIG | 
        BOARD_INIT_MODULE_CLOCK | BOARD_INIT_UART_STDIO;
    Board_init(boardCfg);

    System_printf("Board Initialized\n"); System_flush();

    Error_init(&eb);
    task = Task_create(mmcsd_test, NULL, &eb);
    if (task == NULL) {
        System_printf("Task_create() failed!\n");
        BIOS_exit(0);
    }

    /* Start BIOS */
    BIOS_start();
    return (0);
}

uint32_t txRampBase=0;
int32_t fillMmcPageData(uint8_t *buf, int32_t length, uint8_t flag,uint32_t *rampBase){
    int32_t i;
    uint32_t data = 0x00;
    uint8_t incFlag = 0x00;

    switch(flag)   {
        case DATA_PATTERN_00:
            data = 0;
            break;
        case DATA_PATTERN_FF:
            data = 0xFF;
            break;
        case SDMMC_DATA_PATTERN_AA:
            data = 0xAA;
            break;
        case SDMMC_DATA_PATTERN_55:
            data = 0x55;
            break;
        case SDMMC_DATA_PATTERN_INC:
             data = 0x00;
             incFlag = 0x01;
             break;
        case SDMMC_DATA_PATTERN_RAMP32:
             break;
		default:
            return -1;
    }

    if(flag==SDMMC_DATA_PATTERN_RAMP32) {
         uint32_t *buf_32 = (uint32_t *)buf;
         for(i =0; i < length/sizeof(uint32_t); i++)  {
             *buf_32=*rampBase;
              buf_32++;
             *rampBase = *rampBase+1;
         }

    } else {
     for(i =0; i < length; i++)  {
        *(buf+i) = data;
         if(incFlag) {
            data++;
         }
      }
    }



    return 0;
}

int32_t HSMMCSDReadWriteTest(MMCSD_Handle handle){
  int32_t j;
  unsigned long int blockNum;
  MMCSD_Error retVal = MMCSD_OK;

  profile_reset_results();


  for(blockNum = MMCSTARTSECTOR; blockNum < (MMCTESTSECTORS+MMCSTARTSECTOR); blockNum+=TESTSECTORS)  {

    fillMmcPageData(&tx[0], (SECTORSIZE*TESTSECTORS), SDMMC_DATA_PATTERN_RAMP32,&txRampBase);


     profile_start_point(PROFILE_MMCSD_WRITE); 


    retVal = MMCSD_write(handle, &tx[0], blockNum, TESTSECTORS);

     profile_end_point(PROFILE_MMCSD_WRITE); 


    if(retVal != MMCSD_OK)    {
    	System_printf("FAIL: Write failed on sector %x", blockNum);
        return -1;
    }

    delay(100);

    /* Clear receive buffer */
    fillMmcPageData(&rx[0], (SECTORSIZE*TESTSECTORS), DATA_PATTERN_00,NULL);

    profile_start_point(PROFILE_MMCSD_READ);

    /* Read a page */
     retVal = MMCSD_read(handle, &rx[0], blockNum, TESTSECTORS);


     profile_end_point(PROFILE_MMCSD_READ);

    if(retVal != MMCSD_OK)    {
    	System_printf("FAIL: Read failed on sector %x", blockNum);
        return -1;
    }

    /* Check the pattern */
    for ( j = 0 ; j < (SECTORSIZE*TESTSECTORS) ; j++ )    {
		/* Compare the data read against the expected */
		if ( tx[j] != rx[j])
		{
			System_printf ("FAIL: Data comparison failed @ ");
			System_printf("sector %d", blockNum);
			System_printf(" ,data %d", j);
			System_printf(" ,rx = %x", rx[j]);
			System_printf(" ,tx = %x", tx[j]);
			return -1;
		}

	}
  }

  System_printf ("\nAll tests have PASSED\n");

    profile_display_results(TEST_TOTAL_BYTES_TO_TRANSFER);


  return 0;
}

static void EmmcsReset(void){
    /* EMMC reset */
    GPIO_write(GPIO_PIN_EMMC_RST, GPIO_PIN_VAL_LOW);
    delay(100);
    GPIO_write(GPIO_PIN_EMMC_RST, GPIO_PIN_VAL_HIGH);
    delay(100);
}


/*
 *  ======== Delay function ========
 */
static void delay(unsigned int delayValue) {
    volatile uint32_t delay1 = delayValue*10000;
    while (delay1--) ;
}
