/*
 * helloWorld_bios6.c
 *
 * TCP/IP Stack 'Hello World!' Example ported to use BIOS6 OS.
 *
 * Copyright (C) 2007, 2011 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

//--------------------------------------------------------------------------
// IP Stack 'Hello World!' Example
//
// This is a skeleton application, intended to provide application
// programmers with a basic Stack setup, to which they can start
// adding their code.
//
// To test it as is, use with helloWorld.exe from \winapps directory
//
#include <stdio.h>
#include <ti/ndk/inc/netmain.h>
#include <ti/transport/ndk/nimu/example/helloWorld/src/nimu_cppi_qmss_iface.h>

#ifdef _TMS320C6X
#include <ti/csl/csl_chipAux.h>
#endif

#include <ti/csl/csl_psc.h>
#include <ti/csl/csl_pscAux.h>

#ifdef __ARM_ARCH_7A__
#include <ti/csl/cslr_msmc.h>
#include <ti/csl/csl_bootcfgAux.h>
#endif
/* BIOS6 include */
#include <ti/sysbios/BIOS.h>

#ifdef _TMS320C6X
#include <ti/sysbios/family/c64p/Hwi.h>
#include <ti/sysbios/family/c64p/EventCombiner.h>
#include <ti/sysbios/family/c66/Cache.h>
#else
#include <ti/sysbios/family/arm/gic/Hwi.h>
#include <ti/sysbios/family/arm/a15/Mmu.h>
#endif

#ifdef ADD_UDP_RECV_TASK
#include <ti/sysbios/knl/Task.h>
#endif
#include <xdc/runtime/System.h>


//---------------------------------------------------------------------------
// Title String
//
char *VerStr = "\nTCP/IP Stack 'Hello World! application with OMP extension\n";

// Our NETCTRL callback functions
static void NetworkOpen();
static void NetworkClose();
static void NetworkIPAddr(IPN IPAddr, uint IfIdx, uint fAdd);

// Fun reporting function
static void ServiceReport(uint Item, uint Status, uint Report, HANDLE hCfgEntry);

// External references
extern int dtask_udp_hello();
#ifdef ADD_UDP_RECV_TASK
extern Void task_udp_recv(UArg unused0, UArg unused1);
#endif

//---------------------------------------------------------------------------
// Configuration
//
char *HostName = "tidsp";
char *LocalIPAddr = "10.48.109.84";
char *LocalIPMask = "255.255.254.0";    // Not used when using DHCP
char *GatewayIP = "10.48.108.128";    // Not used when using DHCP
char *DomainName = "demo.net";         // Not used when using DHCP
char *DNSServer = "0.0.0.0";          // Used when set to anything but zero

// Simulator EMAC Switch does not handle ALE_LEARN mode, so please configure the
// MAC address of the PC where you want to launch the webpages and initiate PING to NDK */

Uint8 clientMACAddress[6] = { 0xd4, 0xbe, 0xd9, 0x3d, 0xb6, 0xb8 }; /* MAC address for my PC */

//---------------------------------------------------------------------
// Main Entry Point
//---------------------------------------------------------------------
int main() {
	/* Start the BIOS 6 Scheduler */
	UInt marVal;

	marVal = Cache_getMar((Ptr)0x0c020000);
	marVal &= ~(1 << 3);
	Cache_wbInvAll();
	Cache_setMar((Ptr)0x0c020000, 0x003e0000, marVal);

	BIOS_start();
	return 0;
}

//
// Main Thread
//
Void StackTest(UArg unused0, UArg unused1)
{
	int rc;
	HANDLE hCfg;
	NIMU_QMSS_CFG_T qmss_cfg;
	NIMU_CPPI_CFG_T cppi_cfg;

	// Print out our banner
	System_printf(VerStr);

	if (CSL_chipReadDNUM() != 0)
		System_abort("NDK should be running on Core0\n");

	/* Initialize the components required to run this application:
	 *  (1) QMSS
	 *  (2) CPPI
	 *  (3) Packet Accelerator
	 */
	qmss_cfg.master_core = 1;
	qmss_cfg.max_num_desc = MAX_NUM_DESC;
	qmss_cfg.desc_size = MAX_DESC_SIZE;
	qmss_cfg.mem_region = Qmss_MemRegion_MEMORY_REGION0;
	if (NIMU_initQmss(&qmss_cfg) != 0) {
		System_abort("Failed to initialize the QMSS subsystem \n");
	} else {
		System_printf("QMSS successfully initialized \n");
	}

	cppi_cfg.master_core = 1;
	cppi_cfg.dma_num = Cppi_CpDma_PASS_CPDMA;
	cppi_cfg.num_tx_queues = NUM_PA_TX_QUEUES;
	cppi_cfg.num_rx_channels = NUM_PA_RX_CHANNELS;
	if (NIMU_initCppi(&cppi_cfg) != 0) {
		System_abort("Failed to initialize CPPI subsystem\n");
	} else {
		System_printf("CPPI successfully initialized \n");
	}

	if (NIMU_initPass() != 0) {
		System_abort("Failed to initialize the Packet Accelerator \n");
	} else {
		System_printf("PA successfully initialized \n");
	}

	//
	// THIS MUST BE THE ABSOLUTE FIRST THING DONE IN AN APPLICATION!!
	//
	rc = NC_SystemOpen( NC_PRIORITY_LOW, NC_OPMODE_INTERRUPT);
	if (rc) {
		System_abort("NC_SystemOpen Failed)\n");
	}

	//
	// Create and build the system configuration from scratch.
	//

	// Create a new configuration
	hCfg = CfgNew();
	if (!hCfg) {
		System_abort("Unable to create configuration\n");
	}

	// We better validate the length of the supplied names
	if (strlen(DomainName) >= CFG_DOMAIN_MAX
			|| strlen(HostName) >= CFG_HOSTNAME_MAX) {
		System_abort("Names too long\n");
	}

	// Add our global hostname to hCfg (to be claimed in all connected domains)
	CfgAddEntry(hCfg, CFGTAG_SYSINFO, CFGITEM_DHCP_HOSTNAME, 0,
			strlen(HostName), (UINT8 *) HostName, 0);

	// If the IP address is specified, manually configure IP and Gateway
#if defined(_SCBP6618X_) || defined (DEVICE_C6678) || defined(_EVMTCI6614_) || defined(DEVICE_K2H) || defined(DEVICE_K2K) || defined(DEVICE_K2L) || defined(DEVICE_K2E)

	/* SCBP6618x, EVMTCI6614, EVMK2H, EVMK2K always uses DHCP */
	if (1)
#else
	if (!platform_get_switch_state(1))
#endif
	{
		CI_IPNET NA;
		CI_ROUTE RT;
		IPN IPTmp;

		System_printf("StackTest: using localIp\n");
		// Setup manual IP address
		bzero(&NA, sizeof(NA));
		NA.IPAddr = inet_addr(LocalIPAddr);
		NA.IPMask = inet_addr(LocalIPMask);
		strcpy(NA.Domain, DomainName);
		NA.NetType = 0;

		// Add the address to interface 1
		CfgAddEntry(hCfg, CFGTAG_IPNET, 1, 0, sizeof(CI_IPNET),
				(UINT8 *) &NA, 0);

		// Add the default gateway. Since it is the default, the
		// destination address and mask are both zero (we go ahead
		// and show the assignment for clarity).
		bzero(&RT, sizeof(RT));
		RT.IPDestAddr = 0;
		RT.IPDestMask = 0;
		RT.IPGateAddr = inet_addr(GatewayIP);

		// Add the route
		CfgAddEntry(hCfg, CFGTAG_ROUTE, 0, 0, sizeof(CI_ROUTE),
				(UINT8 *) &RT, 0);

		// Manually add the DNS server when specified
		IPTmp = inet_addr(DNSServer);
		if (IPTmp)
			CfgAddEntry(hCfg, CFGTAG_SYSINFO, CFGITEM_DHCP_DOMAINNAMESERVER,
					0, sizeof(IPTmp), (UINT8 *) &IPTmp, 0);
	}
	// Else we specify DHCP
	else {
		System_printf("StackTest: using dhcp\n");
		CI_SERVICE_DHCPC dhcpc;

		// Specify DHCP Service on IF-1
		bzero(&dhcpc, sizeof(dhcpc));
		dhcpc.cisargs.Mode = CIS_FLG_IFIDXVALID;
		dhcpc.cisargs.IfIdx = 1;
		dhcpc.cisargs.pCbSrv = &ServiceReport;
		CfgAddEntry(hCfg, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT, 0,
				sizeof(dhcpc), (UINT8 *) &dhcpc, 0);
	}

	//
	// Configure IPStack/OS Options
	//

	// We don't want to see debug messages less than WARNINGS
	rc = DBG_WARN;
	CfgAddEntry(hCfg, CFGTAG_OS, CFGITEM_OS_DBGPRINTLEVEL,
	CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *) &rc, 0);

	//
	// This code sets up the TCP and UDP buffer sizes
	// (Note 8192 is actually the default. This code is here to
	// illustrate how the buffer and limit sizes are configured.)
	//

	// UDP Receive limit
	rc = 8192;
	CfgAddEntry(hCfg, CFGTAG_IP, CFGITEM_IP_SOCKUDPRXLIMIT,
	CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *) &rc, 0);

	//
	// Boot the system using this configuration
	//
	// We keep booting until the function returns 0. This allows
	// us to have a "reboot" command.
	//
	do {
		rc = NC_NetStart(hCfg, NetworkOpen, NetworkClose, NetworkIPAddr);
	} while (rc > 0);

	// Delete Configuration
	CfgFree(hCfg);

	NC_SystemClose();

	System_printf("StackTest: exiting\n");

}

//
// System Task Code [ Server Daemon Servers ]
//
static HANDLE hHello = 0;
#ifdef ADD_UDP_RECV_TASK
static Task_Handle g_udpRecvHandle = NULL;
#endif
//
// NetworkOpen
//
// This function is called after the configuration has booted
//
static void NetworkOpen() {

#ifdef ADD_UDP_RECV_TASK
	// Create receive task for incoming UDP packets
	Task_Params tskParams;
	Task_Params_init(&tskParams);
	tskParams.stackSize = 0x1000;
	tskParams.priority = 3;
	g_udpRecvHandle = Task_create(task_udp_recv, &tskParams, NULL);
#endif

	// Create our local server
	hHello = DaemonNew( SOCK_DGRAM, 0, 7, dtask_udp_hello,
			OS_TASKPRINORM, OS_TASKSTKNORM, 0, 1 );

}

//
// NetworkClose
//
// This function is called when the network is shutting down,
// or when it no longer has any IP addresses assigned to it.
//
static void NetworkClose() {

#ifdef ADD_UDP_RECV_TASK
	// Delete receive task for UDP packets
	if (g_udpRecvHandle)
		Task_delete(&g_udpRecvHandle);
#endif

	DaemonFree(hHello);
}

//
// NetworkIPAddr
//
// This function is called whenever an IP address binding is
// added or removed from the system.
//
static void NetworkIPAddr(IPN IPAddr, uint IfIdx, uint fAdd) {
	IPN IPTmp;

	if (fAdd)
		printf("Network Added: ");
	else
		printf("Network Removed: ");

	// Print a message
	IPTmp = ntohl(IPAddr);
	printf("If-%d:%d.%d.%d.%d\n", IfIdx, (UINT8) (IPTmp >> 24) & 0xFF,
			(UINT8) (IPTmp >> 16) & 0xFF, (UINT8) (IPTmp >> 8) & 0xFF,
			(UINT8) IPTmp & 0xFF);
}

//
// Service Status Reports
//
// Here's a quick example of using service status updates
//
static char *TaskName[] = { "Telnet", "HTTP", "NAT", "DHCPS", "DHCPC", "DNS" };
static char *ReportStr[] = { "", "Running", "Updated", "Complete", "Fault" };
static char *StatusStr[] = { "Disabled", "Waiting", "IPTerm", "Failed",
		"Enabled" };
static void ServiceReport(uint Item, uint Status, uint Report, HANDLE h) {
	printf("Service Status: %-9s: %-9s: %-9s: %03d\n", TaskName[Item - 1],
			StatusStr[Status], ReportStr[Report / 256], Report & 0xFF);

	//
	// Example of adding to the DHCP configuration space
	//
	// When using the DHCP client, the client has full control over access
	// to the first 256 entries in the CFGTAG_SYSINFO space.
	//
	// Note that the DHCP client will erase all CFGTAG_SYSINFO tags except
	// CFGITEM_DHCP_HOSTNAME. If the application needs to keep manual
	// entries in the DHCP tag range, then the code to maintain them should
	// be placed here.
	//
	// Here, we want to manually add a DNS server to the configuration, but
	// we can only do it once DHCP has finished its programming.
	//
	if (Item == CFGITEM_SERVICE_DHCPCLIENT && Status == CIS_SRV_STATUS_ENABLED
			&& (Report == (NETTOOLS_STAT_RUNNING | DHCPCODE_IPADD)
					|| Report == (NETTOOLS_STAT_RUNNING | DHCPCODE_IPRENEW))) {
		IPN IPTmp;

		// Manually add the DNS server when specified
		IPTmp = inet_addr(DNSServer);
		if (IPTmp)
			CfgAddEntry(0, CFGTAG_SYSINFO, CFGITEM_DHCP_DOMAINNAMESERVER, 0,
					sizeof(IPTmp), (UINT8 *) &IPTmp, 0);
	}
}


