/*
 * device.c
*/

/*
 * Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/ 
*/
/* 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/* --------------------------------------------------------------------------
  FILE        : device.c                                                   
  PROJECT     : TI Boot and Flash Utils
  AUTHOR      : Daniel Allred
  DESC        : This file provides low-level init functions.
-------------------------------------------------------------------------- */ 


/************************************************************
* Include Files                                             *
************************************************************/

// General type include
#include "tistdtypes.h"

// Utility functions
#include "util.h"

// This module's header file
#include "device.h"


/************************************************************
* Explicit External Declarations                            *
************************************************************/

extern __FAR__ Uint32 EXTERNAL_RAM_START;


/************************************************************
* Local Macro Declarations                                  *
************************************************************/


/************************************************************
* Local Typedef Declarations                                *
************************************************************/


/************************************************************
* Local Function Declarations                               *
************************************************************/


/************************************************************
* Local Variable Definitions                                *
\***********************************************************/

// Boot pin to LUT index mapping: 72103 = 43210
static const Uint8 LOCAL_bootModeLUT[] = 
{                                     //         43210 = 72103
  DEVICE_BOOTMODE_I2C0_MASTER,        // Index = 00000b, Boot pins = 0xxx0000b
  DEVICE_BOOTMODE_I2C0_SLAVE,         // Index = 00001b, Boot pins = 0xxx1000b
  DEVICE_BOOTMODE_NOR_EMIFA,          // Index = 00010b, Boot pins = 0xxx0001b
  DEVICE_BOOTMODE_NOR_EMIFA,          // Index = 00011b, Boot pins = 0xxx1001b
  DEVICE_BOOTMODE_UHPI,               // Index = 00100b, Boot pins = 0xxx0010b
  DEVICE_BOOTMODE_UHPI,               // Index = 00101b, Boot pins = 0xxx1010b
  DEVICE_BOOTMODE_I2C1_MASTER,        // Index = 00110b, Boot pins = 0xxx0011b
  DEVICE_BOOTMODE_I2C1_SLAVE,         // Index = 00111b, Boot pins = 0xxx1011b
  DEVICE_BOOTMODE_SPI0_EEPROM,        // Index = 01000b, Boot pins = 0xxx0100b
  DEVICE_BOOTMODE_SPI1_EEPROM,        // Index = 01001b, Boot pins = 0xxx1100b
  DEVICE_BOOTMODE_SPI0_FLASH,         // Index = 01010b, Boot pins = 0xxx0101b
  DEVICE_BOOTMODE_SPI0_FLASH,         // Index = 01011b, Boot pins = 0xxx1101b
  DEVICE_BOOTMODE_SPI1_FLASH,         // Index = 01100b, Boot pins = 0xxx0110b
  DEVICE_BOOTMODE_SPI1_FLASH,         // Index = 01101b, Boot pins = 0xxx1110b
  DEVICE_BOOTMODE_NAND_EMIFA_8BIT,    // Index = 01110b, Boot pins = 0xxx0111b
  DEVICE_BOOTMODE_NAND_EMIFA_8BIT,    // Index = 01111b, Boot pins = 0xxx1111b
  DEVICE_BOOTMODE_NAND_EMIFA_16BIT,   // Index = 10000b, Boot pins = 1xxx0000b
  DEVICE_BOOTMODE_NONE,               // Index = 10001b, Boot pins = 1xxx1000b
  DEVICE_BOOTMODE_SPI0_SLAVE,         // Index = 10010b, Boot pins = 1xxx0001b
  DEVICE_BOOTMODE_SPI1_SLAVE,         // Index = 10011b, Boot pins = 1xxx1001b
  DEVICE_BOOTMODE_UART2,              // Index = 10100b, Boot pins = 1xxx0010b
  DEVICE_BOOTMODE_THB,                // Index = 10101b, Boot pins = 1xxx1010b
  DEVICE_BOOTMODE_UART0,              // Index = 10110b, Boot pins = 1xxx0011b
  DEVICE_BOOTMODE_UART1,              // Index = 10111b, Boot pins = 1xxx1011b
  DEVICE_BOOTMODE_ESF,                // Index = 11000b, Boot pins = 1xxx0100b
  DEVICE_BOOTMODE_ESF,                // Index = 11001b, Boot pins = 1xxx1100b
  DEVICE_BOOTMODE_USB11,              // Index = 11010b, Boot pins = 1xxx0101b
  DEVICE_BOOTMODE_USB20,              // Index = 11011b, Boot pins = 1xxx1101b
  DEVICE_BOOTMODE_MMC,                // Index = 11100b, Boot pins = 1xxx0110b
  DEVICE_BOOTMODE_RMII,               // Index = 11101b, Boot pins = 1xxx1110b
  DEVICE_BOOTMODE_EMU_DEBUG,          // Index = 11110b, Boot pins = 1xxx0111b
  DEVICE_BOOTMODE_NONE,               // Index = 11111b, Boot pins = 1xxx1111b
};


/************************************************************
* Global Variable Definitions                               *
************************************************************/

#if defined(AM1707)
  const char devString[] = "AM1707";
#elif defined(OMAPL137_v1)
  const char devString[] = "OMAP-L137, Rev1.0";
#elif defined(OMAPL137_v2)
  const char devString[] = "OMAP-L137, Rev2.0";
#elif defined(C6747)
  const char devString[] = "C6747";
#else
  const char devString[] = "OMAP-L137";
#endif

/************************************************************
* Global Function Definitions                               *
************************************************************/

Uint32 DEVICE_init()
{
  Uint32 status = E_PASS;
  
  // Ensure the kick registers are unlocked
  DEVICE_kickLock();
  DEVICE_kickUnlock();

#ifndef SKIP_LOW_LEVEL_INIT
  // System PSC Setup
  DEVICE_PSCInit();

  // System PLL Setup
  if (status == E_PASS)
  {
    #if defined(AM1707)
    // CPU(s) at 456 MHz
     status |= DEVICE_PLL0Init(0, 18, 0, 3, 3, 5, 441);
	//status |= DEVICE_PLL0Init(0, 24, 1, 2, 2, 5, 441);
    #else
    // CPU(s) at 300 MHz
    status |= DEVICE_PLL0Init(0, 24, 1, 2, 2, 5, 441);
    #endif  
  }
  
  
  // External Memory Timing Setup
  if (status == E_PASS)
  {
    status |= DEVICE_ExternalMemInit(0x00018421, 0x10912A08, 0x70090005, 0x000003FA, TRUE);
  }
#endif

  return status;
}

DEVICE_ChipRevIDType DEVICE_chipRevIDType( )
{
  // The code used by the actual ROM boot loader
  Uint32 chipType = (SYSTEM->CHIPREVID & DEVICE_CHIPREVID_TYPE_MASK) >> DEVICE_CHIPREVID_TYPE_SHIFT;
  return ((DEVICE_ChipRevIDType)chipType);
}

void DEVICE_LPSCTransition(Uint8 pscnum, Uint8 module, Uint8 domain, Uint8 state)
{
  DEVICE_PSCRegs *PSC;
  
  if (pscnum == PSCNUM0)
    PSC = PSC0;
  else if(pscnum == PSCNUM1)
    PSC = PSC1;
  else
    return; 

  // Wait for any outstanding transition to complete
  while ( (PSC->PTSTAT) & (0x00000001 << domain) );
  
  // If we are already in that state, just return
  if (((PSC->MDSTAT[module]) & 0x1F) == state) return;
    
  // Perform transition
  PSC->MDCTL[module] = ((PSC->MDCTL[module]) & (0xFFFFFFE0)) | (state);
  PSC->PTCMD |= (0x00000001 << domain);

  // Wait for transition to complete
  while ( (PSC->PTSTAT) & (0x00000001 << domain) );
  
  // Wait and verify the state
  while (((PSC->MDSTAT[module]) & 0x1F) != state);	
}

void DEVICE_pinmuxControl(Uint32 regOffset, Uint32 mask, Uint32 value)
{
  SYSTEM->PINMUX[regOffset] = (SYSTEM->PINMUX[regOffset] & ~mask) | (mask & value);
}

void DEVICE_PSCInit()
{                                                                    
  Uint32 i;
  
  // PSC0, domain 0 init
  while ((PSC0->PTSTAT) & 0x00000001);

  for( i = 0 ; i <= 6 ; i++ )
    PSC0->MDCTL[i] = (PSC0->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;

  for( i = 8 ; i <= 13 ; i++ )
    PSC0->MDCTL[i] = (PSC0->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;
    
  // Do Always-On Power Domain Transitions
  PSC0->PTCMD |= 0x00000001;
  while ((PSC0->PTSTAT) & 0x00000001);
  
  // PSC1, domain 1 init
  while ((PSC1->PTSTAT) & 0x00000001);

  for( i = 1 ; i <= 13 ; i++ )
    PSC1->MDCTL[i] = (PSC1->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;

  for( i = 16 ; i <= 17 ; i++ )
    PSC1->MDCTL[i] = (PSC1->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;

  for( i = 20 ; i <= 21 ; i++ )
    PSC1->MDCTL[i] = (PSC1->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;
  
  for( i = 24 ; i <= 26 ; i++ )
    PSC1->MDCTL[i] = (PSC1->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;

  for( i = 31 ; i <= 31 ; i++ )
    PSC1->MDCTL[i] = (PSC1->MDCTL[i] & 0xFFFFFFE0) | PSC_ENABLE;
    
  // Do Always-On Power Domain Transitions
  PSC1->PTCMD |= 0x00000001;
  while ((PSC1->PTSTAT) & 0x00000001);
}

void DEVICE_kickUnlock(void)
{
  SYSTEM->KICKR[0] = 0x83e70b13;  // Kick0 register + data (unlock)
  SYSTEM->KICKR[1] = 0x95a4f1e0;  // Kick1 register + data (unlock)
}

void DEVICE_kickLock( void )
{
  SYSTEM->KICKR[0] = 0x00000000;  // Kick0 register + data (lock)
  SYSTEM->KICKR[1] = 0x00000000;  // Kick0 register + data (lock)
}

DEVICE_BootPeripheral DEVICE_bootPeripheral( void )
{
  DEVICE_BootPeripheral bp;
  DEVICE_BootMode bm = DEVICE_bootMode();
  switch (bm)
  {
    case DEVICE_BOOTMODE_SPI0_EEPROM:
    case DEVICE_BOOTMODE_SPI1_EEPROM:
    case DEVICE_BOOTMODE_SPI0_FLASH:
    case DEVICE_BOOTMODE_SPI1_FLASH:
      bp = DEVICE_BOOTPERIPHERAL_SPI;
      break;
    case DEVICE_BOOTMODE_I2C0_MASTER:
    case DEVICE_BOOTMODE_I2C0_SLAVE:
    case DEVICE_BOOTMODE_I2C1_MASTER:
    case DEVICE_BOOTMODE_I2C1_SLAVE: 
      bp = DEVICE_BOOTPERIPHERAL_I2C;
      break;
    case DEVICE_BOOTMODE_NAND_EMIFA_8BIT:
    case DEVICE_BOOTMODE_NAND_EMIFA_16BIT:
      bp = DEVICE_BOOTPERIPHERAL_NAND;
      break;
    case DEVICE_BOOTMODE_UART0:
    case DEVICE_BOOTMODE_UART1:
    case DEVICE_BOOTMODE_UART2:
      bp = DEVICE_BOOTPERIPHERAL_UART;
      break;
    case DEVICE_BOOTMODE_NOR_EMIFA:
      bp = DEVICE_BOOTPERIPHERAL_NOR;
      break;
    default:
      bp = DEVICE_BOOTPERIPHERAL_NONE;
      break;
  }
  
  return bp;
}

DEVICE_BootMode DEVICE_bootMode( void )
{
  Uint8 i;

  // Read BOOT pins and set bootMode using a look-up table
  i = (SYSTEM->BOOTCFG & DEVICE_BOOTCFG_BOOTMODE_MASK) >> DEVICE_BOOTCFG_BOOTMODE_SHIFT;

  // Bit sequence used for LUT is 7,2,1,0,3
  // Translate: 76543210 to ___72103
  // (__210 << 1) + (7___3___ >> 3) = _210_ + 7___3 = 72103
  i = ((i & 0x0007) << 1) + ((i & 0x0088) >> 3);      // _210_ + 7___3

  return ((DEVICE_BootMode)LOCAL_bootModeLUT[i]);
}

DEVICE_BusWidth DEVICE_emifBusWidth( void )
{
  if (DEVICE_bootMode() == DEVICE_BOOTMODE_NAND_EMIFA_8BIT)
  {
    return DEVICE_BUSWIDTH_8BIT;
  }
  else
  {
    return DEVICE_BUSWIDTH_16BIT;
  }
}

Uint32 DEVICE_PLL0Init(Uint8 clk_src,Uint8 pllm, Uint8 postdiv, Uint8 div3, Uint8 div5, Uint8 div7, Uint16 lock_time )
{
  // Reset PLL and put in bypass mode
  // Set PLLEN=0 and PLLRST=0
  PLL0->PLLCTL &= 0xFFFFFFFE;

  // wait for 4 cycles to allow PLLEN mux switches properly to bypass clock
  UTIL_waitLoopAccurate (4);

  // Select the Clock Mode bit 8 as External Clock or On Chip Oscilator
  PLL0->PLLCTL &= 0xFFFFFEFF;
  PLL0->PLLCTL |= (clk_src << 8);

  /* Set PLLENSRC '0',bit 5, PLL Enable(PLLEN) selection is controlled through MMR */
  PLL0->PLLCTL &= 0xFFFFFFDF;

  /* PLLCTL.EXTCLKSRC bit 9 should be left at 0 for Primus */
  PLL0->PLLCTL &= 0xFFFFFDFF;

  /* Clear PLLRST bit to 0 -Reset the PLL */
  PLL0->PLLCTL &= 0xFFFFFFF7;

  /* Disable the PLL output */
  PLL0->PLLCTL |= 0x10;

  /* PLL initialization sequence */

  /* Power up the PLL- PWRDN bit set to 0 to bring the PLL out of power down bit */
  PLL0->PLLCTL &= 0xFFFFFFFD;

  /* Enable the PLL from Disable Mode PLLDIS bit to 0 - This is step is not required for Primus */
  PLL0->PLLCTL &= 0xFFFFFFEF;

  /* Program the required multiplier value in PLLM */
  PLL0->PLLM = pllm;

  /* If desired to scale all the SYSCLK frequencies of a given PLLC, program the POSTDIV ratio */
  PLL0->POSTDIV = 0x8000 | postdiv;

  /* Check for the GOSTAT bit in PLLSTAT to clear to 0 to indicate that no GO operation is currently in progress */
  while ((PLL0->PLLSTAT & 0x1) == 1);

  /* Program the RATIO field in PLLDIVx with the desired divide factors. In addition, make sure in this step you leave the PLLDIVx.DxEN bits set so clocks are still enabled (default). */
  PLL0->PLLDIV3 = 0x8000 | div3;
  PLL0->PLLDIV5 = 0x8000 | div5;
  PLL0->PLLDIV7 = 0x8000 | div7;

  /* Set the GOSET bit in PLLCMD to 1 to initiate a new divider transition. */
  PLL0->PLLCMD |= 0x1;

  /* Wait for the GOSTAT bit in PLLSTAT to clear to 0 (completion of phase alignment). */
  while ((PLL0->PLLSTAT & 0x1) == 1);

  /* Set the PLLRST bit in PLLCTL to 1 to bring the PLL out of reset */
  PLL0->PLLCTL |= 0x8;

  /* Wait for PLL to lock. See PLL spec for PLL lock time */
  UTIL_waitLoopAccurate (lock_time);

  /* Set the PLLEN bit in PLLCTL to 1 to remove the PLL from bypass mode */
  PLL0->PLLCTL |= 0x1;

  return E_PASS;
}

Uint32 DEVICE_ExternalMemInit(Uint32 sdcr, Uint32 sdtimr1, Uint32 sdtimr2, Uint32 sdrcr, Bool use45Div)
{
  DEVICE_pinmuxControl(0,0xFFFFFF00,0x11111100);  // nEMB_WE, nEMB_RAS, nEMB_CAS, nEMB_CS[0], EMB_CLK, EMB_SDCKE
  DEVICE_pinmuxControl(1,0xFFFFFFFF,0x11111111);  // EMB_A[5], EMB_A[4], EMB_A[3], EMB_A[2], EMB_A[1], EMB_A[0], EMB_BA[0], EMB_BA[1]
  DEVICE_pinmuxControl(2,0x0FFFFFFF,0x01111111);  // EMB_A[12], EMB_A[11], EMB_A[10], EMB_A[9], EMB_A[8], EMB_A[7], EMB_A[6]
  DEVICE_pinmuxControl(5,0xFFFFFFF0,0x11111110);  // EMB_D[6], EMB_D[5], EMB_D[4], EMB_D[3], EMB_D[2], EMB_D[1], EMB_D[0]
  DEVICE_pinmuxControl(6,0xFFFFFFFF,0x11111111);  // EMB_D[14], EMB_D[13], EMB_D[12], EMB_D[11], EMB_D[10], EMB_D[9], EMB_D[8], EMB_D[7]
  DEVICE_pinmuxControl(7,0x00000FFF,0x00000111);  // nEMB_WE_DQM[0], nEMB_WE_DQM[1], EMB_D[15]

  if (((sdcr & DEVICE_SDCR_NM_MASK) >> DEVICE_SDCR_NM_SHIFT) == 0x0)
  {
    DEVICE_pinmuxControl(2,0xF0000000,0x10000000);  // EMB_D[31]
    DEVICE_pinmuxControl(3,0xFFFFFFFF,0x11111111);  // EMB_D[23], EMB_D[24], EMB_D[25], EMB_D[26], EMB_D[27], EMB_D[28], EMB_D[29], EMB_D[30]
    DEVICE_pinmuxControl(4,0xFFFFFFFF,0x11111111);  // nEMB_WE_DQM[3], EMB_D[16], EMB_D[17], EMB_D[18], EMB_D[19], EMB_D[20], EMB_D[21], EMB_D[22]
    DEVICE_pinmuxControl(5,0x0000000F,0x00000001);  // nEMB_WE_DQM[2]
  }

  // Power ON
  DEVICE_LPSCTransition(PSCNUM1, LPSC_EMIFB, PD0, PSC_ENABLE);

  // Configure
  EMIF3C->SDCR    = sdcr;
  EMIF3C->SDTIMR  = sdtimr1;
  EMIF3C->SDTIMR2 = sdtimr2;
  EMIF3C->SDRCR   = sdrcr;
  
  if (use45Div)
  {
    SYSTEM->CFGCHIP[3] = SYSTEM->CFGCHIP[3] | 0x5;
  }
  
  return E_PASS;
}

Uint32 DEVICE_AsyncMemInit (Uint8 interfaceNum)
{
  if (interfaceNum == 0)
  {
    // Turn on the EMIF LPSC
    DEVICE_LPSCTransition(PSCNUM0, LPSC_EMIFA, PD0, PSC_ENABLE);

    // Set PINMUX for EMIF use
  DEVICE_pinmuxControl(13,0xFFFFFFFF,0x11111111);
  DEVICE_pinmuxControl(14,0xFFFFFFFF,0x88111111);
  DEVICE_pinmuxControl(15,0xFFFFFFFF,0x11888888);
  DEVICE_pinmuxControl(16,0xFFFFFFFF,0x11111111);
  DEVICE_pinmuxControl(17,0xFFFFFFFF,0x08111111);
  DEVICE_pinmuxControl(18,0xFFFFFFFF,0x11111111);

  }
  else
  {
    return E_FAIL;
  }

  return E_PASS;
}

// Device-level SPI initialization
Uint32 DEVICE_SPIInit(Uint8 periphNum)
{
  if (periphNum == 0)
  {
    DEVICE_LPSCTransition(PSCNUM0, LPSC_SPI0, PD0, PSC_ENABLE);
    DEVICE_pinmuxControl(7,0xF0FFF000,0x10111000); // nSPI0_SCS[0], SPI0_CLK, SPI0_SIMO[0], SPI0_SOMI[0]
  }
  else if (periphNum == 1)
  {
    DEVICE_LPSCTransition(PSCNUM1, LPSC_SPI1, PD0, PSC_ENABLE);
    DEVICE_pinmuxControl(8,0x00000FFF,0x00000111); // SPI1_CLK, SPI1_SIMO[0], SPI1_SOMI[0]
    DEVICE_pinmuxControl(9,0x0000000F,0x00000001); // nSPI1_SCS[0]    
  }
  else
  {
    return E_FAIL;
  }
  
  return E_PASS;
}

// Device-level UART initialization
Uint32 DEVICE_UARTInit(Uint8 periphNum)
{
  switch (periphNum)
  {
    case 0:
      DEVICE_LPSCTransition(PSCNUM0, LPSC_UART0, PD0, PSC_ENABLE);
      DEVICE_pinmuxControl(8,0x000FF000,0x00011000); // UART0_TXD, UART0_RXD
      break;
    case 1:
      DEVICE_LPSCTransition(PSCNUM1, LPSC_UART1, PD0, PSC_ENABLE);
      DEVICE_pinmuxControl(11,0x0000FF00,0x00001100); // UART1_TXD, UART1_RXD
      break;
    case 2:
      DEVICE_LPSCTransition(PSCNUM1, LPSC_UART2, PD0, PSC_ENABLE);
      DEVICE_pinmuxControl(8,0xF0000000,0x20000000); // UART2_RXD
      DEVICE_pinmuxControl(9,0x0000000F,0x00000002); // UART2_TXD
      break;
    default:
      return E_FAIL;
  }
  return E_PASS;
}

Uint32 DEVICE_TIMER0Init()
{
  // Power-up the Timer
  //DEVICE_LPSCTransition(LPSC_TIMER0, PD0 ,PSC_ENABLE);

  // Put timer into reset
  //TIMER0->EMUMGT_CLKSPD = 0x00000003;
  //TIMER0->TCR   = 0x00000000;
  
  // Setup as 2 unchained 32-bit timers
  //TIMER0->TGCR  = 0x00000005;

  // Reset timers to zero 
  //TIMER0->TIM12 = 0x00000000;

  // Set period to 5 sec
  //TIMER0->PRD12 = 0x019BFCC0;

  return E_PASS;
}

void DEVICE_TIMER0Start(void)
{
  // Clear interrupt
  //AINTC->IRQ1   |=  0x00000001;

  // Put timer out in reset
  //TIMER0->TGCR  &= ~(0x00000001);
    
  // Reset the timer counter to zero
  //TIMER0->TIM12 = 0x00000000;

  // Set for one-time run
  //TIMER0->TCR   = 0x00000040;
    
  // Take timer out of reset
  //TIMER0->TGCR  = 0x00000005;
  
}

void DEVICE_TIMER0Stop(void)
{
  // Clear interrupt
  //AINTC->IRQ1   |=  0x00000001;

  // Put timer in reset
  //TIMER0->TCR   = 0x00000000;
  //TIMER0->TGCR  = 0x00000000;

  // Reset timer count to zero 
  //TIMER0->TIM12 = 0x00000000;
}

Uint32 DEVICE_TIMER0Status(void)
{
  //return ((AINTC->IRQ1)&0x1);
	return 1;
}


/************************************************************
* Local Function Definitions                                *
************************************************************/


/***********************************************************
* End file                                                 *
***********************************************************/

