
// generic
#include "stdio.h"
#include "string.h"

// psp
//#include "psp_i2s.h"
//#include "psp_i2c.h"
//#include "dda_i2c.h"
//#include "psp_common.h"

// csl
#include <cslr.h>
#include <tistdtypes.h>
#include <csl_general.h>
#include "osd9616_control.h"

// dsplib
//#include "dsplib.h"

// local
//#include "app_globals.h"
//#include "globals.h"
#include "i2c_control.h"
//#include "Arescfg.h"

// defines

#define DISPLAY_I2C_ADDR 0x3C   // OSD9616 I2C address including R/W for Write!
//#define DISPLAY_I2C_ADDR 0x1D   // OSD9616 I2C address

// global variables

//extern PSP_Handle hi2c;

// index to characters in TI DSP ROM
// Characters are made up of dot matrix arrays with bits filling in the columns and sequential bytes filling in the rows
#pragma DATA_SECTION(CharGen_6x8, ".charrom")       // .charrom defined in lnkx.cmd
Uint16 CharGen_6x8[256*3];	                        // dot graphic of ascii characters from TI DSP ROM for display

// Unshared Prototypes

Int16 OSD9616_send( Uint16 regAddr, Uint16 regData);
Int16 OSD9616_multiSend( Uint16 *regData, Uint16 length);

// methods

Int16 oled_init()
{
    /* Initialize OSD9616 display */

    // for setup, keep in mind that the display is in the hardware range
    // of segment 127 to 32
    // and page 6 to 7 (COM 0 to 15) unless reversed

	Int16 i2c_err;
	Uint16 cmd[10];    // For multibyte commands

	// ************ Hardware Paramenters *****************

    // Set multiplex ratio (15 to 63 = MUX 16 to 64) - scanning stops at row specified
    cmd[0] = 0x00;                      // signal command
    cmd[1] = 0xa8;                      // set multiplex ratio
    cmd[2] = 0x0f;                      // set to N+1 MUX or 16
    i2c_err=OSD9616_multiSend( cmd, 3);
    
    if(i2c_err)  // Don't continue to setup display if not connected
        return 1;

    // Set Display (vertical) Offset by pixel row from 0 to 63 (maps ROW to COM0)
    cmd[0] = 0x00;                      // signal command mode
    cmd[1] = 0xd3;                      // Set Offset
    cmd[2] = 0x00;                      // none (ROW0 = COM0)
    OSD9616_multiSend(cmd, 3);

    // Set Display Start Line Address in memory (maps RAM to ROW0)
    OSD9616_send(0x00,0x40);            // set start location to 0 (RAM0=ROW0)

    // Set Segment (horizontal) Remap
    // the display controller has a range of pixels from SEG 0 to 127 and COM 0 to 63 (through pages 0 to 7)
    // our display normally maps to SEG 127 to 32 (left to right) COM 0 to 15 (bottom to top)
    // this remaps the display to SEG 0 to 96 (left to right) but leaves COM 0 to 15 (bottom to top)
    OSD9616_send(0x00,0xa1);            // column address 127 is mapped to SEG0 (flip horizontal mapping)

    // Set COM Output (vertical) Scan Direction (will flip vertically immediately when changed)
    OSD9616_send(0x00,0xc0);            // scan from COM0 to COMN (default)

    // Set COM pins hardware configuration
    cmd[0] = 0x00;                      // signal command
    cmd[1] = 0xda;                      // Set COM Pins Hardware Config
    cmd[2] = 0x02;                      // Sequential COM Pin config, disable COM left right/remap (don't flip vertical mapping)
    OSD9616_multiSend( cmd, 3);

    // Set Contrast Control Register
    cmd[0] = 0x00;                      // signal command mode
    cmd[1] = 0x81;                      // Set Contrast Control
    cmd[2] = 0x7f;                      // half way
    OSD9616_multiSend(cmd, 3);

    oledFullOff();

    // Normal display (not light background)
    OSD9616_send(0x00,0xa6);

    // Set Display Clock Divide Ratio / Clock Freq
    cmd[0] = 0x00;                      // signal command
    cmd[1] = 0xd5;                      // Set Divide Ratio / Freq
    cmd[2] = 0xf0;                      // Ratio=0, Oscillator Freq=maximum (reset=0x8)
    OSD9616_multiSend( cmd, 3);

    // Set pre-charge period
    cmd[0] = 0x00;                      // signal command
    cmd[1] = 0xd9;                      // Set Pre-Charge Period
    cmd[2] = 0x22;                      // 2 clock periods for both phase 1 (discharge) and 2 (charge)
//    OSD9616_multiSend( cmd, 3);

    // Set Vcomh Deselect Level
    cmd[0] = 0x00;                      // signal command
    cmd[1] = 0xdb;                      // Set Vcom Deselect Level
    cmd[2] = 0x30;                      // was 0x49, should be 0x30 for 0.83xVcc ???????????????????????????????????????????
//    OSD9616_multiSend( cmd, 3);

    // Charge Pump Setting
    cmd[0] = 0x00;                      // signal command
    cmd[1] = 0x8d;                      // Charge Pump Setting
    cmd[2] = 0x14;                      // Enable
    OSD9616_multiSend( cmd, 3);
    
    
    // ************ Other Parameters *****************

    // it seems as though the horizontal limit takes prescendence over the page mode and also
    // causes problems with horizontal scrolling, so it can be left at 127 since we don't need truncated horizontal limits

    oled_hv_address_range();            // set the horizontal and vertical mode column and page address range
    oled_scroll_off();                  // Turn off scrolling

    oled_page0();                       // set to page 0 in page mode

    oled_clear();                       // clear the display
    
    oledOn();                           // turn the display on

	return 0;
}

void oledOn()
{
    OSD9616_send(0x00,0xaf);            // Turn display on
}

void oledOff()
{
    OSD9616_send(0x00,0xae);            // Turn display off (sleep)
}

void oledFullOn()
{
    OSD9616_send(0x00,0xa5);            // Turn display on
}

void oledFullOff()
{
    OSD9616_send(0x00,0xa4);            // Turn display on
}

void oled_page_mode()
{
    Uint16 cmd[5];

    cmd[0]= 0x00;                       // control byte
    cmd[1]= 0x20;                       // set addressing mode
    cmd[2]= 0x10;                       // page
    OSD9616_multiSend(cmd,3);
}

Int16 oled_page0()
{
    // for page addressing mode only!
    oled_page_mode();

	// start at page 0 column(segment) 0
    OSD9616_send(0x00,0xb0);    // Set page to to 0 (bottom line)
    oled_page_column0();        // set the column to 0
    
    return 0;		
}

Int16 oled_page1()
{
    // for page addressing mode only!
    oled_page_mode();

	// start at page 1 column(segment) 0
    OSD9616_send(0x00,0xb1);            // Set page to 1 (top line)
    oled_page_column0();                // set the column to 0
    
    return 0;		
}

void oled_page_column0()
{
    // Set Column Start Address (page addressing mode only)
    OSD9616_send(0x00,0x00);            // Set column start address low nibble to 0
    OSD9616_send(0x00,0x10);            // Set column start address high nibble to 0
}

void oled_vertical_mode()
{
    Uint16 cmd[5];

    cmd[0]= 0x00;                       // control byte
    cmd[1]= 0x20;                       // set addressing mode
    cmd[2]= 0x01;                       // vertical
    OSD9616_multiSend(cmd,3);

    oled_hv_address_range();
}

void oled_horizontal_mode()
{
    Uint16 cmd[5];

    cmd[0]= 0x00;                       // control byte
    cmd[1]= 0x20;                       // set addressing mode
    cmd[2]= 0x00;                       // horizontal
    OSD9616_multiSend(cmd,3);

    oled_hv_address_range();
}

void oled_hv_address_range()
{
    Uint16 cmd[5];

    // Set Column Start and End Addresses (horizontal and vertical mode only)
    // set pointer to start
    cmd[0]= 0x00;                       // signal command mode
    cmd[1]= 0x21;                       // set the column start and end address
    cmd[2]= 0x00;                       // start at column 0
    cmd[3]= 0x7F;                       // end at column 95 (5F) or leave at 127 (7F)
    OSD9616_multiSend(cmd,4);

    // Set Page Start and End Addresses (horizontal and vertical mode only)
    // set pointer to start
    cmd[0]= 0x00;                       // signal command mode
    cmd[1]= 0x22;                       // set the page start and end address
    cmd[2]= 0x00;                       // start at page 0
    cmd[3]= 0x01;                       // end at page 1
    OSD9616_multiSend(cmd,4);
}

Int16 oled_clear()
{
    Uint16 i;
    static int endcol=128;

    /* clear page 0 */
    // 0x00 indicates command
    oled_page0();
    for (i=0; i<endcol; i++) {
        OSD9616_send(0x40,0x00);    // clear all pixels in columns 0 to 127
    }

    /* clear page 1 */
    oled_page1();
    for (i=0; i<endcol; i++) {
        OSD9616_send(0x40,0x00);    // clear all pixels in columns 0 to 127
    }

    return 0;
}

Int16   oled_hscroll()
{
    Uint16 cmd[10];

    // make sure off first
    oled_scroll_off();

    cmd[0] = 0x00;  // control byte - command
    cmd[1] = 0x27;  // Left Horizontal Scroll
    cmd[2] = 0x00;  // Dummy byte
    cmd[3] = 0x00;  // Define start page address
    cmd[4] = 0x02;  // Set time interval between each scroll step
    cmd[5] = 0x01;  // Define end page address
    cmd[6] = 0x00;  // Dummy
    cmd[7] = 0xFF;  // Dummy
    OSD9616_multiSend( cmd, 8);

    // turn on scrolling
    oled_scroll_on();

    return 0;
}

Int16   oled_hvscroll()
{
    Uint16 cmd[10];

    // make sure off to make adjustments
    oled_scroll_off();

    /* Set Vertical Scroll Area (default=all rows)
    cmd[0] = 0x00;  // control byte - command
    cmd[1] = 0xa3;  // Set Vertical Scroll Area
    cmd[2] = 0x08;  // Set No. of rows in top fixed area
    cmd[3] = 0x08;  // Set No. of rows in scroll area
    OSD9616_multiSend( cmd, 4 );
    */

    // Set vertical and horizontal scrolling
    cmd[0] = 0x00;  // control byte - command
    cmd[1] = 0x29;  // Vertical and Right Horizontal Scroll
    cmd[2] = 0x00;  // Dummy byte
    cmd[3] = 0x00;  // Define start page address
    cmd[4] = 0x02;  // Set time interval between each scroll step
    cmd[5] = 0x01;  // Define end page address
    cmd[6] = 0x01;  // Vertical scrolling offset
    OSD9616_multiSend( cmd, 7 );

    // turn on scrolling
    oled_scroll_on();

    return 0;
}

Int16 oled_scroll_off()
{
    OSD9616_send(0x00,0x2e);    // Turn off scrolling
    return 0;
}

Int16 oled_scroll_on()
{
    OSD9616_send(0x00,0x2f);    // Turn off scrolling
    return 0;
}


/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Int16 OSD9616_send( Uint16 regAddr, Uint16 regData)                     *
 *                                                                          *
 *      Sends 2 bytes of data to the OSD9616                                *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 OSD9616_send( Uint16 regAddr, Uint16 regData)
{
    // The first I2C byte written is the address plus read/write mode (DISPLAY_I2C_ADDR)
    // I2C Address: |0|0|1|1|1|1|0|R/W*| or 0x3C for write (8 bit) or 0x1D (7 bit + R/W)
    //
    // The second I2C byte written is the control byte (first byte in writeBuff)
    // I2C Data: |Co|D/C*|0|0|0|0|0|0| Co=continue bit, D/C* data or command bit
    //      0x40 Co=0, D/C*=1, data follows and continues until transmission finished
    //      0x00 Co=0, D/C*=0, command follows and contiues until transmission finished
    //
    // The third I2C byte is written is register data, either a command or data for the display as per previous byte
    //
    // All remaining bytes sent are the rest of a command or continuing data
    //

//    PSP_Result    status=1;
    Uint16  writeCount;
    Uint16  writeBuff[2];

    if(i2cStatus == CSL_SOK)
    {
        writeCount  =  2;

        // copy to write buffer and strip off high byte
        writeBuff[0] = (regAddr & 0x00FF);
        writeBuff[1] = (regData & 0x00FF);

        /* Write the data */
        //status = I2C_Write(hi2c, DISPLAY_I2C_ADDR, writeCount, writeBuff);
        i2cStatus = I2C_write(writeBuff, writeCount, DISPLAY_I2C_ADDR, TRUE, CSL_I2C_DEFAULT_STTSTP, CSL_I2C_MAX_TIMEOUT);
        i2c_delay(I2C_DELAY);
    }

    if (i2cStatus==CSL_SOK)
        return 0;
    else
        return 1;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Int16 OSD9616_multiSend(Uint16 *regData, Uint16 length)                 *
 *                                                                          *
 *      Sends multiple bytes of data to the OSD9616                         *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 OSD9616_multiSend( Uint16 *regData, Uint16 length)
{
//  PSP_Result    status=1;
    Uint16  i;
    Uint16  writeBuff[10];

    if(i2cStatus == CSL_SOK)
    {

        // copy to write buffer and strip off high byte
        for (i=0; i<length; i++) {
        	writeBuff[i] = (regData[i] & 0x00FF);
        }

        /* Write the data */
        //status = I2C_Write(hi2c, DISPLAY_I2C_ADDR, length, writeBuff);
        i2cStatus = I2C_write(writeBuff, length, DISPLAY_I2C_ADDR, TRUE, CSL_I2C_DEFAULT_STTSTP, CSL_I2C_MAX_TIMEOUT);    // TODO currently fails - (-200 or -7)timeout??!!!
        i2c_delay(I2C_DELAY);
    }

    if (i2cStatus==CSL_SOK)
        return 0;
    else
        return 1;
}


/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Int16 printchar(unsigned char a)              *
 *                                                                          *
 *      Send 6 bytes representing a Character                               *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 printchar(unsigned char a)
{
	Uint8 data;
	
	// uses character map from ROM which contains array of active LED bits for each char
	// 3 words per character or 6 bytes
	// characters are 6x8 bits (6 cols by 8 rows)
	// mapped (indexed) as per ascii table char no. multiplied by 3
	// 0x40 is data byte: R/W=0=write, D/C=1=data, followed by 000000
	// data is written to oled controller display memory and auto incremented

	data = ((CharGen_6x8[a*3])>>8) & 0x00FF;	// 1st high byte - col 1
    OSD9616_send(0x40,data);
    data = (CharGen_6x8[a*3]) & 0x00FF;			// 1st low byte - col 2
    OSD9616_send(0x40,data);
    data = ((CharGen_6x8[a*3+1])>>8) & 0x00FF;	// 2nd hight byte - col 3 
    OSD9616_send(0x40,data);			
    data = (CharGen_6x8[a*3+1]) & 0x00FF;		// 2nd low byte - col 4
    OSD9616_send(0x40,data);
    data = ((CharGen_6x8[a*3+2])>>8) & 0x00FF;	// 3rd high byte - col 5
    OSD9616_send(0x40,data);
    data = (CharGen_6x8[a*3+2]) & 0x00FF;		// 3rd low byte - col 6
    OSD9616_send(0x40,data);

    return 0;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Int16 printstring(char *a)              *
 *                                                                          *
 *      print a string                               *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 printstring(char *string)
{
	Uint16 i, len;
	
	len = strlen(string);
	if (len>16) len=16;			// limit to 16 characters
	if (len==0) return (0);		// return if none
	
	for (i=0; i<len; i++) {
		printchar(string[i]);
	}
	return 0;	
}	


/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Int16 print_two(s1, s2)                                                 *
 *                                                                          *
 *      Print s1				                            				*
 *            s2                                                            *
 * ------------------------------------------------------------------------ */
Int16 print_two(char *topline, char *bottomline)
{
    oled_page_mode();
	oled_scroll_off(); // Turn off scrolling
	oled_clear();
	
	/* Write to page 1 */
	oled_page1();
	printstring(topline);
	
	/* Write to page 0*/ 
	oled_page0();
	printstring(bottomline);
	
	return 0;
}


/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Int16 print_charROM()                                                       *
 *                                                                          *
 *      Print "Industry lowest"                            				*
 *            "power dsp C5515"                                                             *
 * ------------------------------------------------------------------------ */
Int16 print_charROM()
{
	char crom[16]; 
	Uint16 i,j;
	
	oled_page_mode();
	oled_scroll_off(); // Turn off scrolling
	oled_clear();
	
	for (j=0; j<8; j++) {
		
	    /* Write to page 1 */
		oled_page1();
   		for (i=0; i<16; i++) {
   			crom[i] = i + j*32;
   		}
		printstring(crom);
	 
	    /* Write to page 0*/ 
    	oled_page0();
   		for (i=0; i<16; i++) {
   			crom[i] = i + 16 + j*32;
   		}	
		printstring(crom);
	}
	return 0;
}


/*
 * Name: amplitude2bargraph(int level)
 *
 * Description: takes an input value and converts it to a binary pixel representation
 *
 * level:  level to be converted to a bar display
 *
 * RETURNS: Bargraph pixel value between 0 and FFFFh.
 *
 */

Uint16 amplitude2bargraph(Int32 level, Int32 min, Int32 max)
{
    Int32 range = max - min;
    Int32 step = range / 16;

    if (level > max) {
        return (0xFFFF);                // All LEDs on
    } else if (level > min + step * 16) {
        return (0x7FFF);
    } else if (level > min + step * 15) {
        return (0x3FFF);
    } else if (level > min + step * 14) {
        return (0x1FFF);
    } else if (level > min + step * 13) {
        return (0x0FFF);
    } else if (level > min + step * 12) {
        return (0x07FF);
    } else if (level > min + step * 11) {
        return (0x03FF);
    } else if (level > min + step * 10) {
        return (0x01FF);
    } else if (level > min + step * 9) {
        return (0x00FF);
    } else if (level > min + step * 8) {
        return (0x007F);
    } else if (level > min + step * 7) {
        return (0x003F);
    } else if (level > min + step * 6) {
        return (0x001F);
    } else if (level > min + step * 5) {
        return (0x000F);
    } else if (level > min + step * 4) {
        return (0x0007);
    } else if (level > min + step * 3) {
        return (0x0003);
    } else if (level > min + step * 2) {
        return (0x0001);
    } else if (level > min + step * 1) {
        return (0x0000);
    } else {
        return (0);                     // no LEDs on
    }
}


/*****************************************************************************/
/* oled_display_bargraph()                                                   */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* levels:  Pointer to array containing 96 16-bit input values.         */
/*                                                                           */
/* RETURNS: Nothing                                                          */
/*                                                                           */
/*****************************************************************************/
void display_bargraph (Uint16 *levels, Int16 size, Int32 min, Int32 max)
{
    int i,i2;
    Uint16 bar;
    Uint16 pixels;

//#define TESTPATTERN
#ifdef TESTPATTERN
    size=96;
    min=0;
    max=96;

    Int32 testarray[96]={
            25,25,25,25,25,25,25,25,25,25,25,75,25,25,25,25,25,25,25,25,25,25,25,25,    // column  1 to 24
            25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,    // column 25 to 48
            25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,    // column 49 to 72
            25,25,25,25,25,25,25,25,25,25,25,75,25,25,25,25,25,25,25,25,25,25,25,25     // column 73 to 96
    };

    levels=testarray;
#endif

    oled_scroll_off();
    oled_vertical_mode();
	
    // send 96 vertical lines to bar graph bottom and top

    for (i=0;i<=95;i++)
    {
        if (i<48)
            i2=i+48;        // plot last half of graph (upper frequecies)
        else
            i2=i-48;        // plot lower half of graph (lower frequencies)

        bar = amplitude2bargraph(levels[(int)((i2*size)/96)], min, max);    // repeat values if size is smaller than 96 and skip if greater
     
        pixels = (bar & 0xFF);       // Extract low byte
        OSD9616_send(0x40,pixels);   // send bottom half (page 0)
       
        pixels = ((bar>>8) & 0xFF );        // Extract high byte
        OSD9616_send(0x40,pixels);   // send top half (page 1)
    }

	//return 0;
}


/***************************************************************
 * print_SEI_message()
 *
 * Desctiption:
 * -print the SEI startup message
 ***************************************************************/

Int16 print_SEI_message()
{
    char message[23];       // 16 character maximum for buffer

    oled_scroll_off();      // Turn off scrolling
    oled_clear();           // clear oled

    // clear message buffer
    int count;
    for (count=0;count<23;++count)
        message[count]=0x20;

    // write top line (page 1)
    oled_page1();                           // set page mode on page 1 seg 0
    sprintf(message,"Ares V0.1       ");
    printstring(message);

    // write bottom line (page 0)
    oled_page0();                           // set page mode on page 0 seg 0
    sprintf(message,"Sigma Eight Inc.");
    printstring(message);

//  oled_hscroll(); // Turn on scrolling

    return 0;
}


