/*
*******************************************************************************
*  File name:        FS_RamTst.c
*  Copyright:        Neusoft Corporation. All rights reserved.
*  Notes:            FS RamTst
*  History:
*  Revision        Data        Name              Comment
*  ----------------------------------------------------------------------------
*  1.0            2022.04.02   SuiGuoqing     New Create  *
*******************************************************************************
*/

/*
*******************************************************************************
*  Includes
*******************************************************************************
*/
#include "FS_RamTst.h"

/*
*******************************************************************************
*  Global variable
*******************************************************************************
*/

/*
*******************************************************************************
*  Static variable
*******************************************************************************
*/
static uint32_t gU32RamTstErrInfo = 0U;

/* ESM MCU Interrupt ID  RamTstģĶӦϵ */
static const FS_RamTst_INTRPTIdToMod_st stArrMcuINTRPTIdToMod[FSRAMTST_MCU_INTRPTID_CNT] = {
    {FS_MCU_ADC0_ECC_CORRECTED_ERR_LEVEL_0  ,	SDL_ECC_MEMTYPE_MCU_ADC0},
    {FS_MCU_ADC0_ECC_UNCORRECTED_ERR_LEVEL_0,	SDL_ECC_MEMTYPE_MCU_ADC0},
    {FS_MCU_ADC1_ECC_CORRECTED_ERR_LEVEL_0  ,	SDL_ECC_MEMTYPE_MCU_ADC1},
    {FS_MCU_ADC1_ECC_UNCORRECTED_ERR_LEVEL_0,	SDL_ECC_MEMTYPE_MCU_ADC1},
    {FS_MCU_R5FSS0_CORE0_ECC_CORRECTED_LEVEL_0  ,	SDL_ECC_MEMTYPE_MCU_R5F0_CORE},
    {FS_MCU_R5FSS0_CORE0_ECC_UNCORRECTED_LEVEL_0,	SDL_ECC_MEMTYPE_MCU_R5F0_CORE},
    {FS_MCU_R5FSS0_CORE1_ECC_CORRECTED_LEVEL_0  ,	SDL_ECC_MEMTYPE_MCU_R5F1_CORE},
    {FS_MCU_R5FSS0_CORE1_ECC_UNCORRECTED_LEVEL_0,	SDL_ECC_MEMTYPE_MCU_R5F1_CORE},
    {FS_MCU_NAVSS0_UDMASS_ECC_AGGR_0_ECC_CORRECTED_ERR_LEVEL_0  ,	SDL_MCU_NAVSS0_UDMASS_ECC_AGGR0},
    {FS_MCU_NAVSS0_UDMASS_ECC_AGGR_0_ECC_UNCORRECTED_ERR_LEVEL_0,	SDL_MCU_NAVSS0_UDMASS_ECC_AGGR0},
    {FS_MCU_ECC_AGGR0_CORR_LEVEL_0  ,	SDL_ECC_MEMTYPE_MCU_CBASS_ECC_AGGR0},
    {FS_MCU_ECC_AGGR0_UNCORR_LEVEL_0,	SDL_ECC_MEMTYPE_MCU_CBASS_ECC_AGGR0},
};

/* ESM WKUP Interrupt ID  RamTstģĶӦϵ */
static const FS_RamTst_INTRPTIdToMod_st stArrWkupINTRPTIdToMod[FSRAMTST_WKUP_INTRPTID_CNT] = {
    {FS_WKUP_VTM0_CORR_LEVEL_0  ,	SDL_WKUP_VTM0_K3VTM_NC_ECCAGGR},
    {FS_WKUP_VTM0_UNCORR_LEVEL_0,	SDL_WKUP_VTM0_K3VTM_NC_ECCAGGR},
};

/*
*******************************************************************************
*  Prototype
*******************************************************************************
*/

/*
*******************************************************************************
*  Function name:        fsRamTst_RdEccReg
*  Explanation:          read Ecc Register
*  Timming:              
*  Inputs:               const InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              uint32_t *pU32RdValue
*  Returns:              uint8_t (Ƿȡɹ 1ɹ0ʧ)
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_RdEccReg(const InfoOfEccReg_st *pStInfoOfEccReg, 
                             uint32_t *pU32RdValue)
{
    uint32_t u32RegVal = 0U;
    uint8_t bNoErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegEccVectorAddr = pStInfoOfEccReg->u32EccRegBaseAddr + OADDR_ECC_VECTOR;
    uint32_t u32RegEccAddr = pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr;
    
    u32RegVal = HW_RD_REG32(u32RegEccVectorAddr);
    
    /* 1) ECCIDţECC_VECTOR.ECC_VECTOR = [ECCID] */
    HW_SET_FIELD32(u32RegVal, ECC_VECTOR_ECC_VECTOR, pStInfoOfEccReg->u32EccId);

    /* 2) ѡҪECCĴַECC_VECTOR.RD_SVBUS_ADDRESS = [ECCĴַ] */
    HW_SET_FIELD32(u32RegVal, ECC_VECTOR_RD_SVBUS_ADDRESS, pStInfoOfEccReg->u32EccRegOffsetAddr);

    /* 3) ECCĴĲECC_VECTOR.RD_SVBUS = 1 */
    HW_SET_FIELD32(u32RegVal, ECC_VECTOR_RD_SVBUS, FSRAMTST_VALUE_RD_SVBUS_SETRD);

    HW_WR_REG32(u32RegEccVectorAddr, u32RegVal);

    /* 4) ȴɣȴECC_VECTOR.RD_SVBUS_DONE = 1 */
    uint32_t u32Cnt = FSRAMTST_WAIT_CNT;
    uint32_t u32Ret = 0U;
    while( u32Cnt > 0U ) 
    {
        u32Ret = HW_RD_FIELD32(u32RegEccVectorAddr, ECC_VECTOR_RD_SVBUS_DONE);
        if( u32Ret == FSRAMTST_VALUE_RD_SVBUS_DONE_RDDONE )
        {
            break;
        }
        else
        {
            u32Cnt--;
        }
    }
    if( u32Cnt > 0U )
    {
        /* 5) ӵڶеļĴжȡ */
        *pU32RdValue = HW_RD_REG32(u32RegEccAddr);
    }
    else
    {
        bNoErr = FSRAMTST_ERROR;
        gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_INTERFACE_ERR;    /* ¼ʱ */
    }

    return bNoErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_WrEccReg
*  Explanation:          write Ecc Register
*  Timming:              
*  Inputs:               const InfoOfEccReg_st *pStInfoOfEccReg
*                        uint32_t u32WrValue
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_WrEccReg(const InfoOfEccReg_st *pStInfoOfEccReg, 
                     uint32_t u32WrValue)
{
    uint32_t u32RegVal = 0U;
    uint32_t u32RegEccVectorAddr = pStInfoOfEccReg->u32EccRegBaseAddr + OADDR_ECC_VECTOR;
    uint32_t u32RegEccAddr = pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr;

    u32RegVal = HW_RD_REG32(u32RegEccVectorAddr);

    /* 1) ECCIDţECC_VECTOR.ECC_VECTOR = [ECCID] */
    HW_SET_FIELD32(u32RegVal, ECC_VECTOR_ECC_VECTOR, pStInfoOfEccReg->u32EccId);
    
    /* 2) ECCдĴĲECC_VECTOR.RD_SVBUS = 0 */
    HW_SET_FIELD32(u32RegVal, ECC_VECTOR_RD_SVBUS, FSRAMTST_VALUE_RD_SVBUS_SETWR);

    HW_WR_REG32(u32RegEccVectorAddr, u32RegVal);

    /* 3) ĿĴдֵ */
    HW_WR_REG32(u32RegEccAddr, u32WrValue);

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ChoseRegN 
*  Explanation:          ECC IDҪļĴ
*  Timming:
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        RegN_en enRegN
*  Outputs:              InfoOfEccReg_st *pStInfoOfEccReg
*  Returns:              void
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_ChoseRegN(InfoOfEccReg_st *pStInfoOfEccReg, 
                               RegN_en enRegN)
{
    uint32_t n = (uint32_t)pStInfoOfEccReg->u32EccId >> (uint32_t)5U;
    
    switch (enRegN)
    {
    case ECC_SEC_ENABLE_SET_REG_N:
            pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_SEC_ENABLE_SET_REG0 + (n * (uint32_t)4U);
        break;
    case ECC_DED_ENABLE_SET_REG_N:
            pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_DED_ENABLE_SET_REG0 + (n * (uint32_t)4U);;
        break;    
    case ECC_SEC_STATUS_REG_N:
            pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_SEC_STATUS_REG0 + (n * (uint32_t)4U);;
        break;
    case ECC_DED_STATUS_REG_N:
            pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_DED_STATUS_REG0 + (n * (uint32_t)4U);;
        break;
    default:
        break;
    }

    pStInfoOfEccReg->u32FeildShift = (uint32_t)pStInfoOfEccReg->u32EccId & (uint32_t)0x1FU;
    pStInfoOfEccReg->u32FeildMask = (uint32_t)1U << pStInfoOfEccReg->u32FeildShift;

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ErrorReport 
*  Explanation:          RAM error report
*  Timming:
*  Inputs:               void
*  Outputs:              void
*  Returns:              void
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_ErrorReport(void)
{
    if(gU32RamTstErrInfo != 0U)
    {
        /* <TBD> */
        (void)Rte_Write_RpDriver1_ErrDetect_RAM_0(gU32RamTstErrInfo);

        /* ϣԼϼPBISTϲ */
        gU32RamTstErrInfo &= ( (uint32_t)FSRAMTST_ECC_CORR_SELFTST_ERR 
                             | (uint32_t)FSRAMTST_ECC_UNCORR_SELFTST_ERR 
                             | (uint32_t)FSRAMTST_PBSIT_HW_POST_FAILED 
                             | (uint32_t)FSRAMTST_PBSIT_MOD_SELFTST_FAILED 
                             | (uint32_t)FSRAMTST_PBSIT_MOD_FAILED );
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_SearchMod 
*  Explanation:          Interrupt IDģ
*  Timming:
*  Inputs:               const FS_RamTst_INTRPTIdToMod_st* pStArrINTRPTIdToMod
*                        uint8_t u8ArrLen
*                        uint32_t u32EsmINTRPTId
*  Outputs:              SDL_ECC_MemType* pEccMemType
*  Returns:              uint8_t
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_SearchMod(const FS_RamTst_INTRPTIdToMod_st* pStArrINTRPTIdToMod, 
                                  uint8_t u8ArrLen, 
                                  uint32_t u32EsmINTRPTId,
                                  SDL_ECC_MemType* pEccMemType)
{
    uint8_t u8SearchResult = FSRAMTST_ERROR;
    uint8_t length = u8ArrLen;
    uint8_t first = 0U;
    uint8_t last = length - 1U;
    uint8_t middle = (first + last) / 2U;

    while (first <= last) 
    {
        if (pStArrINTRPTIdToMod[middle].u32INTRPTId < u32EsmINTRPTId)
        {
            first = middle + 1U;
        } 
        else if (pStArrINTRPTIdToMod[middle].u32INTRPTId == u32EsmINTRPTId) 
        {
            *pEccMemType = (uint8_t)pStArrINTRPTIdToMod[middle].eccMemType;
            u8SearchResult = FSRAMTST_NO_ERROR;
            break;
        } 
        else 
        {
            last = middle - 1U;
        }

        middle = (first + last) / 2U;
    }

    return u8SearchResult;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_getMemType 
*  Explanation:          ȡmemTypeеλ
*  Timming:
*  Inputs:               FS_ESM_InstanceType enInstType
*                        uint32_t u32EsmINTRPTId
*  Outputs:              SDL_ECC_MemType* pEccMemType
*  Returns:              uint8_t
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_getMemType(FS_ESM_InstanceType enInstType, 
                                      uint32_t u32EsmINTRPTId, 
                                      SDL_ECC_MemType* pEccMemType)
{
    uint8_t u8GetErr = FSRAMTST_NO_ERROR;

    switch (enInstType)
    {
    case FS_ESM_WKUP_ESM0:
        u8GetErr = fsRamTst_SearchMod(stArrWkupINTRPTIdToMod, FSRAMTST_WKUP_INTRPTID_CNT, 
                                      u32EsmINTRPTId, pEccMemType);
        break;
    
    case FS_ESM_MCU_ESM0:
        u8GetErr = fsRamTst_SearchMod(stArrMcuINTRPTIdToMod, FSRAMTST_MCU_INTRPTID_CNT, 
                                      u32EsmINTRPTId, pEccMemType);
        break;

    default:
        u8GetErr = FSRAMTST_ERROR;
        break;
    }
    
    return u8GetErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_IsToDoParity
*  Explanation:          жǷִżУز
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*  Outputs:              void
*  Returns:              uint8_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_IsToDoParity(SDL_ECC_MemType eccMemType)
{
    uint8_t u8IsToDo = FSRAMTST_TO_DO;

    if( (eccMemType == SDL_ECC_MEMTYPE_MCU_ADC0)
        || (eccMemType == SDL_ECC_MEMTYPE_MCU_ADC1) )
    {
        u8IsToDo = FSRAMTST_NOT_TO_DO;
    }

    return u8IsToDo;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_Init_DoWrapper
*  Explanation:          RAM init
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*                        InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_Init_DoWrapper(SDL_ECC_MemType eccMemType, 
                                    InfoOfEccReg_st *pStInfoOfEccReg)
{
    uint32_t u32RegVal = 0U;
    uint32_t u8RdErr = FSRAMTST_NO_ERROR;
    uint8_t u8IsDoParity = FSRAMTST_TO_DO;

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CTRL;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        /* ʹżУ飬ECC_CTRL.CHECK_PARITY = 1 */
        /* ʹECCӿڷʳʱƣECC_CTRL.CHECK_SVBUS_TIMEOUT = 1 */
        u8IsDoParity = fsRamTst_IsToDoParity(eccMemType);
        if( u8IsDoParity == FSRAMTST_TO_DO )
        {
            HW_SET_FIELD32(u32RegVal, ECC_CTRL_CHECK_PARITY, FSRAMTST_VALUE_CHECK_PARITY_EN);
            HW_SET_FIELD32(u32RegVal, ECC_CTRL_CHECK_SVBUS_TIMEOUT, FSRAMTST_VALUE_CHECK_SVBUS_TIMEOUT_EN);
        }

        /* ʹECCĶ-޸-дƣECC_CTRL.ENABLE_RMW = 1 */
        HW_SET_FIELD32(u32RegVal, ECC_CTRL_ENABLE_RMW, FSRAMTST_VALUE_ENABLE_RMW_EN);

        /* ʹECCܣECC_CTRL.ECC_CHECK = 1ECC_CTRL.ECC_ENABLE = 1 */
        HW_SET_FIELD32(u32RegVal, ECC_CTRL_ECC_CHECK, FSRAMTST_VALUE_ECC_CHECK_EN);
        HW_SET_FIELD32(u32RegVal, ECC_CTRL_ECC_ENABLE, FSRAMTST_VALUE_ECC_ENABLE_EN);

        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_Init_DoInterconnect
*  Explanation:          RAM init
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_Init_DoInterconnect(InfoOfEccReg_st *pStInfoOfEccReg)
{
    uint32_t u32RegVal = 0U;
    uint32_t u8RdErr = FSRAMTST_NO_ERROR;

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_CTRL;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        /* ʹECCܣECC_CBASS_CTRL.ECC_CHECK = 1 */
        HW_SET_FIELD32(u32RegVal, ECC_CBASS_CTRL_ECC_CHECK, FSRAMTST_VALUE_ECC_CHECK_EN);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_Init_EnIntrpt1
*  Explanation:          RAM init
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_Init_EnIntrpt1(InfoOfEccReg_st *pStInfoOfEccReg)
{
    /* ʹECC CorrectableжϣECC_SEC_ENABLE_SET_REG0ӦλΪ1 */
    fsRamTst_ChoseRegN(pStInfoOfEccReg, ECC_SEC_ENABLE_SET_REG_N);
    HW_WR_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                      pStInfoOfEccReg->u32FeildMask, 
                      pStInfoOfEccReg->u32FeildShift, 
                      1U);
    
    /* ʹECC UncorrectableжϣECC_DED_ENABLE_SET_REG0ӦλΪ1 */
    fsRamTst_ChoseRegN(pStInfoOfEccReg, ECC_DED_ENABLE_SET_REG_N);
    HW_WR_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                      pStInfoOfEccReg->u32FeildMask, 
                      pStInfoOfEccReg->u32FeildShift, 
                      1U);

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_Init_EnIntrpt2
*  Explanation:          RAM init
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*                        InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_Init_EnIntrpt2(SDL_ECC_MemType eccMemType, 
                                    InfoOfEccReg_st *pStInfoOfEccReg)
{
    uint32_t u32RegVal = 0U;
    uint8_t u8IsDoParity = FSRAMTST_TO_DO;
    
    u8IsDoParity = fsRamTst_IsToDoParity(eccMemType);
    if( u8IsDoParity == FSRAMTST_TO_DO )
    {
        /* ʹECCӿڷʳʱжϣECC_AGGR_ENABLE_SET.TIMEOUT = 1  */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_AGGR_ENABLE_SET;
        u32RegVal = HW_RD_REG32(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr);
        HW_SET_FIELD32(u32RegVal, ECC_AGGR_ENABLE_SET_TIMEOUT, FSRAMTST_VALUE_TIMEOUT_EN);

        /* ʹECCżУжϣECC_AGGR_ENABLE_SET.PARITY = 1 */
        HW_SET_FIELD32(u32RegVal, ECC_AGGR_ENABLE_SET_PARITY, FSRAMTST_VALUE_PARITY_EN);
        HW_WR_REG32(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, u32RegVal);
    }    
    
    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_Init
*  Explanation:          RAM init
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_Init(SDL_ECC_MemType eccMemType)
{
    InfoOfEccReg_st stInfoOfEccReg = {0U};
    uint32_t u32NumRams = SDL_ECC_getNumRams(eccMemType);
    stInfoOfEccReg.u32EccRegBaseAddr = SDL_ECC_getAggrBaseAddr(eccMemType);

    SDL_ECC_MemSubType memSubType = 0U;
    uint32_t u32RamIdType = 0U;
    for(memSubType = 0U; memSubType < u32NumRams; ++memSubType)
    {
        stInfoOfEccReg.u32EccId = SDL_ECC_getRamId(eccMemType, memSubType);
        u32RamIdType = SDL_ECC_getRamIdType(eccMemType, memSubType);
        if(u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER)
        {
            fsRamTst_Init_DoWrapper(eccMemType, &stInfoOfEccReg);
        }
        else    /* do interconnect */
        {
            fsRamTst_Init_DoInterconnect(&stInfoOfEccReg);
        }

        fsRamTst_Init_EnIntrpt1(&stInfoOfEccReg);
    }
    
    fsRamTst_Init_EnIntrpt2(eccMemType, &stInfoOfEccReg);

    return ;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_Init
*  Explanation:          RAM init
*  Timming:              
*  Inputs:               void
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
void FS_RamTst_Init(void)
{
    SDL_ECC_MemType eccMemType;
    for(eccMemType = 0U; eccMemType < SDL_ECC_MEMTYPE_MAX; ++eccMemType)
    {
        fsRamTst_Init(eccMemType);
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccCorrISR_CheckErrPend
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              uint32_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint32_t fsRamTst_EccCorrISR_CheckErrPend(InfoOfEccReg_st *pStInfoOfEccReg)
{
    uint32_t u32CorrErr = FSRAMTST_NO_ERROR;

    /* ǷCorrectable󣬶ȡĴECC_SEC_STATUS_REG0ݣǷӦλ1 */
    fsRamTst_ChoseRegN(pStInfoOfEccReg, ECC_SEC_STATUS_REG_N);
    u32CorrErr = HW_RD_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                                   pStInfoOfEccReg->u32FeildMask, 
                                   pStInfoOfEccReg->u32FeildShift);
    return u32CorrErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccISR_RecordErrAddr
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*                        SDL_ECC_MemSubType memSubType
*  Outputs:              void
*  Returns:              uint8_t
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_EccISR_RecordErrAddr(InfoOfEccReg_st *pStInfoOfEccReg, 
                                                 SDL_ECC_MemType eccMemType, 
                                                 SDL_ECC_MemSubType memSubType)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;
    SDL_MemConfig_t memConfig;

    /* ¼Ϸĵַ[Ecc Error Address] = ECC_ERR_STAT2.ECC_ROW */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_STAT2;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    if( u8RdErr ==  FSRAMTST_NO_ERROR )
    {
        /* ַ */
        SDL_ECC_getMemConfig(eccMemType, memSubType, &memConfig);
        gU32RamTstErrAddr = pStInfoOfEccReg->u32EccRegBaseAddr + (u32RegVal * memConfig.rowSize);
    }

    return u8RdErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccCorrISR_ConfirmErr_Wrapper
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*  Outputs:              uint8_t* pU8CorrErrPend
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccCorrISR_ConfirmErr_Wrapper(InfoOfEccReg_st *pStInfoOfEccReg, 
                                              SDL_ECC_MemType eccMemType, 
                                              uint8_t* pU8CorrErrPend)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;
    uint32_t u32CorrErrCnt = 0U;

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_STAT1;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    /* ȡCorrectableϷECC_ERR_STAT1.ECC_SEC = 3
        1¼[RAMTst_ECCErrorSts] = [ModuleX ECC ID Correctable Error]
        2ϢECC_ERR_STAT1.CLR_ECC_SEC = 3 */
    if( (u8RdErr ==  FSRAMTST_NO_ERROR)
       && ((eccMemType == SDL_ECC_MEMTYPE_MCU_ADC0)
            || (eccMemType == SDL_ECC_MEMTYPE_MCU_ADC1)) )
    {
        u32CorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_SEC_ADC);
        if(u32CorrErrCnt == FSRAMTST_VALUE_ECC_SEC_CONFIRM_ERR_ADC)
        {
            *pU8CorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_CORR_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_SEC_ADC, FSRAMTST_VALUE_CLR_ECC_SEC_CLR_ERR_ADC);
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }
    }
    else if(u8RdErr ==  FSRAMTST_NO_ERROR) /* ADCģ */
    {
        u32CorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_SEC);
        if(u32CorrErrCnt == FSRAMTST_VALUE_ECC_SEC_CONFIRM_ERR)
        {
            *pU8CorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_CORR_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_SEC, FSRAMTST_VALUE_CLR_ECC_SEC_CLR_ERR);
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }
    }
    else
    {
        /* do nothing */
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccCorrISR_ConfirmErr_Interconnect
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              uint8_t* pU8CorrErrPend
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccCorrISR_ConfirmErr_Interconnect(InfoOfEccReg_st *pStInfoOfEccReg, 
                                              uint8_t* pU8CorrErrPend)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;
    uint32_t u32CorrErrCnt = 0U;

    /* ECC_CBASS_ERR_STAT1.COR_PEND = 1
        1¼[RAMTst_ECCErrorSts] = [ECC ID Correctable Error]
        2ϢECC_CBASS_ERR_STAT1.COR_PEND_CLR = 1 */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_ERR_STAT1;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    if( u8RdErr ==  FSRAMTST_NO_ERROR )
    {
        u32CorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_CBASS_ERR_STAT1_COR_PEND);
        if( u32CorrErrCnt == FSRAMTST_VALUE_COR_PEND )
        {
            *pU8CorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_CORR_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_STAT1_COR_PEND_CLR, FSRAMTST_VALUE_COR_PEND_CLR);
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccCorrISR_EndIntrpt
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccCorrISR_EndIntrpt(InfoOfEccReg_st *pStInfoOfEccReg)
{
    /* жϷECC_SEC_EOI_REG.EOI_WR = 1 */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_SEC_EOI_REG;
    HW_WR_FIELD32(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                  ECC_SEC_EOI_REG_EOI_WR, 
                  FSRAMTST_VALUE_SEC_EOI_WR_END_INT);

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccCorrISR
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccCorrISR(SDL_ECC_MemType eccMemType)
{
    uint32_t u32CorrErr = FSRAMTST_NO_ERROR;
    uint8_t u8CorrErrPend = FSRAMTST_NO_ERROR;
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    InfoOfEccReg_st stInfoOfEccReg = {0U};
    stInfoOfEccReg.u32EccRegBaseAddr = SDL_ECC_getAggrBaseAddr(eccMemType);
    uint32_t u32NumRams = SDL_ECC_getNumRams(eccMemType);
    SDL_ECC_MemSubType memSubType = 0U;
    uint32_t u32RamIdType = 0U;

    for(memSubType = 0U; memSubType < u32NumRams; ++memSubType)
    {
        stInfoOfEccReg.u32EccId = SDL_ECC_getRamId(eccMemType, memSubType);
        u32RamIdType = SDL_ECC_getRamIdType(eccMemType, memSubType);

        u32CorrErr = fsRamTst_EccCorrISR_CheckErrPend(&stInfoOfEccReg);

        if( (u32CorrErr == FSRAMTST_ERROR)
           && (u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER) )
        {
            u8RdErr = fsRamTst_EccISR_RecordErrAddr(&stInfoOfEccReg, eccMemType, memSubType);

            if( u8RdErr ==  FSRAMTST_NO_ERROR )
            {
                fsRamTst_EccCorrISR_ConfirmErr_Wrapper(&stInfoOfEccReg, eccMemType, &u8CorrErrPend);
            }
        }
        else if(u32CorrErr == FSRAMTST_ERROR)  /* INTERCONNECT */
        {
            fsRamTst_EccCorrISR_ConfirmErr_Interconnect(&stInfoOfEccReg, &u8CorrErrPend);
        }
        else
        {
            /* do nothing */
        }

        if( u8CorrErrPend ==  FSRAMTST_ERROR )
        {
            break;
        }
    }
        
    fsRamTst_EccCorrISR_EndIntrpt(&stInfoOfEccReg);

    return ;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_ECCCorrISR
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               FS_ESM_InstanceType enInstType
*                        uint32_t u32EsmINTRPTId
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
void FS_RamTst_ECCCorrISR(FS_ESM_InstanceType enInstType, uint32_t u32EsmINTRPTId)
{
    SDL_ECC_MemType eccMemType = 0U;
    uint8_t u8GetErr = FSRAMTST_NO_ERROR;
    u8GetErr = fsRamTst_getMemType(enInstType, u32EsmINTRPTId, &eccMemType);

    if(u8GetErr == FSRAMTST_NO_ERROR)
    {
        fsRamTst_EccCorrISR(eccMemType);
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccUncorrISR_CheckErrPend
*  Explanation:          check Uncorrectable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              uint32_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint32_t fsRamTst_EccUncorrISR_CheckErrPend(InfoOfEccReg_st *pStInfoOfEccReg)
{
    uint32_t u32UncorrErr = FSRAMTST_NO_ERROR;

    /* ǷUncorrectable󣬶ȡĴECC_DED_STATUS_REG0ݣǷӦλ1 */
    fsRamTst_ChoseRegN(pStInfoOfEccReg, ECC_DED_STATUS_REG_N);
    u32UncorrErr = HW_RD_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                                     pStInfoOfEccReg->u32FeildMask, 
                                     pStInfoOfEccReg->u32FeildShift);
    
    return u32UncorrErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccUncorrISR_ConfirmErr_Wrapper
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*  Outputs:              uint8_t* pU8UncorrErrPend
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccUncorrISR_ConfirmErr_Wrapper(InfoOfEccReg_st *pStInfoOfEccReg, 
                                              SDL_ECC_MemType eccMemType, 
                                              uint8_t* pU8UncorrErrPend)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;
    uint32_t u32UncorrErrCnt = 0U;
    uint8_t u8IsDoParity = fsRamTst_IsToDoParity(eccMemType);

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_STAT1;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    /* ȡUncorrectableϷECC_ERR_STAT1.ECC_DED = 13
        1¼[RAMTst_ECCErrorSts] = [ModuleX ECC ID Uncorrectable Error]
        2ϢECC_ERR_STAT1.CLR_ECC_DED = 13 
        ע⣺ĴECC_ERR_STAT1.ECC_DEDֵ13ȡھʹõģ */
    if( (u8RdErr ==  FSRAMTST_NO_ERROR)
       && ((eccMemType == SDL_ECC_MEMTYPE_MCU_ADC0)
           || (eccMemType == SDL_ECC_MEMTYPE_MCU_ADC1)) )
    {
        u32UncorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_DED_ADC);
        if(u32UncorrErrCnt == FSRAMTST_VALUE_ECC_DED_CONFIRM_ERR_ADC)
        {
            *pU8UncorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_UNCORR_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_DED_ADC, FSRAMTST_VALUE_CLR_ECC_DED_CLR_ERR_ADC);
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }
    }
    else if(u8RdErr ==  FSRAMTST_NO_ERROR) /* ADCģ */
    {
        u32UncorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_DED);
        if(u32UncorrErrCnt == FSRAMTST_VALUE_ECC_DED_CONFIRM_ERR)
        {
            *pU8UncorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_UNCORR_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_DED, FSRAMTST_VALUE_CLR_ECC_DED_CLR_ERR);
        }
    }
    else
    {
        /* do nothing */
    }

    if( (u8RdErr ==  FSRAMTST_NO_ERROR)
       && (u8IsDoParity == FSRAMTST_TO_DO) )
    {
        /* ȡżУϷECC_ERR_STAT1.PARITY_ERR = 13
        1¼[RAMTst_ECCErrorSts] = [ModuleX ECC ID Parity Error]
        2ϢECC_ERR_STAT1.CLR_PARITY_ERR = 13
        ע⣺ĴECC_ERR_STAT1.PARITY_ERRֵ13ȡھʹõģ */
        u32UncorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_PARITY_ERR);
        if(u32UncorrErrCnt == FSRAMTST_VALUE_PARITY_ERR_CONFIRM_ERR)
        {
            *pU8UncorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_PARITY_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_PARITY_ERR, FSRAMTST_VALUE_CLR_PARITY_ERR_CLR_ERR);
        }

        if(*pU8UncorrErrPend == FSRAMTST_ERROR)
        {
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }

        /* ȡECCӿڷʴECC_ERR_STAT3.SVBUS_TIMEOUT_ERR = 1
            1¼[RAMTst_ECCErrorSts] = [ModuleX ECC ID Interface Error]
            2ϢECC_ERR_STAT1.CLR_SVBUS_TIMEOUT_ERR = 1 */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_STAT3;
        u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
        u32UncorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT3_SVBUS_TIMEOUT_ERR);

        if( (u8RdErr ==  FSRAMTST_NO_ERROR) 
           && (u32UncorrErrCnt == FSRAMTST_VALUE_SVBUS_TIMEOUT_CONFIRM_ERR) )
        {
            *pU8UncorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_INTERFACE_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT3_CLR_SVBUS_TIMEOUT_ERR, FSRAMTST_VALUE_SVBUS_TIMEOUT_ERR_CLR_ERR);
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccUncorrISR_ConfirmErr_Interconnect
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              uint8_t* pU8UncorrErrPend
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccUncorrISR_ConfirmErr_Interconnect(InfoOfEccReg_st *pStInfoOfEccReg, 
                                              uint8_t* pU8UncorrErrPend)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;
    uint32_t u32UncorrErrCnt = 0U;

    /* ECC_CBASS_ERR_STAT1.UNC_PEND = 1
        1¼[RAMTst_ECCErrorSts] = [ECC ID Uncorrectable Error]
        2ϢECC_CBASS_ERR_STAT1.UNC_PEND_CLR = 1 */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_ERR_STAT1;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    if( u8RdErr ==  FSRAMTST_NO_ERROR )
    {
        u32UncorrErrCnt = HW_GET_FIELD(u32RegVal, ECC_CBASS_ERR_STAT1_UNC_PEND);
        if( u32UncorrErrCnt == FSRAMTST_VALUE_UNC_PEND )
        {
            *pU8UncorrErrPend = FSRAMTST_ERROR;
            gU32RamTstErrInfo |= FSRAMTST_MODX_ECCID_UNCORR_ERR;
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_STAT1_UNC_PEND_CLR, FSRAMTST_VALUE_UNC_PEND_CLR);
            fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
        }
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccUncorrISR_EndIntrpt
*  Explanation:          check Correctable err
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccUncorrISR_EndIntrpt(InfoOfEccReg_st *pStInfoOfEccReg)
{
    /* жϷECC_DED_EOI_REG.EOI_WR = 1 */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_DED_EOI_REG;
    HW_WR_FIELD32(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                  ECC_DED_EOI_REG_EOI_WR, 
                  FSRAMTST_VALUE_DED_EOI_WR_END_INT);

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_EccUncorrISR
*  Explanation:          check Uncorrectable err
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_EccUncorrISR(SDL_ECC_MemType eccMemType)
{
    uint32_t u32UncorrErr = FSRAMTST_NO_ERROR;
    uint8_t u8UncorrErrPend = FSRAMTST_NO_ERROR;
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    InfoOfEccReg_st stInfoOfEccReg = {0U};
    stInfoOfEccReg.u32EccRegBaseAddr = SDL_ECC_getAggrBaseAddr(eccMemType);
    uint32_t u32NumRams = SDL_ECC_getNumRams(eccMemType);
    SDL_ECC_MemSubType memSubType = 0U;
    uint32_t u32RamIdType = 0U;

    for(memSubType = 0U; memSubType < u32NumRams; ++memSubType)
    {
        stInfoOfEccReg.u32EccId = SDL_ECC_getRamId(eccMemType, memSubType);
        u32RamIdType = SDL_ECC_getRamIdType(eccMemType, memSubType);

        u32UncorrErr = fsRamTst_EccUncorrISR_CheckErrPend(&stInfoOfEccReg);

        if( (u32UncorrErr == FSRAMTST_ERROR)
           && (u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER) )
        {
            u8RdErr = fsRamTst_EccISR_RecordErrAddr(&stInfoOfEccReg, eccMemType, memSubType);
            if(u8RdErr == FSRAMTST_NO_ERROR)
            {
                fsRamTst_EccUncorrISR_ConfirmErr_Wrapper(&stInfoOfEccReg, eccMemType, &u8UncorrErrPend);
            }
        }
        else if(u32UncorrErr == FSRAMTST_ERROR)  /* INTERCONNECT */
        {
            fsRamTst_EccUncorrISR_ConfirmErr_Interconnect(&stInfoOfEccReg, &u8UncorrErrPend);
        }
        else
        {
            /* do nothing */
        }
            
        if( u8UncorrErrPend == FSRAMTST_ERROR )
        {
            break;
        }
    }

    fsRamTst_EccUncorrISR_EndIntrpt(&stInfoOfEccReg);

    return ;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_ECCUncorrISR
*  Explanation:          check Uncorrectable err
*  Timming:              
*  Inputs:               FS_ESM_InstanceType enInstType
*                        uint32_t u32EsmINTRPTId
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
void FS_RamTst_ECCUncorrISR(FS_ESM_InstanceType enInstType, uint32_t u32EsmINTRPTId)
{
    SDL_ECC_MemType eccMemType = 0U;
    uint8_t u8GetErr = FSRAMTST_NO_ERROR;
    u8GetErr = fsRamTst_getMemType(enInstType, u32EsmINTRPTId, &eccMemType);

    if(u8GetErr == FSRAMTST_NO_ERROR)
    {
        fsRamTst_EccUncorrISR(eccMemType);
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst_WaitErrComplete
*  Explanation:          ȴע
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        FS_RamTst_ErrType_en enErrType
*                        uint32_t u32RamIdType
*  Outputs:              uint8_t *pU8RdErr
*  Returns:              uint8_t
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_ECCSelfTst_WaitErrComplete(InfoOfEccReg_st *pStInfoOfEccReg, FS_RamTst_ErrType_en enErrType, 
                                                   uint32_t u32RamIdType, uint8_t *pU8RdErr)
{
    uint8_t u8WaitErr = FSRAMTST_NO_ERROR;  /* ȴ־ */
    uint32_t u32RegVal = 0U;
    uint32_t u32WaitComplete = 0U;

    if( (enErrType == FS_RAMTST_SELFTST_CORR) 
        && (u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER) )
    {
        /* Corr Wrapper 5ȴECC_CTRL.FORCE_SEC = 0ʾע */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CTRL;
        pStInfoOfEccReg->u32FeildMask = ECC_CTRL_FORCE_SEC_MASK;
        pStInfoOfEccReg->u32FeildShift = ECC_CTRL_FORCE_SEC_SHIFT;
    }
    else if((enErrType == FS_RAMTST_SELFTST_CORR) 
            && (u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_INTERCONNECT))
    {
        /* Corr Interconnect ȴECC_CBASS_CTRL.FORCE_SE = 0ʾע */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_CTRL;
        pStInfoOfEccReg->u32FeildMask = ECC_CBASS_CTRL_FORCE_SE_MASK;
        pStInfoOfEccReg->u32FeildShift = ECC_CBASS_CTRL_FORCE_SE_SHIFT;
    }
    else if( (enErrType == FS_RAMTST_SELFTST_UNCORR) 
        && (u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER) )
    {
        /* Uncorr Wrapper 5ȴECC_CTRL.FORCE_DED = 0ʾע */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CTRL;
        pStInfoOfEccReg->u32FeildMask = ECC_CTRL_FORCE_DED_MASK;
        pStInfoOfEccReg->u32FeildShift = ECC_CTRL_FORCE_DED_SHIFT;
    }
    else if( (enErrType == FS_RAMTST_SELFTST_UNCORR) 
        && (u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_INTERCONNECT) )
    {
        /* Uncorr Interconnect ȴECC_CBASS_CTRL.FORCE_DE = 0ʾע */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_CTRL;
        pStInfoOfEccReg->u32FeildMask = ECC_CBASS_CTRL_FORCE_DE_MASK;
        pStInfoOfEccReg->u32FeildShift = ECC_CBASS_CTRL_FORCE_DE_SHIFT;
    }
    else
    {
        /* do nothing */
    }

    uint32_t u32Cnt = FSRAMTST_WAIT_ERR_CNT;
    while( u32Cnt > 0U )
    {
        *pU8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
        u32WaitComplete = (u32RegVal & pStInfoOfEccReg->u32FeildMask) >> pStInfoOfEccReg->u32FeildShift;
        if( (*pU8RdErr == FSRAMTST_NO_ERROR)
             && (u32WaitComplete == FSRAMTST_VALUE_FORCE_WAIT_ERR_COMPLETE ) )
        {
            break;
        }
        else
        {
            u32Cnt--;
        }
    }

    if( u32Cnt == 0U )
    {
        u8WaitErr = FSRAMTST_ERROR;
    }

    return u8WaitErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst_InjectErr_Wrapper
*  Explanation:          ECCԼ ע
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        FS_RamTst_ErrType_en enErrType
*  Outputs:              void
*  Returns:              uint8_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_ECCSelfTst_InjectErr_Wrapper(InfoOfEccReg_st *pStInfoOfEccReg,
                                                     FS_RamTst_ErrType_en enErrType)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;
    
    /* 1óĵַüĴECC_ERR_CTRL1.ECC_ROW */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_CTRL1;
    fsRamTst_WrEccReg(pStInfoOfEccReg, FSRAMTST_VALUE_ERR_ECC_ROW);

    /* 2ģһγüĴECC_CTRL.ERROR_ONCE = 1 */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CTRL;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    if( u8RdErr == FSRAMTST_NO_ERROR )
    {
        HW_SET_FIELD32(u32RegVal, ECC_CTRL_ERROR_ONCE, FSRAMTST_VALUE_ERROR_ONCE_SIM_ERR);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }

    /* 3Correctable óλλãüĴECC_ERR_CTRL2.ECC_BIT1 */
    if( u8RdErr == FSRAMTST_NO_ERROR )
    {
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_CTRL2;
        u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    }

    if( (u8RdErr == FSRAMTST_NO_ERROR) && (enErrType == FS_RAMTST_SELFTST_CORR) )
    {
        HW_SET_FIELD32(u32RegVal, ECC_ERR_CTRL2_ECC_BIT1, FSRAMTST_VALUE_ERR_BIT);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }
    /* 3Uncorrectable óλλãüĴECC_ERR_CTRL2.ECC_BIT2 */
    else if( (u8RdErr == FSRAMTST_NO_ERROR) && (enErrType == FS_RAMTST_SELFTST_UNCORR) )
    {
        HW_SET_FIELD32(u32RegVal, ECC_ERR_CTRL2_ECC_BIT2, FSRAMTST_VALUE_ERR_BIT);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }
    else
    {
        /* do nothing */
    }

    /* 4CorrectableüĴECC_CTRL.FORCE_SEC = 1 */
    if( u8RdErr == FSRAMTST_NO_ERROR )
    {
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CTRL;
        u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    }

    if( (u8RdErr == FSRAMTST_NO_ERROR) && (enErrType == FS_RAMTST_SELFTST_CORR) )
    {
        HW_SET_FIELD32(u32RegVal, ECC_CTRL_FORCE_SEC, FSRAMTST_VALUE_SET_CORR_ERR);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }
    /* 4UncorrectableüĴECC_CTRL.FORCE_DED = 1 */
    else if( (u8RdErr == FSRAMTST_NO_ERROR) && (enErrType == FS_RAMTST_SELFTST_UNCORR) )
    {
        HW_SET_FIELD32(u32RegVal, ECC_CTRL_FORCE_DED, FSRAMTST_VALUE_SET_UNCORR_ERR);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }
    else
    {
        /* do nothing */
    }

    return u8RdErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst_ClearErr_Wrapper
*  Explanation:          ECCԼ 
*  Timming:              
*  Inputs:               const InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*                        FS_RamTst_ErrType_en enErrType
*  Outputs:              uint8_t* u8SelfTstErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_ECCSelfTst_ClearErr_Wrapper(const InfoOfEccReg_st *pStInfoOfEccReg, 
                                                   SDL_ECC_MemType eccMemType, 
                                                   FS_RamTst_ErrType_en enErrType,
                                                   uint32_t u32RegVal,
                                                   uint32_t u32ErrCnt)
{
    if((eccMemType == SDL_ECC_MEMTYPE_MCU_ADC0)
       || (eccMemType == SDL_ECC_MEMTYPE_MCU_ADC1))  /* ADCģ */
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_SEC_ADC, u32ErrCnt);
        }
        else
        {
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_DED_ADC, u32ErrCnt);
        }
    }
    else
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_SEC, u32ErrCnt);
        }
        else
        {
            HW_SET_FIELD32(u32RegVal, ECC_ERR_STAT1_CLR_ECC_DED, u32ErrCnt);
        }
    }
    fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst_CheckErr_Wrapper
*  Explanation:          ECCԼ 
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*                        FS_RamTst_ErrType_en enErrType
*  Outputs:              uint8_t* u8SelfTstErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_ECCSelfTst_CheckErr_Wrapper(InfoOfEccReg_st *pStInfoOfEccReg, 
                                                   SDL_ECC_MemType eccMemType, 
                                                   FS_RamTst_ErrType_en enErrType,
                                                   uint8_t* u8SelfTstErr)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32ErrCnt = 0U;
    uint32_t u32RegVal = 0U;
    
    /* 6ǷCorrectable
    aECC_ERR_STAT1.ECC_SECΪ㣬ECC־λ
        ECC_ERR_STAT1.CLR_ECC_SEC = ECC_ERR_STAT1.ECC_SEC
    bECC_ERR_STAT1.ECC_SECΪ㣬ECC쳣 */

    /* 6ǷUncorrectable
    aECC_ERR_STAT1.ECC_DEDΪ㣬ECC־λ
        ECC_ERR_STAT1.CLR_ECC_DED = ECC_ERR_STAT1.ECC_DED
    bECC_ERR_STAT1.ECC_DEDΪ㣬ECC쳣 */

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_ERR_STAT1;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    if((eccMemType == SDL_ECC_MEMTYPE_MCU_ADC0)
       || (eccMemType == SDL_ECC_MEMTYPE_MCU_ADC1))  /* ADCģ */
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            u32ErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_SEC_ADC);
        }
        else
        {
            u32ErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_DED_ADC);
        }
    }
    else
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            u32ErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_SEC);
        }
        else
        {
            u32ErrCnt = HW_GET_FIELD(u32RegVal, ECC_ERR_STAT1_ECC_DED);
        }
    }

    if( (u8RdErr == FSRAMTST_NO_ERROR)
       && (u32ErrCnt != FSRAMTST_VALUE_ECC_NO_ERR) )    /*  */
    {
        /* ־λ */
        fsRamTst_ECCSelfTst_ClearErr_Wrapper(pStInfoOfEccReg, eccMemType, enErrType,
                                             u32RegVal, u32ErrCnt);
    }
    else
    {
        *u8SelfTstErr = FSRAMTST_ERROR;
    }
    
    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst_InjectErr_Interconnect
*  Explanation:          ECCԼ Correctable
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        FS_RamTst_ErrType_en enErrType
*  Outputs:              void
*  Returns:              uint8_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_ECCSelfTst_InjectErr_Interconnect(InfoOfEccReg_st *pStInfoOfEccReg, 
                                                          FS_RamTst_ErrType_en enErrType)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32RegVal = 0U;

    /* üĴECC_CBASS_CTRL.FORCE_N_BIT = 0 */
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_CTRL;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        HW_SET_FIELD32(u32RegVal, ECC_CBASS_CTRL_FORCE_N_BIT, FSRAMTST_VALUE_FORCE_N_BIT);
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }

    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            /* üĴECC_CBASS_ERR_CTRL1.ECC_BIT1  ECC_CBASS_ERR_CTRL1.ECC_GRP */
            pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_ERR_CTRL1;
        }
        else
        {
            /* üĴECC_CBASS_ERR_CTRL2.ECC_BIT2 */
            pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_ERR_CTRL2;
        }

        u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    }

    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_CTRL1_ECC_BIT1, FSRAMTST_VALUE_ERR_BIT);
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_CTRL1_ECC_GRP, FSRAMTST_VALUE_ERR_GRP);
        }
        else
        {
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_CTRL2_ECC_BIT2, FSRAMTST_VALUE_ERR_BIT);
        }
        
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }

    /* üĴECC_CBASS_CTRL.FORCE_SE = 1 */
    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_CTRL;
        u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);
    }

    if(u8RdErr == FSRAMTST_NO_ERROR)
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_CTRL_FORCE_SE, FSRAMTST_VALUE_FORCE_SE);
        }
        else
        {
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_CTRL_FORCE_DE, FSRAMTST_VALUE_FORCE_DE);
        }
        
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }

    return u8RdErr;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst_CheckErr_Interconnect
*  Explanation:          ECCԼ Correctable
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        FS_RamTst_ErrType_en enErrType
*  Outputs:              uint8_t* pU8CorrErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_ECCSelfTst_CheckErr_Interconnect(InfoOfEccReg_st *pStInfoOfEccReg, 
                                                      FS_RamTst_ErrType_en enErrType, 
                                                      uint8_t* pU8CorrErr)
{
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint32_t u32ErrCnt = 0U;
    uint32_t u32RegVal = 0U;

    /* aECC_CBASS_ERR_STAT1.INJ_COR_PEND_CLRΪ㣬ECC־λ
        ECC_CBASS_ERR_STAT1.INJ_COR_PEND_CLR = ECC_CBASS_ERR_STAT1.INJ_COR_PEND_CLR
       bECC_CBASS_ERR_STAT1.INJ_COR_PEND_CLRΪ㣬ECC쳣 */

    /*  aECC_CBASS_ERR_STAT1.INJ_UNC_PEND_CLRΪ㣬ECC־λ
         ECC_CBASS_ERR_STAT1.INJ_UNC_PEND_CLR = ECC_CBASS_ERR_STAT1.INJ_UNC_PEND_CLR
        bECC_CBASS_ERR_STAT1.INJ_UNC_PEND_CLRΪ㣬ECC쳣 */

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_ERR_STAT1;
    u8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32RegVal);

    if( enErrType == FS_RAMTST_SELFTST_CORR )
    {
        u32ErrCnt = HW_GET_FIELD(u32RegVal, ECC_CBASS_ERR_STAT1_INJ_COR_PEND_CLR);
    }
    else
    {
        u32ErrCnt = HW_GET_FIELD(u32RegVal, ECC_CBASS_ERR_STAT1_INJ_UNC_PEND_CLR);
    }
    
    if( (u8RdErr == FSRAMTST_NO_ERROR)
       && (u32ErrCnt != FSRAMTST_VALUE_ECC_NO_ERR) )
    {
        if( enErrType == FS_RAMTST_SELFTST_CORR )
        {
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_STAT1_INJ_COR_PEND_CLR, u32ErrCnt);
        }
        else
        {
            HW_SET_FIELD32(u32RegVal, ECC_CBASS_ERR_STAT1_INJ_UNC_PEND_CLR, u32ErrCnt);
        }
        
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32RegVal);
    }
    else
    {
        *pU8CorrErr = FSRAMTST_ERROR;
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_ECCSelfTst
*  Explanation:          ECCԼ Correctable
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*                        FS_RamTst_ErrType_en enErrType
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_ECCSelfTst(SDL_ECC_MemType eccMemType, 
                                FS_RamTst_ErrType_en enErrType)
{
    uint8_t u8SelfTstErr = FSRAMTST_NO_ERROR;
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    uint8_t u8WaitErr = FSRAMTST_NO_ERROR;
    InfoOfEccReg_st stInfoOfEccReg = {0U};
    stInfoOfEccReg.u32EccRegBaseAddr = SDL_ECC_getAggrBaseAddr(eccMemType);
    uint32_t u32NumRams = SDL_ECC_getNumRams(eccMemType);
    SDL_ECC_MemSubType memSubType = 0U;
    uint32_t u32RamIdType = 0U;

    uint8_t u8Cnt = FSRAMTST_SELFTST_CNT; 
    while( u8Cnt > 0U )
    {
        for(memSubType = 0U; memSubType < u32NumRams; ++memSubType)
        {
            u8SelfTstErr = FSRAMTST_NO_ERROR;
            u8RdErr = FSRAMTST_NO_ERROR;
            u8WaitErr = FSRAMTST_NO_ERROR;
            
            stInfoOfEccReg.u32EccId = SDL_ECC_getRamId(eccMemType, memSubType);
            u32RamIdType = SDL_ECC_getRamIdType(eccMemType, memSubType);
            
            if(u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER)
            {
                u8RdErr = fsRamTst_ECCSelfTst_InjectErr_Wrapper(&stInfoOfEccReg, enErrType);
                
                if( u8RdErr == FSRAMTST_NO_ERROR )
                {
                    u8WaitErr = fsRamTst_ECCSelfTst_WaitErrComplete(&stInfoOfEccReg, enErrType, 
                                                                    u32RamIdType, &u8RdErr);
                }

                if( (u8RdErr == FSRAMTST_NO_ERROR)
                   && (u8WaitErr == FSRAMTST_NO_ERROR) )
                {
                    fsRamTst_ECCSelfTst_CheckErr_Wrapper(&stInfoOfEccReg, eccMemType, 
                                                         enErrType, &u8SelfTstErr);
                }
                else  /* ȴʧܻĴʧΪϹ */
                {
                    u8SelfTstErr = FSRAMTST_ERROR;
                }

            }
            else  /* do interconnect */
            {
                u8RdErr = fsRamTst_ECCSelfTst_InjectErr_Interconnect(&stInfoOfEccReg, enErrType);

                /* ȴECC_CBASS_CTRL.FORCE_SE = 0ʾע */
                if( u8RdErr == FSRAMTST_NO_ERROR )
                {
                    u8WaitErr = fsRamTst_ECCSelfTst_WaitErrComplete(&stInfoOfEccReg, enErrType, 
                                                                    u32RamIdType, &u8RdErr);
                }

                if( (u8RdErr == FSRAMTST_NO_ERROR)
                   && (u8WaitErr == FSRAMTST_NO_ERROR) )
                {
                    fsRamTst_ECCSelfTst_CheckErr_Interconnect(&stInfoOfEccReg, enErrType, &u8SelfTstErr);
                }
                else  /* ȴʧܻĴʧΪϹ */
                {
                    u8SelfTstErr = FSRAMTST_ERROR;
                }                
            }

            if(u8SelfTstErr == FSRAMTST_ERROR)
            {
                break;
            }
        }

        if(u8SelfTstErr == FSRAMTST_ERROR)  /* й */
        {
            --u8Cnt;
        }
        else
        {
            break;
        }
    }

    if(u8Cnt == 0U)
    {
        gU32RamTstErrInfo |= FSRAMTST_ECC_CORR_SELFTST_ERR;
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_ECCSelfTst
*  Explanation:          ECCԼ
*  Timming:              
*  Inputs:               void
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
void FS_RamTst_ECCSelfTst(void)
{
    SDL_ECC_MemType eccMemType;
    for(eccMemType = 0U; eccMemType < SDL_ECC_MEMTYPE_MAX; ++eccMemType)
    {
        fsRamTst_ECCSelfTst(eccMemType, FS_RAMTST_SELFTST_CORR);
        fsRamTst_ECCSelfTst(eccMemType, FS_RAMTST_SELFTST_UNCORR);
    }

    return ;
}


/*
*******************************************************************************
*  Function name:        fsRamTst_ResetState
*  Explanation:          ԺΪreset state
*  Timming:              
*  Inputs:               uint32_t instanceId
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static int32_t fsRamTst_ResetState(uint32_t instanceId)
{
    int32_t status = FSRAMTST_PBIST_PASS;

    /**--- Step 2a: Request Primary core ---*/
    if( PBIST_TestHandleArray[instanceId].procRstNeeded )
    {
        if (PBIST_TestHandleArray[instanceId].tisciProcId != 0u)
        {
            /* Request Primary core */
            status = Sciclient_procBootRequestProcessor((uint8_t)PBIST_TestHandleArray[instanceId].tisciProcId,
                                                        SCICLIENT_SERVICE_WAIT_FOREVER);
        }
    }

    /**--- Step 2b: Request Secondary core ---*/
    if ((status == FSRAMTST_PBIST_PASS) && (PBIST_TestHandleArray[instanceId].procRstNeeded))
    {
        if ((PBIST_TestHandleArray[instanceId].secondaryCoreNeeded)
            && (PBIST_TestHandleArray[instanceId].tisciSecProcId != 0u))
        {
            /* Request secondary core */
            status = Sciclient_procBootRequestProcessor((uint8_t)PBIST_TestHandleArray[instanceId].tisciSecProcId,
                                                        SCICLIENT_SERVICE_WAIT_FOREVER);
        }
    }

    /**--- Step 2c: Put Primary core in local reset ---*/
    if ((status == FSRAMTST_PBIST_PASS) && (PBIST_TestHandleArray[instanceId].procRstNeeded))
    {
        if (PBIST_TestHandleArray[instanceId].tisciDeviceId != 0u)
        {
            /* Set Local reset for Primary core */
            status =  Sciclient_pmSetModuleRst(PBIST_TestHandleArray[instanceId].tisciDeviceId,
                                               0x1, /* Local Reset asserted */
                                               SCICLIENT_SERVICE_WAIT_FOREVER);
        }
    }

    /**--- Step 2d: Put Secondary core in local reset ---*/
    if ((status == FSRAMTST_PBIST_PASS) && (PBIST_TestHandleArray[instanceId].procRstNeeded))
    {
        if ((PBIST_TestHandleArray[instanceId].secondaryCoreNeeded)
            && (PBIST_TestHandleArray[instanceId].tisciSecDeviceId != 0u))
        {
            /* Set Local reset for Secondary core */
            status =  Sciclient_pmSetModuleRst(PBIST_TestHandleArray[instanceId].tisciSecDeviceId,
                                               0x1, /* Local Reset asserted */
                                               SCICLIENT_SERVICE_WAIT_FOREVER);
        }
    }

    /**--- Step 2l: Power up PBIST */
    if ((status == FSRAMTST_PBIST_PASS) && (PBIST_TestHandleArray[instanceId].tisciPBISTDeviceId != 0u))
    {
        status = Sciclient_pmSetModuleState(PBIST_TestHandleArray[instanceId].tisciPBISTDeviceId,
                                            TISCI_MSG_VALUE_DEVICE_SW_STATE_ON,
                                            TISCI_MSG_FLAG_AOP,
                                            SCICLIENT_SERVICE_WAIT_FOREVER);
    }

    return status;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_PBSITTst
*  Explanation:          PBSIT
*  Timming:              
*  Inputs:               uint32_t instanceId
*  Outputs:              void
*  Returns:              void
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_PBSITTst(uint32_t instanceId)
{
    int32_t status = FSRAMTST_PBIST_PASS;
    SDL_PBIST_postResult result;
    int32_t i32TstResult = 0;

    bool PBISTResult;

    uint8_t u8Cnt = FSRAMTST_PBISTTST_CNT;
    while( u8Cnt > 0U )
    {
        i32TstResult = 0;
        /* 1HW POSTTIӿSDL_PBIST_getPOSTStatus()ֵPass
            ¼[RAMTst_PBISTSts] = [PBSIT HW POST Failed] */
        status = SDL_PBIST_getPOSTStatus(&result);
        if( status != FSRAMTST_PBIST_PASS )
        {
            if( u8Cnt == 1U )
            {
                gU32RamTstErrInfo |= FSRAMTST_PBSIT_HW_POST_FAILED;
            }
            i32TstResult = -1;
        }

        /* 2ԺΪreset state */
        if( status == FSRAMTST_PBIST_PASS )
        {
            (void)fsRamTst_ResetState(instanceId);
        }

        /* 3ԸInstancePBIST ϣú
            SDL_PBIST_selfTest(SDL_PBIST_INST_MAINR5F0, SDL_PBIST_NEG_TEST, NULL, &PBISTResult);  */
        if( status == FSRAMTST_PBIST_PASS )
        {
            status = SDL_PBIST_selfTest(PBIST_TestHandleArray[instanceId].pbistInst, SDL_PBIST_NEG_TEST, NULL_PTR, &PBISTResult);
        }

        /* 4ֵPass߲Խʧܣ
            ¼[RAMTst_PBISTSts] = [PBSIT MAINR5F0 SelfTest Failed] */
        if( ((status != FSRAMTST_PBIST_PASS) || (PBISTResult == (bool) false)) 
           && (i32TstResult == 0) )
        {
            if( u8Cnt == 1U )
            {
                gU32RamTstErrInfo |= FSRAMTST_PBSIT_MOD_SELFTST_FAILED;
            }   
            i32TstResult = -1;
        }

        /* 5PBISTԼɹԸInstancePBSITԣ 
            SDL_PBIST_selfTest(SDL_PBIST_INST_MAINR5F0, SDL_PBIST_TEST, NULL, &PBISTResult); */
        if( status == FSRAMTST_PBIST_PASS )
        {
            status = SDL_PBIST_selfTest(PBIST_TestHandleArray[instanceId].pbistInst, SDL_PBIST_TEST, NULL_PTR, &PBISTResult);
        }

        /* 6ֵPass߲Խʧܣ
        ¼[RAMTst_PBISTSts] = [PBSIT MAINR5F0 Failed] */
        if( ((status != FSRAMTST_PBIST_PASS) || (PBISTResult == (bool) false)) 
           && (i32TstResult == 0) )
        {
            if( u8Cnt == 1U )
            {
                gU32RamTstErrInfo |= FSRAMTST_PBSIT_MOD_FAILED;
            }
            i32TstResult = -1;
        }

        if( i32TstResult == -1 )
        {
            --u8Cnt;
        }
        else
        {
            break;
        }
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_PBSITTst
*  Explanation:          PBSIT
*  Timming:              
*  Inputs:               void
*  Outputs:              void
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
void FS_RamTst_PBSITTst(void)
{
    uint32_t i;

    /* PBIST */
    for(i = 0U; i < SDL_PBIST_NUM_INSTANCES; ++i)
    {
        fsRamTst_PBSITTst(i);
    }
    
    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_RegCheck_DoWrapper
*  Explanation:          Ĵ
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*  Outputs:              uint8_t* pU8MemTypeRegCheckErr
*                        uint8_t* pU8RdErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_RegCheck_DoWrapper(InfoOfEccReg_st *pStInfoOfEccReg, 
                                           SDL_ECC_MemType eccMemType, 
                                           uint8_t* pU8MemTypeRegCheckErr, 
                                           uint8_t* pU8RdErr)
{
    uint8_t u8RegCheckErr = FSRAMTST_NO_ERROR;
    uint32_t u32FieldValue = 0U;
    uint8_t u8IsDoParity = fsRamTst_IsToDoParity(eccMemType);
    
    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CTRL;
    *pU8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32FieldValue);

    if( (*pU8RdErr == FSRAMTST_NO_ERROR) 
        && (u8IsDoParity == FSRAMTST_TO_DO) )
    {
        /* ʹżУ飬ECC_CTRL.CHECK_PARITY = 1 */
        if(HW_GET_FIELD(u32FieldValue, ECC_CTRL_CHECK_PARITY) != FSRAMTST_VALUE_CHECK_PARITY_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_CTRL_CHECK_PARITY, FSRAMTST_VALUE_CHECK_PARITY_EN);
        }

        /* ʹECCӿڷʳʱƣECC_CTRL.CHECK_SVBUS_TIMEOUT = 1 */
        if(HW_GET_FIELD(u32FieldValue, ECC_CTRL_CHECK_SVBUS_TIMEOUT) != FSRAMTST_VALUE_CHECK_SVBUS_TIMEOUT_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_CTRL_CHECK_SVBUS_TIMEOUT, FSRAMTST_VALUE_CHECK_SVBUS_TIMEOUT_EN);
        }            
    }

    if(*pU8RdErr == FSRAMTST_NO_ERROR)
    {
        /* ʹECCĶ-޸-дƣECC_CTRL.ENABLE_RMW = 1 */
        if(HW_GET_FIELD(u32FieldValue, ECC_CTRL_ENABLE_RMW) != FSRAMTST_VALUE_ENABLE_RMW_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_CTRL_ENABLE_RMW, FSRAMTST_VALUE_ENABLE_RMW_EN);
        }

        /* ʹECCܣECC_CTRL.ECC_CHECK = 1ECC_CTRL.ECC_ENABLE = 1 */
        if(HW_GET_FIELD(u32FieldValue, ECC_CTRL_ECC_CHECK) != FSRAMTST_VALUE_ECC_CHECK_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_CTRL_ECC_CHECK, FSRAMTST_VALUE_ECC_CHECK_EN);
        }

        if(HW_GET_FIELD(u32FieldValue, ECC_CTRL_ECC_ENABLE) != FSRAMTST_VALUE_ECC_ENABLE_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_CTRL_ECC_ENABLE, FSRAMTST_VALUE_ECC_ENABLE_EN);
        }
    }

    if( (*pU8RdErr == FSRAMTST_NO_ERROR)
        && (u8RegCheckErr == FSRAMTST_ERROR) )
    {
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32FieldValue);
        *pU8MemTypeRegCheckErr = FSRAMTST_ERROR;
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_RegCheck_DoInterconnect
*  Explanation:          Ĵ
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              uint8_t* pU8MemTypeRegCheckErr
*                        uint8_t* pU8RdErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_RegCheck_DoInterconnect(InfoOfEccReg_st *pStInfoOfEccReg, 
                                                uint8_t* pU8MemTypeRegCheckErr, 
                                                uint8_t* pU8RdErr)
{
    uint8_t u8RegCheckErr = FSRAMTST_NO_ERROR;
    uint32_t u32FieldValue = 0U;

    pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_CBASS_CTRL;
    *pU8RdErr = fsRamTst_RdEccReg(pStInfoOfEccReg, &u32FieldValue);

    if( *pU8RdErr == FSRAMTST_NO_ERROR )
    {
        /* ʹECCܣECC_CBASS_CTRL.ECC_CHECK = 1 */
        if(HW_GET_FIELD(u32FieldValue, ECC_CBASS_CTRL_ECC_CHECK) != FSRAMTST_VALUE_ECC_CHECK_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_CBASS_CTRL_ECC_CHECK, FSRAMTST_VALUE_ECC_CHECK_EN);
        }
    }

    if( (*pU8RdErr == FSRAMTST_NO_ERROR)
        && (u8RegCheckErr == FSRAMTST_ERROR) )
    {
        fsRamTst_WrEccReg(pStInfoOfEccReg, u32FieldValue);
        *pU8MemTypeRegCheckErr = FSRAMTST_ERROR;
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_RegCheck_EnIntrpt1
*  Explanation:          Ĵ
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*  Outputs:              uint8_t* pU8MemTypeRegCheckErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_RegCheck_EnIntrpt1(InfoOfEccReg_st *pStInfoOfEccReg, 
                                            uint8_t* pU8MemTypeRegCheckErr)
{
    uint32_t u32FieldValue = 0U;
    
    /* ʹECC CorrectableжϣECC_SEC_ENABLE_SET_REG0ӦλΪ1 */
    fsRamTst_ChoseRegN(pStInfoOfEccReg, ECC_SEC_ENABLE_SET_REG_N);
    u32FieldValue = HW_RD_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                                      pStInfoOfEccReg->u32FeildMask, 
                                      pStInfoOfEccReg->u32FeildShift);
    if(u32FieldValue != 1U)
    {
        HW_WR_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                          pStInfoOfEccReg->u32FeildMask, 
                          pStInfoOfEccReg->u32FeildShift,
                          1U);
        *pU8MemTypeRegCheckErr = FSRAMTST_ERROR;
    }

    /* ʹECC UncorrectableжϣECC_DED_ENABLE_SET_REG0ӦλΪ1 */
    fsRamTst_ChoseRegN(pStInfoOfEccReg, ECC_DED_ENABLE_SET_REG_N);
    u32FieldValue = HW_RD_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                                      pStInfoOfEccReg->u32FeildMask, 
                                      pStInfoOfEccReg->u32FeildShift);
    if(u32FieldValue != 1U)
    {
        HW_WR_FIELD32_RAW(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                          pStInfoOfEccReg->u32FeildMask, 
                          pStInfoOfEccReg->u32FeildShift,
                          1U);
        *pU8MemTypeRegCheckErr = FSRAMTST_ERROR;
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_RegCheck_EnIntrpt2
*  Explanation:          Ĵ
*  Timming:              
*  Inputs:               InfoOfEccReg_st *pStInfoOfEccReg
*                        SDL_ECC_MemType eccMemType
*  Outputs:              uint8_t* pU8MemTypeRegCheckErr
*  Returns:              void             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static void fsRamTst_RegCheck_EnIntrpt2(InfoOfEccReg_st *pStInfoOfEccReg, 
                                           SDL_ECC_MemType eccMemType, 
                                           uint8_t* pU8MemTypeRegCheckErr)
{
    uint8_t u8RegCheckErr = FSRAMTST_NO_ERROR;
    uint32_t u32FieldValue = 0U;
    uint8_t u8IsDoParity = fsRamTst_IsToDoParity(eccMemType);

    if( u8IsDoParity == FSRAMTST_TO_DO )
    {
        /* ʹECCӿڷʳʱжϣECC_AGGR_ENABLE_SET.TIMEOUT = 1  */
        pStInfoOfEccReg->u32EccRegOffsetAddr = OADDR_ECC_AGGR_ENABLE_SET;
        u32FieldValue = HW_RD_REG32(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr);

        if(HW_GET_FIELD(u32FieldValue, ECC_AGGR_ENABLE_SET_TIMEOUT) != FSRAMTST_VALUE_TIMEOUT_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_AGGR_ENABLE_SET_TIMEOUT, FSRAMTST_VALUE_TIMEOUT_EN);
        }
        
        /* ʹECCżУжϣECC_AGGR_ENABLE_SET.PARITY = 1 */
        if(HW_GET_FIELD(u32FieldValue, ECC_AGGR_ENABLE_SET_PARITY) != FSRAMTST_VALUE_PARITY_EN)
        {
            u8RegCheckErr = FSRAMTST_ERROR;
            HW_SET_FIELD32(u32FieldValue, ECC_AGGR_ENABLE_SET_PARITY, FSRAMTST_VALUE_PARITY_EN);
        }
        
        if( u8RegCheckErr == FSRAMTST_ERROR )
        {
            HW_WR_REG32(pStInfoOfEccReg->u32EccRegBaseAddr + pStInfoOfEccReg->u32EccRegOffsetAddr, 
                        u32FieldValue);
            *pU8MemTypeRegCheckErr = FSRAMTST_ERROR;
        }
    }

    return ;
}

/*
*******************************************************************************
*  Function name:        fsRamTst_RegCheck
*  Explanation:          Ĵ
*  Timming:              
*  Inputs:               SDL_ECC_MemType eccMemType
*  Outputs:              void
*  Returns:              uint8_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
static uint8_t fsRamTst_RegCheck(SDL_ECC_MemType eccMemType)
{    
    uint8_t u8RegCheckErrTemp = FSRAMTST_NO_ERROR;
    uint8_t u8MemTypeRegCheckErr = FSRAMTST_NO_ERROR;
    uint8_t u8RdErr = FSRAMTST_NO_ERROR;
    InfoOfEccReg_st stInfoOfEccReg = {0U};
    stInfoOfEccReg.u32EccRegBaseAddr = SDL_ECC_getAggrBaseAddr(eccMemType);
    uint32_t u32NumRams = SDL_ECC_getNumRams(eccMemType);
    SDL_ECC_MemSubType memSubType = 0U;
    uint32_t u32RamIdType = 0U;

    for(memSubType = 0U; memSubType < u32NumRams; ++memSubType)
    {
        stInfoOfEccReg.u32EccId = SDL_ECC_getRamId(eccMemType, memSubType);
        u32RamIdType = SDL_ECC_getRamIdType(eccMemType, memSubType);

        if(u32RamIdType == (uint32_t)SDL_ECC_RAM_ID_TYPE_WRAPPER)
        {
            fsRamTst_RegCheck_DoWrapper(&stInfoOfEccReg, eccMemType, 
                                        &u8MemTypeRegCheckErr, &u8RdErr);
        }
        else    /* do interconnect */
        {
            fsRamTst_RegCheck_DoInterconnect(&stInfoOfEccReg, &u8MemTypeRegCheckErr, &u8RdErr);
        }

        if( u8RdErr == FSRAMTST_ERROR )
        {
            u8MemTypeRegCheckErr = FSRAMTST_ERROR;
        }

        fsRamTst_RegCheck_EnIntrpt1(&stInfoOfEccReg, &u8MemTypeRegCheckErr);
    }

    fsRamTst_RegCheck_EnIntrpt2(&stInfoOfEccReg, eccMemType, &u8MemTypeRegCheckErr);

    return u8MemTypeRegCheckErr;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_RegCheck
*  Explanation:          Ĵ
*  Timming:              
*  Inputs:               void
*  Outputs:              void
*  Returns:              uint8_t             
*  Author:               SuiGuoqing
*  CreatTime:            2022.04.02
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.04.02          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
uint8_t FS_RamTst_RegCheck(void)
{
    SDL_ECC_MemType eccMemType;
    uint8_t u8RegCheckErr = FSRAMTST_NO_ERROR;
    uint8_t u8MemTypeRegCheckErr = FSRAMTST_NO_ERROR;
    
    for(eccMemType = 0U; eccMemType < SDL_ECC_MEMTYPE_MAX; ++eccMemType)
    {
        u8MemTypeRegCheckErr = fsRamTst_RegCheck(eccMemType);
        if( u8MemTypeRegCheckErr == FSRAMTST_ERROR )
        {
            u8RegCheckErr = FSRAMTST_ERROR;
        }
    }

    return u8RegCheckErr;
}

/*
*******************************************************************************
*  Function name:        FS_RamTst_MainFunc
*  Explanation:          ں㱨
*  Timming:              
*  Inputs:               void
*  Outputs:              void
*  Returns:              viod             
*  Author:               SuiGuoqing
*  CreatTime:            2022.07.22
*  History:
*  Revision        ModifiedDate        Modifier       ModifiedReason    Comment
*  ----------------------------------------------------------------------------
*  1.0             2022.07.22          SuiGuoqing         created
*  ----------------------------------------------------------------------------
*******************************************************************************
*/
void FS_RamTst_MainFunc(void)
{
    fsRamTst_ErrorReport();

    return ;
}

