/**
 * @file  FS_RamTst_pbist.c
*/

#include "FS_RamTst_pbist.h"

PBIST_TestHandle_t PBIST_TestHandleArray[SDL_PBIST_NUM_INSTANCES] = 
{
    /* MCU PBIST - Not run as part of BootApp since PBIST changes the memory upon which it is run. */
    {
        .testName               = "MCU PBIST",
        .pbistInst              = SDL_PBIST_INST_MCUR5F0,
        .numPostPbistToCheck    = 0u,
        .tisciPBISTDeviceId     = TISCI_DEV_MCU_PBIST0, /* PBIST device id  */
        .procRstNeeded          = false,
        .secondaryCoreNeeded    = false,                /* Secondary core needed */
        .coreCustPwrSeqNeeded   = false,
        .numAuxDevices          = 0u,                   /* No Aux devices */
        .auxInitRestoreFunction = NULL,                 /* Auxilliary init/restore function */
    },

    /* MCU PULSAR PBIST */
    {
        .testName               = "MCU PULSAR PBIST",
        .pbistInst              = SDL_PBIST_INST_MCUR5F1,
        .numPostPbistToCheck    = 0u,
        .tisciPBISTDeviceId     = TISCI_DEV_MCU_PBIST1,       /* PBIST device id */
        .procRstNeeded          = false,
        .secondaryCoreNeeded    = false,                   /* Secondary core needed */
        .coreCustPwrSeqNeeded   = false,
        .numAuxDevices          = 0u,                      /* No Aux devices */
        .auxInitRestoreFunction = NULL,                    /* Auxilliary init/restore function */
    },

    /* NAVSS - Not supported in BootApp */
    {
        .testName               = "NAVSS PBIST",
        .pbistInst              = SDL_PBIST_INST_NAVSS,
        .numPostPbistToCheck    = 0u,
        .tisciPBISTDeviceId     = TISCI_DEV_PBIST7,         /* PBIST device id  */
        .procRstNeeded          = true,
        .secondaryCoreNeeded    = false,                    /* Secondary core needed */
        .coreName               = "NAVSS",                  /* Primary core   */
        .tisciProcId            = 0x0u,                     /* No Proc Id for NAVSS */
        .tisciDeviceId          = TISCI_DEV_NAVSS0,         /* NAVSS Device Id */
        .coreCustPwrSeqNeeded   = false,
        .numAuxDevices          = 0u,                       /* No Aux devices */
        .auxInitRestoreFunction = NULL,                     /* Auxilliary init/restore function */
    },
};

static sdlClecEvtMap clecLookupTable[SDL_PBIST_MAX_CLEC_EVTS] =
{
    {
        SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_8,
        SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_8
    },

    {
        SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_9,
        SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_9
    },

    {
        SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_12,
        SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_12
    },

    {
        SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_13,
        SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_13
    }
};


/*
 * Note: the order for the below must match order of SDL_PBIST_inst
 */
static SDL_pbistInstInfo SDL_PBIST_InstInfoArray[SDL_PBIST_NUM_INSTANCES] =
{
    /* MCU R5F Instance */
    {
        .PBISTRegsHiAddress  = 0u,
        .pPBISTRegs          = (SDL_pbistRegs *)SDL_MCU_PBIST0_BASE,
        /*
         * Golden vectors not currently supported running from MCU core because
         * this will be self destructive. Below select memory which is guaranteed
         * not to be utilized can be run.
         */
        .numPBISTRuns        = 1u,
        .PBISTConfigRun = 
        {
            [0] = 
            {
                .override           = 0x0u,
                /* Override bit set to 0 to use memoryGroupsBitMap & algorithmsBitMap */
                /* NOTE: As the MCU R5f is running this code override bit cannot be set to enable all memory test */
                .algorithmsBitMap   = 0x00002000u,          /* Choose Algorithm 14 */
                .memoryGroupsBitMap = 0x0080000000000000u,  /* Choose ADC RAM */
                .scrambleValue      = 0xFEDCBA9876543210U,  /* Scramble Value */
            },
            [1] = 
            {
                .override           = 0x0u,
                .algorithmsBitMap   = 0x0u,
                .memoryGroupsBitMap = 0x0u,
                .scrambleValue      = 0x0u,
            }
        },
        .clecInputEvt           = 0x0u,
        .interruptNumber        = SDLR_MCU_R5FSS0_CORE0_INTR_MCU_PBIST0_DFT_PBIST_CPU_0,
        .doneFlag               = PBIST_NOT_DONE,
        .PBISTNegConfigRun =
        {
            .CA0   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA0,
            .CA1   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA1,
            .CA2   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA2,
            .CA3   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA3,
            .CL0   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL0,
            .CL1   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL1,
            .CL2   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL2,
            .CL3   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL3,
            .CMS   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CMS,
            .CSR   = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CSR,
            .I0    = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I0,
            .I1    = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I1,
            .I2    = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I2,
            .I3    = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I3,
            .RAMT  = SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_RAMT
        }
    },

    /* MCU Pulsar Instance */
    {
        .PBISTRegsHiAddress  = 0u,
        .pPBISTRegs          = (SDL_pbistRegs *)SDL_MCU_PBIST1_BASE,
        .numPBISTRuns        = SDL_MCU_PBIST1_NUM_TEST_VECTORS,
        .PBISTConfigRun = 
        {
            [0] = 
            {
                .override           = 0x0u,
                /* Override bit set to 0 to use memoryGroupsBitMap & algorithmsBitMap */
                .algorithmsBitMap   = SDL_MCU_PBIST1_ALGO_BITMAP_0,   /* Choose recommended Algo bitmap */
                /* NOTE: As the MCU R5f is running this code override bit cannot be set to enable all memory test */
                .memoryGroupsBitMap = SDL_MCU_PBIST1_MEM_BITMAP_0,    /* Choose recommended mem bitmap  */
                .scrambleValue      = 0xFEDCBA9876543210U,  /* Scramble Value       */
            },
            [1] = 
            {
                .override           = 0x0u,
                .algorithmsBitMap   = 0x0u,
                .memoryGroupsBitMap = 0x0u,
                .scrambleValue      = 0x0u,
            }
        },
        .clecInputEvt           = 0x0u,
        .interruptNumber        = SDLR_MCU_R5FSS0_CORE0_INTR_MCU_PBIST1_DFT_PBIST_CPU_0,
        .doneFlag               = PBIST_NOT_DONE,
        .PBISTNegConfigRun = 
        {
            .CA0   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA0,
            .CA1   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA1,
            .CA2   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA2,
            .CA3   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA3,
            .CL0   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL0,
            .CL1   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL1,
            .CL2   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL2,
            .CL3   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL3,
            .CMS   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CMS,
            .CSR   = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CSR,
            .I0    = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I0,
            .I1    = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I1,
            .I2    = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I2,
            .I3    = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I3,
            .RAMT  = SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_RAMT
        }
    },

    /* NAVSS Instance */
    {
        .PBISTRegsHiAddress  = 0u,
        .pPBISTRegs          = (SDL_pbistRegs *)SDL_PBIST7_BASE,
        .numPBISTRuns        = SDL_PBIST7_NUM_TEST_VECTORS,
        .PBISTConfigRun = 
        {
            /* Golden Vector Part 1 */
            [0] = 
            {
                .override           = 0x0u,
                /* Override bit set to 0 to use memoryGroupsBitMap & algorithmsBitMap */
                .algorithmsBitMap   = SDL_PBIST7_ALGO_BITMAP_0,   /* Choose recommended Algo bitmap 0 */
                .memoryGroupsBitMap = SDL_PBIST7_MEM_BITMAP_0,    /* Choose recommended mem bitmap 0  */
                .scrambleValue      = 0xFEDCBA9876543210U,  /* Scramble Value */
            },
            /* Golden Vector Part 2 */
            [1] = 
            {
                .override           = 0x0u,
                /* Override bit set to 0 to use memoryGroupsBitMap & algorithmsBitMap */
                .algorithmsBitMap   = SDL_PBIST7_ALGO_BITMAP_1,   /* Choose recommended Algo bitmap 1 */
                .memoryGroupsBitMap = SDL_PBIST7_MEM_BITMAP_1,    /* Choose recommended mem bitmap 1  */
                .scrambleValue      = 0xFEDCBA9876543210U,  /* Scramble Value */
            }
        },
        .clecInputEvt           = 0x0u,
        .interruptNumber        = SDLR_MCU_R5FSS0_CORE0_INTR_PBIST7_DFT_PBIST_CPU_0,
        .doneFlag               = PBIST_NOT_DONE,
        .PBISTNegConfigRun = 
        {
            .CA0   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA0,
            .CA1   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA1,
            .CA2   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA2,
            .CA3   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA3,
            .CL0   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL0,
            .CL1   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL1,
            .CL2   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL2,
            .CL3   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL3,
            .CMS   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CMS,
            .CSR   = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CSR,
            .I0    = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I0,
            .I1    = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I1,
            .I2    = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I2,
            .I3    = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I3,
            .RAMT  = SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_RAMT
        }
    },
};

static SDL_OSAL_Interface *gOSAL_Interface = NULL_PTR;

static int32_t SDL_PBIST_getDevicePostStatus(SDL_PBIST_postResult *pResult);
static void SDL_PBIST_runPostPbistCheck(SDL_PBIST_hwpostInst instance, SDL_PBIST_postStatus *pPostStatus);
static void SDL_PBIST_isPostPbistDone(uint32_t postStatMmrRegVal,
                                      SDL_PBIST_hwpostInst instance,
                                      bool *pIsDone);
static void SDL_PBIST_isPostPbistTimeout(uint32_t postStatMmrRegVal,
                                         SDL_PBIST_hwpostInst instance,
                                         bool *pIsTimedOut);
static void SDL_PBIST_postCheckResult(uint32_t postStatMmrRegVal,
                                      SDL_PBIST_hwpostInst instance,
                                      bool *pResult);
static SDL_pbistInstInfo * SDL_PBIST_getInstInfo(SDL_PBIST_inst instance);
static int32_t SDL_OSAL_getClecOutputEvent(uint32_t *clecOutputEvt);
static uint32_t SDL_PBIST_translateClecToIntr(uint32_t outputEvt);
static int32_t SDL_PBIST_prepareTest(SDL_PBIST_inst instance, const SDL_pbistInstInfo *pInfo,
                                     SDL_pbistRegs **pRegs, pSDL_OSAL_hwipHandle *PBIST_intrHandle);
static int32_t SDL_OSAL_disableInterrupt(int32_t intNum);
static void SDL_PBIST_eventHandler( uint32_t instanceId);
static int32_t SDL_OSAL_registerInterrupt(SDL_OSAL_hwipParams *pParams, pSDL_OSAL_hwipHandle *handle);
static void* SDL_OSAL_addrTranslate(uint64_t addr, uint32_t size);
static int32_t SDL_PBIST_runTest(const SDL_PBIST_config *pConfig, SDL_PBIST_testType testType,
                                 SDL_pbistRegs *pRegs, SDL_pbistInstInfo *pInfo, bool *pResult);
static int32_t SDL_PBIST_start(SDL_pbistRegs *pPBISTRegs,
                        const SDL_PBIST_config * pConfig);
static int32_t SDL_PBIST_startNeg(SDL_pbistRegs *pPBISTRegs,
                           const SDL_PBIST_configNeg * const pConfig);
static int32_t SDL_PBIST_getResult(SDL_PBIST_testType testType, const SDL_pbistRegs *pRegs, bool *pResult);
static int32_t SDL_PBIST_checkResult (const SDL_pbistRegs *pPBISTRegs, bool *pResult);
static int32_t SDL_PBIST_softReset(SDL_pbistRegs *pPBISTRegs);
static int32_t SDL_PBIST_releaseTestMode(SDL_pbistRegs *pPBISTRegs);
static int32_t SDL_PBIST_cleanupTest(pSDL_OSAL_hwipHandle PBIST_intrHandle);
static int32_t SDL_OSAL_deregisterInterrupt(pSDL_OSAL_hwipHandle handle);

static int32_t SDL_OSAL_deregisterInterrupt(pSDL_OSAL_hwipHandle handle)
{
    SDL_ErrType_t ret = SDL_PASS;

    // Check HwiP function is valid
    if (handle == NULL_PTR)
    {
        ret = SDL_EINVALID_PARAMS;
    }
    else if ((gOSAL_Interface == NULL_PTR) || (gOSAL_Interface->deregisterInterrupt == NULL_PTR))
    {
        ret = SDL_EINVALID_PARAMS;
    }
    else
    {
        ret = gOSAL_Interface->deregisterInterrupt(handle);
    }

    return ret;
}

static int32_t SDL_PBIST_cleanupTest(pSDL_OSAL_hwipHandle PBIST_intrHandle)
{
    int32_t ret = SDL_PASS;

    /* Destroy the interrupt handler */
    if (PBIST_intrHandle != NULL)
    {
        (void)SDL_OSAL_deregisterInterrupt(PBIST_intrHandle);
    }

    return ret;
}

static int32_t SDL_PBIST_releaseTestMode(SDL_pbistRegs *pPBISTRegs)
{
    int32_t sdlResult = SDL_PASS;
    if (pPBISTRegs == NULL_PTR)
    {
        sdlResult = SDL_EFAIL;
    }
    else
    {
        /*
         * Set Margin mode register to relese test mode and switch to
         * Functional mode
         */
        pPBISTRegs->MARGIN_MODE = SDL_PBIST_FUNCTIONAL_MODE;

        /* Put RAM Group Select (RGS) = 0 */
        pPBISTRegs->RAMT = ((uint32_t)0x0u);

        /* Bit 0: Disable PBIST access */
        pPBISTRegs->PACT = ((uint32_t)0x0u);
    }

    return  sdlResult;

}

static int32_t SDL_PBIST_softReset(SDL_pbistRegs *pPBISTRegs)
{
    int32_t sdlResult= SDL_PASS;

    if (pPBISTRegs == NULL_PTR)
    {
        sdlResult = SDL_EBADARGS;
    }
    else
    {
        /* Turns on PBIST clock */
        pPBISTRegs->PACT = SDL_PBIST_PACT_PACT_MASK;

        /* Zero out Loop counter 0 */
        pPBISTRegs->L0 = ((uint32_t)0x0u);

        /* Zero out Pbist Id register */
        pPBISTRegs->PID = ((uint32_t)0x0u);

        /* Set override register to all 1 */
        pPBISTRegs->OVER = (SDL_PBIST_OVER_RINFO_MASK
                           | SDL_PBIST_OVER_READ_MASK
                           | SDL_PBIST_OVER_MM_MASK
                           | SDL_PBIST_OVER_ALGO_MASK);

        /* Zero out Data logger 0 */
        pPBISTRegs->DLR = ((uint32_t)0x0u);

        /* Zero out Clock Mux Select register */
        pPBISTRegs->CMS = ((uint32_t)0x0u);
    }
    return  sdlResult;
}

static int32_t SDL_PBIST_checkResult (const SDL_pbistRegs *pPBISTRegs, bool *pResult)
{
    int32_t sdlResult = SDL_PASS;

    if ((pPBISTRegs == (SDL_pbistRegs *)NULL_PTR) || (pResult == NULL_PTR))
    {
        sdlResult = SDL_EBADARGS;
    }
    else
    {
        /* Check Fail Status Fail Register: expected to be all zeros.
           If any bit set indicates error */
        if (pPBISTRegs->FSRF == ((uint64_t)0x00000000u))
        {
            *pResult = TRUE;
        }
        else
        {
            *pResult = FALSE;
        }
    }

    return  sdlResult;
}

static int32_t SDL_PBIST_getResult(SDL_PBIST_testType testType, const SDL_pbistRegs *pRegs, bool *pResult)
{
    int32_t ret = SDL_PASS;
    bool PBISTResult;

    ret = SDL_PBIST_checkResult(pRegs, &PBISTResult);

    if (ret == SDL_PASS)
    {
        /* Check the PBIST result */
        if ((testType == SDL_PBIST_TEST) && (PBISTResult == (bool) true))
        {
            *pResult = (bool) true;
        }
        else if ((testType == SDL_PBIST_NEG_TEST) && (PBISTResult == (bool) false))
        {
            *pResult = (bool) true;
        }
        else
        {
            *pResult = (bool) false;
            ret = SDL_EFAIL;
        }
    }

    return ret;
}

static int32_t SDL_PBIST_startNeg(SDL_pbistRegs *pPBISTRegs,
                           const SDL_PBIST_configNeg * const pConfig)
{
    int32_t sdlResult = SDL_PASS;

    if ((pPBISTRegs == NULL_PTR) || (pConfig == (SDL_PBIST_configNeg *)NULL_PTR))
    {
        sdlResult = SDL_EBADARGS;
    }
    else
    {
        pPBISTRegs->PACT = SDL_PBIST_PACT_PACT_MASK;
        pPBISTRegs->MARGIN_MODE = SDL_PBIST_FAILURE_INSERTION_TEST_MODE;
        pPBISTRegs->L0 = ((uint32_t)0x0u);
        pPBISTRegs->DLR = ((uint32_t)0x00000010u);
        pPBISTRegs->RF0L = ((uint32_t)0x00000001u);
        pPBISTRegs->RF0U = ((uint32_t)0x00003123u);
        pPBISTRegs->RF1L = ((uint32_t)0x0513FC02u);
        pPBISTRegs->RF1U = ((uint32_t)0x00000002u);
        pPBISTRegs->RF2L = ((uint32_t)0x00000003u);
        pPBISTRegs->RF2U = ((uint32_t)0x00000000u);
        pPBISTRegs->RF3L = ((uint32_t)0x00000004u);
        pPBISTRegs->RF3U = ((uint32_t)0x00000028u);
        pPBISTRegs->RF4L = ((uint32_t)0x64000044u);
        pPBISTRegs->RF4U = ((uint32_t)0x00000000u);
        pPBISTRegs->RF5L = ((uint32_t)0x0006A006u);
        pPBISTRegs->RF5U = ((uint32_t)0x00000000u);
        pPBISTRegs->RF6L = ((uint32_t)0x00000007u);
        pPBISTRegs->RF6U = ((uint32_t)0x0000A0A0u);
        pPBISTRegs->RF7L = ((uint32_t)0x00000008u);
        pPBISTRegs->RF7U = ((uint32_t)0x00000064u);
        pPBISTRegs->RF8L = ((uint32_t)0x00000009u);
        pPBISTRegs->RF8U = ((uint32_t)0x0000A5A5u);
        pPBISTRegs->RF9L = ((uint32_t)0x0000000Au);
        pPBISTRegs->RF9U = ((uint32_t)0x00000079u);
        pPBISTRegs->RF10L = ((uint32_t)0x00000000u);
        pPBISTRegs->RF10U = ((uint32_t)0x00000001u);
        pPBISTRegs->D = ((uint32_t)0xAAAAAAAAu);
        pPBISTRegs->E = ((uint32_t)0xAAAAAAAAu);

        pPBISTRegs->CA2 = pConfig->CA2;
        pPBISTRegs->CL0 = pConfig->CL0;
        pPBISTRegs->CA3 = pConfig->CA3;
        pPBISTRegs->I0 = pConfig->I0;
        pPBISTRegs->CL1 = pConfig->CL1;
        pPBISTRegs->I3 = pConfig->I3;
        pPBISTRegs->I2 = pConfig->I2;
        pPBISTRegs->CL2 = pConfig->CL2;
        pPBISTRegs->CA1 = pConfig->CA1;
        pPBISTRegs->CA0 = pConfig->CA0;
        pPBISTRegs->CL3 = pConfig->CL3;
        pPBISTRegs->I1 = pConfig->I1;
        pPBISTRegs->RAMT = pConfig->RAMT;
        pPBISTRegs->CSR = pConfig->CSR;
        pPBISTRegs->CMS = pConfig->CMS;

        pPBISTRegs->STR = ((uint32_t)0x00000009u);

        /* Start PBIST */
        pPBISTRegs->STR = ((uint32_t)0x00000001u);
    }

    return  sdlResult;
}

static int32_t SDL_PBIST_start(SDL_pbistRegs *pPBISTRegs,
                        const SDL_PBIST_config * pConfig)
{
    int32_t sdlResult= SDL_PASS;

    if ((pPBISTRegs == NULL_PTR) || (pConfig == (SDL_PBIST_config *)NULL_PTR))
    {
        sdlResult = SDL_EBADARGS;
    }
    else
    {
        /* Turns on PBIST clock in PBIST ACTivate register */
        pPBISTRegs->PACT = SDL_PBIST_PACT_PACT_MASK;

        /* Set Margin mode register for Test mode */
        pPBISTRegs->MARGIN_MODE = SDL_PBIST_TEST_MODE;

        /* Zero out Loop counter 0 */
        pPBISTRegs->L0 = ((uint32_t)0x0u);

        /* Set algorithm bitmap */
        pPBISTRegs->ALGO = pConfig->algorithmsBitMap;

        /* Set Memory group bitmap */
        pPBISTRegs->RINFO = pConfig->memoryGroupsBitMap;

        /* Zero out override register */
        pPBISTRegs->OVER = pConfig->override;

        /* Set Scramble value */
        pPBISTRegs->SCR = pConfig->scrambleValue;

        /* Set DLR register for ROM based testing and Config Access */
        pPBISTRegs->DLR = (SDL_PBIST_DLR_DLR0_ROM_MASK
                           | SDL_PBIST_DLR_DLR0_CAM_MASK);
    }

    return  sdlResult;
}

static int32_t SDL_PBIST_runTest(const SDL_PBIST_config *pConfig, SDL_PBIST_testType testType,
                                 SDL_pbistRegs *pRegs, SDL_pbistInstInfo *pInfo, bool *pResult)
{
    int32_t ret = SDL_PASS;
    uint32_t numRuns = 1;
    uint32_t i;
    uint32_t timeoutCount = 0;

    if ((pConfig == (SDL_PBIST_config *)NULL) && (testType == SDL_PBIST_TEST))
    {
        numRuns = pInfo->numPBISTRuns;
    }
    pInfo->doneFlag = PBIST_NOT_DONE;

    for (i = 0; i < numRuns; i++)
    {
        if (testType == SDL_PBIST_TEST)
        {
            if (pConfig != (SDL_PBIST_config *)NULL)
            {
                ret = SDL_PBIST_start(pRegs, pConfig);
            }
            else
            {
                ret = SDL_PBIST_start(pRegs, &pInfo->PBISTConfigRun[i]);
            }
        }
        else // (testType == SDL_PBIST_NEG_TEST)
        {
            ret = SDL_PBIST_startNeg(pRegs, &pInfo->PBISTNegConfigRun);
        }

        if (ret == SDL_PASS)
        {
            timeoutCount = 0;
            /* Timeout if exceeds time */
            while ((pInfo->doneFlag == PBIST_NOT_DONE)
                   && (timeoutCount < PBIST_MAX_TIMEOUT_VALUE))
            {
                timeoutCount++;
            }

            if (pInfo->doneFlag == PBIST_NOT_DONE)
            {
                ret = SDL_EFAIL;
            }
            else
            {
                ret = SDL_PBIST_getResult(testType, pRegs, pResult);

                /* Do a Soft Reset */
                if (ret == SDL_PASS)
                {
                    ret = SDL_PBIST_softReset(pRegs);

                    /* Execute exit sequence */
                    if (ret == SDL_PASS)
                    {
                        ret = SDL_PBIST_releaseTestMode(pRegs);
                    }
                }
            }

            /* reset Done flag so we can run again */
            pInfo->doneFlag = PBIST_NOT_DONE;
        }
    }

    return ret;
}

static void* SDL_OSAL_addrTranslate(uint64_t addr, uint32_t size)
{
    void *ret = (void *)(-1);

    if ((gOSAL_Interface != NULL_PTR) && (gOSAL_Interface->addrTranslate != NULL_PTR))
    {
        ret = gOSAL_Interface->addrTranslate(addr, size);
    }

    return ret;
}

static int32_t SDL_OSAL_registerInterrupt(SDL_OSAL_hwipParams *pParams, pSDL_OSAL_hwipHandle *handle)
{
    SDL_ErrType_t ret = SDL_PASS;

    // Check HwiP function is valid
    if ((handle == NULL_PTR) || (pParams == NULL_PTR))
    {
        ret = SDL_EINVALID_PARAMS;
    }
    else if ((gOSAL_Interface == NULL_PTR) || (gOSAL_Interface->registerInterrupt == NULL_PTR))
    {
        ret = SDL_EINVALID_PARAMS;
    }
    else
    {
        *handle = gOSAL_Interface->registerInterrupt(pParams);
    }

    return ret;
}

static void SDL_PBIST_eventHandler( uint32_t instanceId)
{
    SDL_PBIST_InstInfoArray[instanceId].doneFlag = PBIST_DONE;

    return;
}

static int32_t SDL_OSAL_disableInterrupt(int32_t intNum)
{
    SDL_ErrType_t ret = SDL_PASS;

    // Check HwiP function is valid
    if ((gOSAL_Interface == NULL_PTR) || (gOSAL_Interface->disableInterrupt == NULL_PTR))
    {
        ret = SDL_EINVALID_PARAMS;
    }
    else
    {
        ret = gOSAL_Interface->disableInterrupt(intNum);
    }

    return ret;
}

static int32_t SDL_PBIST_prepareTest(SDL_PBIST_inst instance, const SDL_pbistInstInfo *pInfo,
                                     SDL_pbistRegs **pRegs, pSDL_OSAL_hwipHandle *PBIST_intrHandle)
{
    int32_t ret = SDL_PASS;
    SDL_OSAL_hwipParams intrParams;
    void *localAddr = NULL;

    /* Disable interrupt */
    ret = SDL_OSAL_disableInterrupt((int32_t)pInfo->interruptNumber);

    if (ret == SDL_PASS)
    {
        intrParams.intNum = (int32_t)pInfo->interruptNumber;
        intrParams.callback = (pSDL_OSAL_interruptCallbackFunction)SDL_PBIST_eventHandler;
        intrParams.callbackArg = (uintptr_t)instance;

        /* Register call back function for PBIST Interrupt */
        ret = SDL_OSAL_registerInterrupt(&intrParams,
                                         PBIST_intrHandle);
    }
    if (*PBIST_intrHandle == NULL)
    {
        ret = SDL_EFAIL;
    }

    /* Get PBIST register space Pointer */
    *pRegs = pInfo->pPBISTRegs;

    if ((ret == SDL_PASS) && (pInfo->PBISTRegsHiAddress != (uint64_t)0))
    {
        localAddr = SDL_OSAL_addrTranslate(pInfo->PBISTRegsHiAddress, PBIST_REG_REGION_SIZE);
        if (localAddr == (void *)(-1))
        {
            ret = SDL_EFAIL;
        }
        *pRegs = (SDL_pbistRegs *)(localAddr);
    }

    return ret;
}

static uint32_t SDL_PBIST_translateClecToIntr(uint32_t outputEvt)
{
    uint32_t intNum = 0;
    uint32_t i;

    for (i = 0; i < SDL_PBIST_MAX_CLEC_EVTS; i++)
    {
        if (outputEvt == clecLookupTable[i].outputEvt)
        {
            intNum = clecLookupTable[i].intNum;
            break;
        }
    }

    return intNum;
}

static int32_t SDL_OSAL_getClecOutputEvent(uint32_t *clecOutputEvt)
{
    int32_t retVal = SDL_PASS;

    *clecOutputEvt = CSLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_12;

    return retVal;
}

static SDL_pbistInstInfo * SDL_PBIST_getInstInfo(SDL_PBIST_inst instance)
{
    SDL_pbistInstInfo *pInfo = NULL;

    if ((uint32_t)instance < SDL_PBIST_NUM_INSTANCES)
    {
        pInfo = &SDL_PBIST_InstInfoArray[instance];
    }

    return pInfo;
}

static void SDL_PBIST_postCheckResult(uint32_t postStatMmrRegVal,
                                      SDL_PBIST_hwpostInst instance,
                                      bool *pResult)
{
    uint32_t mask = 0xFFFFFFFFu;
    (void)instance;

    /* Only one instance supported for J721E: SDL_PBIST_HWPOST_INST_MCU */
    mask = SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT_POST_MCU_PBIST_FAIL_MASK;

    if ((postStatMmrRegVal & mask) == ((uint32_t)0x00000000u))
    {
        *pResult = (bool)true;
    }
    else
    {
        *pResult = (bool)false;
    }

    return;
}

static void SDL_PBIST_isPostPbistTimeout(uint32_t postStatMmrRegVal,
                                         SDL_PBIST_hwpostInst instance,
                                         bool *pIsTimedOut)
{
    uint32_t shift;
    (void)instance;

    /* Only one instance supported for J721E: SDL_PBIST_HWPOST_INST_MCU */
    shift = SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT_POST_MCU_PBIST_TIMEOUT_SHIFT;

    *pIsTimedOut = (((postStatMmrRegVal >> shift) & 0x1u) == 0x1u) ? (bool)true : (bool)false;

    return;
}

static void SDL_PBIST_isPostPbistDone(uint32_t postStatMmrRegVal,
                                      SDL_PBIST_hwpostInst instance,
                                      bool *pIsDone)
{
    uint32_t shift;
    (void)instance;

    /* Only one instance supported for J721E: SDL_PBIST_HWPOST_INST_MCU */
    shift = SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT_POST_MCU_PBIST_DONE_SHIFT;

    *pIsDone = (((postStatMmrRegVal >> shift) & 0x1u) == 0x1u) ? (bool)true : (bool)false;

    return;
}

static void SDL_PBIST_runPostPbistCheck(SDL_PBIST_hwpostInst instance, SDL_PBIST_postStatus *pPostStatus)
{
    uint32_t postRegVal;
    bool     PBISTResult;

    /* Get PBIST register space Pointer */
    *pPostStatus = SDL_PBIST_POST_COMPLETED_SUCCESS;

    /* Read HW POST status register */
    postRegVal = SDL_REG32_RD(SDL_WKUP_CTRL_MMR0_CFG0_BASE +
                              SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT);

    /* Check if HW POST PBIST was performed */
    SDL_PBIST_isPostPbistDone(postRegVal, instance, &PBISTResult);

    if (PBISTResult != (bool) true)
    {
        /* HW POST: PBIST not completed, check if it timed out */
        SDL_PBIST_isPostPbistTimeout(postRegVal,
                                     instance,
                                     &PBISTResult);
        if (PBISTResult != (bool) true)
        {
            /* HW POST: PBIST was not performed at all on this device
             * for this core */
            *pPostStatus = SDL_PBIST_POST_NOT_RUN;
        }
        else
        {
            /* HW POST: PBIST was attempted but timed out for this section */
            *pPostStatus = SDL_PBIST_POST_TIMEOUT;
        }
    }
    else
    {
        /* HW POST: PBIST was completed on this device, check the result */
        SDL_PBIST_postCheckResult(postRegVal, instance, &PBISTResult);
        if (PBISTResult != (bool) true)
        {
            /* HW POST: PBIST was completed, but the test failed */
            *pPostStatus = SDL_PBIST_POST_COMPLETED_FAILURE;
        }
    } /* if (PBISTResult != (bool) true) */

    return;
}

static int32_t SDL_PBIST_getDevicePostStatus(SDL_PBIST_postResult *pResult)
{
    int32_t ret = SDL_PASS;

    /* Get status for MCU */
    SDL_PBIST_runPostPbistCheck(SDL_PBIST_HWPOST_INST_MCU, &pResult->mcuPostStatus);

    return ret;
}

int32_t SDL_PBIST_getPOSTStatus(SDL_PBIST_postResult *pResult)
{
    int32_t ret = SDL_PASS;

    if (pResult == NULL)
    {
        ret = SDL_EBADARGS;
    }

    if (ret == SDL_PASS)
    {
        /* HW POST test flow - test already run, checking the result */
        ret = SDL_PBIST_getDevicePostStatus(pResult);
    }

    return ret;
}

int32_t SDL_PBIST_selfTest(SDL_PBIST_inst instance, SDL_PBIST_testType testType,
                           const SDL_PBIST_config *pConfig, bool *pResult)
{
    int32_t ret = SDL_PASS;
    SDL_pbistRegs *pRegs;
    SDL_pbistInstInfo *pInfo;
    uint32_t clecOutputEvt = 0;
    uint32_t intNum = 0;

    pSDL_OSAL_hwipHandle PBIST_intrHandle = NULL;

    /* Get the PBIST Instance Info */
    pInfo = SDL_PBIST_getInstInfo(instance);

    if ((pResult == NULL) ||
        ((testType != SDL_PBIST_TEST) && (testType != SDL_PBIST_NEG_TEST)) ||
        (pInfo == NULL))
    {
        ret = SDL_EBADARGS;
    }

    if ((ret == SDL_PASS) && (pInfo->clecInputEvt != (uint32_t)0))
    {
        /* Request a clec event through OSAL */
        ret = SDL_OSAL_getClecOutputEvent(&clecOutputEvt);
        if (ret == SDL_PASS)
        {
            intNum = SDL_PBIST_translateClecToIntr(clecOutputEvt);
            if (intNum == (uint32_t)0)
            {
                ret = SDL_EFAIL;
            }
            else
            {
                pInfo->interruptNumber = intNum;
            }
        }
    }

    if (ret == SDL_PASS)
    {
        ret = SDL_PBIST_prepareTest(instance, pInfo, &pRegs, &PBIST_intrHandle);

        if (ret == SDL_PASS)
        {
            ret = SDL_PBIST_runTest(pConfig, testType, pRegs, pInfo, pResult);
        }
    }

    (void)SDL_PBIST_cleanupTest(PBIST_intrHandle);

    return ret;
}


/* Nothing past this point */
