/***************************************************************************************************************
*
*  File name:        FS_RamTst_pbist.h
*  Copyright:        Neusoft Corporation. All rights reserved.
*  Notes:            FS RamTst
*  History:
*  Revision        Data        Name              Comment
*  ----------------------------------------------------------------------------------------------------------
*  1.0            2022.04.02  SuiGuoqing     New Create  *
**************************************************************************************************************/

#ifndef FS_RAMTST_PBIST_H_
#define FS_RAMTST_PBIST_H_

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************************************************
*  Includes
**************************************************************************************************************/
#include <stdint.h>
#include <stdbool.h>
#include <ti/drv/sciclient/soc/sysfw/include/j721e/tisci_devices.h>

/**************************************************************************************************************
*  Symbols definition
**************************************************************************************************************/
/**************************************************************************
* Base Macros
**************************************************************************/
#define SDL_WKUP_CTRL_MMR0_CFG0_BASE                                                               (0x43000000UL)
#define SDL_MCU_PBIST0_BASE                                                                        (0x40e00000UL)
#define SDL_MCU_PBIST1_BASE                                                                        (0x40e10000UL)
#define SDL_PBIST7_BASE                                                                            (0x3360000UL)

/**************************************************************************
* Register Macros
**************************************************************************/
#define SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT                               (0x0000C2C0U)

/**************************************************************************
* Field Macros
**************************************************************************/
#define SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT_POST_MCU_PBIST_DONE_SHIFT     (0x00000008U)

#define SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT_POST_MCU_PBIST_TIMEOUT_SHIFT  (0x00000009U)

#define SDL_WKUP_CTRL_MMR_CFG0_WKUP_POST_STAT_POST_MCU_PBIST_FAIL_MASK      (0x00008000U)

#define SDL_PBIST_PACT_PACT_MASK                                               (0x00000001U)

#define SDL_PBIST_MARGIN_MODE_PBIST_DFT_WRITE_MASK                             (0x00000003U)

#define SDL_PBIST_MARGIN_MODE_PBIST_DFT_READ_SHIFT                             (0x00000002U)

#define SDL_PBIST_DLR_DLR0_ROM_MASK                                            (0x00000004U)

#define SDL_PBIST_DLR_DLR0_CAM_MASK                                            (0x00000010U)

#define SDL_PBIST_MARGIN_MODE_PBIST_DFT_READ_MASK                              (0x0000000CU)

#define SDL_PBIST_OVER_RINFO_MASK                                              (0x00000001U)

#define SDL_PBIST_OVER_READ_MASK                                               (0x00000002U)

#define SDL_PBIST_OVER_MM_MASK                                                 (0x00000004U)

#define SDL_PBIST_OVER_ALGO_MASK                                               (0x00000008U)

/* Properties of PBIST instances in: MCU_PBIST0 */
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA0                                              (0x00000000U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA1                                              (0x000000FFU)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA2                                              (0x000000FFU)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CA3                                              (0x00000000U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL0                                              (0x0000003FU)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL1                                              (0x00000003U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL2                                              (0x00000007U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CL3                                              (0x000000FFU)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CMS                                              (0x00000000U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_CSR                                              (0x00000001U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I0                                               (0x00000001U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I1                                               (0x00000004U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I2                                               (0x00000007U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_I3                                               (0x00000000U)
#define SDL_MCU_PBIST0_FAIL_INSERTION_TEST_VECTOR_RAMT                                             (0x35001628U)

/* Properties of PBIST instances in: MCU_PBIST1 */
#define SDL_MCU_PBIST1_NUM_TEST_VECTORS                                                            (1U)
#define SDL_MCU_PBIST1_ALGO_BITMAP_0                                                               (0x0000000000000006U)
#define SDL_MCU_PBIST1_MEM_BITMAP_0                                                                (0x000000000000002AU)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA0                                              (0x00000000U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA1                                              (0x0000007FU)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA2                                              (0x0000007FU)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CA3                                              (0x00000000U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL0                                              (0x0000001FU)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL1                                              (0x00000003U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL2                                              (0x00000006U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CL3                                              (0x0000007FU)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CMS                                              (0x00000000U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_CSR                                              (0x00000001U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I0                                               (0x00000001U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I1                                               (0x00000004U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I2                                               (0x00000006U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_I3                                               (0x00000000U)
#define SDL_MCU_PBIST1_FAIL_INSERTION_TEST_VECTOR_RAMT                                             (0x04001C28U)

/* Properties of PBIST instances in: PBIST7 */
#define SDL_PBIST7_NUM_TEST_VECTORS                                                                (2U)
#define SDL_PBIST7_ALGO_BITMAP_0                                                                   (0x0000000000000005U)
#define SDL_PBIST7_MEM_BITMAP_0                                                                    (0x0000000000135AB5U)
#define SDL_PBIST7_ALGO_BITMAP_1                                                                   (0x000000000000000AU)
#define SDL_PBIST7_MEM_BITMAP_1                                                                    (0x000000068DC00000U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA0                                                  (0x00000000U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA1                                                  (0x0000003FU)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA2                                                  (0x0000003FU)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CA3                                                  (0x00000000U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL0                                                  (0x0000000FU)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL1                                                  (0x00000003U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL2                                                  (0x00000005U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CL3                                                  (0x0000003FU)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CMS                                                  (0x00000000U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_CSR                                                  (0x00000003U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I0                                                   (0x00000001U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I1                                                   (0x00000004U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I2                                                   (0x00000005U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_I3                                                   (0x00000000U)
#define SDL_PBIST7_FAIL_INSERTION_TEST_VECTOR_RAMT                                                 (0x40004028U)

/*
* List of intr sources for receiver: MCU_R5FSS0_CORE0
*/
#define SDLR_MCU_R5FSS0_CORE0_INTR_MCU_PBIST0_DFT_PBIST_CPU_0                                      (47U)
#define SDLR_MCU_R5FSS0_CORE0_INTR_MCU_PBIST1_DFT_PBIST_CPU_0                                      (37U)
#define SDLR_MCU_R5FSS0_CORE0_INTR_PBIST7_DFT_PBIST_CPU_0                                          (308U)
#define SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_8                    (288U)
#define SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_9                    (289U)
#define SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_12                   (293U)
#define SDLR_MCU_R5FSS0_CORE0_INTR_COMPUTE_CLUSTER0_CLEC_SOC_EVENTS_OUT_LEVEL_13                   (294U)

/*
* List of intr sources for receiver: COMPUTE_CLUSTER0_CLEC
*/
#define CSLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_12           (12U)
#define SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_8            (8U)
#define SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_9            (9U)
#define SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_12           (12U)
#define SDLR_COMPUTE_CLUSTER0_CLEC_MSMC_EVENT_IN_COMPUTE_CLUSTER0_CORE_CORE_MSMC_INTR_13           (13U)

/**
 *  \anchor SDL_ErrType_t
 *  \name SDL Error Types
 *
 *  SDL function return error codes.
 *
 *  @{
 */
#define SDL_PASS                        ( (int32_t) (0))
#define SDL_EFAIL                       (-(int32_t) (1))
#define SDL_EBADARGS                    (-(int32_t) (2))
#define SDL_EINVALID_PARAMS             (-(int32_t) (3))
#define SDL_ETIMEOUT                    (-(int32_t) (4))
#define SDL_EOUT_OF_RANGE               (-(int32_t) (5))
#define SDL_EUNSUPPORTED_CMD            (-(int32_t) (6))
#define SDL_EUNSUPPORTED_OPS            (-(int32_t) (7))
#define SDL_EALLOC                      (-(int32_t) (8))
/* @} */

#ifndef NULL
#define NULL            ((void*)0)
#endif

#ifndef NULL_PTR
#define NULL_PTR            ((void*)0)
#endif

#ifndef TRUE
#define TRUE		((bool) 1)
#define FALSE		((bool) 0)
#endif

#define SDL_PBIST_NUM_INSTANCES 3U

#define SDL_PBIST_MAX_CLEC_EVTS 4U

#define PBIST_NOT_DONE        (0U)
#define PBIST_DONE            (1U)

#define PBIST_REG_REGION_SIZE             (0x400u)

#define PBIST_MAX_TIMEOUT_VALUE           (100000000u)

#define PBIST_MAX_NUM_RUNS    2

#define PBIST_INSTANCE_NAME_MAX_LENGTH    16

/* PBIST test mode */
#define SDL_PBIST_TEST_MODE ((uint32_t)SDL_PBIST_MARGIN_MODE_PBIST_DFT_WRITE_MASK \
                             | ((uint32_t)(1u) << (uint32_t)SDL_PBIST_MARGIN_MODE_PBIST_DFT_READ_SHIFT))

/* PBIST Failure Insertion test mode */
#define SDL_PBIST_FAILURE_INSERTION_TEST_MODE ((uint32_t)SDL_PBIST_MARGIN_MODE_PBIST_DFT_WRITE_MASK \
                                               | (uint32_t)SDL_PBIST_MARGIN_MODE_PBIST_DFT_READ_MASK)

/* PBIST Functional mode  */
#define SDL_PBIST_FUNCTIONAL_MODE       (0x0U)

/**
 *  \brief   This macro reads a 32-bit value from a hardware register
 *           and returns the value.
 *
 *  \param   p      Address of the memory mapped hardware register.
 *
 *  \return  Unsigned 32-bit value read from a register.
 */
#define SDL_REG32_RD(p)         (SDL_REG32_RD_RAW((volatile uint32_t *) (p)))

/*************************************************************************************************************
*  Type definition
**************************************************************************************************************/
typedef struct {
    uint32_t outputEvt;
    uint32_t intNum;
} sdlClecEvtMap;

/**
 *  \brief PBIST self test type
 *
 *  This enum defines the test types supported by the SDL_PBIST_selfTest API.
 */
typedef enum {
    /*!
     * PBIST Test
     * Performs the PBIST test for a the instance specified.
     * When a valid pConfig is passed, those values will be used to execute the
     * test. If pConfig is NULL, then the default "golden vectors" will be used
     * to execute the test.
     */
    SDL_PBIST_TEST,
    /*!
     * PBIST Negative Test
     * Performs the PBIST negative test to verify the PBIST logic for the
     * instance specified
     */
    SDL_PBIST_NEG_TEST
} SDL_PBIST_testType;

/**
 *  @addtogroup SDL_PBIST_ENUM
    @{
 *
 */

/**
 *  \brief PBIST instance
 *
 *  This enum defines the PBIST instances supported by the SDL_PBIST_selfTest API.
 */
typedef enum {
    /*!
     * MCU Instance
     */
    SDL_PBIST_INST_MCUR5F0,
    /*!
     * MCU Pulsar Instance
     */
    SDL_PBIST_INST_MCUR5F1,
    /*!
     * NAVSS Instance
     */
    SDL_PBIST_INST_NAVSS,
} SDL_PBIST_inst;

typedef enum {
    /*!
     * MCU Instance
     */
    SDL_PBIST_HWPOST_INST_MCU,
} SDL_PBIST_hwpostInst;

typedef enum {
    /*!
     * The HW POST PBIST completed and the test passed
     */
    SDL_PBIST_POST_COMPLETED_SUCCESS,
    /*!
     * The HW POST PBIST completed and the test failed
     */
    SDL_PBIST_POST_COMPLETED_FAILURE,
    /*!
     * The HW POST PBIST was run but timed out
     */
    SDL_PBIST_POST_TIMEOUT,
    /*!
     * The HW POST PBIST was not performed on this device
     */
    SDL_PBIST_POST_NOT_RUN
} SDL_PBIST_postStatus;

/** @} */

/**
 *  @addtogroup SDL_PBIST_DATASTRUCT
    @{
 *
 */

typedef struct {
    /*!
     * HW POST PBIST status for MCU
     */
    SDL_PBIST_postStatus mcuPostStatus;
} SDL_PBIST_postResult;

/** @} */

/** ---------------------------------------------------------------------------
 * @brief   This structure contains the different configuration used for PBIST
 *
 *
 * ----------------------------------------------------------------------------
 */

typedef struct
{
    /** Override value for memory configuration.
        Check SOC documentation for supported override mode for each PBIST instance. */
    uint32_t override;

    /** Bitmap to select algorithms to use for test.

        NOTE: For a given PBIST instance, there is a selected
        set of memory groups a particular Algorithm can be run.
        Check SOC documentation for supported algorithms for each PBIST instance. */
    uint32_t algorithmsBitMap;

    /** Bitmap to select memory groups to run test on.
        Check SOC documentation for supported memory groups for each PBIST instance. */
    uint64_t memoryGroupsBitMap;

    /** Scramble value to be used for test.
        Check SOC documentation for scramble value to be used. */
    uint64_t scrambleValue;
} SDL_PBIST_config;

typedef struct {
    volatile uint32_t RF0L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF1L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF2L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF3L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF4L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF5L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF6L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF7L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF8L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF9L;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF10L;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF11L;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF12L;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF13L;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF14L;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF15L;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF0U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF1U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF2U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF3U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF4U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF5U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF6U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF7U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF8U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF9U;                      /* Register Files / Instruction Registers */
    volatile uint32_t RF10U;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF11U;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF12U;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF13U;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF14U;                     /* Register Files / Instruction Registers */
    volatile uint32_t RF15U;                     /* Register Files / Instruction Registers */
    volatile uint8_t  Resv_256[128];
    volatile uint32_t A0;                        /* Variable Address Registers */
    volatile uint32_t A1;                        /* Variable Address Registers */
    volatile uint32_t A2;                        /* Variable Address Registers */
    volatile uint32_t A3;                        /* Variable Address Registers */
    volatile uint32_t L0;                        /* Variable Loop Count Registers */
    volatile uint32_t L1;                        /* Variable Loop Count Registers */
    volatile uint32_t L2;                        /* Variable Loop Count Registers */
    volatile uint32_t L3;                        /* Variable Loop Count Registers */
    volatile uint32_t D;                         /* Data Registers */
    volatile uint32_t E;                         /* Data Registers */
    volatile uint8_t  Resv_304[8];
    volatile uint32_t CA0;                       /* Constant Address Registers */
    volatile uint32_t CA1;                       /* Constant Address Registers */
    volatile uint32_t CA2;                       /* Constant Address Registers */
    volatile uint32_t CA3;                       /* Constant Address Registers */
    volatile uint32_t CL0;                       /* Constant Loop Count Registers */
    volatile uint32_t CL1;                       /* Constant Loop Count Registers */
    volatile uint32_t CL2;                       /* Constant Loop Count Registers */
    volatile uint32_t CL3;                       /* Constant Loop Count Registers */
    volatile uint32_t I0;                        /* Constant Increment Registers */
    volatile uint32_t I1;                        /* Constant Increment Registers */
    volatile uint32_t I2;                        /* Constant Increment Registers */
    volatile uint32_t I3;                        /* Constant Increment Registers */
    volatile uint32_t RAMT;                      /* RAM Configuration Register */
    volatile uint32_t DLR;                       /* Datalogger Register */
    volatile uint32_t CMS;                       /* Clock-Mux Select Register */
    volatile uint32_t STR;                       /* Program Control Register */
    volatile uint64_t SCR;                       /* Address Scrambling Register */
    volatile uint32_t CSR;                       /* Chip Select Register */
    volatile uint32_t FDLY;                      /* Fail Delay Register */
    volatile uint32_t PACT;                      /* PACT Register */
    volatile uint32_t PID;                       /* PBIST_ID Register */
    volatile uint32_t OVER;                      /* Override Register */
    volatile uint8_t  Resv_400[4];
    volatile uint64_t FSRF;                      /* Fail Status Fail Register */
    volatile uint64_t FSRC;                      /* Fail Status Count Register */
    volatile uint64_t FSRA;                      /* Fail Status Address Register */
    volatile uint32_t FSRDL0;                    /* Fail Status Data Registers */
    volatile uint8_t  Resv_432[4];
    volatile uint32_t FSRDL1;                    /* Fail Status Data Registers */
    volatile uint32_t MARGIN_MODE;               /* Fail Status Fail Register */
    volatile uint32_t WRENZ;                     /* Fail Status Fail Register */
    volatile uint32_t PAGE_PGS;                  /* Fail Status Fail Register */
    volatile uint32_t ROM;                       /* ROM Mask Register */
    volatile uint32_t ALGO;                      /* Algorithm Mask Register */
    volatile uint64_t RINFO;                     /* RAM Info Mask Register */
} SDL_pbistRegs;

typedef struct
{
    /** Failure insertion value for CA0
        Value for the Failure Insertion Test Vector CA0 */
    uint32_t CA0;

    /** Failure insertion value for CA1
        Value for the Failure Insertion Test Vector CA1 */
    uint32_t CA1;

    /** Failure insertion value for CA2
        Value for the Failure Insertion Test Vector CA2 */
    uint32_t CA2;

    /** Failure insertion value for CA3
        Value for the Failure Insertion Test Vector CA3 */
    uint32_t CA3;

    /** Failure insertion value for CL0
        Value for the Failure Insertion Test Vector CL0 */
    uint32_t CL0;

    /** Failure insertion value for CL1
        Value for the Failure Insertion Test Vector CL1 */
    uint32_t CL1;

    /** Failure insertion value for CL2
        Value for the Failure Insertion Test Vector CL2 */
    uint32_t CL2;

    /** Failure insertion value for CL3
        Value for the Failure Insertion Test Vector CL3 */
    uint32_t CL3;

    /** Failure insertion value for CMS
        Value for the Failure Insertion Test Vector CMS */
    uint32_t CMS;

    /** Failure insertion value for CSR
        Value for the Failure Insertion Test Vector CSR */
    uint32_t CSR;

    /** Failure insertion value for I0
        Value for the Failure Insertion Test Vector I0 */
    uint32_t I0;

    /** Failure insertion value for I1
        Value for the Failure Insertion Test Vector I1 */
    uint32_t I1;

    /** Failure insertion value for I2
        Value for the Failure Insertion Test Vector I2 */
    uint32_t I2;

    /** Failure insertion value for I3
        Value for the Failure Insertion Test Vector I3 */
    uint32_t I3;

    /** Failure insertion value for RAMT
        Value for the Failure Insertion Test Vector RAMT */
    uint32_t RAMT;
} SDL_PBIST_configNeg;

/**
 *  @addtogroup SDL_OSAL_DATASTRUCT
    @{
 *
 */

typedef struct
{
    uint64_t PBISTRegsHiAddress;
    SDL_pbistRegs *pPBISTRegs;
    uint32_t numPBISTRuns;
    SDL_PBIST_config PBISTConfigRun[PBIST_MAX_NUM_RUNS];
    uint32_t clecInputEvt;
    uint32_t interruptNumber;
    volatile uint32_t doneFlag;
    SDL_PBIST_configNeg PBISTNegConfigRun;
} SDL_pbistInstInfo;

// Data Structures and Types

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the interrupt callback function
 * ----------------------------------------------------------------------------
 */
typedef void (*pSDL_OSAL_interruptCallbackFunction)(uint32_t instanceId);

/** ---------------------------------------------------------------------------
 * @brief   This structure contains the parameters for interrupt registration
 *          through the SDL OSAL interface.
 * ----------------------------------------------------------------------------
 */
typedef struct SDL_OSAL_hwipParams_s
{
    int32_t intNum;
    /**< Interrupt Vector Id */
    pSDL_OSAL_interruptCallbackFunction callback;
    /**< Callback function for the interrupt */
    uintptr_t callbackArg;
    /**< Argument passed to the callback function */
} SDL_OSAL_hwipParams;

typedef void* pSDL_OSAL_hwipHandle;

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the interrupt enable/disable functions
 * ----------------------------------------------------------------------------
 */
typedef int32_t (*pSDL_OSAL_interruptFunction)(int32_t intNum);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the interrupt registration function
 * ----------------------------------------------------------------------------
 */
typedef pSDL_OSAL_hwipHandle (*pSDL_OSAL_registerFunction)(SDL_OSAL_hwipParams *pParams);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the interrupt de-register function
 * ----------------------------------------------------------------------------
 */
typedef int32_t (*pSDL_OSAL_deregisterFunction)(pSDL_OSAL_hwipHandle handle);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the interrupt global disable function
 * ----------------------------------------------------------------------------
 */
typedef int32_t (*pSDL_OSAL_globalDisableInterruptsFunction)(uintptr_t *key);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the interrupt global restore function
 * ----------------------------------------------------------------------------
 */
typedef int32_t (*pSDL_OSAL_globalRestoreInterruptsFunction)(uintptr_t key);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the print function
 * ----------------------------------------------------------------------------
 */
typedef void (*pSDL_OSAL_printFunction)(const char *traceBuf);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for the delay function
 * ----------------------------------------------------------------------------
 */
typedef int32_t (*pSDL_OSAL_delayFunction)(int32_t ndelay);

/** ---------------------------------------------------------------------------
 * @brief   Prototype for address translation function
 */
typedef void* (*pSDL_OSAL_addrTranslateFunction)(uint64_t addr, uint32_t size);

/** ---------------------------------------------------------------------------
 * @brief   This structure contains the pointers for the OSAL interfaces 
 *          provided by the application to SDL_OSAL_init.
 * ----------------------------------------------------------------------------
 */
typedef struct SDL_OSAL_Interface_s
{
    /**< Pointer to interrupt enable function */
    pSDL_OSAL_interruptFunction enableInterrupt;
    /**< Pointer to interrupt disable function */
    pSDL_OSAL_interruptFunction disableInterrupt;
    /**< Pointer to interrupt registration function */
    pSDL_OSAL_registerFunction registerInterrupt;
    /**< Pointer to inerrupt de-register function */
    pSDL_OSAL_deregisterFunction deregisterInterrupt;
    /**< Pointer to global interrupt disable function */
    pSDL_OSAL_globalDisableInterruptsFunction globalDisableInterrupts;
    /**< Pointer to global interrupt enable function */
    pSDL_OSAL_globalRestoreInterruptsFunction globalRestoreInterrupts;
    /**< Pointer to print function */
    pSDL_OSAL_printFunction printFxn;
    /**< Pointer to delay function */
    pSDL_OSAL_delayFunction delay;
    /**< Pointer to address translation function */
    pSDL_OSAL_addrTranslateFunction addrTranslate;
} SDL_OSAL_Interface;

/** @} */

typedef int32_t SDL_ErrType_t;


typedef int32_t (*PBIST_auxInitRestoreFunctionPtr)(bool init);

typedef struct PBIST_TestHandle_s
{
    char               testName[PBIST_INSTANCE_NAME_MAX_LENGTH];
    SDL_PBIST_inst     pbistInst;
    uint32_t           tisciPBISTDeviceId;
    bool               procRstNeeded;
    bool               secondaryCoreNeeded;
    char               coreName[16];
    char               secCoreName[16];
    uint32_t           tisciProcId;
    uint32_t           tisciSecProcId;
    uint32_t           tisciDeviceId;
    uint32_t           tisciSecDeviceId;
    bool               coreCustPwrSeqNeeded;
    uint8_t            numPostPbistToCheck;
    uint32_t           numAuxDevices;
    uint32_t          *auxDeviceIdsP;
    PBIST_auxInitRestoreFunctionPtr auxInitRestoreFunction;
} PBIST_TestHandle_t;

/*************************************************************************************************************
*  Global variable
**************************************************************************************************************/
extern PBIST_TestHandle_t PBIST_TestHandleArray[SDL_PBIST_NUM_INSTANCES];

/*************************************************************************************************************
*  Declare
**************************************************************************************************************/
/**
 *  \brief   This function reads a 32-bit value from a hardware register
 *           and returns the value.
 *
 *  \param   p      Address of the memory mapped hardware register.
 *
 *  \return  Unsigned 32-bit value read from a register.
 */
static inline uint32_t SDL_REG32_RD_RAW(volatile const uint32_t * const p);
static inline uint32_t SDL_REG32_RD_RAW(volatile const uint32_t * const p)
{
    return (*p);
}

/**
 *  \brief PBIST POST result status
 *
 *  This function retrieves the status of the PBIST POST test(s) for all
 *  the supported PBIST instances which have POST support on the device.
 *  PBIST is a feature that is used for self test of the memory regions
 *  in the SoC. Multiple instances may be supported.
 *
 *  If supported, HW POST may execute for one or more instances. This
 *  function will check if the test was executed, and if it was executed,
 *  what was the result of the test.
 *
 *  See the definition of SDL_PBIST_postResult for more information.
 *
 *  \param pResult          [OUT] Pointer to variable to indicate the result
 */
int32_t SDL_PBIST_getPOSTStatus(SDL_PBIST_postResult *pResult);


/**
 *  \brief PBIST self test
 *
 *  This function executes a PBIST self test of the specified type for a
 *  specific PBIST instance. The test types supported are defined in
 *  SDL_PBIST_testType.
 *  PBIST is a feature that is used for self test of the memory regions
 *  in the SoC. Multiple instances may be supported.
 *  Note: The PBIST test is destructive to the cores/IP which are being
 *  tested. Therefore it should be executed from a different core than
 *  the one under test.
 *  It is important to note that execution of PBIST self-tests require
 *  preparation of the IPS under test by bringing them to a certain
 *  power and reset state before executing the test. The application
 *  must take care of this sequence before executing the self-test.
 *  The SDL examples give a reference for the necessary sequences.
 *  The PBIST instances supported are specific to the SoC, and are
 *  defined in the soc-specific header file.
 *
 *  \param instance         [IN]  PBIST instance
 *  \param testType         [IN]  PBIST test type to run.
 *  \param pConfig          [IN]  Pointer to PBIST test configuration (optional)
 *  \param pResult          [OUT] Pointer to variable to indicate result
 *
 *  \return The SDL error code for the API.
 *                                 If pResult is NULL: SDL_EBADARGS
 *                                 If testType is invalid: SDL_EBADARGS
 *                                 If other error happened: SDL_EFAIL
 *                                 Success: SDL_PASS
 */
int32_t SDL_PBIST_selfTest(SDL_PBIST_inst instance, SDL_PBIST_testType testType,
                           const SDL_PBIST_config *pConfig, bool *pResult);

#ifdef __cplusplus
}
#endif

#endif /* FS_RAMTST_PBIST_H_ */
