/**********************************************************************
***********************************************************************
**
**         Filename: UART_Messages.c
**
**           Author: J. Cely
**          Created: 09/12/2017
**
**      Description:
**   This file contains the routines used to initialize the settings and
**   peripherals of the TM4C129.
**
**
** Revision History:
**
**
***********************************************************************
**********************************************************************/

/*===================================================================*/
/*  I N C L U D E    F I L E S                                       */
/*===================================================================*/
//-- Required --
#include <xdc/std.h>
#include <xdc/cfg/global.h>
#include <xdc/runtime/System.h>



//-- Additional --
/* UART Header files */
#include <ti/drv/uart/UART.h>
#include <ti/drv/uart/UART_stdio.h>
#include <ti/starterware/include/hw/hw_uart.h>
#include <ti/starterware/include/hw/hw_types.h>
#include <ti/starterware/include/uart.h>
#include <ti/starterware/include/error.h>
#include <ti/starterware/include/types.h>
#include <ti/drv/uart/soc/UART_soc.h>
#include "UART_Messages.h"
#include "Main.h"
#include "Utility_Routines.h"


/*===================================================================*/
/*  D E F I N I T I O N S                                            */
/*===================================================================*/
#define  OK   1
#define  FAIL 0

#define UART3_BASEADDRESS 0x481A6000

#define UART_REG_MODE_A                  (0x80U)

/** \brief Macro representing Register Access Mode B.*/
#define UART_REG_MODE_B                  (0xBFU)

/** \brief Macro representing Register Access Operational Mode. */
#define UART_REG_MODE_OPER               (0x7FU)

/*===================================================================*/
/*  F U N C T I O N   P R O T O T Y P E S                            */
/*===================================================================*/
void UART_Opening_Msg(void);
//void UART_RAWMeasurements_Msg(void);
void UART_Stop_Msg(void);
void UART_Program_Msg(void);
void UART_StartMeasurements_Msg(void);
void UART_Status_Msg(void);
void UART_Error_Msg(void);
void UART_Program_Status(uint8_t);

static void delay(void);
//void UART_CommTest_Msg(void);
/*===================================================================*/
/*  V A R I A B L E S                                                */
/*===================================================================*/
//UART_Params   uart_params;


/*===================================================================*/
/*  T Y P E   D E C L A R A T I O N S                                */
/*===================================================================*/
// See Typedefs.h


/*===================================================================*/
/*  M A C R O S                                                      */
/*===================================================================*/


/*===================================================================*/
/*  M A J O R    R O U T I N E S                                     */
/*===================================================================*/

/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_CheckForMessage
**
**              Author: Joe Cely
**             Created: 5/7/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: TRUE - Data in RX FIFO
**                      FALSE - No Data
**
**         Description:
** This routine checks the RXFIFO bit in the LSR_UART register to determine
** if it has at least one character.
**
***********************************************************************/
uint32_t UART_CheckForMessage(void)
{
    uint16_t  UartRXFifo_FULL=0;
    uint32_t   UARTCommandRXed = 0;
    int32_t   CommandLength = 1;


    UartRXFifo_FULL = HW_RD_FIELD16(UART3_BASEADDRESS + UART_LSR_CIR, UART_IIR_IT_TYPE);;

    if(UartRXFifo_FULL)
       {
       UART_gets(&UARTCommandRXed,CommandLength);
       g_dwEventHandler.s_dwEventHandler.EVENT_NewUARTMessage = 1;
       }

    return UARTCommandRXed;
}


/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_CheckForMessage
**
**              Author: Joe Cely
**             Created: 5/7/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: TRUE - Data in RX FIFO
**                      FALSE - No Data
**
**         Description:
** This routine checks the RXFIFO bit in the LSR_UART register to determine
** if it has at least one character.
**
***********************************************************************/
void UART_Message(uint32_t dwMessage)
{

   //--  Determine which Message to Display
   switch(dwMessage)
      {
      case UART_OPENING_MSG:
          UART_Opening_Msg();
          break;

//-- Operating Messages
      case UART_START_MSG:
          UART_StartMeasurements_Msg();
          break;

      case UART_MEASUREMENT_MSG:
          //UART_RAWMeasurements_Msg();
          break;

      case UART_STOP_MSG:
          UART_Stop_Msg();
          break;

      case UART_COMMTEST_MSG:
         //UART_CommTest_Msg();
          break;

      case UART_VERSION_MSG:
          //UART_Version_Msg();
          break;


//-- Programming
      case UART_PROGRAM_MSG:
          UART_Program_Msg();
          break;

      case UART_PROGRAM_OK:
          UART_Program_Status(OK);
          break;

      case UART_PROGRAM_FAIL:
          UART_Program_Status(FAIL);
          break;

      case UART_STATUS_MSG:
          UART_Status_Msg();
          break;

      case UART_ERROR_MSG:
          UART_Error_Msg();
          break;


      default:
          break;
      }

}


/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_Opening_Msg
**
**              Author: Joe Cely
**             Created: 9/13/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine displays the Introduction message on the UART.
**
***********************************************************************/
void UART_Opening_Msg(void)
{
    UART_printf("****************************\n\r`");
    UART_printf("* Tire Detection UART App  *\n\r`");
    UART_printf("*                          *\n\r`");
    UART_printf("*   P - Prog all Devices   *\n\r`");
    UART_printf("*   S - Start measurements *\n\r`");
    UART_printf("*   #- Stop Measurements   *\n\r`");
    UART_printf("*   V- Report Version      *\n\r`");
    UART_printf("*   A- Report Status       *\n\r`");
    UART_printf("*   C- Communication Test  *\n\r`");
    UART_printf("****************************\n\r`");
    UART_printf("Enter Command:\n\r`");
}


/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_RAWMeasurements_Msg
**
**              Author: Joe Cely
**             Created: 9/13/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine is used to display the RAW Measurements from the 8
** LDC1101 devices.
**
***********************************************************************/

/*
void UART_RAWMeasurements_Msg(void)
{
    UARTSendRAWData((uint32_t *) &g_aryMISO_SENSOR1[3],g_dwNumberOfDataBits);
    UARTSendRAWData((uint32_t *) &g_aryMISO_SENSOR2[3],g_dwNumberOfDataBits);
    UARTSendRAWData((uint32_t *) &g_aryMISO_SENSOR3[3],g_dwNumberOfDataBits);
    UARTSendRAWData((uint32_t *) &g_aryMISO_SENSOR4[3],g_dwNumberOfDataBits);
}
*/

/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_CommTest_Msg
**
**              Author: Joe Cely
**             Created: 9/13/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine compares the 10 bytes in the MISO_SENSOR arrays to the
** 1st 10 values in the LDC1101Settings structure.  If the 10 bytes match,
** a "PASS" message is sent, if they don't, a "FAIL" Message is sent.
**
***********************************************************************/
/*
void UART_CommTest_Msg(void)
{
    const uint32_t* ptrSettings;
    volatile uint32_t  dwLoopCnt;
    volatile uint32_t  dwSettingsValue;
    uint32_t  dwError = NO_ERROR;

    //-- Sensor 1 Validation
    ptrSettings = &LDC1101Settings_1[3];
    for(dwLoopCnt=0; dwLoopCnt < 10; dwLoopCnt++)
       {
        dwSettingsValue = *ptrSettings;
        ptrSettings = ptrSettings + 2;
         if(g_aryMISO_SENSOR1[dwLoopCnt] != dwSettingsValue)
            {
           dwError = ERR_TEST_ERROR;
            }
       }

    if(dwError == NO_ERROR)
       {
        UARTSend((uint8_t *)"SENSOR1 Comm:  PASS\n\r`");
       }
    else
       {
        UARTSend((uint8_t *)"SENSOR1 Comm:  FAIL\n\r`");
       }


    //-- Sensor 2 Validation
    ptrSettings = &LDC1101Settings_1[3];
    for(dwLoopCnt=0; dwLoopCnt < 10; dwLoopCnt++)
       {
        dwSettingsValue = *ptrSettings;
        ptrSettings = ptrSettings + 2;
         if(g_aryMISO_SENSOR2[dwLoopCnt] != dwSettingsValue)
            {
           dwError = ERR_TEST_ERROR;
            }
       }

    if(dwError == NO_ERROR)
       {
        UARTSend((uint8_t *)"SENSOR2 Comm:  PASS\n\r`");
       }
    else
       {
        UARTSend((uint8_t *)"SENSOR2 Comm:  FAIL\n\r`");
       }


    //-- Sensor 3 Validation
    ptrSettings = &LDC1101Settings_1[3];
    for(dwLoopCnt=0; dwLoopCnt < 10; dwLoopCnt++)
       {
        dwSettingsValue = *ptrSettings;
        ptrSettings = ptrSettings + 2;
         if(g_aryMISO_SENSOR3[dwLoopCnt] != dwSettingsValue)
            {
           dwError = ERR_TEST_ERROR;
            }
       }

    if(dwError == NO_ERROR)
       {
        UARTSend((uint8_t *)"SENSOR3 Comm:  PASS\n\r`");
       }
    else
       {
        UARTSend((uint8_t *)"SENSOR3 Comm:  FAIL\n\r`");
       }


    //-- Sensor 4 Validation
    ptrSettings = &LDC1101Settings_1[3];
    for(dwLoopCnt=0; dwLoopCnt < 10; dwLoopCnt++)
       {
        dwSettingsValue = *ptrSettings;
        ptrSettings = ptrSettings + 2;
         if(g_aryMISO_SENSOR4[dwLoopCnt] != dwSettingsValue)
            {
           dwError = ERR_TEST_ERROR;
            }
       }

    if(dwError == NO_ERROR)
       {
        UARTSend((uint8_t *)"SENSOR4 Comm:  PASS\n\r`");
       }
    else
       {
        UARTSend((uint8_t *)"SENSOR4 Comm:  FAIL\n\r`");
       }

}

*/
/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_Stop_Msg
**
**              Author: Joe Cely
**             Created: 9/13/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine is used to inform the user that the Measurement Mode
** has stopped.
**
***********************************************************************/
void UART_Stop_Msg(void)
{
    UART_printf("== MEASUREMENTS STOPPED\n\r`");
    UART_printf("\n\r`");
    StopSensorRefClk();
    delay();

}

/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_Program_Msg
**
**              Author: Joe Cely
**             Created: 9/18/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine displays the Programming Message indicating that the 8
** devices are being programmed.
**
***********************************************************************/
void UART_Program_Msg(void)
{
    UART_printf("== PROGRAMMING THE DEVICES == \n\r`");
#if DIRECTIVE_SENSOR_TYPE == SENSOR1
    UART_printf(" 35mm Disk Profile \n\r'");
#elif DIRECTIVE_SENSOR_TYPE == SENSOR2
    UART_printf(" .8 inch Disk Profile \n\r'");
#elif DIRECTIVE_SENSOR_TYPE == SENSOR3
    UART_printf(" 14mm Disk Profile \n\r'");
#endif
    UART_printf("\n\r`");

}


/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_Program_Status
**
**              Author: Joe Cely
**             Created: 9/21/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine displays the Programming Message indicating that the 8
** devices are being programmed.
**
***********************************************************************/
void UART_Program_Status(uint8_t byMsg)
{
    if(byMsg)
       {
        UART_printf("== PROGRAMMING SUCCESSFUL == \n\r`");
       }
    else
       {
        UART_printf("== PROGRAMMING ERROR == \n\r`");
       }

    UART_printf("\n\r`");
}


/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_StartMeasurements_Msg
**
**              Author: Joe Cely
**             Created: 9/18/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine displays the message indicating that measurements from the LDC1101
** will begin collecting.
**
***********************************************************************/
void UART_StartMeasurements_Msg(void)
{

    UART_printf("== MEASUREMENTS STARTING\n\r`");
    UART_printf("\n\r`");
    UART_printf("== Reference Clock OK\n\r");
    StartSensorRefClk();
    delay();

}



/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_Status_Msg
**
**              Author: Joe Cely
**             Created: 9/18/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine displays the message indicating that measurements from the LDC1101
** will begin collecting.
**
***********************************************************************/
void UART_Status_Msg(void)
{
    UART_printf("****************************\n\r`");
    UART_printf("      SENSOR STATUS         \n\r`");
    UART_printf("      =============         \n\r`");
    UART_printf(" Sensor 1:  \n\r`");
    UART_printf(" Sensor 2:  \n\r`");
    UART_printf(" Sensor 3:  \n\r`");
    UART_printf(" Sensor 4:  \n\r`");
    UART_printf(" Sensor 5:  \n\r`");
    UART_printf(" Sensor 6:  \n\r`");
    UART_printf(" Sensor 7:  \n\r`");
    UART_printf(" Sensor 8:  \n\r`");
    UART_printf("****************************\n\r`");
    UART_printf("\n\r`");
}


/***********************************************************************
**
**            Filename: UART_Messages.c
**        Routine Name: UART_Error_Msg
**
**              Author: Joe Cely
**             Created: 9/18/2017
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This routine displays the Error Message meaning that a command was issued
** but is not defined.
**
***********************************************************************/
void UART_Error_Msg(void)
{
    UART_printf("!!!!!!!!  ERROR   !!!!!!!!!!\n\r`");
    UART_printf("\n\r`");
    UART_printf("Command Not Recognized\n\r`");
    UART_printf("\n\r`");
}


void delay(void)
{
    uint32_t  i;

    for(i=0;i<500000;i++)
    {

    }
}
