/**********************************************************************
***********************************************************************
**
**         Filename: TD_Main.c
**
**           Author: J. Cely
**          Created: 05/3/2018
**
**      Description:
**   This file contains all of the SYSBIOS modules that are used in this
**   application.  The SYSBIOS modules are all created statically in the
**   TireDetecSYSBIOSConfig.cfg file.  The following modules are used.
**
**   SYSBIOS:CLOCK - Based on MEASUREMENT_INTERVAL
**   Take_Measurement()
**
**   SYSBIOS:STARTUP
**   TD_SW_Initialization()
**
**   SYSBIOS:TASKS
**   ProcessNewMeasurement() - Controlled by TDEvent.NewMeasurement
**   EtherNet_IP_Comm() - Controlled by TDEvent.NewEtherNetMsg
**   SendUARTMessages() - Controlled by TDEvent.SendUARTMessage
**   ReceiveUARTMessage() - Controlled by TDEvent.NewUARTMessage
**
**   SYSBIOS.EVENTS
**   TDEvents - NewMeasurement
**              NewEtherNetMsg
**              SendUARTMessage
**              NewUARTMessage
**
**   SYSBIOS.HWI
**   UARTReceiveISR
**
**
** Revision History:
**
**
***********************************************************************
**********************************************************************/

/*===================================================================*/
/*  I N C L U D E    F I L E S                                       */
/*===================================================================*/

/* XDCtools Header files */
#include <xdc/std.h>
#include <xdc/cfg/global.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Error.h>


/* BIOS Header files */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/knl/Idle.h>

#include <stdio.h>

/* TI-RTOS Driver Header Files*/
#include <ti/drv/gpio/GPIO.h>
#include <ti/drv/gpio/soc/GPIO_soc.h>
#include <ti/drv/gpio/test/led_blink/src/GPIO_log.h>
#include <ti/board/board.h>

/* UART Header files */
//#include <ti/drv/uart/UART.h>
//#include <ti/drv/uart/UART_stdio.h>

/* Tire Detect Custom Board Header Files */
#include "TireDetectGPIOConfig.h"
#include "Initialization.h"
#include "UART_Messages.h"
#include "Main.h"
#include "Utility_Routines.h"


/*===================================================================*/
/*  D E F I N I T I O N S                                            */
/*===================================================================*/


/*===================================================================*/
/*  F U N C T I O N   P R O T O T Y P E S                            */
/*===================================================================*/
static void delay(void);


/*===================================================================*/
/*  V A R I A B L E S                                                */
/*===================================================================*/
volatile u_EventHandler_t  g_dwEventHandler;


/*===================================================================*/
/*  M A J O R    R O U T I N E S                                     */
/*===================================================================*/

/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: Main()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is the entry point for the Tire Detection Application
**
***********************************************************************/

int main(void)
{
    /* Call board init functions */
    TD_HW_Initialization();

/*    while(1)
    {
        UART_Message(UART_OPENING_MSG);
        StartSensorRefClk();
        delay();
        StopSensorRefClk();
        delay();
    }

*/

    /* Start BIOS */
    BIOS_start();
    return (0);
}


void delay(void)
{
    uint32_t  i;

    for(i=0;i<500000;i++)
    {

    }
}
/*---------------------------------------------------------------------
                             STARTUP
 ----------------------------------------------------------------------*/
/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: TD_SW_Initialization()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is the SYSBIOS.Startup module.  This routine is entered directly
** after the BIOS_start() routine is called and runs only once.
**
** This routine initializes all of the SW parameters and some of the
** peripherals.
**
***********************************************************************/
void TD_SW_Initialization(void)
{
    TD_SW_Init_Routine();

    UART_Message(UART_OPENING_MSG);

    /* Write high to gpio pin to control LED1 */
    GPIO_write((USER_LED1), GPIO_PIN_VAL_LOW);
    GPIO_write((USER_LED2), GPIO_PIN_VAL_LOW);
    GPIO_write((USER_LED3), GPIO_PIN_VAL_LOW);
    GPIO_write((USER_LED4), GPIO_PIN_VAL_HIGH);


}


/*---------------------------------------------------------------------
                             TASKS
 ----------------------------------------------------------------------*/

/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: EtherNetIP_Comm()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is a SYSBIOS.TASK module.  This routine is controlled by
** TDEvent.NewEtherNetMsg and is used to process any new EtherNet
** messages
**
** Handle = Communications
** Priority = 12 (2nd Highest)
***********************************************************************/
void EtherNetIP_Comm(UArg arg0, UArg arg1)
{

     while(1)
     {
         GPIO_toggle(USER_LED1);
         if(g_dwEventHandler.s_dwEventHandler.EVENT_NewEtherNetMsg)
         {
             UART_printf("\n TASK:  EtherNetIP_Comm \n");
             g_dwEventHandler.s_dwEventHandler.EVENT_NewEtherNetMsg=0;

         }
         Task_sleep(10);
     }

}

/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: ProcessNewMeasurement()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is a SYSBIOS.TASK module.  This routine is controlled by
** TDEvent.NewMeasurement and is used to process the new measurement
** made in the Take_Measurement() Timer Routine.
**
** Handle = NewMeas
** Priority = 15 (Highest)
***********************************************************************/
void ProcessNewMeasurement(UArg arg0, UArg arg1)
{

    while(1)
    {
        GPIO_toggle(USER_LED2);
        if(g_dwEventHandler.s_dwEventHandler.EVENT_NewMeasurement)
        {
            UART_printf("\n TASK:  ProcessNewMeasurement \n");
            g_dwEventHandler.s_dwEventHandler.EVENT_NewMeasurement=0;

            g_dwEventHandler.s_dwEventHandler.EVENT_SendUARTMessage=1;
            g_dwEventHandler.s_dwEventHandler.EVENT_NewEtherNetMsg=1;
        }
        Task_sleep(10);
    }
}


/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: SendUARTMessageFxn()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is a SYSBIOS.TASK module.  This routine is controlled by
** TDEvent.SendUARTMessage and is used to send new measurements
** out of UART0.
**
** Handle = UART_TX
** Priority = 1 (Lowest)
***********************************************************************/
void SendUARTMessageFxn(UArg arg0, UArg arg1)
{

    while(1)
    {
        GPIO_toggle(USER_LED3);
        if(g_dwEventHandler.s_dwEventHandler.EVENT_SendUARTMessage)
        {
            UART_printf("\n TASK:  SendUARTMessageFxn \n");
            g_dwEventHandler.s_dwEventHandler.EVENT_SendUARTMessage=0;
        }
        Task_sleep(10);
    }
}


/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: ReceiveUARTMessage()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is a SYSBIOS.TASK module.  This routine is controlled by
** TDEvent.NewUARTMessage and is used to process any new UART messages
** received from UART0.
**
** Handle = UART_RX
** Priority = 1 (Lowest)
***********************************************************************/
void ReceiveUARTMessage(UArg arg0, UArg arg1)
{
char  NewMessage = 0;

    while(1)
    {

        NewMessage = UART_CheckForMessage();
        GPIO_toggle(USER_LED4);
        if(g_dwEventHandler.s_dwEventHandler.EVENT_NewUARTMessage)
        {
            UART_printf("\n TASK:  ReceiveUARTMessage \n");
            UART_Message(NewMessage);
            g_dwEventHandler.s_dwEventHandler.EVENT_NewUARTMessage=0;
        }
        Task_sleep(10);
    }
}




/*---------------------------------------------------------------------
                            CLOCKS
 ----------------------------------------------------------------------*/
/***********************************************************************
**
**            Filename: TD_Main.c
**        Routine Name: Take_Measurement()
**
**              Author: Joe Cely
**             Created: 5/3/2018
**
**        Last Edit By:
**           Last Edit:
**
**  Function Arguments: None
**    Function Returns: None
**
**         Description:
** This is a SYSBIOS.CLOCK module.  The period of this routine is controlled
** by the MEASUREMENT_INTERVAL variable.
** Handle = SensorRead
**
***********************************************************************/
void Take_Measurement(UArg arg0)
{
    GPIO_log("\n Measurement Timer \n");
    GPIO_toggle(USER_LED1);
    GPIO_toggle(USER_LED3);
    g_dwEventHandler.s_dwEventHandler.EVENT_NewMeasurement=1;
}



/*---------------------------------------------------------------------
                         MAIN UTILITIES
 ----------------------------------------------------------------------*/

/*--------------------------------------
 *    Function:  Start/StopSensor Clock
 *
 *    These functions are external routine
 *    that are used by other modules to start
 *    and stop the Sensor Clock.
 *--------------------------------------*/
void SensorClockStart(void)
{
    Clock_start(SensorRead);
}

void SensorClockStop(void)
{
    Clock_stop(SensorRead);
}
