/***************************************************************************************
 * MAIN.C
 *
 * Description: main source file for PRU development
 *
 * Rafael de Souza
 * (C) 2015 Texas Instruments, Inc
 * 
 * Built with Code Composer Studio v6
 **************************************************************************************/

#include <stdint.h>
#include <pru_cfg.h>
#include <pru_iep.h>
#include <pru_intc.h>
#include <rsc_types.h>
#include <pru_virtqueue.h>
#include <pru_rpmsg.h>
#include <sys_mailbox.h>

#include "resource_table_empty.h"
//#include "PRU_audio.h"


/* PRU0 is mailbox module user 1 */
#define MB_USER							1
/* CROSSBAR will map MBX3 user 1 event to Int Number 60 */
#define MB_INT_NUMBER					60

/* Host-0 Interrupt sets bit 30 in register R31 */
#define HOST_INT						0x40000000

/* The mailboxes used for RPMsg are defined in the Linux device tree
 * PRU0 uses mailboxes 0 (From ARM) and 1 (To ARM)
 * PRU1 uses mailboxes 2 (From ARM) and 3 (To ARM)
 */
#define MB_TO_ARM_HOST					1
#define MB_FROM_ARM_HOST				0

/*
 * Using the name 'rpmsg-pru' will probe the rpmsg_pru driver found
 * at linux-x.y.z/drivers/rpmsg/rpmsg_pru.c
 */
#define CHAN_NAME						"rpmsg-pru"
#define CHAN_DESC						"Channel 32"
#define CHAN_PORT						32

/*
 * Need to program the crossbar to bring the correct events into the
 * PRUSS INTC
 */
#define CTRL_CORE_PRUSS2_IRQ_60_61		*(volatile unsigned int *) 0x4A002940
#define MAILBOX4_IRQ_USER1				246

/*
 * Used to make sure the Linux drivers are ready for RPMsg communication
 * Found at linux-x.y.z/include/uapi/linux/virtio_config.h
 */
#define VIRTIO_CONFIG_S_DRIVER_OK		4


#define RUN	1
#define	STOP	2
#define LOAD_START	3
#define LOAD_STOP	4

uint8_t payload[RPMSG_BUF_SIZE];
uint8_t state = 0;

struct  sequ {
	uint16_t duration;
	uint16_t y_scan;
	uint8_t led;
}__attribute__((__packed__));


//struct  sequ {
//	uint16_t duration;
//	uint16_t y_scan;
//	union led{
//		uint8_t led;
//		uint8_t red:1;
//		uint8_t blue:1;
//		uint8_t green:1;
//		uint8_t nm:1;
//		uint8_t upper:1;
//		uint8_t lower:1;
//		uint8_t trigger:1;
//		uint8_t frame_start:1;
//	}led;
//}__attribute__((__packed__));

#pragma DATA_SECTION(sequencer, ".sequencer")
struct sequ sequencer[1024];
//struct sequ payload[90];
struct sequ *temp_seq = &sequencer[0];

volatile register uint32_t __R30;
volatile register uint32_t __R31;

#define PRU_IEP_EVT   7
#define CMP0_VAL        0x186A0

uint16_t row = 0;
void iep_timer_config(uint32_t cmp_val){

        CT_IEP.GLB_CFG_bit.CNT_EN = 0;             /* Disable counter */
        CT_IEP.CNT = 0xFFFFFFFF;               	   /* Reset Count register */
        CT_IEP.GLB_STS_bit.CNT_OVF = 0x1;          /* Clear overflow status register */
        CT_IEP.CMP0 = cmp_val*200;                    /* Set compare0 value */
        CT_IEP.CMP_STS_bit.CMP_HIT = 0xFF;         /* Clear compare status */
        CT_IEP.COMPEN_bit.COMPEN_CNT = 0x0;        /* Disable compensation */
        CT_IEP.CMP_CFG_bit.CMP0_RST_CNT_EN = 0x1;  /* Disable CMP0 and reset on event */
        CT_IEP.CMP_CFG_bit.CMP_EN = 0x1;
        CT_IEP.GLB_CFG_bit.DEFAULT_INC = 0x1;		/* Configure incr value */
        CT_INTC.SECR0 = 0xFFFFFFFF;                     /* Clear the status of all interrupts */
        CT_INTC.SECR1 = 0xFFFFFFFF;
        CT_IEP.GLB_CFG_bit.CNT_EN = 0x1;
}




void intc_config(void){

	/* Set interrupts 0 to 31 to active high */
	CT_INTC.SIPR0 = 0xFFFFFFFF;

	/* Clear mapping for sys evt 4 to 7 */
	CT_INTC.CMR1 = 0x00000000;

	/* Set event 7 to channel 0 */
	CT_INTC.CMR1_bit.CH_MAP_7 = 0x0;

	/* Map channel 0 to host 0 */
	CT_INTC.HMR0_bit.HINT_MAP_0 = 0x0;

	/* Enable system event (7th bit = sys interrupt 7) */
	CT_INTC.ESR0 = 0x80;

	/* Enable host interrupt 0 */
	CT_INTC.HIER = 0x1;

	/* Globally enable interrupts */
	CT_INTC.GER = 0x1;
}


void init_virt_channel(struct pru_rpmsg_transport *trnsprt) {
	uint32_t regValue;
	volatile uint8_t *status;

	/* allow OCP master port access by the PRU so the PRU can read external memories */
	CT_CFG.SYSCFG_bit.STANDBY_INIT = 0;

	/* need to program the CROSSBAR to map MBX3 User1 event to PRUSS INTC event 60 */
	regValue = CTRL_CORE_PRUSS2_IRQ_60_61;
	regValue &= 0xFFFFFE00;
	CTRL_CORE_PRUSS2_IRQ_60_61 = regValue | MAILBOX4_IRQ_USER1;

	/* clear the status of event MB_INT_NUMBER (the mailbox event) and enable the mailbox event */
	CT_INTC.SICR_bit.STATUS_CLR_INDEX = MB_INT_NUMBER;
	MBX4.IRQ[MB_USER].ENABLE_SET |= 1 << (MB_FROM_ARM_HOST * 2);

	/* Make sure the Linux drivers are ready for RPMsg communication */
	status = &resourceTable.rpmsg_vdev.status;
	while (!(*status & VIRTIO_CONFIG_S_DRIVER_OK));

	/* Initialize pru_virtqueue corresponding to vring0 (PRU to ARM Host direction) */
	pru_virtqueue_init(&(*trnsprt).virtqueue0, &resourceTable.rpmsg_vring0, &MBX4.MESSAGE[MB_TO_ARM_HOST], &MBX4.MESSAGE[MB_FROM_ARM_HOST]);

	/* Initialize pru_virtqueue corresponding to vring1 (ARM Host to PRU direction) */
	pru_virtqueue_init(&(*trnsprt).virtqueue1, &resourceTable.rpmsg_vring1, &MBX4.MESSAGE[MB_TO_ARM_HOST], &MBX4.MESSAGE[MB_FROM_ARM_HOST]);

	/* Create the RPMsg channel between the PRU and ARM user space using the transport structure. */
	while(pru_rpmsg_channel(RPMSG_NS_CREATE, trnsprt, CHAN_NAME, CHAN_DESC, CHAN_PORT) != PRU_RPMSG_SUCCESS);
}

void verify_command(struct pru_rpmsg_transport *trnsprt) {

	uint16_t src, dst, len;

	/* Check bit 30 of register R31 to see if the mailbox interrupt has occurred */
	if(__R31 & HOST_INT){
		/* Clear the mailbox interrupt */
		MBX4.IRQ[MB_USER].STATUS_CLR |= 1 << (MB_FROM_ARM_HOST * 2);
		/* Clear the event status, event MB_INT_NUMBER corresponds to the mailbox interrupt */
		CT_INTC.SICR_bit.STATUS_CLR_INDEX = MB_INT_NUMBER;
		/* Use a while loop to read all of the current messages in the mailbox */
		/* Check to see if the message corresponds to a receive event for the PRU */
		if(MBX4.MESSAGE[MB_FROM_ARM_HOST] == 1){
			/* Receive the message */
			if(pru_rpmsg_receive(trnsprt, &src, &dst, payload, &len) == PRU_RPMSG_SUCCESS){
				if(state != LOAD_START) {
					if(len >= 2 && payload[0] == 'O' && payload[1] == 'N')
						state = RUN;
					if(len >= 2 && payload[0] == 'O' && payload[1] == 'F')
						state = STOP;
					if(len >= 2 && payload[0] == 'L' && payload[3] == 'D')
						state = LOAD_START;
				}
				else {
					/*Store Data*/
					memcpy(temp_seq, &payload, len);
					if(payload[len - 5] == 0) {
						temp_seq = &sequencer[0];
						state = LOAD_STOP;
					}
					else
						temp_seq += len/5;

					row += len/5;
				}
		//		pru_rpmsg_send(&transport, dst, src, payload, len);


			}
		}
	}

}

uint8_t led_toggle(struct sequ led){

	__R30 = led.led;

	iep_timer_config(led.duration);

	/* Poll until R31.30 is set */
//	do{
		 while((__R31 & 0x40000000) == 0){ }
//	/* Verify that the IEP is the source of the interrupt */
//	} while (CT_INTC.HIPIR0 != PRU_IEP_EVT);

	if(CT_INTC.HIPIR0 != PRU_IEP_EVT)
		return 1;
	/* Clear Compare0 status */
	CT_IEP.GLB_STS = 0x1;

	__delay_cycles(2);

	/* Clear the status of the interrupt */
	CT_INTC.SECR0 = (1 << PRU_IEP_EVT);

	return 0;
	//__delay_cycle(delay);
}

void stop_sequencer() {
	//dac_cmd(0x00);
	__R30 = 0x00;
}
int main(void)
 {
	int i;

	struct pru_rpmsg_transport transport;

	init_virt_channel(&transport);
	intc_config();
	while(1){

		verify_command(&transport);

		if(state == RUN) {
			for(i = 0; i < (row -1); i++) {
	//			dac_cmd(sequencer[i].y_scan);
				if(led_toggle(sequencer[i]))
					verify_command(&transport);
				if(state != RUN)
					break;
				//TODO: What to do if LOAD command comes after RUN
				}
			}
		if(state == STOP)
			stop_sequencer();

	}

	return 0;
}
