//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//
//---------------------------------------------------------------------------------------------------------------------
#include <stdint.h>
#include <ti/csl/hw_types.h>
#include <ti/csl/csl_types.h>
#include "lld_hsi2c_edma.h"
#include <ti/drv/stw_lld/i2clld/lld_hsi2c.h>
#include <ti/csl/csl_edma.h>
#include <ti/csl/arch/csl_arch.h>
#include <ti/csl/soc.h>
//---------------------------------------------------------------------------------------------------------------------
/* Edma OPT Field specific defines */
#define EDMA3CC_U_BASE      (SOC_EDMA_TPCC_BASE_VIRT)
#define EDMA3CC_IPR_OFFSET  ((uint32_t) EDMA_TPCC_IPR)
#define EDMA3CC_IPRH_OFFSET ((uint32_t) EDMA_TPCC_IPRH)

#define EDMA3CC_ICR_OFFSET  ((uint32_t) EDMA_TPCC_ICR)
#define EDMA3CC_ICRH_OFFSET ((uint32_t) EDMA_TPCC_ICRH)

#define LLD_I2C_EDMA_OPT_SYNCDIM_SHIFT      ((uint32_t) 0x00000002U)
#define LLD_I2C_EDMA_OPT_TCC_MASK           ((uint32_t) 0x0003F000U)
#define LLD_I2C_EDMA_OPT_TCC_SHIFT          ((uint32_t) 0x0000000CU)
#define LLD_I2C_EDMA_OPT_ITCINTEN_SHIFT     ((uint32_t) 0x00000015U)
#define LLD_I2C_EDMA_OPT_TCINTEN_SHIFT      ((uint32_t) 0x00000014U)
//---------------------------------------------------------------------------------------------------------------------
static lldHsi2cEdmaCallback callbackFn;
//---------------------------------------------------------------------------------------------------------------------
void EDMA_callbackFn(void *arg)
{
    uint32_t regVal, tcc, i;
    LLD_hsi2cEdmaStatus i2cEdmaErrStatus = EDMA_TRANSFER_SUCCESS;

    regVal = HW_RD_REG32(EDMA3CC_U_BASE + EDMA3CC_IPR_OFFSET);
    for (i = 0; i < 4U; i++)
    {
        if ((uint32_t) 0U !=
            (regVal & ((uint32_t) 1U << (LLD_I2C_EDMA_CHA_I2CTXEVT0 + i))))
        {
            tcc = LLD_I2C_EDMA_CHA_I2CTXEVT0 + i;

            /*I2C LLD EDMA call back function should be called here*/
            callbackFn(tcc, i2cEdmaErrStatus, arg);

            HW_WR_REG32(
                EDMA3CC_U_BASE + EDMA3CC_ICR_OFFSET,
                (uint32_t) 1U << (LLD_I2C_EDMA_CHA_I2CTXEVT0 + i));
        }
    }
}
//---------------------------------------------------------------------------------------------------------------------
LLD_hsi2cEdmaStatus EDMA_requestChannel(hsI2cHandle          handle,
                                        uint32_t            *i2cEventNo,
                                        lldHsi2cEdmaCallback i2cCallback,
                                        void                *edmaParam)
{
    uint32_t chanNum = *i2cEventNo;
    LLD_hsi2cEdmaParam *edmaParams    = (LLD_hsi2cEdmaParam *) edmaParam;
    LLD_hsi2cEdmaStatus i2cEdmaStatus = EDMA_TRANSFER_SUCCESS;

    callbackFn = i2cCallback;

    *i2cEventNo = chanNum;

    if (TRUE !=
        EDMA3RequestChannel(edmaParams->baseAddr,
                            EDMA3_CHANNEL_TYPE_DMA,
                            *i2cEventNo,
                            *i2cEventNo,
                            edmaParams->eventQueue))
    {
        i2cEdmaStatus = EDMA_TRANSFER_FAIL;
    }

    return i2cEdmaStatus;
}

LLD_hsi2cEdmaStatus EDMA_freeChannel(uint32_t i2cEventNo, void *edmaParam)
{
    LLD_hsi2cEdmaStatus status     = EDMA_TRANSFER_SUCCESS;
    LLD_hsi2cEdmaParam *edmaParams = (LLD_hsi2cEdmaParam *) edmaParam;

    if (TRUE !=
        (EDMA3FreeChannel(edmaParams->baseAddr,
                          EDMA3_CHANNEL_TYPE_DMA,
                          i2cEventNo,
                          EDMA3_TRIG_MODE_EVENT,
                          i2cEventNo,
                          edmaParams->eventQueue)))
    {
        status = EDMA_TRANSFER_FAIL;
    }

    return status;
}

LLD_hsi2cEdmaStatus EDMA_disableLogicalChannel(uint32_t tcc, void *edmaParam)
{
    LLD_hsi2cEdmaStatus status     = EDMA_TRANSFER_SUCCESS;
    LLD_hsi2cEdmaParam *edmaParams = (LLD_hsi2cEdmaParam *) edmaParam;

    EDMA3DisableDmaEvt(edmaParams->baseAddr, tcc);

    return status;
}

LLD_hsi2cEdmaStatus EDMA_clearErrorBits(uint32_t tcc, void *edmaParam)
{
    LLD_hsi2cEdmaStatus status     = EDMA_TRANSFER_SUCCESS;
    LLD_hsi2cEdmaParam *edmaParams = (LLD_hsi2cEdmaParam *) edmaParam;

    EDMA3ClearErrorBits(edmaParams->baseAddr, tcc, edmaParams->eventQueue);

    return status;
}

LLD_hsi2cEdmaStatus EDMA_setPaRAM(const void *srcPtr, const void *dstPtr,
                                  uint32_t rdWrflag,
                                  uint32_t chan, uint32_t bufLen,
                                  void *edmaParam,
                                  uint8_t fifoThreshold)
{
    uint16_t            srcIndex   = 0U;
    uint16_t            dstIndex   = 1U;
    LLD_hsi2cEdmaStatus status     = EDMA_TRANSFER_SUCCESS;
    LLD_hsi2cEdmaParam *edmaParams = (LLD_hsi2cEdmaParam *) edmaParam;
    EDMA3CCPaRAMEntry   i2cEdmaParam;

    i2cEdmaParam.opt = 0U;
    if (LLD_HSI2C_WRITE == rdWrflag)
    {
        srcIndex = 1U;
        dstIndex = 0U;

        i2cEdmaParam.srcCIdx  = fifoThreshold;
        i2cEdmaParam.destCIdx = 0U;
    }
    else
    {
        i2cEdmaParam.srcCIdx  = 0U;
        i2cEdmaParam.destCIdx = fifoThreshold;
    }

    i2cEdmaParam.srcAddr  = (uint32_t) srcPtr;
    i2cEdmaParam.destAddr = (uint32_t) dstPtr;
    i2cEdmaParam.aCnt     = 0x1U;

    if (bufLen >= (uint32_t) fifoThreshold)
    {
        i2cEdmaParam.bCnt =
            ((uint16_t) ((uint32_t) 0xFFFFU & (uint32_t) fifoThreshold));
        i2cEdmaParam.cCnt =
            ((uint16_t) ((uint32_t) 0xFFFFU &
                         (bufLen / (uint32_t) fifoThreshold)));
    }
    else
    {
        i2cEdmaParam.bCnt = ((uint16_t) 0xFFFFU & bufLen);
        i2cEdmaParam.cCnt = 1U;
    }

    i2cEdmaParam.srcBIdx  = srcIndex;
    i2cEdmaParam.destBIdx = dstIndex;

    i2cEdmaParam.linkAddr   = 0xFFFFU;
    i2cEdmaParam.bCntReload = 0U;
    i2cEdmaParam.opt       &= (uint32_t) 0xFFFFFFFCU;

    /* FIFO width is 8 bit                                                */
    i2cEdmaParam.opt &= (uint32_t) 0xFFFFF8FFU;

    /* EDMA3_DRV_SYNC_AB                                                  */
    i2cEdmaParam.opt |= (uint32_t) 0x04U;

    /* EDMA3_DRV_OPT_FIELD_TCINTEN                                        */
    i2cEdmaParam.opt |= ((uint32_t) 1U << LLD_I2C_EDMA_OPT_TCINTEN_SHIFT);

    /* update the transfer completion code                                */
    i2cEdmaParam.opt &= (~LLD_I2C_EDMA_OPT_TCC_MASK);
    i2cEdmaParam.opt |= (chan << LLD_I2C_EDMA_OPT_TCC_SHIFT);

    EDMA3SetPaRAM(edmaParams->baseAddr, chan, &i2cEdmaParam);

    return status;
}

LLD_hsi2cEdmaStatus EDMA_enableTransfer(uint32_t chan, void *edmaParam)
{
    LLD_hsi2cEdmaStatus edmaStatus = EDMA_TRANSFER_SUCCESS;
    uint32_t status;
    LLD_hsi2cEdmaParam *edmaParams = (LLD_hsi2cEdmaParam *) edmaParam;

    status = EDMA3EnableTransfer(edmaParams->baseAddr, chan,
                                 EDMA3_TRIG_MODE_EVENT);
    if (FALSE == status)
    {
        edmaStatus = EDMA_TRANSFER_FAIL;
    }
    return edmaStatus;
}

