//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//
//---------------------------------------------------------------------------------------------------------------------
/**
 *  \ingroup BSP_DRV_VPS_CORE_API
 *  \addtogroup BSP_DRV_VPS_CORE_VIP_API VPS - VIP Capture Core API
 *
 *  @{
 */

/**
 *  \file vpscore_vip.h
 *
 *  \brief VPS capture VIP core interface.
 *  This file defines the interface for the VIP capture core.
 *
 */

#ifndef VPSCORE_VIP_H_
#define VPSCORE_VIP_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "bsp_types.h"
#include "vps_graph.h"
#include "vps_common.h"
#include "vps_evtMgr.h"
#include "vpscore.h"

#include "vps_captureVipDataTypes.h"

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/** \brief Maximum number of output streams from a VIP. */
#define VPSCORE_VIP_STREAM_ID_MAX  4U

/**
 *  \brief Maximum number of channels per VIP port - used in mux-mode capture.
 */
#define VPSCORE_VIP_CH_ID_MAX      (8U)

/** \brief IOCTL base address for the common IOCTLs listed below. */
#define VPSCORE_VIP_COMMON_IOCTL_BASE     (uint32_t) (0x00000000U)
/** \brief IOCTL base address for the custom IOCTLs of each cores. */
#define VPSCORE_VIP_CUSTOM_IOCTL_BASE     (uint32_t) (0x00000100U)

/** \brief Control command to set MAX FRAME configuration. */
#define VPSCORE_VIP_IOCTL_SET_MF_CFG      (VPSCORE_VIP_COMMON_IOCTL_BASE + \
                                           0x0000U)

/** \brief Control command to get Sub Frame Information. */
#define VPSCORE_VIP_IOCTL_GET_SUBFRM_INFO (VPSCORE_VIP_COMMON_IOCTL_BASE + \
                                           0x0001U)

/** \brief Control command to Reset PORTs. This is required only for
 *  TI814X_BUILD. */
#define VPSCORE_VIP_IOCTL_RESET_PORT      (VPSCORE_VIP_COMMON_IOCTL_BASE + \
                                           0x0002U)

/** \brief Control command to get the current frame being captured.*/
#define VPSCORE_VIP_IOCTL_GET_CURCAPTFRM_INFO    (VPSCORE_VIP_COMMON_IOCTL_BASE \
                                                  +                             \
                                                  0x0003U)

/** \brief Control command to get the params to set the interrupt for
 *    overflow.
 *
 *  \param appArgs     VpsCore_VipObj *
 *  \param drvArgs      VpsCore_VipOpenRetParams *
 *
 *  \return BSP_SOK on success, else failure.
 */

#define VPSCORE_VIP_IOCTL_GET_VEM_PARAMS  ( \
        VPSCORE_VIP_COMMON_IOCTL_BASE +     \
        0x0004U)

/** \brief Control command to get the status if the overflow interrupt is set or
 * not
 *
 *
 *  \param appArgs      NULL
 *  \param drvArgs      UInt32 *
 *
 *  \return BSP_SOK on success, else failure.
 */

#define VPSCORE_VIP_IOCTL_GET_OVERFLOW_REG_STATUS ( \
        VPSCORE_VIP_COMMON_IOCTL_BASE +             \
        0x0005U)

/** \brief Control command to reset the port.Needed in case of overflow.
 *      This is required to stop the overflow and come out of the ISR.
 *      This can be called from the ISR context. It just sets a reset bit
 *      for reset of the port
 *
 *  \param appArgs      NULL
 *  \param drvArgs      NULL
 *
 *  \return BSP_SOK on success, else failure.
 */

#define VPSCORE_VIP_IOCTL_RESET_VIP_PORT ( \
        VPSCORE_VIP_COMMON_IOCTL_BASE +    \
        0x0006U)

typedef enum
{
    S0_PORTA_08 = 0,
    S0_PORTA_16,
    S0_PORTA_24,
    S0_PORTB_08,
    S1_PORTA_08,
    S1_PORTA_16,
    S1_PORTA_24,
    S1_PORTB_08,
    S0_PORTA_MULT,
    S0_PORTB_MULT,
    S1_PORTA_MULT,
    S1_PORTB_MULT
} VpsCore_VipSrcNode;

typedef enum
{
    GRAPH_VIP_INVALID,
    /* Slice 0 */
    GRAPH_VIP_S0_PARSER_PORTA,
    GRAPH_VIP_S0_PARSER_PORTB,
    GRAPH_VIP_S0_PARSER_PORTA_INTF,
    GRAPH_VIP_S0_PARSER_PORTB_INTF,
    GRAPH_VIP_S0_PARSER_PORTA_08,
    GRAPH_VIP_S0_PARSER_PORTA_16,
    GRAPH_VIP_S0_PARSER_PORTA_24,
    GRAPH_VIP_S0_PARSER_PORTB_08,
    GRAPH_VIP_S0_PARSER_PORTA_08_16,
    GRAPH_VIP_S0_SEC0,
    GRAPH_VIP_S0_COMP0,
    GRAPH_VIP_S0_CSC_SRC,
    GRAPH_VIP_S0_SC_SRC,
    GRAPH_VIP_S0_CSC,
    GRAPH_VIP_S0_SC,
    GRAPH_VIP_S0_CHRDS0_ON,
    GRAPH_VIP_S0_CHRDS1_ON,
    GRAPH_VIP_S0_CHRDS0_OFF,
    GRAPH_VIP_S0_CHRDS1_OFF,
    GRAPH_VIP_S0_CHRDS0,
    GRAPH_VIP_S0_CHRDS1,
    GRAPH_VIP_S0_RGB_HI,
    GRAPH_VIP_S0_Y_UP,
    GRAPH_VIP_S0_UV_UP,
    GRAPH_VIP_S0_Y_LO,
    GRAPH_VIP_S0_UV_LO,
    GRAPH_VIP_S0_PARSER_PORTA_ANC,
    GRAPH_VIP_S0_PARSER_PORTB_ANC,
    GRAPH_VIP_S0_PORTA_LUMA,
    GRAPH_VIP_S0_PORTA_CHROMA,
    GRAPH_VIP_S0_PORTB_LUMA,
    GRAPH_VIP_S0_PORTB_CHROMA,
    GRAPH_VIP_S0_PORTA_RGB,
    GRAPH_VIP_S0_PORTB_RGB,
    GRAPH_VIP_S0_MULT_ACTA_SRC0,
    GRAPH_VIP_S0_MULT_ACTB_SRC0,
    GRAPH_VIP_S0_MULT_ANCA_SRC0,
    GRAPH_VIP_S0_MULT_ANCB_SRC0,
    /* Slice 1*/
    GRAPH_VIP_S1_PARSER_PORTA,
    GRAPH_VIP_S1_PARSER_PORTB,
    GRAPH_VIP_S1_PARSER_PORTA_INTF,
    GRAPH_VIP_S1_PARSER_PORTB_INTF,
    GRAPH_VIP_S1_PARSER_PORTA_08,
    GRAPH_VIP_S1_PARSER_PORTA_16,
    GRAPH_VIP_S1_PARSER_PORTA_24,
    GRAPH_VIP_S1_PARSER_PORTB_08,
    GRAPH_VIP_S1_PARSER_PORTA_08_16,
    GRAPH_VIP_S1_SEC0,
    GRAPH_VIP_S1_COMP0,
    GRAPH_VIP_S1_CSC_SRC,
    GRAPH_VIP_S1_SC_SRC,
    GRAPH_VIP_S1_CSC,
    GRAPH_VIP_S1_SC,
    GRAPH_VIP_S1_CHRDS0_ON,
    GRAPH_VIP_S1_CHRDS1_ON,
    GRAPH_VIP_S1_CHRDS0_OFF,
    GRAPH_VIP_S1_CHRDS1_OFF,
    GRAPH_VIP_S1_CHRDS0,
    GRAPH_VIP_S1_CHRDS1,
    GRAPH_VIP_S1_RGB_HI,
    GRAPH_VIP_S1_Y_UP,
    GRAPH_VIP_S1_UV_UP,
    GRAPH_VIP_S1_Y_LO,
    GRAPH_VIP_S1_UV_LO,
    GRAPH_VIP_S1_PARSER_PORTA_ANC,
    GRAPH_VIP_S1_PARSER_PORTB_ANC,
    GRAPH_VIP_S1_PORTA_LUMA,
    GRAPH_VIP_S1_PORTA_CHROMA,
    GRAPH_VIP_S1_PORTB_LUMA,
    GRAPH_VIP_S1_PORTB_CHROMA,
    GRAPH_VIP_S1_PORTA_RGB,
    GRAPH_VIP_S1_PORTB_RGB,
    GRAPH_VIP_S1_MULT_ACTA_SRC0,
    GRAPH_VIP_S1_MULT_ACTB_SRC0,
    GRAPH_VIP_S1_MULT_ANCA_SRC0,
    GRAPH_VIP_S1_MULT_ANCB_SRC0,
    GRAPH_VIP_NUM_NODES
} VpsCore_VipGraphNodes;

typedef enum
{
    VPSCORE_VIP_INTRTYPE_TIMER,
    VPSCORE_VIP_INTRTYPE_VEM,
    VPSCORE_VIP_INTRTYPE_OS,
    VPSCORE_VIP_INTRTYPE_MAX
} VpsCore_VipIntrType;

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

typedef struct
{
    VpsHal_Handle vipTopHandle;
    VpsHal_Handle vpdmaHandle;
    VpsHal_Handle s0PortAHandle;
    VpsHal_Handle s0PortBHandle;
    VpsHal_Handle s1PortAHandle;
    VpsHal_Handle s1PortBHandle;
    VpsHal_Handle sc0Handle;
    VpsHal_Handle sc1Handle;
    VpsHal_Handle csc0Handle;
    VpsHal_Handle csc1Handle;
    Vem_InstId    vemInstId;
    Vrm_InstId    vrmInstId;
} VpsCore_VipInitParams;

typedef struct
{
    UInt32 src;
    /**< #VpsCore_VipSrcNode */
    UInt32 numChannels;
    /**< For multi-channel capture. */
    UInt32 muxModeStartChId;
    /**< [IN] Start channel ID in pixel or line mux mode. Used to add an
     *   offset to start channel mapping. This will be used when the decoder
     *   start channel ID is other than 0.
     *   For example some decoder's channel 0 CHID starts from 4 instead of 0.
     *   This is valid only in multi-channel mode and is ignored in single
     *   channel or other modes. */
} VpsCore_VipOpenParams;

typedef struct
{
    /*
     * Below parameters are valid only if intrType is VEM.
     */
    Vem_InstId     subFrmVemInstId;
    Vem_EventGroup subFrmVemEventGroup;
    /**< #Vem_EventGroup */
    UInt32         subFrmVemEvent[VPSCORE_VIP_STREAM_ID_MAX]
    [FVID2_MAX_PLANES];
    /**< #VpsHal_VpdmaClient for every stream. */
} VpsCore_VipSetCfgRetPrms;

typedef struct
{
    UInt32 isWrDescValid;
    /**< Status which returns if invalid write descriptors
     *   observed in last DeQueue() */
    UInt32 isOutputOverflow;
    /**< Status which returns if output-port overflow
     *   was observed in last DeQueue() */
    UInt32 isAsyncOverflow;
    /**< Status which returns if async-port overflow
     *   was observed in last DeQueue() */
    UInt32 isActProtViolation;
    /**< Status which returns if protocol
     *   violations in active data was observed in last DeQueue() */
    UInt32 isAncProtViolation;
    /**< Status which returns number of times protocol
     *   violations in ancillary data was observed in last DeQueue() */
} VpsCore_VipCoreStats;

typedef struct
{
    VpsCore_VipIntrType intrType[VPSCORE_VIP_CH_ID_MAX];
    /**< VPSCORE_VIP_INTRTYPE_TIMER: Driver will use timer interrupt
     *   for the instance.
     *   VPSCORE_VIP_INTRTYPE_VEM: Driver will register to the VPS
     *   event manager.
     *   VPSCORE_VIP_INTRTYPE_OS: Driver will register to the OSAL. */

    /*
     * Below parameters are valid only if intrType is VEM.
     */
    Vem_InstId          vemInstId[VPSCORE_VIP_CH_ID_MAX];
    Vem_EventGroup      vemEventGroup[VPSCORE_VIP_CH_ID_MAX];
    UInt32              vemEvent[VPSCORE_VIP_CH_ID_MAX];

    /*
     * Below parameters are valid only if intrType is OS.
     */
    UInt32              intNum[VPSCORE_VIP_CH_ID_MAX];
} VpsCore_VipOpenRetParams;

typedef struct
{
    UInt32                 srcColorSpace;

    UInt32                 srcWidth;
    /**< Expected source width, only used if scaler is active */
    UInt32                 srcHeight;
    /**< Expected source height, only used if scaler is active */
    UInt32                 numStreams;
    /**< Maximum 4 streams supported (anc-data is one of the streams). */
    UInt32                 dstColorSpace[VPSCORE_VIP_STREAM_ID_MAX];
    /**< Will contain FVID2 datatypes per stream, will contain FVID2_RAW_VBI for
     *   anc-data. */
    UInt32                 tarWidth[VPSCORE_VIP_STREAM_ID_MAX];
    /**< Target width for output per stream, only used if scaler is active */
    UInt32                 tarHeight[VPSCORE_VIP_STREAM_ID_MAX];
    /**< Target height for output per stream, only used if scaler is active */
    UInt32                 pitch[VPSCORE_VIP_STREAM_ID_MAX][FVID2_MAX_PLANES
    ];
    /**< 2D pitch for captured data per stream, per plane */
    UInt32                 scActive[VPSCORE_VIP_STREAM_ID_MAX];
    /**< Specify whether scaler is active for a stream */

    Vgraph_edgeInfoList    edgeInfoList;
    /**< Input edge list - if number of edges is zero, VIP core will attempt to
     *   create an edgeList internally. */

    Vps_VipPortConfig      portCfg;
    /**< This will be as per vps_cfgVipParser.h. */
    Fvid2_CropConfig       inCropCfg;
    /**< Scaler input crop config. */
    Vps_ScConfig           scCfg;
    /**< This will be as per vps_scCfg.h. */
    Vps_CscConfig          cscCfg;
    /**< This will be as per vps_cscCfg.h. */
    VpsCore_SubFrameParams subFrmPrms[VPSCORE_VIP_STREAM_ID_MAX];
    /**< SubFrame mode parameters. */

    UInt32                 maxOutWidth[VPSCORE_VIP_STREAM_ID_MAX]
    [FVID2_MAX_PLANES];
    /**< Set limit on the max possible width of the output frame
     *   For valid values see #Vps_VpdmaMaxOutWidth. */
    UInt32                 maxOutHeight[VPSCORE_VIP_STREAM_ID_MAX]
    [FVID2_MAX_PLANES];
    /**< Set limit on the max possible height of the output frame
     *   For valid values see #Vps_VpdmaMaxOutHeight. */
    UInt32                 memType[VPSCORE_VIP_STREAM_ID_MAX];
    /**< Tiled or non-tiled memory selection for output.
     *   For valid values see #Vps_VpdmaMemoryType.*/
} VpsCore_VipParams;

/**
 *  struct VpsCore_VipSubFrameInfo
 *  \param streamId     [IN]  Stream ID information needed by the VIP Core
 *                            to return the appropriate FID, Width and Height.
 *  \param channelId    [IN]  Channel ID information needed by the VIP Core
 *                            to return the appropriate FID, Width and Height.
 *  \param currFid      [OUT] Current FID.
 *  \param prevFid      [OUT] Previou FID.
 *  \param width        [OUT] Width of the Field.
 *  \param height       [OUT] Height of the Field.
 */
typedef struct
{
    UInt32 streamId;
    UInt32 channelId;
    UInt32 currFid;
    UInt32 prevFid;
    UInt32 width;
    UInt32 height;
} VpsCore_VipSubFrameInfo;

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  VpsCore_vipInit
 *  \brief VIP core init function.
 *  Initializes core objects, allocates memory for one instance of VIP.
 *  This function should be called before calling any of VIP core API's.
 *
 *  \param initPrms     [IN] Pointer to the init parameter containing
 *                      instance specific information. #VpsCore_VipInitParams
 *
 *  \return             Returns a handle of type #VpsCore_Inst to the
 *                      VIP instance
 */
VpsCore_Inst VpsCore_vipInit(const VpsCore_VipInitParams *initParams);

/**
 *  VpsCore_vipDeInit
 *  \brief VIP core exit function.
 *  Resets all SW state inside the driver for the current VIP instance.
 *
 *  \param pVipHandle   [IN] Handle for the VIP instance to be closed.
 *                      #VpsCore_Inst
 *
 *  \return             Returns BSP_SOK on success else returns error value.
 */
Int32 VpsCore_vipDeInit(VpsCore_Inst pVipHandle);

/**
 *  VpsCore_vipGetCoreOps
 *  \brief Returns the pointer to core function pointer table.
 *
 *  \return             Returns the pointer to core function pointer table.
 */
const VpsCore_Ops *VpsCore_vipGetCoreOps(void);

/**
 *  VpsCore_vipGetProperty
 *  \brief Gets the core properties of the VIP core.
 *
 *  \param handle       [IN] VIP instance handle.
 *  \param property     [OUT] Pointer to which the VIP path core properties
 *                      to be copied.
 *
 *  \return             Returns BSP_SOK on success else returns error value.
 */
Int32 VpsCore_vipGetProperty(VpsCore_Inst      instObj,
                             VpsCore_Property *property);

/**
 *  VpsCore_vipOpen
 *  \brief Creates a specified path on the specified VIP instance.
 *
 *  \param pVipHandle     [IN] VIP Instance handle #VpsCore_Inst
 *  \param openPrms       [IN] Callback function information #VpsCore_OpenPrms
 *  \param coreOpenPrms   [IN] Basic source information  #VpsCore_VipOpenParams
 *  \param coreReturnPrms [OUT] Returns interrupt requirement for this path.
 *
 *  \return             If success returns the path handle else returns NULL.
 */
VpsCore_Handle VpsCore_vipOpen(VpsCore_Inst            pVipHandle,
                               const VpsCore_OpenPrms *openPrms,
                               const void             *coreOpenPrms,
                               void                   *coreReturnPrms);
/**
 *  VpsCore_vipClose
 *  \brief Closes an already opened handle.
 *
 *  \param handle       [IN] VIP Path handle.
 *
 *  \return             Returns BSP_SOK on success else returns error value.
 */
Int32 VpsCore_vipClose(VpsCore_Handle pathHandle);

/**
 *  VpsCore_vipPathGetConfig
 *  \brief Provides the current configuration for the given path handle.
 *         If no configuration has been done, it will provide a fixed default
 *         configuration.
 *
 *  \param pathHandle   [IN] VIP Path handle. #VpsCore_Handle
 *  \param params       [OUT] Pointer where the configuration is to be copied
 *                      #Vcore_GetVipParams. This parameter should be non-NULL.
 *
 *  \return             Returns BSP_SOK on success else returns error value.
 */
Int32 VpsCore_vipPathSetConfig(VpsCore_Handle pathHandle,
                               const void    *params,
                               void          *setCfgRetPrms);

/**
 *  VpsCore_vipPathGetConfig
 *  \brief Provides the current configuration for the given path handle.
 *         If no configuration has been done, it will provide a fixed default
 *         configuration.
 *
 *  \param pathHandle   [IN] VIP Path handle. #VpsCore_Handle
 *  \param params       [OUT] Pointer where the configuration is to be copied
 *                      #Vcore_GetVipParams. This parameter should be non-NULL.
 *
 *  \return             Returns BSP_SOK on success else returns error value.
 */
Int32 VpsCore_vipPathGetConfig(VpsCore_Handle pathHandle,
                               void          *params);

/**
 *  VpsCore_vipStart
 *  \brief Start capture.
 *  This will post the initial VPDMA descriptors to enable the VIP port
 *
 *  \param pathHandle       Path handle #VpsCore_Handle returned by
 *                          VpsCore_vipOpen
 *
 *  \return                 success: BSP_SOK
 *                          failure: BSP_EFAIL
 */
Int32 VpsCore_vipStart(VpsCore_Handle pathHandle);

/**
 *  VpsCore_vipStop
 *  \brief Stop capture.
 *  This will post the abort descriptors for the capture channel and disable the
 *  VIP port
 *
 *  \param pathHandle       Path handle #VpsCore_Handle returned by
 *                          VpsCore_vipOpen
 *
 *  \return                 success: BSP_SOK
 *                          failure: BSP_EFAIL
 */
Int32 VpsCore_vipStop(VpsCore_Handle pathHandle);

/**
 *  VpsCore_vipIsr
 *  \brief ISR callback to be called after recieving the configured interrupt.
 *  Currently, this supports only single channel capture.
 *  ISR registration will be handled by upper layer of SW
 *
 *  \param pathHandle       Path handle #VpsCore_Handle returned by
 *                          VpsCore_vipOpen
 *  \param chId             Channel index. Used in multi-channel capture.
 *
 *  \return                 success: BSP_SOK
 *                          failure: BSP_EFAIL
 */
Int32 VpsCore_vipIsr(VpsCore_Handle pathHandle, UInt32 chId);

/**
 *  VpsCore_vipControl
 *  \brief This is a control call to SET/GET VIP Core related parameters/status
 *  which are not supported by the default API
 *
 *  \param pathHandle       Path handle #VpsCore_Handle returned by
 *                          VpsCore_vipOpen
 *  \param cmd              IOCTL Command to SET/GET parameters/status.
 *
 *  \param appArgs          IOCTL specific arguments provided by the
 *                          application.
 *                          To SET the VIP core parameters this args shall
 *                          point to parameters that shall be provided by
 *                          the Application.
 *                          To GET the VIP core status this args shall
 *                          point to parameters that shall be provided to
 *                          the Application by the VIP Core.
 *  \param drvArgs          IOCTL specific arguments that which are provided by
 *                          the driver to the Core.
 *
 *  \return                 success: BSP_SOK, control call serviced.
 *                          failure: BSP_EFAIL
 */
Int32 VpsCore_vipControl(VpsCore_Handle handle,
                         UInt32         cmd,
                         Ptr            appArgs,
                         Ptr            drvArgs);

#ifdef __cplusplus
}
#endif

#endif /* VPSCORE_VIP_H_ */

/* @} */
