//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//  File containing the DCTRL driver APIs.
//---------------------------------------------------------------------------------------------------------------------
#include <stdint.h>
#include <stdio.h>
#include <stddef.h>
#include "tistdtypes.h"
#include "vps_common.h"
#include "bsp_utils.h"
#include "vps_dctrlDrv.h"
//---------------------------------------------------------------------------------------------------------------------
static VpsDrv_DctrlInstObj gDctrlDrvObj;
static UInt32 gDrvInit = 0;
static VpsDrv_DctrlInstObj *VpsDrv_DctrlGetInst(void);
//---------------------------------------------------------------------------------------------------------------------
/**
 *  VpsDrv_dctrlInit
 *  \brief VPS DSS capture driver init function.
 *
 *  \param numInst            Number of instances to initialize.
 *  \param initPrms           Pointer to #VpsDrv_DispInitParams.
 *
 *  \return                   success    0
 *                            failure    error value
 *
 */
Int32 VpsDrv_dctrlInit(const VpsDrv_DctrlInitParams *initPrms)
{
    gDctrlDrvObj.drvInstId   = 0;
    gDctrlDrvObj.coreInstObj = initPrms->coreInstObj;
    gDctrlDrvObj.coreOps     = initPrms->coreOps;

    gDrvInit = 1U;
    return BSP_SOK;
}
//---------------------------------------------------------------------------------------------------------------------
/**
 *  VpsDrv_dispDeInit
 *  \brief VPS DSS capture driver de-initialization function.
 *
 *  \return                   success    0
 *                            failure    error value
 *
 */
Int32 VpsDrv_dctrlDeInit(void)
{
    gDctrlDrvObj.drvInstId   = 0U;
    gDctrlDrvObj.coreInstObj = NULL;
    gDctrlDrvObj.coreOps     = NULL;
    gDrvInit = 0;

    return BSP_SOK;
}
//---------------------------------------------------------------------------------------------------------------------
/**
 *  VpsDrv_DctrlGetInst
 *  \brief Get Handle to the Driver control instance as per specified ID
 *
 *  \param instId             Driver control Instance ID
 *
 *  \return                   success    BSP_SOK
 *                            failure    error value
 *
 */
static VpsDrv_DctrlInstObj *VpsDrv_DctrlGetInst(void)
{
    VpsDrv_DctrlInstObj *pObj = NULL;
    if (gDrvInit)
    {
        if (NULL != gDctrlDrvObj.coreInstObj)
        {
            pObj = &gDctrlDrvObj;
        }
    }
    return pObj;
}
//---------------------------------------------------------------------------------------------------------------------
/**
 *  VpsDrv_DctrlOpen
 *  \brief Open a capture driver instance as per specified parameters
 *
 *  \param captureOpenParams Capture Driver parameters #VpsCore_OpenPrms
 *  \param vipOpenParams     VIP specific parameters #VpsCore_VipOpenParams
 *
 *  \return                  Capture handle #VpsDrv_CaptInstObj on success
 *                           NULL pointer on failure
 *
 */
VpsDrv_DctrlInstObj *VpsDrv_dctrlOpen(void)
{
    VpsDrv_DctrlInstObj *pObj;

    pObj = (VpsDrv_DctrlInstObj *) VpsDrv_DctrlGetInst();
    return pObj;
}
//---------------------------------------------------------------------------------------------------------------------
/**
 *  VpsDrv_DctrlClose
 *  \brief Close the specified capture handle
 *
 *  \param pObj         Capture Handle to close
 *
 *  \return             success    BSP_SOK
 *                      failure    error value
 *
 */
Int32 VpsDrv_dctrlClose(VpsDrv_DctrlInstObj *pObj)
{
    Int32 retVal = BSP_SOK;

    return retVal;
}

/**
 *  vpsDrv_DctrlSetConfig
 *  \brief Gets the current display status information.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns BSP_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlSetConfig(VpsDrv_DctrlInstObj *pObj,
                            Vps_DctrlConfig     *dctrlCfg)

{
    Int32 retVal = BSP_SOK;

    if (NULL == pObj)
    {
        printf("Invalid argument!!\r\n");
        retVal = BSP_EBADARGS;
    }
    else
    {
        retVal = pObj->coreOps->control(
            pObj,
            IOCTL_VPSCORE_DCTRL_SET_CONFIG,
            dctrlCfg,
            (Ptr) pObj->coreInstObj);

        if (BSP_SOK != retVal)
        {
            printf("Core control: Set Config failed!!\r\n");
        }
    }
    return (retVal);
}

/**
 *  vpsDrv_DctrlCleatConfig
 *  \brief Gets the current display status information.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns BSP_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlClearConfig(VpsDrv_DctrlInstObj *pObj,
                              Vps_DctrlConfig     *dctrlCfg)

{
    Int32 retVal = BSP_SOK;

    /* Check for NULL pointers */
    /* GT_assert(VpsDrvDctrlTrace, (NULL != dctrlCfg)); */

    retVal = pObj->coreOps->control(
        pObj,
        IOCTL_VPSCORE_DCTRL_CLEAR_CONFIG,
        dctrlCfg,
        (Ptr) pObj->coreInstObj);
    if (BSP_SOK != retVal)
    {
        printf("Core control: clear config failed!!\r\n");
    }

    return (retVal);
}

/**
 *  vpsDrv_DctrlSetVencOutput
 *  \brief Gets the current display status information.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns FVID2_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlSetVencOutput(VpsDrv_DctrlInstObj *pObj,
                                Vps_DctrlOutputInfo *dctrl_info)

{
    Int32 retVal = BSP_SOK;

    retVal = pObj->coreOps->control(
        pObj,
        IOCTL_VPSCORE_DCTRL_SET_VENC_OUTPUT,
        dctrl_info,
        (Ptr) pObj->coreInstObj);
    if (BSP_SOK != retVal)
    {
        printf("Core control: Set venc output failed!!\r\n");
    }

    return (retVal);
}

/**
 *  vpsDrv_DctrlGetVencOutput
 *  \brief Gets the current display status information.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns FVID2_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlGetVencOutput(VpsDrv_DctrlInstObj *pObj,
                                Vps_DctrlOutputInfo *dctrl_info)

{
    Int32 retVal = BSP_SOK;

    retVal = pObj->coreOps->control(
        pObj,
        IOCTL_VPSCORE_DCTRL_GET_VENC_OUTPUT,
        dctrl_info,
        (Ptr) pObj->coreInstObj);
    if (FVID2_SOK != retVal)
    {
        printf("Core control: Get venc output failed!!\r\n");
    }

    return (retVal);
}

/**
 *  vpsDrvDctrlSetVencDivisorIoctl
 *  \brief Sets the Venc divisor values to get desired pixel clock.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns BSP_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlSetVencDivisor(
    VpsDrv_DctrlInstObj *pObj,
    Vps_DctrlVencDivisorInfo *
    vencDivisor)

{
    Int32 retVal = BSP_SOK;

    /* Check for NULL pointers */
    /* GT_assert(VpsDrvDctrlTrace, (NULL != vencDivisor)); */

    retVal = pObj->coreOps->control(
        pObj,
        IOCTL_VPSCORE_DCTRL_SET_VENC_PCLK_DIVISORS,
        vencDivisor,
        (Ptr) pObj->coreInstObj);
    if (FVID2_SOK != retVal)
    {
        printf("Core control: Set panelconfig failed!!\r\n");
    }

    return (retVal);
}

/**
 *  vpsDrv_DctrlSetOvlyParms
 *  \brief Gets the current display status information.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns BSP_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlSetOvlyParams(
    VpsDrv_DctrlInstObj *pObj,
    Vps_DssDispcOvlyPanelConfig *
    panelconfig)

{
    Int32 retVal = BSP_SOK;

    retVal = pObj->coreOps->control(
        pObj,
        IOCTL_VPSCORE_DCTRL_SET_OVLY_PARAMS,
        panelconfig,
        (Ptr) pObj->coreInstObj);
    if (BSP_SOK != retVal)
    {
        printf("Core control: Set panelconfig failed!!\r\n");
    }

    return (retVal);
}

/**
 *  vpsDrvDctrlSetPipelineParmsIoctl
 *  \brief Gets the current display status information.
 *
 *  NOTE: This can't be be called from ISR context.
 *
 *  Returns BSP_SOK on success else returns appropriate error code.
 */
Int32 vpsDrv_DctrlSetPipelineParams(
    VpsDrv_DctrlInstObj        *pObj,
    Vps_DssDispcOvlyPipeConfig *ovlpipeconfig)

{
    Int32 retVal = BSP_SOK;

    /* Check for NULL pointers */
    /* GT_assert(VpsDrvDctrlTrace, (NULL != ovlpipeconfig)); */

    retVal = pObj->coreOps->control(
        pObj,
        IOCTL_VPSCORE_DCTRL_SET_PIPELINE_PARAMS,
        ovlpipeconfig,
        (Ptr) pObj->coreInstObj);

    if (BSP_SOK != retVal)
    {
        printf("Core control: Set pipelineconfig failed!!\r\n");
    }

    return (retVal);
}

