//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//  File containing the VPS driver init and deinit functions.
//---------------------------------------------------------------------------------------------------------------------
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <ti/csl/tistdtypes.h>
#include <ti/drv/vps/include/common/bsp_types.h>
#include <ti/drv/vps/include/common/trace.h>
#include <ti/drv/vps/include/common/bsp_config.h>
#include <ti/csl/soc.h>
#include "fvid2_dataTypes.h"
#include <ti/drv/vps/include/vps.h>
#include <ti/drv/vps/include/vps_control.h>
#include <ti/drv/vps/include/vps_dataTypes.h>
#include <ti/drv/vps/src/vpslib/common/vps_common.h>
#include <ti/drv/vps/src/vpslib/common/vpscore.h>
#include <ti/drv/vps/src/vpsdrv/vpsdrv_control.h>

#ifdef VPS_CAPT_BUILD
#ifdef VPS_VIP_BUILD
#include <ti/drv/vps/include/vps_capture.h>
#include <ti/drv/vps/src/vpslib/captcore/vpscore_vip.h>
#endif /* VPS_VIP_BUILD */
#include <ti\drv\vps\src\vpsdrv\captdrv/vpsdrv_capture.h>
#include <ti/drv/vps/include/vps_capture.h>
#endif /* VPS_CAPT_BUILD */

#ifdef VPS_DISP_BUILD
#include <ti/drv/vps/include/vps_display.h>
#include <ti/drv/vps/src/vpslib/dispcore/vpscore_dss.h>
#include <ti/drv/vps/src/vpslib/dispcore/vpscore_dctrl.h>
#include <ti/drv/vps/src/vpslib/dispcore/vpscore_dssWb.h>
#include <ti/drv/vps/src/vpsdrv/dispdrv/vpsdrv_display.h>
#include <ti/drv/vps/src/vpsdrv/dispdrv/vpsdrv_dctrl.h>
#include <ti/drv/vps/include/dss/vps_m2mDss.h>
#include <ti/drv/vps/src/vpsdrv/m2mdrv_dss/vpsdrv_m2mDss.h>
#endif /* VPS_DISP_BUILD */

#ifdef VPS_CAL_BUILD
#include <ti/drv/vps/src/vpslib/common/iem_evtmgr.h>
#include <ti/drv/vps/src/vpsdrv/captdrv/vpsdrv_capture.h>
#include <ti/drv/vps/src/vpslib/isscore/vpscore_capture.h>
#endif /* VPS_CAL_BUILD */

#ifdef VPS_SIMCOP_BUILD
#include <ti/drv/vps/src/vpslib/isscore/vpscore_simcop.h>
#endif /* VPS_SIMCOP_BUILD */

#ifdef VPS_M2M_BUILD
#include <ti/drv/vps/include/vps_m2mIntf.h>
#include <m2mdrv/vpsdrv_m2mintf.h>
#ifdef VPS_ISS_BUILD
#include <ti/drv/vps/include/iss/vps_m2mIss.h>
#include <ti/drv/vps/src/vpslib/isscore/vpscore_m2m.h>
#endif /* VPS_ISS_BUILD */
#ifdef VPS_SIMCOP_BUILD
#include <ti/drv/vps/src/vpslib/isscore/vpscore_simcop.h>
#endif /* VPS_SIMCOP_BUILD */
#endif /* VPS_M2M_BUILD */
//---------------------------------------------------------------------------------------------------------------------
// Platform specific data containing base address information of various modules.
typedef struct
{
#ifdef VPS_CAPT_BUILD
    VpsDrv_CaptInitParams     captDrvInitPrms[VPS_CFG_CAPT_NUM_INST_OBJS];
    /**< Capture driver init parameters. */
#endif

#ifdef VPS_DISP_BUILD
    VpsDrv_DispInitParams     dispDrvInitPrms[VPS_DISP_INST_MAX];
    /**< Display driver init parameters. */
    VpsMdrv_DssInitParams     m2mDssDrvInitPrms[VPS_M2M_DSS_INST_MAX];
    /**< M2M DSS driver init parameters. */
#endif

#ifdef VPS_CAL_BUILD
    VpsDrv_CaptInitParams     issCaptDrvInitPrms[VPS_ISS_CAPT_INST_MAX_ID];
#endif /* VPS_CAL_BUILD */

#ifdef VPS_M2M_BUILD
    VpsMdrv_M2mIntfInitParams m2mIntfDrvInitPrms[VPS_M2M_INTF_INST_MAX];
#endif

    UInt32                    reserved;
    /**< Needed in case both VIP and DSS are not defined!! */
} VpsDrv_PlatformData;

//---------------------------------------------------------------------------------------------------------------------
void vpsInitTda3xxPlatData(void);

const VpsDrv_PlatformData *VpsDrv_getPlatformData(void);

#ifdef VPS_CAPT_BUILD
static Int32 Vps_captInit(void);
static Int32 Vps_captDeInit(void);
#endif
#ifdef VPS_DISP_BUILD
static Int32 Vps_dispInit(void);
static Int32 Vps_dispDeInit(void);
#endif
#ifdef VPS_M2M_BUILD
static Int32 Vps_m2mInit(void);
static Int32 Vps_m2mDeInit(void);
#endif

extern  Int32 VpsLib_init(const VpsLib_InitParams *initPrms);

static VpsDrv_PlatformData gVpsDrvPlatDataTda3xx;
//---------------------------------------------------------------------------------------------------------------------
// VPS init function.
//---------------------------------------------------------------------------------------------------------------------
Int32 Vps_init( const Vps_InitParams *initPrms )
{
    Int32             retVal = BSP_SOK;
    Vps_InitParams    localInitPrms;
    VpsLib_InitParams vpsLibInitPrms;

    printf( "Inside Vps_init - try vpsInitTda3xxPlatData\r\n" );
    vpsInitTda3xxPlatData();

    if( NULL == initPrms )
    {
        printf( "Vps_InitParams entry is empty - Try VpsInitParams_init...\r\n" );
        initPrms = &localInitPrms;
        VpsInitParams_init( &localInitPrms );
        printf( "VpsInitParams_init done.\r\n" );
    }

    vpsLibInitPrms.isAddrTransReq           = initPrms->isAddrTransReq;
    vpsLibInitPrms.virtBaseAddr             = initPrms->virtBaseAddr;
    vpsLibInitPrms.physBaseAddr             = initPrms->physBaseAddr;
    vpsLibInitPrms.isCacheOpsReq            = initPrms->isCacheOpsReq;
    vpsLibInitPrms.isCacheFlushReq          = initPrms->isCacheFlushReq;
    vpsLibInitPrms.irqParams.vipIrqNum[0U]  = initPrms->irqParams.vipIrqNum[0U];
    vpsLibInitPrms.irqParams.vipIrqNum[1U]  = initPrms->irqParams.vipIrqNum[1U];
    vpsLibInitPrms.irqParams.vipIrqNum[2U]  = initPrms->irqParams.vipIrqNum[2U];
    vpsLibInitPrms.irqParams.vpeIrqNum      = initPrms->irqParams.vpeIrqNum;
    vpsLibInitPrms.irqParams.dssIrqNum      = initPrms->irqParams.dssIrqNum;
    vpsLibInitPrms.irqParams.calIrqNum      = initPrms->irqParams.calIrqNum;
    vpsLibInitPrms.irqParams.issIrqNum      = initPrms->irqParams.issIrqNum;


    printf( "Try VpsLib_init...\r\n" );
                   retVal = VpsLib_init(&vpsLibInitPrms);
    if( BSP_SOK == retVal )
    {
        printf("VpsLib_init done.\r\n" );
        retVal += VpsDrv_ctrlInit();
        printf("VpsDrv_ctrlInit done.\r\n" );

#ifdef VPS_CAPT_BUILD
        retVal += Vps_captInit();
#endif
#ifdef VPS_DISP_BUILD
        retVal += Vps_dispInit();
        printf("Vps_dispInit done.\r\n" );
#endif
#ifdef VPS_M2M_BUILD
        retVal += Vps_m2mInit();
#endif
    }
    else printf( "ERROR! VpsLib_init failed!\r\n" );

    return (retVal);
}
//---------------------------------------------------------------------------------------------------------------------
/**
 *  \brief VPS deinit function.
 */
Int32 Vps_deInit(Ptr args)
{
    Int32 retVal = BSP_SOK;

#ifdef VPS_M2M_BUILD
    retVal += Vps_m2mDeInit();
#endif

#ifdef VPS_DISP_BUILD
    retVal += Vps_dispDeInit();
#endif

#ifdef VPS_CAPT_BUILD
    retVal += Vps_captDeInit();
#endif

    retVal += VpsDrv_ctrlDeInit();
    retVal += VpsLib_deInit(NULL);

    return (retVal);
}

void VpsInitParams_init(Vps_InitParams *initPrms)
{
    if (NULL != initPrms)
    {
        initPrms->isAddrTransReq  = (UInt32) FALSE;
        initPrms->virtBaseAddr    = 0U;
        initPrms->physBaseAddr    = 0U;
        initPrms->isCacheOpsReq   = (UInt32) FALSE;
        initPrms->isCacheFlushReq = (UInt32) FALSE;

        BspUtils_memset(&initPrms->irqParams, 0, sizeof (Vps_IrqParams));

    #if defined (VPS_VIP1_BUILD)
        initPrms->irqParams.vipIrqNum[0U] = CSL_INTC_EVENTID_VIP1INT0;
    #endif
    #if defined (VPS_VIP2_BUILD)
        initPrms->irqParams.vipIrqNum[1U] = CSL_INTC_EVENTID_VIP2INT0;
    #endif
    #if defined (VPS_VIP3_BUILD)
        initPrms->irqParams.vipIrqNum[2U] = CSL_INTC_EVENTID_VIP3INT0;
    #endif
        initPrms->irqParams.dssIrqNum    = CSL_INTC_EVENTID_DSS_DISPC;
    #if defined (VPS_CAL_BUILD)
        initPrms->irqParams.calIrqNum    = CSL_CAL_IRQ_NUM;
    #endif
    #if defined (VPS_ISS_BUILD)
        initPrms->irqParams.issIrqNum    = CSL_ISS_IRQ0_NUM;
    #endif
    }
    return;
}

const VpsDrv_PlatformData *VpsDrv_getPlatformData(void)
{
    VpsDrv_PlatformData *platData = NULL;
    platData = &gVpsDrvPlatDataTda3xx;
    return (platData);
}

#ifdef VPS_CAPT_BUILD
static Int32 Vps_captInit(void)
{
    Int32  retVal  = BSP_SOK;
    UInt32 instCnt = 0, numInst = 0;

#ifdef VPS_VIP_BUILD
    UInt32 numVipPort;
#endif

    UInt32 numWbPipe;

    UInt32 captInstCnt = 0U;
    const VpsLib_PlatformData *libPlatData;
    const VpsDrv_PlatformData *drvPlatData;
#ifdef VPS_CAL_BUILD
    VpsDrv_CaptInitParams      captDrvInitPrms[VPS_CFG_CAPT_NUM_INST_OBJS +
                                               VPS_ISS_CAPT_INST_MAX_ID];
#else
    VpsDrv_CaptInitParams      captDrvInitPrms[VPS_CFG_CAPT_NUM_INST_OBJS];
#endif /* VPS_CAL_BUILD */

    /* Get Platform data */
    libPlatData = VpsLib_getPlatformData();

#ifdef VPS_VIP_BUILD
    numVipPort = libPlatData->hal->numVipPort;
    numInst   += numVipPort;
    GT_assert(VpsDrvTrace,
              (numVipPort <= VPS_CFG_CAPT_NUM_INST_OBJS));
#endif

    numWbPipe = libPlatData->hal->numWbPipe;
    numInst  += numWbPipe;
    GT_assert(VpsDrvTrace,
              (numWbPipe <= VPS_DISP_INST_MAX));

#ifdef VPS_CAL_BUILD
    numInst += VPS_ISS_CAPT_INST_MAX_ID;
    GT_assert(VpsDrvTrace,
              (numInst <= (VPS_CFG_CAPT_NUM_INST_OBJS + VPS_ISS_CAPT_INST_MAX_ID)));
#endif /* VPS_CAL_BUILD */

    GT_assert(VpsDrvTrace, (NULL != libPlatData));

    drvPlatData = VpsDrv_getPlatformData();
    GT_assert(VpsDrvTrace, (NULL != drvPlatData));

#ifdef VPS_VIP_BUILD
    for (instCnt = 0U; instCnt < numVipPort; instCnt++)
    {
        captDrvInitPrms[captInstCnt].drvInstId =
            drvPlatData->captDrvInitPrms[instCnt].drvInstId;
        captDrvInitPrms[captInstCnt].coreInstObj =
            VpsLib_getVipCoreInstObj(
                Vps_captGetVipId(captDrvInitPrms[instCnt].drvInstId));

        GT_assert(VpsDrvTrace,
                  (NULL != captDrvInitPrms[captInstCnt].coreInstObj));

        captDrvInitPrms[captInstCnt].coreOps = VpsCore_vipGetCoreOps();
        captInstCnt++;
    }
#endif

    for (instCnt = 0U; instCnt < numWbPipe; instCnt++)
    {
        captDrvInitPrms[captInstCnt].drvInstId =
            VPS_CAPT_INST_DSS_WB_BASE + instCnt;
        captDrvInitPrms[captInstCnt].coreInstObj =
            VpsLib_getDssWbCoreInstObj();
        GT_assert(VpsDrvTrace,
                  (NULL != captDrvInitPrms[captInstCnt].coreInstObj));

        captDrvInitPrms[captInstCnt].coreOps = VpsCore_dssWbGetCoreOps();
        captInstCnt++;
    }

#if defined (VPS_CAL_BUILD)
    {
        UInt32 coreInst = 0;
        for (instCnt = 0U; instCnt < VPS_ISS_CAPT_INST_MAX_ID; instCnt++)
        {
            captDrvInitPrms[captInstCnt].drvInstId = VPS_CAPT_INST_ISS_BASE +
                                                     drvPlatData->
                                                     issCaptDrvInitPrms[instCnt
                                                     ].drvInstId;
            /* Get core instance objects for a given instance */
            captDrvInitPrms[captInstCnt].coreInstObj =
                VpsCore_issCaptGetCoreInstObj((vpsissCaptInstId_t) coreInst);

            GT_assert(VpsDrvTrace,
                      (NULL != captDrvInitPrms[captInstCnt].coreInstObj));

            captDrvInitPrms[captInstCnt].coreOps = VpsCore_issCaptGetCoreOps();
            captInstCnt++;
            coreInst++;
        }
    }
#endif /* VPS_CAL_BUILD */

    /* Call Capture init only if required */
    if (captInstCnt > 0U)
    {
        retVal =
            VpsDrv_captInit(captInstCnt, &captDrvInitPrms[0U]);
    }

    return (retVal);
}

static Int32 Vps_captDeInit(void)
{
    Int32 retVal = BSP_SOK;

    retVal = VpsDrv_captDeInit();

    return (retVal);
}

#endif /* VPS_CAPT_BUILD */

#ifdef VPS_M2M_BUILD
static Int32 Vps_m2mInit(void)
{
    Int32  retVal  = BSP_SOK;
    UInt32 instCnt = 0U;
    VpsMdrv_M2mIntfInitParams m2mDrvInitPrms[VPS_M2M_INTF_INST_MAX];

#ifdef VPS_ISS_BUILD
    m2mDrvInitPrms[instCnt].drvInstId          = VPS_M2M_ISS_INST_CAL_ISP;
    m2mDrvInitPrms[instCnt].isMultiChSupported = FALSE;
    m2mDrvInitPrms[instCnt].numCores           = 1U;
    m2mDrvInitPrms[instCnt].coreInstId[0U]     =
        VpsCore_issM2mGetCoreInstObj(VPSCORE_ISS_M2M_ISP_INST_0);
    m2mDrvInitPrms[instCnt].coreOps[0U] = VpsCore_issM2mGetCoreOps();
    instCnt++;
#endif

#ifdef VPS_SIMCOP_BUILD
    m2mDrvInitPrms[instCnt].drvInstId          = VPS_M2M_ISS_INST_SIMCOP;
    m2mDrvInitPrms[instCnt].isMultiChSupported = FALSE;
    m2mDrvInitPrms[instCnt].numCores           = 1U;
    m2mDrvInitPrms[instCnt].coreInstId[0U]     =
        (VpsCore_Inst *) VPSCORE_ISS_SIMCOP_INST_0;
    m2mDrvInitPrms[instCnt].coreOps[0U] = VpsCore_simcopGetCoreOps();

    instCnt++;
#endif

    if (0 < instCnt)
    {
        retVal = VpsMdrv_intfInit(instCnt, &m2mDrvInitPrms[0U]);
    }
    else
    {
        retVal = BSP_EFAIL;
    }

    return (retVal);
}

static Int32 Vps_m2mDeInit(void)
{
    Int32 retVal = BSP_SOK;

    retVal = VpsMdrv_intfDeInit();

    return (retVal);
}

#endif /* VPS_M2M_BUILD */

#ifdef VPS_DISP_BUILD
static Int32 Vps_dispInit(void)
{
    Int32  retVal = BSP_SOK;
    UInt32 instCnt;
    const VpsLib_PlatformData *libPlatData;
    const VpsDrv_PlatformData *drvPlatData;
    VpsDrv_DispInitParams      dispDrvInitPrms[VPS_DISP_INST_MAX];
    VpsDrv_DctrlInitParams     dctrlDrvInitPrms;
    VpsMdrv_DssInitParams      m2mDssDrvInitPrms[VPS_M2M_DSS_INST_MAX];


    /* Get Platform data */
    libPlatData = VpsLib_getPlatformData();
    GT_assert(VpsDrvTrace, (NULL != libPlatData));
    drvPlatData = VpsDrv_getPlatformData();
    GT_assert(VpsDrvTrace, (NULL != drvPlatData));

    dctrlDrvInitPrms.coreInstObj = VpsLib_getDctrlCoreInstObj();
    dctrlDrvInitPrms.coreOps     = VpsCore_dctrlGetCoreOps();

    VpsDrv_dctrlInit(&dctrlDrvInitPrms);

    GT_assert(VpsDrvTrace,
              ((libPlatData->hal->numVidPipe) <= VPS_DISP_INST_MAX));
    for (instCnt = 0U; instCnt < libPlatData->hal->numVidPipe; instCnt++)
    {
        dispDrvInitPrms[instCnt].drvInstId =
            drvPlatData->dispDrvInitPrms[instCnt].drvInstId;
        dispDrvInitPrms[instCnt].coreInstObj = VpsLib_getDssCoreInstObj();
        GT_assert(VpsDrvTrace, (NULL != dispDrvInitPrms[instCnt].coreInstObj));
        dispDrvInitPrms[instCnt].coreOps = VpsCore_dssGetCoreOps();
    }


    /* Call display init only if required */
    if (libPlatData->hal->numVidPipe > 0U)
    {
        retVal = VpsDrv_dispInit(libPlatData->hal->numVidPipe,
                                 &dispDrvInitPrms[0U]);
    }

    if (BSP_SOK == retVal)
    {
        m2mDssDrvInitPrms[0U].drvInstId         =
            drvPlatData->m2mDssDrvInitPrms[0U].drvInstId;
        m2mDssDrvInitPrms[0U].inCoreInstObj     = VpsLib_getDssCoreInstObj();
        m2mDssDrvInitPrms[0U].inCoreOps         = VpsCore_dssGetCoreOps();
        m2mDssDrvInitPrms[0U].outCoreInstObj    = VpsLib_getDssWbCoreInstObj();
        m2mDssDrvInitPrms[0U].outCoreOps        = VpsCore_dssWbGetCoreOps();
        retVal = VpsMdrv_dssInit(VPS_M2M_DSS_INST_MAX, &m2mDssDrvInitPrms[0U]);
    }

    return (retVal);
}

static Int32 Vps_dispDeInit(void)
{
    Int32 retVal = BSP_SOK;
    const VpsLib_PlatformData *platData;

    /* Get Platform data */
    platData = VpsLib_getPlatformData();
    GT_assert(VpsDrvTrace, (NULL != platData));

    retVal += VpsMdrv_dssDeInit();

    /* Call display deinit only if required */
    if (platData->hal->numVidPipe > 0U)
    {
        retVal += VpsDrv_dispDeInit();
    }

    VpsDrv_dctrlDeInit();

    return (retVal);
}

#endif

void vpsInitTda3xxPlatData(void)
{
    printf( "inside vpsInitTda3xxPlatData...\r\n" );
#if (defined (VPS_VIP_BUILD) || defined (VPS_DSS_BUILD) || defined (VPS_ISS_BUILD) || defined (VPS_CAL_BUILD))
    UInt32 idx = 0;
#ifdef VPS_CAPT_BUILD
#ifdef VPS_VIP_BUILD
#if defined (VPS_VIP1_BUILD)
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].drvInstId =
        VPS_CAPT_VIP_MAKE_INST_ID(VPS_VIP1, VPS_VIP_S0, VPS_VIP_PORTA);
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreOps     = NULL;
    idx++;

    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].drvInstId =
        VPS_CAPT_VIP_MAKE_INST_ID(VPS_VIP1, VPS_VIP_S0, VPS_VIP_PORTB);
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreOps     = NULL;
    idx++;

    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].drvInstId =
        VPS_CAPT_VIP_MAKE_INST_ID(VPS_VIP1, VPS_VIP_S1, VPS_VIP_PORTA);
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreOps     = NULL;
    idx++;

    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].drvInstId =
        VPS_CAPT_VIP_MAKE_INST_ID(VPS_VIP1, VPS_VIP_S1, VPS_VIP_PORTB);
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreOps     = NULL;
    idx++;
    GT_assert(VpsDrvTrace, (VPS_CFG_CAPT_NUM_INST_OBJS > idx));
#endif /* #if defined (VPS_VIP1_BUILD) */
#endif /* VPS_VIP_BUILD */

    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].drvInstId =
        VPS_CAPT_INST_DSS_WB1;
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.captDrvInitPrms[idx].coreOps     = NULL;
    GT_assert(VpsDrvTrace, (VPS_CFG_CAPT_NUM_INST_OBJS > idx));
#endif /* VPS_CAPT_BUILD */

    /* Display driver */
    idx = 0;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].drvInstId =
        VPS_DISP_INST_DSS_VID1;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].coreOps     = NULL;
    idx++;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].drvInstId =
        VPS_DISP_INST_DSS_VID2;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].coreOps     = NULL;
    idx++;

    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].drvInstId =
        VPS_DISP_INST_DSS_GFX1;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.dispDrvInitPrms[idx].coreOps     = NULL;
    GT_assert(VpsDrvTrace, (VPS_DISP_INST_MAX > idx));

    /* DSS M2M driver */
    gVpsDrvPlatDataTda3xx.m2mDssDrvInitPrms[0U].drvInstId       =
        VPS_M2M_DSS_INST_WB1;
    gVpsDrvPlatDataTda3xx.m2mDssDrvInitPrms[0U].inCoreInstObj   = NULL;
    gVpsDrvPlatDataTda3xx.m2mDssDrvInitPrms[0U].inCoreOps       = NULL;
    gVpsDrvPlatDataTda3xx.m2mDssDrvInitPrms[0U].outCoreInstObj  = NULL;
    gVpsDrvPlatDataTda3xx.m2mDssDrvInitPrms[0U].outCoreOps      = NULL;


#ifdef VPS_CAL_BUILD

    /* Generic capture driver should be included in the build, else we cannot
     *  support ISS based capture */
#ifndef VPS_CAPT_BUILD
    #error "ReBuild after including build option -DVPS_CAPT_BUILD"
#endif /* VPS_CAPT_BUILD */

    idx = 0;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].drvInstId =
        VPS_ISS_CAPT_CAL_A_ID;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].coreOps     = NULL;
    idx++;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].drvInstId =
        VPS_ISS_CAPT_CAL_A_CPI;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].coreOps     = NULL;
    idx++;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].drvInstId =
        VPS_ISS_CAPT_CAL_A_OTF0_ID;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].coreInstObj = NULL;
    gVpsDrvPlatDataTda3xx.issCaptDrvInitPrms[idx].coreOps     = NULL;
    GT_assert(VpsDrvTrace, (VPS_ISS_CAPT_INST_MAX_ID > idx));

#endif /* VPS_CAL_BUILD */
#endif /* (defined(VPS_VIP_BUILD) ||... */

#ifdef VPS_M2M_BUILD

    BspUtils_memset(&gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms, 0U, sizeof (gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms));
    idx = 0U;

#ifdef VPS_ISS_BUILD
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].drvInstId = VPS_M2M_ISS_INST_CAL_ISP;
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].isMultiChSupported = FALSE;
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].numCores           = 1U;
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].coreInstId[0U]     = VpsCore_issM2mGetCoreInstObj(VPSCORE_ISS_M2M_ISP_INST_0);
    idx++;
#endif /* VPS_ISS_BUILD */

#ifdef VPS_SIMCOP_BUILD
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].drvInstId = VPS_M2M_ISS_INST_SIMCOP;
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].isMultiChSupported = FALSE;
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].numCores           = 1U;
    gVpsDrvPlatDataTda3xx.m2mIntfDrvInitPrms[idx].coreInstId[0U]     = (VpsCore_Inst *) VPSCORE_ISS_SIMCOP_INST_0;
    idx++;
#endif /* VPS_SIMCOP_BUILD */

#endif /* VPS_M2M_BUILD */
    printf( "End vpsInitTda3xxPlatData...\r\n" );
}

