//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
// File containing the VPS HAL init, deinit and other common functions.
//---------------------------------------------------------------------------------------------------------------------
#include <stdint.h>
#include <stddef.h>
#include <ti/csl/tistdtypes.h>
#include <ti/drv/vps/include/common/trace.h>
#include <ti/drv/vps/include/common/stw_config.h>
#include <ti/csl/soc.h>
#include <ti/drv/vps/src/vpslib/hal/vpshal_platform.h>
#include <ti/drv/vps/src/vpslib/common/vps_common.h>
//---------------------------------------------------------------------------------------------------------------------
static VpsHal_PlatformData           gVpsHalPlatDataTda3xx =
{
#if defined (VPS_VIP_BUILD)
    0U,                                                 /* numVpdma */
    {
        /* vpdmaInstPrms:halId,             vpdmaInstPrms:baseAddr */
#ifdef VPS_VIP_BUILD
  #if defined (VPS_VIP1_BUILD)
        {
            VPSHAL_VIP1_VPDMA, SOC_VIP1_VIP_VPDMA_REGS_BASE, 0U, 0U, 0U, 0U, 0U
        },
  #endif    /* #if defined (VPS_VIP1_BUILD) */
#endif
    },

    0U,                                                 /* numCsc */
    {
        /* cscInstPrms:halId,    cscInstPrms:baseAddr */
#ifdef VPS_VIP_BUILD
  #if defined (VPS_VIP1_BUILD)
        {VPSHAL_VIP1_S0_CSC,   SOC_VIP1_VIP_CSC_SLICE0_REGS_BASE   },
        {VPSHAL_VIP1_S1_CSC,   SOC_VIP1_VIP_CSC_SLICE1_REGS_BASE   },
  #endif    /* #if defined (VPS_VIP1_BUILD) */
#endif
    },

    0U,                                                 /* numSc */
    {
        /*  scInstPrms:halId,               scInstPrms:baseAddr,
         *  scInstPrms:coeffConfigDest,     scInstPrms:isHqScaler */
#ifdef VPS_VIP_BUILD
  #if defined (VPS_VIP1_BUILD)
        {
            VPSHAL_VIP1_S0_SC, SOC_VIP1_VIP_SC_SLICE0_REGS_BASE,
            VPSHAL_VPDMA_CONFIG_DEST_SC3, FALSE
        },
        {
            VPSHAL_VIP1_S1_SC, SOC_VIP1_VIP_SC_SLICE1_REGS_BASE,
            VPSHAL_VPDMA_CONFIG_DEST_SC4, FALSE
        },
  #endif    /* #if defined (VPS_VIP1_BUILD) */
#endif
    },

    0U,                                                 /* numVipTop */
    {
        /* vipTopInstPrms:halId, vipTopInstPrms:baseAddr */
  #if defined (VPS_VIP1_BUILD)
        {VPSHAL_VIP1,          SOC_VIP1_VIP_REGS_BASE              }
  #endif    /* #if defined (VPS_VIP1_BUILD) */
    },

    0U,                                                 /* numVipPort */
    {
        /* vipPortInstPrms:halId,           vipPortInstPrms:baseAddr */
  #if defined (VPS_VIP1_BUILD)
        {VPSHAL_VIP1_S0_PORTA, SOC_VIP1_VIP_PARSER_SLICE0_REGS_BASE},
        {VPSHAL_VIP1_S0_PORTB, SOC_VIP1_VIP_PARSER_SLICE0_REGS_BASE},
        {VPSHAL_VIP1_S1_PORTA, SOC_VIP1_VIP_PARSER_SLICE1_REGS_BASE},
        {VPSHAL_VIP1_S1_PORTB, SOC_VIP1_VIP_PARSER_SLICE1_REGS_BASE}
  #endif    /* #if defined (VPS_VIP1_BUILD) */
    },
#endif

    CSL_VPS_DSS_VID_PIPE_PER_CNT,                   /* numVidPipe */
    {
        /* VidInstParams:halId */
        {VPSHAL_DSS_DISPC_VID1},
        {VPSHAL_DSS_DISPC_VID2},
        {VPSHAL_DSS_DISPC_GFX}
    },

    CSL_VPS_DSS_WB_PIPE_PER_CNT,                           /* numWbPipe */
    {
        {VPSHAL_DSS_DISPC_WB1}
    },

    CSL_VPS_DSS_OVLY_MGR_PER_CNT,                   /* numOvlyMgrs */
    {
        /* DispcOvlInstParams:halId */
        {VPSHAL_DSS_DISPC_LCD1},
        {VPSHAL_DSS_DISPC_LCD2}
    },

#if !(defined (VPS_VIP_BUILD)) && !(defined (VPS_DSS_BUILD))
    0U
#endif
};

#ifdef VPS_SIMCOP_BUILD
static vpshalissSimcopPlatformData_t gVpsHalSimcopPlatDataTda3xx =
{
    CSL_ISS_SIMCOP_PER_CNT,                /* numSimcop */
    {                                      /* simcopInitPrms[] */
        {
            ISSHAL_SIMCOP_INST_0,          /* instId              */
            ISS_SIMCOP_BASE_REGISTER,      /* simcopBaseAddress   */
            ISS_SIMCOP_DMA_REGISTERS,      /* simcopDmaBaseAddress*/
            CSL_ISS_SIMCOP_NUM_ACT_CTX,    /* numActiveContext*/
            CSL_ISS_SIMCOP_NUM_DMA_CH,     /* numDmaChannels  */
            CSL_ISS_SIMCOP_NUM_IRQ,        /* numIrq          */
            CSL_ISS_SIMCOP_MAX_HWSEQ_STEPS /* maxHwSeqSteps   */
        }
    },
    {                                /* ldcInitPrms[ISSHAL_SIMCOP_MAX_INST] */
        {
            ISSHAL_LDC_INST_0,       /* instId              */
            ISS_SIMCOP_LDC_REGISTERS /* simcopBaseAddress   */
        }
    },
    {                                 /* vtnfInitPrms[ISSHAL_SIMCOP_MAX_INST] */
        {
            ISSHAL_VTNF_INST_0,       /* instId              */
            ISS_SIMCOP_VTNF_REGISTERS /* simcopBaseAddress   */
        }
    }
};
#endif /* DVPS_SIMCOP_BUILD */

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

/**
 *  \brief VPS HAL init function.
 */
Int32 VpsHal_init(const VpsHal_InitParams *initPrms)
{
    Int32  retVal = BSP_SOK;
    VpsHal_PlatformData *platData;
#if defined (VPS_VIP_BUILD) || defined (VPS_DSS_BUILD)
    UInt32 cnt;
#endif
#ifdef VPS_DSS_BUILD
    VpsHal_DispcVidInstParams pipeInstPrms[VPSHAL_DSS_DISPC_PIPE_MAX_INST];
#endif

    GT_assert(VpsHalTrace, NULL != initPrms);

    /* Get Platform data */
    platData = VpsHal_getPlatformData();
    GT_assert(VpsHalTrace, NULL != platData);

#ifdef VPS_VIP_BUILD
    if(platData->numVipTop > 0)
    {
        /* Init VIP Top HAL */
        GT_assert(VpsHalTrace,
                  platData->numVipTop <= VPSHAL_VIP_TOP_MAX_INST);
        retVal = VpsHal_vipTopInit(
            platData->numVipTop,
            &platData->vipTopInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(VpsHalTrace, GT_ERR, "VPS HAL: VIP Top init failed\r\n");
        }
    }
    else
    {
        retVal = BSP_EFAIL;
    }
#endif

#if defined (VPS_VIP_BUILD)
    if((BSP_SOK == retVal) && (platData->numVpdma > 0))
    {
        for(cnt = 0U; cnt < platData->numVpdma; cnt++)
        {
            platData->vpdmaInstPrms[cnt].isAddrTransReq =
                (Bool) initPrms->isAddrTransReq;
            platData->vpdmaInstPrms[cnt].virtBaseAddr  = initPrms->virtBaseAddr;
            platData->vpdmaInstPrms[cnt].physBaseAddr  = initPrms->physBaseAddr;
            platData->vpdmaInstPrms[cnt].isCacheOpsReq =
                initPrms->isCacheOpsReq;
            platData->vpdmaInstPrms[cnt].isCacheFlushReq =
                initPrms->isCacheFlushReq;
        }

        /* Init VPDMA HAL */
        GT_assert(VpsHalTrace, platData->numVpdma <= VPSHAL_VPDMA_MAX_INST);
        retVal = VpsHal_vpdmaInit(
            platData->numVpdma,
            &platData->vpdmaInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(VpsHalTrace, GT_ERR, "VPS HAL: VPDMA init failed\r\n");
        }
    }
    else
    {
        retVal = BSP_EFAIL;
    }

    if((BSP_SOK == retVal) && (platData->numCsc > 0))
    {
        /* Init CSC HAL */
        GT_assert(VpsHalTrace, platData->numCsc <= VPSHAL_CSC_MAX_INST);
        retVal = VpsHal_cscInit(
            platData->numCsc,
            &platData->cscInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(VpsHalTrace, GT_ERR, "VPS HAL: CSC init failed\r\n");
        }
    }

    if((BSP_SOK == retVal) && (platData->numSc > 0))
    {
        /* Init SC HAL */
        GT_assert(VpsHalTrace, platData->numSc <= VPSHAL_SC_MAX_INST);
        retVal = VpsHal_scInit(
            platData->numSc,
            &platData->scInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(VpsHalTrace, GT_ERR, "VPS HAL: SC init failed\r\n");
        }
    }
#endif

#ifdef VPS_VIP_BUILD
    if((BSP_SOK == retVal) && (platData->numVipPort > 0))
    {
        /* Init VIP Port HAL */
        GT_assert(VpsHalTrace,
                  platData->numVipPort <= VPSHAL_VIP_PORT_MAX_INST);
        retVal = VpsHal_vipPortInit(
            platData->numVipPort,
            &platData->vipPortInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(VpsHalTrace, GT_ERR, "VPS HAL: VIP PORT init failed\r\n");
        }
    }
#endif

#ifdef VPS_DSS_BUILD
    if((BSP_SOK == retVal) && ((platData->numVidPipe + platData->numWbPipe) > 0))
    {
        /* Init Video Pipeline HAL */
        GT_assert(VpsHalTrace,
                  platData->numVidPipe <= VPSHAL_DSS_DISPC_VID_MAX_INST);

        GT_assert(VpsHalTrace,
                  platData->numWbPipe <= VPSHAL_DSS_DISPC_WB_MAX_INST);

        /* We have same HAL for VID,GFX and WB pipelines, but libdata is
           sepatrate for vid/gfx and WB pipeliens, so combining the initparams
           and calling single init function. */
        for(cnt = 0; cnt < platData->numVidPipe; cnt++)
        {
            pipeInstPrms[cnt] = platData->vidPipeInstPrms[cnt];
        }

        for(cnt = 0; cnt < platData->numWbPipe; cnt++)
        {
            pipeInstPrms[platData->numVidPipe +
                         cnt] = platData->wbPipeInstPrms[cnt];
        }

        retVal = VpsHal_dssVidInit(
            (platData->numVidPipe + platData->numWbPipe),
            &pipeInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(
                VpsHalTrace, GT_ERR,
                "VPS HAL: Video Pipeline HAL init failed\r\n");
        }
    }

    if((BSP_SOK == retVal) && (platData->numOvlyMgr > 0))
    {
        /* Init Overlay Manager HAL */
        GT_assert(VpsHalTrace,
                  platData->numOvlyMgr <= VPSHAL_DSS_DISPC_OVLY_MAX_INST);
        retVal = VpsHal_dssOvlyInit(
            platData->numOvlyMgr,
            &platData->ovlyMgrInstPrms[0U],
            NULL);
        if(retVal != BSP_SOK)
        {
            GT_0trace(VpsHalTrace, GT_ERR,
                      "VPS HAL: Overlay Manager init failed\r\n");
        }
    }
#endif

    return (retVal);
}

/**
 *  \brief VPS HAL deinit function.
 */
Int32 VpsHal_deInit(void)
{
    Int32 retVal = BSP_SOK;
    VpsHal_PlatformData *platData;

    /* Get Platform data */
    platData = VpsHal_getPlatformData();
    GT_assert(VpsHalTrace, NULL != platData);

#if defined (VPS_VIP_BUILD)
    retVal += VpsHal_scDeInit(NULL);
    retVal += VpsHal_cscDeInit(NULL);
    retVal += VpsHal_vpdmaDeInit(NULL);
#endif

#ifdef VPS_VIP_BUILD
    retVal += VpsHal_vipPortDeInit(NULL);
    retVal += VpsHal_vipTopDeInit(NULL);
#endif

    return (retVal);
}

/**
 *  \brief Returns HAL platform data.
 */
VpsHal_PlatformData *VpsHal_getPlatformData(void)
{
    VpsHal_PlatformData *platData = NULL;

  #ifdef VPS_VIP_BUILD
    UInt32 numVip;
  #endif

    {
        platData = &gVpsHalPlatDataTda3xx;

    #if defined (VPS_VIP_BUILD)
        platData->numVpdma = 0U;
        platData->numCsc   = 0U;
        platData->numSc    = 0U;

        numVip = 0U;
      #if defined (VPS_VIP1_BUILD)
        numVip++;
      #endif    /* #if defined (VPS_VIP1_BUILD) */

        platData->numVipTop = numVip;
        platData->numVipPort = numVip * 4U;
        platData->numVpdma += numVip;
        /* 2 CSC per VIP */
        platData->numCsc += (numVip * 2U);
        /* 2 SC per VIP */
        platData->numSc += (numVip * 2U);
    #endif
    }

    return (platData);
}

#ifdef VPS_SIMCOP_BUILD
/**
 *  \brief Returns HAL platform data for SIMCOP instances
 */
vpshalissSimcopPlatformData_t *VpsHal_getSimcopPlatformData(void)
{
    vpshalissSimcopPlatformData_t *halData = NULL;

    {
        halData = &gVpsHalSimcopPlatDataTda3xx;
    }
    return halData;
}

#endif

