//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
// File containing the VPS library init and deinit functions.
//---------------------------------------------------------------------------------------------------------------------
#include <stdint.h>
#include <stdio.h>
#include <stddef.h>
#include "tistdtypes.h"
#include "bsp_types.h"
#include "trace.h"
#include "stw_config.h"
#include "bsp_utils.h"
#include "soc.h"
#include "fvid2_dataTypes.h"
#include "vps_dataTypes.h"
#include "vps_common.h"
#include "vpscore.h"
#include "cal_evtmgr.h"

#ifdef VPS_VIP_BUILD
#include "vpscore_vip.h"
#endif

#include "vpscore_dss.h"
#include "vpscore_dssWb.h"
#include "vpscore_dctrl.h"

#ifdef VPS_CAL_BUILD
#include "irm_resrcMgr.h"
#include "vpscore_capture_cal.h"
#endif

#include "vps_common.h"
//---------------------------------------------------------------------------------------------------------------------
// VPS Init object to store HAL handles and other variables.
typedef struct
{
#if defined (VPS_VIP_BUILD)
    VpsHal_Handle vpdmaHandle[VPSHAL_VPDMA_INST_ID_MAX];
    VpsHal_Handle cscHandle[VPSHAL_CSC_INST_ID_MAX];
    VpsHal_Handle scHandle[VPSHAL_SC_INST_ID_MAX];
#endif

#ifdef VPS_VIP_BUILD
    VpsHal_Handle vipTopHandle[VPSHAL_VIP_TOP_MAX_INST];
    VpsHal_Handle vipPortHandle[VPSHAL_VIP_PORT_INST_ID_MAX];

    VpsCore_Inst  vipCoreInstObj[VPSHAL_VIP_TOP_MAX_INST];
#endif

#ifdef VPS_DSS_BUILD
    VpsHal_Handle vidPipeHandle[VPSHAL_DSS_DISPC_VID_INST_ID_MAX];
    VpsHal_Handle ovlHandle[VPSHAL_DSS_DISPC_OVLY_INST_ID_MAX];
    UInt32        dctrlHandle;

    VpsCore_Inst  dssCoreInstObj;
    VpsCore_Inst  dssWbCoreInstObj;
    VpsCore_Inst  dctrlCoreInstObj;
#endif
#if !(defined (VPS_VIP_BUILD)) && !(defined (VPS_DSS_BUILD))
    UInt32        dummy;
#endif
} VpsLibInit_Obj;

//---------------------------------------------------------------------------------------------------------------------
static Int32 Vps_commonInit(const VpsLib_InitParams *initPrms);
static Int32 Vps_commonDeInit(void);
static Int32 VpsHal_open(void);
static Int32 VpsHal_close(void);
#ifdef VPS_VIP_BUILD
static Int32 Vps_captVipCoreInit(void);
static Int32 Vps_captVipCoreDeInit(void);
#endif

#ifdef VPS_DSS_BUILD
static Int32 Vps_dispCoreInit(const VpsLib_InitParams *initPrms);
static Int32 Vps_dispCoreDeInit(void);
#endif

#ifdef VPS_CAL_BUILD
static Int32 Vps_issCaptCoreInit(void);
static Int32 Vps_issCaptCoreDeInit(void);
#endif


static VpsLibInit_Obj      gVpsLibInitObj;

/**
 *  \brief VPS Lib init function.
 */
Int32 VpsLib_init(const VpsLib_InitParams *initPrms)
{
    Int32 retVal = BSP_SOK;
    VpsLib_InitParams localInitPrms;
    VpsHal_InitParams halInitPrms;

    /* Memset to zero is must as this sets the pointers to NULL */
    printf( "Inside VpsLib_init...->BspUtils_memset\r\n" );
    BspUtils_memset(&gVpsLibInitObj, 0, sizeof (gVpsLibInitObj));

    /* If NULL is passed use default */
    if (NULL == initPrms)
    {
        initPrms = &localInitPrms;
        VpsLibInitParams_init( &localInitPrms );
        printf( "NULL Params -> VpsLibInitParams_init \r\n" );
    }

    halInitPrms.isAddrTransReq  = initPrms->isAddrTransReq;
    halInitPrms.virtBaseAddr    = initPrms->virtBaseAddr;
    halInitPrms.physBaseAddr    = initPrms->physBaseAddr;
    halInitPrms.isCacheOpsReq   = initPrms->isCacheOpsReq;
    halInitPrms.isCacheFlushReq = initPrms->isCacheFlushReq;

    retVal += VpsHal_init( &halInitPrms );
    printf( "VpsHal_init done.\r\n" );
    retVal += VpsHal_open();
    printf( "VpsHal_open done.\r\n" );
    retVal += Vps_commonInit(initPrms);
    printf( "Vps_commonInit done.\r\n" );

#ifdef VPS_VIP_BUILD
    retVal += Vps_captVipCoreInit();
#endif

#ifdef VPS_DSS_BUILD
    retVal += Vps_dispCoreInit(initPrms);
#endif

#ifdef VPS_CAL_BUILD
    retVal += Vps_issCaptCoreInit();
#endif /* VPS_CAL_BUILD */

    return (retVal);
}

/**
 *  \brief VPS Lib deinit function.
 */
Int32 VpsLib_deInit(Ptr args)
{
    Int32 retVal = BSP_SOK;

#ifdef VPS_CAL_BUILD
    retVal += Vps_issCaptCoreDeInit();
#endif /* VPS_CAL_BUILD */

#ifdef VPS_DSS_BUILD
    retVal += Vps_dispCoreDeInit();
#endif

#ifdef VPS_VIP_BUILD
    retVal += Vps_captVipCoreDeInit();
#endif
    retVal += Vps_commonDeInit();
    retVal += VpsHal_close();
    retVal += VpsHal_deInit();

    return (retVal);
}

VpsLib_PlatformData *VpsLib_getPlatformData(void)
{
    VpsLib_PlatformData *platData = NULL;
            platData->hal = VpsHal_getPlatformData();
    return (platData);
}

#ifdef VPS_VIP_BUILD
VpsCore_Inst VpsLib_getVipCoreInstObj(UInt32 vipInstId)
{
    VpsCore_Inst    instObj = NULL;
    VpsLibInit_Obj *pObj    = &gVpsLibInitObj;

    if (vipInstId < VPSHAL_VIP_TOP_MAX_INST)
    {
        instObj = pObj->vipCoreInstObj[vipInstId];
    }

    return (instObj);
}

#endif

#ifdef VPS_DSS_BUILD
VpsCore_Inst VpsLib_getDssCoreInstObj(void)
{
    VpsCore_Inst    instObj = NULL;
    VpsLibInit_Obj *pObj    = &gVpsLibInitObj;

    instObj = pObj->dssCoreInstObj;

    return (instObj);
}

VpsCore_Inst VpsLib_getDssWbCoreInstObj(void)
{
    VpsCore_Inst    instObj = NULL;
    VpsLibInit_Obj *pObj    = &gVpsLibInitObj;

    instObj = pObj->dssWbCoreInstObj;

    return (instObj);
}

VpsCore_Inst VpsLib_getDctrlCoreInstObj(void)
{
    VpsCore_Inst    instObj = NULL;
    VpsLibInit_Obj *pObj    = &gVpsLibInitObj;

    instObj = pObj->dctrlCoreInstObj;

    return (instObj);
}

VpsHal_Handle VpsLib_getDssOvlyHalHandle(UInt32 ovlyInstId)
{
    VpsHal_Handle   halHandle = NULL;
    VpsLibInit_Obj *pObj      = &gVpsLibInitObj;

    if (ovlyInstId < VPSHAL_DSS_DISPC_OVLY_INST_ID_MAX)
    {
        halHandle = pObj->ovlHandle[ovlyInstId];
    }

    return (halHandle);
}

#endif

#if defined (VPS_VIP_BUILD)
VpsHal_Handle VpsLib_getVpdmaHalHandle(UInt32 vpdmaInstId)
{
    VpsHal_Handle   halHandle = NULL;
    VpsLibInit_Obj *pObj      = &gVpsLibInitObj;

    if (vpdmaInstId < VPSHAL_VPDMA_INST_ID_MAX)
    {
        halHandle = pObj->vpdmaHandle[vpdmaInstId];
    }

    return (halHandle);
}

#endif

static Int32 Vps_commonInit(const VpsLib_InitParams *initPrms)
{
    Int32 retVal = BSP_SOK;
    VpsLib_PlatformData *platData;

    /* Get Platform data */
    platData = VpsLib_getPlatformData();
    GT_assert(VpsTrace, (NULL != platData));

#if defined (VPS_VIP_BUILD)
    if (BSP_SOK == retVal)
    {
        /* Resource Manager Init */
        retVal = Vrm_init(
            platData->numResrcMgr,
            &platData->resrcMgrInstPrms[0U]);
        if (retVal != BSP_SOK)
        {
            GT_0trace(VpsTrace, GT_ERR, "VPS: Resource Manager init failed\r\n");
        }
    }

    if (BSP_SOK == retVal)
    {
        UInt32 i;
        /* Update the interrupt numbers from init params. */
        for (i= 0U; i< VEM_MAX_INST; i++)
        {
            switch (platData->evtMgrInstPrms[i].instId)
            {
                case VEM_INST_ID_VIP1:
                    platData->evtMgrInstPrms[i].irqNumber = initPrms->irqParams.vipIrqNum[0U];
                    break;
                case VEM_INST_ID_VIP2:
                    platData->evtMgrInstPrms[i].irqNumber = initPrms->irqParams.vipIrqNum[1U];
                    break;
                case VEM_INST_ID_VIP3:
                    platData->evtMgrInstPrms[i].irqNumber = initPrms->irqParams.vipIrqNum[2U];
                    break;
                case VEM_INST_ID_VPE1:
                    platData->evtMgrInstPrms[i].irqNumber = initPrms->irqParams.vpeIrqNum;
                    break;
                default:
                    break;
            }
        }
        /* Event Manager Init */
        retVal = Vem_init(platData->numEvtMgr, &platData->evtMgrInstPrms[0U]);
        if (retVal != BSP_SOK)
        {
            GT_0trace(VpsTrace, GT_ERR, "VPS: Event Manager init failed\r\n");
        }
    }
#endif

#ifdef VPS_CAL_BUILD
    if (BSP_SOK == retVal)
    {
        platData->calHal = (const Ptr *) VpsHal_getCalPlatformData();

        retVal = Vps_calEmInit(platData->numCalEmInst,
                           (const calemInitParams_t *)platData->calEmInitPrms);
        if (retVal != BSP_SOK)
        {
            GT_0trace(VpsTrace, GT_ERR, "VPS: CAL EM Init Failed\r\n");
        }
    }
#endif

    return (retVal);
}

static Int32 Vps_commonDeInit(void)
{
    Int32 retVal = BSP_SOK;

#if defined (VPS_VIP_BUILD)
    retVal += Vem_deInit();
    if (retVal != BSP_SOK)
    {
        GT_0trace(VpsTrace, GT_ERR, "VPS: Event Manager deinit failed\r\n");
    }

    retVal += Vrm_deInit();
    if (retVal != BSP_SOK)
    {
        GT_0trace(VpsTrace, GT_ERR, "VPS: Resource Manager deinit failed\r\n");
    }
#endif

#ifdef VPS_CAL_BUILD
    retVal += Vps_calEmDeInit();
    if (retVal != BSP_SOK)
    {
        GT_0trace(VpsTrace, GT_ERR, "VPS: CAL EM DeInit Failed\r\n");
    }
#endif

    return (retVal);
}

static Int32 VpsHal_open(void)
{
    Int32  retVal = BSP_SOK;
    VpsLib_PlatformData *platData;
#if defined (VPS_VIP_BUILD) || defined (VPS_DSS_BUILD)
    UInt32 halCnt, halId;
    VpsLibInit_Obj            *pObj = &gVpsLibInitObj;
#endif

#if defined (VPS_VIP_BUILD)
    VpsHal_VpdmaOpenParams     vpdmaOpenPrms;
    VpsHal_CscOpenParams       cscOpenPrms;
    VpsHal_ScOpenParams        scOpenPrms;
    VpsHal_VipTopOpenParams    vipTopOpenPrms;
    VpsHal_VipPortOpenParams   vipPortOpenPrms;
#endif

#ifdef VPS_DSS_BUILD
    VpsHal_DispcVidOpenParams  vidPipeOpenPrms;
    VpsHal_DispcOvlyOpenParams ovlyOpenPrms;
#endif

    /* Get Platform data */
    platData = VpsLib_getPlatformData();
    GT_assert(VpsTrace, (NULL != platData));
    printf("VpsLib_getPlatformData done.\n");


#if defined (VPS_VIP_BUILD)
    if ((BSP_SOK == retVal) && (platData->hal->numVpdma > 0))
    {
        /* Open VPDMA HALs */
        for (halCnt = 0U; halCnt < platData->hal->numVpdma; halCnt++)
        {
            halId = platData->hal->vpdmaInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_VPDMA_INST_ID_MAX));
            vpdmaOpenPrms.halId = halId;
            vpdmaOpenPrms.ver   = VPSHAL_VPDMA_VER_DEFAULT;
            pObj->vpdmaHandle[halId] = VpsHal_vpdmaOpen(&vpdmaOpenPrms);
            GT_assert(VpsTrace, (NULL != pObj->vpdmaHandle[halId]));
        }
    }

    if ((BSP_SOK == retVal) && (platData->hal->numCsc > 0))
    {
        /* Open CSC HALs */
        for (halCnt = 0U; halCnt < platData->hal->numCsc; halCnt++)
        {
            halId = platData->hal->cscInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_CSC_INST_ID_MAX));
            cscOpenPrms.halId      = halId;
            pObj->cscHandle[halId] = VpsHal_cscOpen(&cscOpenPrms);
            GT_assert(VpsTrace, (NULL != pObj->cscHandle[halId]));
        }
    }

    if ((BSP_SOK == retVal) && (platData->hal->numSc > 0))
    {
        /* Open SC HALs */
        for (halCnt = 0U; halCnt < platData->hal->numSc; halCnt++)
        {
            halId = platData->hal->scInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_SC_INST_ID_MAX));
            scOpenPrms.halId      = halId;
            pObj->scHandle[halId] = VpsHal_scOpen(&scOpenPrms);
            GT_assert(VpsTrace, (NULL != pObj->scHandle[halId]));
        }
    }
#endif

#ifdef VPS_VIP_BUILD
    if ((BSP_SOK == retVal) && (platData->hal->numVipTop > 0))
    {
        /* Open VIP Top HALs */
        for (halCnt = 0U; halCnt < platData->hal->numVipTop; halCnt++)
        {
            halId = platData->hal->vipTopInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_VIP_TOP_MAX_INST));
            vipTopOpenPrms.halId      = halId;
            pObj->vipTopHandle[halId] = VpsHal_vipTopOpen(&vipTopOpenPrms);
            GT_assert(VpsTrace, (NULL != pObj->vipTopHandle[halId]));
        }
    }

    if ((BSP_SOK == retVal) && (platData->hal->numVipPort > 0))
    {
        /* Open VIP Port HALs */
        for (halCnt = 0U; halCnt < platData->hal->numVipPort; halCnt++)
        {
            halId = platData->hal->vipPortInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_VIP_PORT_INST_ID_MAX));
            vipPortOpenPrms.halId      = halId;
            pObj->vipPortHandle[halId] = VpsHal_vipPortOpen(&vipPortOpenPrms);
            GT_assert(VpsTrace, (NULL != pObj->vipPortHandle[halId]));
        }
    }
#endif

#ifdef VPS_DSS_BUILD
    printf("*** VIDEO PIPES ***\n");
    printf("The number of Video Pipes - %d", platData->hal->numVidPipe );
    if( (BSP_SOK == retVal) && (platData->hal->numVidPipe > 0) )
    {
        // Open VIDEO Pipeline HALs
        printf("Start to Open VIDEO Pipeline HALs...\n");
        printf("The number of video Pipes - %d\n", platData->hal->numVidPipe );
        for( halCnt = 0U; halCnt < platData->hal->numVidPipe; halCnt++ )
        {
            halId = platData->hal->vidPipeInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_DSS_DISPC_VID_MAX_INST));
            vidPipeOpenPrms.halId      = halId;
            printf("Trying to dssVid Open HAL count - %d\n", halCnt );
            pObj->vidPipeHandle[halId] = VpsHal_dssVidOpen( &vidPipeOpenPrms );
            GT_assert(VpsTrace, (NULL != pObj->vidPipeHandle[halId]));
            printf("Open VIDEO Pipeline HALs done.\n");
        }
    }
    /* WB   
    printf("The number of WB Pipes - %d", platData->hal->numWbPipe);
    if( (BSP_SOK == retVal) && (platData->hal->numWbPipe > 0) )
    {
        // Open WB Pipeline HALs
        printf("Start to Open WB Pipeline HALs...\n");

        for( halCnt = 0U; halCnt < platData->hal->numWbPipe; halCnt++ )
        {
            halId = platData->hal->wbPipeInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_DSS_DISPC_PIPE_MAX_INST));
            vidPipeOpenPrms.halId      = halId;
            pObj->vidPipeHandle[halId] = VpsHal_dssVidOpen( &vidPipeOpenPrms );
            GT_assert(VpsTrace, (NULL != pObj->vidPipeHandle[halId]));
            printf("Open WB Pipeline HALs done.\n");
        }
    }*/

    printf("--- The number of Overlay Managers - %d\n", platData->hal->numOvlyMgr );
    if ((BSP_SOK == retVal) && (platData->hal->numOvlyMgr > 0))
    {
        // Open Overlay HALs
        printf("Start to Open Overlay HALs...\n");
        for( halCnt = 0U; halCnt < platData->hal->numOvlyMgr; halCnt++ )
        {
            halId = platData->hal->ovlyMgrInstPrms[halCnt].halId;
            GT_assert(VpsTrace, (halId < VPSHAL_DSS_DISPC_OVLY_INST_ID_MAX));
            ovlyOpenPrms.halId     = halId;
            pObj->ovlHandle[halId] = VpsHal_dssOvlyOpen(&ovlyOpenPrms);
            GT_assert(VpsTrace, (NULL != pObj->ovlHandle[halId]));
            printf("Open Overlay HALs done.\n");
        }
    }
#endif

    return (retVal);
}

static Int32 VpsHal_close(void)
{
    Int32           retVal = BSP_SOK;
#if defined (VPS_VIP_BUILD) || defined (VPS_DSS_BUILD)
    UInt32          halCnt;
    VpsLibInit_Obj *pObj = &gVpsLibInitObj;
#endif

#ifdef VPS_VIP_BUILD
    /* Close VIP Port HALs */
    for (halCnt = 0U; halCnt < VPSHAL_VIP_PORT_INST_ID_MAX; halCnt++)
    {
        if (pObj->vipPortHandle[halCnt] != NULL)
        {
            retVal = VpsHal_vipPortClose(pObj->vipPortHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->vipPortHandle[halCnt] = NULL;
        }
    }

    /* Close VIP Top HALs */
    for (halCnt = 0U; halCnt < VPSHAL_VIP_TOP_MAX_INST; halCnt++)
    {
        if (pObj->vipTopHandle[halCnt] != NULL)
        {
            retVal = VpsHal_vipTopClose(pObj->vipTopHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->vipTopHandle[halCnt] = NULL;
        }
    }
#endif

#if defined (VPS_VIP_BUILD)
    /* Close SC HALs */
    for (halCnt = 0U; halCnt < VPSHAL_SC_INST_ID_MAX; halCnt++)
    {
        if (pObj->scHandle[halCnt] != NULL)
        {
            retVal = VpsHal_scClose(pObj->scHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->scHandle[halCnt] = NULL;
        }
    }

    /* Close CSC HALs */
    for (halCnt = 0U; halCnt < VPSHAL_CSC_INST_ID_MAX; halCnt++)
    {
        if (pObj->cscHandle[halCnt] != NULL)
        {
            retVal = VpsHal_cscClose(pObj->cscHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->cscHandle[halCnt] = NULL;
        }
    }

    /* Close VPDMA HALs */
    for (halCnt = 0U; halCnt < VPSHAL_VPDMA_INST_ID_MAX; halCnt++)
    {
        if (pObj->vpdmaHandle[halCnt] != NULL)
        {
            retVal = VpsHal_vpdmaClose(pObj->vpdmaHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->vpdmaHandle[halCnt] = NULL;
        }
    }
#endif

#ifdef VPS_DSS_BUILD
    /* Close VIDEO Pipeline HALs */
    for (halCnt = 0U; halCnt < VPSHAL_DSS_DISPC_VID_INST_ID_MAX; halCnt++)
    {
        if (pObj->vidPipeHandle[halCnt] != NULL)
        {
            retVal = VpsHal_dssVidClose(pObj->vidPipeHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->vidPipeHandle[halCnt] = NULL;
        }
    }

    /* Close overlay HALs */
    for (halCnt = 0U; halCnt < VPSHAL_DSS_DISPC_OVLY_INST_ID_MAX; halCnt++)
    {
        if (pObj->ovlHandle[halCnt] != NULL)
        {
            retVal = VpsHal_dssOvlyClose(pObj->ovlHandle[halCnt]);
            GT_assert(VpsTrace, (BSP_SOK == retVal));
            pObj->ovlHandle[halCnt] = NULL;
        }
    }

#endif

    return (retVal);
}

#ifdef VPS_VIP_BUILD
static Int32 Vps_captVipCoreInit(void)
{
    Int32  retVal = BSP_SOK;
    UInt32 coreCnt;
    VpsLibInit_Obj            *pObj = &gVpsLibInitObj;
    VpsLib_PlatformData *platData;
    VpsCore_VipInitParams      vipCoreInitPrms;

    /* Get Platform data */
    platData = VpsLib_getPlatformData();
    GT_assert(VpsTrace, (NULL != platData));

    /* Open VIP Core */
    GT_assert(VpsTrace, (0U != platData->hal->numVipTop));
    GT_assert(VpsTrace, (platData->hal->numVipTop <= VPSHAL_VIP_TOP_MAX_INST));
    for (coreCnt = 0U; coreCnt < VPSHAL_VIP_TOP_MAX_INST; coreCnt++)
    {
        if (pObj->vipTopHandle[VPSHAL_VIP1 + coreCnt] != NULL)
        {
            vipCoreInitPrms.vipTopHandle =
                pObj->vipTopHandle[VPSHAL_VIP1 + coreCnt];
                vipCoreInitPrms.vpdmaHandle =
                    pObj->vpdmaHandle[VPSHAL_VIP1_VPDMA + coreCnt];
                vipCoreInitPrms.vemInstId =
                    (Vem_InstId) (VEM_INST_ID_VIP1 + coreCnt);
                vipCoreInitPrms.vrmInstId =
                    (Vrm_InstId) (VRM_INST_ID_VIP1 + coreCnt);
            vipCoreInitPrms.s0PortAHandle =
                pObj->vipPortHandle[VPSHAL_VIP1_S0_PORTA + (4U * coreCnt)];
            vipCoreInitPrms.s0PortBHandle =
                pObj->vipPortHandle[VPSHAL_VIP1_S0_PORTB + (4U * coreCnt)];
            vipCoreInitPrms.s1PortAHandle =
                pObj->vipPortHandle[VPSHAL_VIP1_S1_PORTA + (4U * coreCnt)];
            vipCoreInitPrms.s1PortBHandle =
                pObj->vipPortHandle[VPSHAL_VIP1_S1_PORTB + (4U * coreCnt)];
            vipCoreInitPrms.sc0Handle =
                pObj->scHandle[VPSHAL_VIP1_S0_SC + (2U * coreCnt)];
            vipCoreInitPrms.sc1Handle =
                pObj->scHandle[VPSHAL_VIP1_S1_SC + (2U * coreCnt)];
            vipCoreInitPrms.csc0Handle =
                pObj->cscHandle[VPSHAL_VIP1_S0_CSC + (2U * coreCnt)];
            vipCoreInitPrms.csc1Handle =
                pObj->cscHandle[VPSHAL_VIP1_S1_CSC + (2U * coreCnt)];
            pObj->vipCoreInstObj[coreCnt] = VpsCore_vipInit(&vipCoreInitPrms);
            GT_assert(VpsTrace, (NULL != pObj->vipCoreInstObj[coreCnt]));
        }
    }

    return (retVal);
}

static Int32 Vps_captVipCoreDeInit(void)
{
    Int32           retVal = BSP_SOK;
    UInt32          coreCnt;
    VpsLibInit_Obj *pObj = &gVpsLibInitObj;

    for (coreCnt = 0U; coreCnt < VPSHAL_VIP_TOP_MAX_INST; coreCnt++)
    {
        if (NULL != pObj->vipCoreInstObj[coreCnt])
        {
            retVal += VpsCore_vipDeInit(pObj->vipCoreInstObj[coreCnt]);
            pObj->vipCoreInstObj[coreCnt] = NULL;
        }
    }

    return (retVal);
}

#endif

#ifdef VPS_DSS_BUILD
static Int32 Vps_dispCoreInit(const VpsLib_InitParams *initPrms)
{
    Int32  retVal = BSP_SOK;
    VpsLib_PlatformData *platData;
    VpsCore_DssInitParams dssCoreInitPrms;
    VpsCore_DssWbInitParams dssWbCoreInitPrms = {0};
    VpsCore_DctrlInitParams dctrlCoreInitPrms;
    VpsLibInit_Obj            *pObj = &gVpsLibInitObj;
    UInt32 halId, cnt;

    /* Get Platform data */
    platData = VpsLib_getPlatformData();
    GT_assert(VpsTrace, (NULL != platData));

    /* TODO - Check if this sequence is preoper or not, find a way to avoid
     *             hard codings of array index*/

    /* Init DSS Core */
    GT_assert(VpsTrace, ((platData->hal->numVidPipe) <= \
                         VPSHAL_DSS_DISPC_VID_MAX_INST));
    GT_assert(VpsTrace, ((platData->hal->numWbPipe) <= \
                         VPSHAL_DSS_DISPC_WB_MAX_INST));
    GT_assert(VpsTrace, ((platData->hal->numOvlyMgr) <= \
                         VPSHAL_DSS_DISPC_OVLY_INST_ID_MAX));

    dssCoreInitPrms.dispcIrqNumber = 0U;
    dssCoreInitPrms.dispcIrqNumber = initPrms->irqParams.dssIrqNum;
    for (cnt = 0U; cnt < platData->hal->numVidPipe; cnt++)
    {
        halId = platData->hal->vidPipeInstPrms[cnt].halId;
        dssCoreInitPrms.vidHandle[halId] = pObj->vidPipeHandle[halId];
    }
    for (cnt = 0U; cnt < platData->hal->numOvlyMgr; cnt++)
    {
        halId = platData->hal->ovlyMgrInstPrms[cnt].halId;
        dssCoreInitPrms.ovlHandle[halId] = pObj->ovlHandle[halId];
    }

    pObj->dssCoreInstObj = VpsCore_dssInit(&dssCoreInitPrms);
    GT_assert(VpsTrace, (NULL != pObj->dssCoreInstObj));

    dssWbCoreInitPrms.dispcIrqNumber = 0U;
    dssWbCoreInitPrms.dispcIrqNumber = initPrms->irqParams.dssIrqNum;
    for (cnt = 0U; cnt < platData->hal->numWbPipe; cnt++)
    {
        halId = platData->hal->wbPipeInstPrms[cnt].halId;
        dssWbCoreInitPrms.wbHandle[cnt] = pObj->vidPipeHandle[halId];
    }

    pObj->dssWbCoreInstObj = VpsCore_dssWbInit(&dssWbCoreInitPrms);
    GT_assert(VpsTrace, (NULL != pObj->dssWbCoreInstObj));

    dctrlCoreInitPrms.dispcIrqNumber = 0U;
    dctrlCoreInitPrms.dispcIrqNumber = initPrms->irqParams.dssIrqNum;
    dctrlCoreInitPrms.dctrlId        = 0;

    for (halId = 0U; halId < platData->hal->numOvlyMgr; halId++)
    {
        dctrlCoreInitPrms.ovlHandle[halId] = pObj->ovlHandle[halId];
    }

    pObj->dctrlCoreInstObj = VpsCore_dctrlInit(&dctrlCoreInitPrms);
    GT_assert(VpsTrace, (NULL != pObj->dssCoreInstObj));
    return (retVal);
}

static Int32 Vps_dispCoreDeInit(void)
{
    Int32           retVal = BSP_SOK;
    VpsLibInit_Obj *pObj   = &gVpsLibInitObj;

    if (NULL != pObj->dssCoreInstObj)
    {
        retVal += VpsCore_dssDeInit(pObj->dssCoreInstObj);
        pObj->dssCoreInstObj = NULL;
    }
    if (NULL != pObj->dssWbCoreInstObj)
    {
        retVal += VpsCore_dssWbDeInit(pObj->dssWbCoreInstObj);
        pObj->dssWbCoreInstObj = NULL;
    }
    if (NULL != pObj->dctrlCoreInstObj)
    {
        retVal += VpsCore_dctrlDeInit(pObj->dctrlCoreInstObj);
        pObj->dctrlCoreInstObj = NULL;
    }

    return (retVal);
}

#endif

Bool VpsLib_platformIsTda2xxFamilyBuild(void)
{
    return (TRUE);
}

Bool VpsLib_platformIsTI814xFamilyBuild(void)
{
    return (FALSE);
}

Bool VpsLib_platformIsOmap5FamilyBuild(void)
{
    return (FALSE);
}

Bool VpsLib_platformIsTda3xxFamilyBuild(void)
{
    return (FALSE);
}

#ifdef VPS_CAL_BUILD
static Int32 Vps_issCaptCoreInit(void)
{
    int32_t  status;
    uint32_t instCnt = 0U;
    const VpsLib_PlatformData *platData;
    vcoreissCaptInitParams_t   coreInitParams[VPS_ISS_CAPT_INST_MAX_ID];

    platData = VpsLib_getPlatformData();
    if (NULL != platData)
    {
        GT_assert(VpsTrace, (NULL != platData->calHal));

        coreInitParams[instCnt].instId          = VPS_ISS_CAPT_CAL_A_ID;
        coreInitParams[instCnt].halPlatformData = platData;
        coreInitParams[instCnt].subModules      = SUB_MODULE_CAL_A;
        coreInitParams[instCnt].irqNum = CAL_EM_INST_ID_CAL0;
        instCnt++;
        GT_assert(VpsTrace, instCnt < VPS_ISS_CAPT_INST_MAX_ID);

        /** TODO - The number of core & IP block could be different.
         *      Right now, we have 1 capture block with multiple modes
         *      Each mode represented as a different instance of capture
         *      core.
         *      Require to pass number of IP and number of core instances
         *      desired.
         *      Right now passing the core instances only */
        GT_assert(VpsTrace, instCnt <= VPS_ISS_CAPT_INST_MAX_ID);
        status = VpsCore_issCaptInit(instCnt,
                                     &coreInitParams[0], NULL);
    }
    else
    {
        status = FVID2_EFAIL;
        GT_assert(VpsTrace, FALSE);
/* MISRA.UNREACH.GEN
 * Un Reachable code
 * Name 'status = FVID2_EFAIL;'
 * KW State: Not A Problem -> Waiver -> Case by case
 * MISRAC_WAIVER:
 * In cases where value in the if condition  is dependent on the return of a
 * function and currently the function is hardcoded to return a value. Code is
 * currently unreachable but as the implementation of the function changes, it
 * will not be unreachable
 */
    }
    return status;
}

static Int32 Vps_issCaptCoreDeInit(void)
{
    return (VpsCore_issCaptDeInit(NULL));
}

#endif /* VPS_CAL_BUILD */

