//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//
//---------------------------------------------------------------------------------------------------------------------
/**
 *  \ingroup BSPUTILS_API
 *  \defgroup BSPUTILS_UART_API Application UART helper APIs
 *  @{
 */

/**
 *  \file bsputils_uart.h
 *
 *  \brief Header file to use application level uart helper functions.
 */

#ifndef BSPUTILS_UART_H_
#define BSPUTILS_UART_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/* None */

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/** \brief Uart default timeout */
#define BSP_UTILS_UART_RX_DEFAULT_TIMEOUT   (10U)
/** \brief Timeout value to be passed to have blocking uart read */
#define BSP_UTILS_UART_RX_WAIT_FOREVER      (0xFFFFFFFFU)

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief Initiates Uart module.
 *
 *  This API should be called before calling read or print functions.
 */
void BspUtils_uartInit(void);

/**
 *  \brief Deinitiates Uart module.
 *
 */
void BspUtils_uartDeinit(void);

/**
 *  \brief Returns if the UART initialization is done.
 *
 *  \return TRUE if initialization is done, else FALSE
 */
Bool BspUtils_isUartInitDone(void);

/**
 *  \brief Prints the string on UART console.
 *
 *  \param string   [IN] String to be printed
 *
 */
void BspUtils_uartPrint(char *string);

/**
 *  \brief Reads string from UART console. This breaks on carraige return or
 *  timeout or char read exceeds the buffer size. This will also terminate the
 *  string with NULL character.
 *
 * \param  pRxBuffer       [OUT] Character read from UART
 *
 * \param  bufSize         [IN] Length of the pRxBuffer in Bytes
 *
 * \param  timeoutVal      [IN] Timeout value in seconds. Pass
 *                              BSP_UTILS_UART_RX_WAIT_FOREVER to wait
 *                              forever.
 *
 * \return  None
 */
Int32 BspUtils_uartGetStr(char *pRxBuffer, UInt32 bufSize, UInt32 timeoutVal);

/**
 *  \brief Reads a char from UART console.
 *
 * \param  pRxBuffer       [OUT] Character read from UART
 *
 * \param  timeoutVal      [IN] Timeout value in seconds. Pass
 *                              BSP_UTILS_UART_RX_WAIT_FOREVER to wait
 *                              forever.
 *
 * \return  None
 */
Int32 BspUtils_uartGetChar(char *pRxBuffer, UInt32 timeoutVal);

/**
 * \brief Reads integer from the UART console
 *        BspUtils_uartInit needs to be called prior to any uarRead
 *
 * \param  pInt            [OUT] Character read from UART
 *
 * \param  timeoutVal      [IN] timeout value in seconds
 *
 * \return BSP_SOK upon success or return error
 */
Int32 BspUtils_uartGetNum(Int32 *pInt, UInt32 timeoutVal);

#ifdef __cplusplus
}
#endif

#endif  /* #define BSPUTILS_APP_H_ */

/* @} */
