/*
 *  Copyright (c) Texas Instruments Incorporated 2013-2016
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

 /**
 *  \defgroup HS_I2C_LLD_API HS I2C Driver API
 *
 *  @{
 */
/**
 * \file lld_hsi2c.h
 *
 * This file contains the device LLD APIs for High Speed I2c
 *
 */

#ifndef LLD_HSI2C_H_
#define LLD_HSI2C_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdint.h>
#include <string.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/**
 * \name Definitions stating maximum limitations of I2C
 *  @{
 */
/**
 * \brief  Max number of handles for each instance
 */
#define HSI2C_NUM_OPENS    ((uint32_t) 4U)
/**
 * \brief  Minimum bus frequency
 */
#define HSI2C_MIN_BUS_FREQ ((uint32_t) 100U)
/**
 * \brief  Maximum bus frequency
 */
#define HSI2C_MAX_BUS_FREQ ((uint32_t) 400U)
/**
 * \brief  Maximum buffer length
 */
#define HSI2C_BUFFER_MAXLENGTH 32768
/** @} */

/**
 * \name The following flags offer the user maximum flexibility in terms
 *       of making the right I2C transaction. In case the user does not want
 *       to set the flags on his own, the default read/write flag can be specified
 *  @{
 */
/**
 * \brief  Read from I2C bus (device)
 */
#define LLD_HSI2C_READ                ((uint32_t) 0x0001U)
/**
 * \brief  Write to I2C bus (device)
 */
#define LLD_HSI2C_WRITE               ((uint32_t) 0x0002U)
/**
 * \brief  Configure I2C in Master mode
 */
#define LLD_HSI2C_MASTER              ((uint32_t) 0x0004U)
/**
 * \brief  Configure I2C in Slave mode
 */
#define LLD_HSI2C_SLAVE               ((uint32_t) 0x0008U)
/**
 * \brief  Generate Start - valid in master mode only
 */
#define LLD_HSI2C_START               ((uint32_t) 0x0010U)
/**
 * \brief  Generate Stop - valid in master mode only
 */
#define LLD_HSI2C_STOP                ((uint32_t) 0x0020U)
/**
 * \brief  Re-Start is generated by Master
 */
#define LLD_HSI2C_RESTART             ((uint32_t) 0x0040U)
/**
 * \brief  Repeat mode
 */
#define LLD_HSI2C_REPEAT              ((uint32_t) 0x0080U)
/**
 * \brief  Ignore Bus Busy condition
 */
#define LLD_HSI2C_IGNORE_BUS_BUSY     ((uint32_t) 0x0100U)
/**
 * \brief  Ignore NAK
 */
#define LLD_HSI2C_M_IGNORE_NAK        ((uint32_t) 0x0200U)
/**
 * \brief  Repeat TX buffer flag
 */
#define LLD_HSI2C_REPEAT_TX_BUF       ((uint32_t) 0x0400U)
/**
 * \brief  Overwrite RX buffer flag
 */
#define LLD_HSI2C_OVERWRITE_RX_BUF    ((uint32_t) 0x0400U)
/** @} */
/**
 * \name Use these flags for simple MASTER read/write transactions on the I2C bus
 *  @{
 */
/**
 * \brief  Default read flag
 */
#define LLD_HSI2C_DEFAULT_READ \
    (LLD_HSI2C_READ | LLD_HSI2C_MASTER | LLD_HSI2C_START | LLD_HSI2C_STOP)
/**
 * \brief  Default write flag
 */
#define LLD_HSI2C_DEFAULT_WRITE \
    (LLD_HSI2C_WRITE | LLD_HSI2C_MASTER | LLD_HSI2C_START | LLD_HSI2C_STOP)
/** @} */

/**
 * \name Local values that can be passed to hal_hsi2c_delay  API
 *  @{
 */
/**
 * \brief  Small delay
 */
#define     HSI2C_DELAY_SMALL   ((uint32_t) 5000U)
/**
 * \brief  Average delay
 */
#define     HSI2C_DELAY_MED     ((uint32_t) 10000U)
/**
 * \brief  Large delay
 */
#define     HSI2C_DELAY_BIG     ((uint32_t) 20000U)
/**
 * \brief  Huge delay
 */
#define     HSI2C_DELAY_HUGE    ((uint32_t) 50000U)
/** @} */

/**
 * \name Error base value. All error values returned from lib are smaller than
 *       this
 *  @{
 */
/**
 * \brief  Error base value
 *
 * \note   Upto 16 are used/reserved by IOM model of SysBios
 */
#define     HSI2C_ERROR_BASE    (-((int32_t) 16))
/** @} */

/**
 * \name Local values that can be passed to lld_hsi2c_initParam_t -> isMasterMode
 *       Only Interrupt mode operation supported in slave trans receive
 *  @{
 */
/**
 * \brief  Slave Mode
 */
#define     LLD_HSI2C_MODE_SLAVE                (0U)
/**
 * \brief  Master Mode
 */
#define     LLD_HSI2C_MODE_MASTER               (1U)
/**
 * \brief  Slave Trans Receive Mode
 */
#define     LLD_HSI2C_MODE_SLAVE_TRANSRECEIVE   (2U)
/** @} */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/**
 * \brief lld_i2c_busspeed defines the I2C bus frequencies that can be operated
 */
typedef enum
{
    I2C_NORMAL = 100,
    /**< I2C device operates at 100 kHz */
    I2C_FAST   = 400,
    /**< I2C device operates at 400 kHz */
    I2C_HIGH   = 200000
    /**< I2C device operates at 200000 kHz - Not supported */
}lld_i2c_busspeed;

/**
 * \brief lld_hsi2c_opMode defines operating mode of the driver
 */
typedef enum
{
    HSI2C_OPMODE_POLLED = (uint8_t) 0,
    /**< Polled mode operation */
    HSI2C_OPMODE_INTERRUPT,
    /**< Interrupt mode operation */
    HSI2C_OPMODE_DMA
    /**< DMA mode of operation */
} lld_hsi2c_opMode;

/**
 * \brief LLD_hsi2cErrorCode_t status and error codes that are returned from LLD
 */
typedef enum {
    LLD_HSI2C_SUCCESS = (int32_t) 0,
    /**< Operation is successful */
    LLD_HSI2C_FAIL = ((int32_t)HSI2C_ERROR_BASE - 1),
    /**< Operation failed */
    LLD_HSI2C_BADARGS = -(int32_t) 10,
    /**< Bad argument is passed
     * Value same as IOM error IOM_EBADARGS as defined in IOM.h of SysBios */
    LLD_HSI2C_ETIMEOUT = -(int32_t) 2,
    /**< Time out occurs
     * Value same as IOM error IOM_ETIMEOUT as defined in IOM.h of SysBios */
    LLD_HSI2C_HANDLE_UNAVAIL = ((int32_t)HSI2C_ERROR_BASE - 2),
    /**<  Handle not available */
    LLD_HSI2C_BUS_BUSY_ERR = ((int32_t)HSI2C_ERROR_BASE - 3),
    /**<  Bus is busy
     * Returned when the I2C bus find that the bus is busy */
    LLD_HSI2C_ARBITRATION_LOSS_ERR = ((int32_t)HSI2C_ERROR_BASE - 4),
    /**<  arbitration error occurred
     * Returned when the I2C driver lost the bus arbitration */
    LLD_HSI2C_NACK_ERR = ((int32_t)HSI2C_ERROR_BASE - 5),
    /**<  No ACK received from slave
     * Returned when the I2C slave did not acknowledgement */
    LLD_HSI2C_TRANSMIT_UNDERFLOW_ERR = ((int32_t)HSI2C_ERROR_BASE - 6),
    /**<  transmit underflow occurred
     * Returned in case of an transmit underflow error */
    LLD_HSI2C_RECEIVE_OVERFLOW_ERR = ((int32_t)HSI2C_ERROR_BASE - 7),
    /**<  receiver overflow occurred
     *  Returned in case of an receive overflow error */
    LLD_HSI2C_SLAVE_RX_PENDING = (int32_t) 1,
    /**<  Current Transaction completed Successful but transfer
     * is not complete. Restart received with read */
    LLD_HSI2C_SLAVE_TX_PENDING = (int32_t) 2,
    /**<  Current Transaction completed Successful but transfer
     * is not complete. Restart received with write */
    LLD_HSI2C_SLAVE_TRANSFER_PENDING = (int32_t) 3,
    /**<  Transfer request is submitted but transfer not started by the
     * external I2C master */
    LLD_HSI2C_SLAVE_BUFFER_UPDATED = (int32_t) 4
    /**<  This status will be set by application when the buffers are
     * updated in callback of restart condition */
}LLD_hsi2cErrorCode_t;

/**
 * \brief EDMA transfer status
 */
typedef enum {
    EDMA_TRANSFER_SUCCESS,
    /**< EDMA transfer is successful */
    EDMA_TRANSFER_FAIL
    /**< EDMA transfer is failed */
}LLD_hsi2cEdmaStatus;

/**
 * \brief LLD_Hsi2cInstId_t defines I2C instances that can be selected to
 *        operated These values are passed as argument to lld_i2c_init() and
 *        lld_i2c_open fn
 */
typedef enum
{
    HSI2C_INST_0 = 0,
    /**< 0th instance */
    HSI2C_INST_1,
    /**< 1st instance */
#if defined (TI814X_BUILD) || defined (SOC_TDA2XX) || defined (SOC_TDA2PX) || defined (SOC_TDA2EX)
    HSI2C_INST_2,
    /**< 2nd instance */
    HSI2C_INST_3,
    /**< 3rd instance */
#endif
    /*TI814x has only 4 instance*/
#if defined (SOC_TDA2XX) || defined (SOC_TDA2PX) || defined (SOC_TDA2EX)
    HSI2C_INST_4,
    /**< 4th instance */
#if defined (SOC_TDA2EX)
    HSI2C_INST_5,
    /**< 5th instance */
#endif
#endif
    /* TODO Many places instance is asserted with < HSI2C_INST_MAX.
     *      This asserts. Please check and fix */
    HSI2C_INST_MAX_1,
    /**< Maximum instances that are supported */
    HSI2C_INST_MAX
    /**< Maximum instances that are supported */
}LLD_Hsi2cInstId_t;

/**
 * \brief Defines the I2C lld handle type
 */
typedef void *hsI2cHandle;

/**
 * \brief Defines the I2C lld Obj handle type
 */
typedef void *hsI2cObjHandle;

/**
 * \brief I2C lld application callback function format
 */
typedef void (*lldHsi2cAppCallback)(hsI2cHandle appHandle,
                                    LLD_hsi2cErrorCode_t status, void *appData);

/**
 * \brief I2C lld application EDMA callback function format
 */
typedef void (*lldHsi2cEdmaCallback)(uint32_t            tcc,
                                     LLD_hsi2cEdmaStatus i2cEdmaErrStatus,
                                     void               *arg);

/**
 * \brief I2C lld application ISR callback function format
 */
typedef void (*lld_hsi2c_isr_handler)(void *handle);

/**
 * \brief I2C lld application transreceive restart callback function format
 */
typedef void (*lldHsi2cTransreceiveRestartCallback)(hsI2cHandle appHandle,
                                                    void       *i2cXferParams);

/**
 * \brief lld_hsi2c_initParam_s contains the I2C initialisation parameters.
 *        This is passed as an argument to lld_i2c_init() API.
 *        see #lld_hsi2c_opMode
 */
typedef struct lld_hsi2c_initParam_s {
    lld_hsi2c_opMode opMode;
    /**< Driver operating mode - polled ,interrupt or DMA
     * For valid values see #lld_hsi2c_opMode */

    uint32_t         isMasterMode;
    /**< Operating in Master/Slave mode:
     * 0: Slave mode
     * 1: Master mode
     * 2: Slave Trans receive mode */

    uint32_t         is10BitAddr;
    /**< Addressing mode:
     * 0: 7 bit addressing mode
     * 1: 10 bit addressing mode */

    lld_i2c_busspeed i2cBusFreq;
    /**< I2C Bus Frequency (in KHz) */

    uint32_t         i2cOwnAddr;
    /**< Own address (7 or 10 bits) */

    uint32_t         i2cIntNum;
    /**< Interrupt number */

    /*
     * Edma Handle
     */
    /* void *                    hEdma;*/

    /*
     * Event Number
     */

    /* int32_t                   edma3EventQueue;*/

    void                 *edmaParam;
    /**< EDMA parameters */

    lld_hsi2c_isr_handler i2cIsr;
    /**< Interrupt handler */
}lld_hsi2c_initParam_t;

/**
 * \brief lld_hsi2c_datatfr_params_s contains the I2C transfer parameters.
 *        This is passed as an argument to lld_i2c_transfer() API.
 */
typedef struct lld_hsi2c_datatfr_params_s
{
    uint8_t  slaveAddr;
    /**< Address of the slave to talk to, not valid in Slave mode */

    uint8_t *buffer;
    /**< Data buffer */

    uint32_t bufLen;
    /**< Length of buffer */

    uint32_t flags;
    /**< Flags to indicate the various modes of operation */

    uint32_t timeout;
    /**< Timeout value */
}lld_hsi2c_datatfr_params_t;

/**
 * \brief lld_hsi2c_transreceive_params_t contains the transfer parameters.
 *        This is passed as an argument to lld_i2c_transReceive() API.
 */
typedef struct lld_hsi2c_transreceive_params_s
{
    uint8_t                             slaveAddr;
    /**< Address of the slave to talk to, not valid in Slave mode. Kept for future
     * use if the trans receive API is extended for master mode also. */

    uint8_t                            *txBuffer;
    /**< Address of the transmit buffer */

    uint8_t                            *rxBuffer;
    /**< Address of the receive buffer */

    uint32_t                            txBufLen;
    /**< Expected length of transmit buffer in bytes. This is the length of the
     * transmit buffer pointed by txBuffer */

    uint32_t                            rxBufLen;
    /**< Expected length of Receive buffer in bytes. This is the length of the
     * receive buffer pointed by rxBuffer */

    uint32_t                            processedTxBufLen;
    /**< Processed transmit Buffer length. Driver will update this parameter and
     * it will be reset at the start of a transaction. */

    uint32_t                            processedRxBufLen;
    /**< Processed received Buffer length. Driver will update this parameter and
     * it will be reset at the start of a transaction. */

    uint32_t                            flags;
    /**< Flags to indicate the various modes of operation */

    uint32_t                            timeout;
    /**< Timeout value */

    int32_t                             transferStatus;
    /**< Transfer Status. Driver will update the status. If the buffer is updated
     * by the applicaiton after restart, The status should be updated
     * accordingly to intimate the driver. */

    lldHsi2cTransreceiveRestartCallback txRxRestartCb;
    /**< Callback function for getting the restart condition. This will be
     * intermediate completion. If application wants to update the buffer this
     * function will be used. If this is NULL lib will call the completion
     * callback lldHsi2cAppCallback and new transfer should be issued after
     * restart to continue the transfer. */
}lld_hsi2c_transreceive_params_t;

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */
/**
 * @brief   lld_i2c_init() Initializes i2c instance object properties.
 *          Read the application initialization parameters & initialize
 *          the data structure. Changes the state to created after successful
 *          initialization.
 *
 * @param   i2cObjHandle  Handle to the instance object
 * @param   numInstance   Instance ID(i2c port number).
 * @param   initParams    Initialization parameters.
 *
 * @return  status    status of the initialization.
 *                    HAL_HSI2C_SUCCESS - if initialization is successful
 *                    HAL_HSI2C_FAIL -    if initialization fails
 *                    HAL_HSI2C_BADARGS - if wrong argument is passed
 */
LLD_hsi2cErrorCode_t lld_i2c_init(hsI2cObjHandle        *i2cObjHandle,
                                  LLD_Hsi2cInstId_t      numInstance,
                                  lld_hsi2c_initParam_t *initParams);

/**
 * @brief   lld_i2c_open() creates the handle for I2C port for a open.
 *          There can be maximum of HSI2C_NUM_OPENS opens
 *          Initialize the hardware port & change the state to open
 *          Funtion is not Re-entrant. Please use semaphore in the application
 *
 * @param   instanceId     Instance Id(i2c port number)
 * @param   appCallback    Application call back function
 * @param   appData        Application data to be passed to call back function
 *
 * @return  hsI2cHandle    valid handle is successful
 *                         NULL if failed to open
 */
hsI2cHandle          lld_i2c_open(LLD_Hsi2cInstId_t   instanceId,
                                  lldHsi2cAppCallback appCallback,
                                  void               *appData);

/**
 * @brief   lld_i2c_transfer() is used to send or receive data
 *          Data is transfered depending on the mode of operation(DMA, Interrupt
 *          or polled)
 *          Function is not Re-entrant. Please use semaphore lock in the
 *          application
 *
 *
 * @param   handle           Handle to I2C port
 * @param   i2cXferParams    Data transfer parameters
 *
 * @return  status    Status of transfer or error code
 *                    LLD_HSI2C_SUCCESS - if initialization is successful
 *                    LLD_HSI2C_FAIL    -    if initialization fails
 *                    LLD_HSI2C_BADARGS - if wrong argument is passed
 */
LLD_hsi2cErrorCode_t lld_i2c_transfer(
    hsI2cHandle                       handle,
    const lld_hsi2c_datatfr_params_t *i2cXferParams);

/**
 * @brief   lld_i2c_transReceive() is used to send or receive data
 *          as I2C slave. Only Interrupt mode is supported.
 *          Function is not Re-entrant. Please use semaphore lock in the
 *          application. In case any error is returned then the application
 *          should abort the transfer by calling lld_i2c_abort_transfer API.
 *
 *
 * @param   handle           Handle to I2C port
 * @param   i2cXferParams    Data transfer parameters
 *
 * @return  status    Status of transfer or error code
 *                    LLD_HSI2C_SUCCESS - if initialization is successful
 *                    LLD_HSI2C_FAIL    - if initialization fails
 *                    LLD_HSI2C_BADARGS - if wrong argument is passed
 */
LLD_hsi2cErrorCode_t lld_i2c_transReceive(
    hsI2cHandle                      handle,
    lld_hsi2c_transreceive_params_t *i2cXferParams);

/**
 * @brief   lld_i2c_close() closes the handle for I2C port for a open.
 *          Decreaments number of open handles. Changes the state of object
 *          to CLOSED. Clears the handle
 *
 * @param   handle    i2c handle
 *
 * @return  status    status of close
 *                    LLD_HSI2C_SUCCESS - close successful
 *                    LLD_HSI2C_FAIL    - close failed
 */
LLD_hsi2cErrorCode_t lld_i2c_close(hsI2cHandle *handle);

/**
 * @brief      lld_i2c_probe() probes the i2c device for the slave address passed
 *
 * @param   handle      i2c handle
 * @param   slaveAddr   address of the device to be probed
 *
 * @return  status    status of probe
 *                    LLD_HSI2C_BADARGS - if arguments are incorrect
 *                    LLD_HSI2C_SUCCESS - close successful
 *                    LLD_HSI2C_FAIL    - close failed
 */
LLD_hsi2cErrorCode_t lld_i2c_probe(hsI2cHandle handle, uint8_t slaveAddr);

/**
 * @brief   lld_i2c_recoverBus() Generate 9 CLK pulses on SCL
 *
 * @param   handle      i2c handle
 * @param   i2cDelay    used as busy loop count down value to generate delay
 *                      in toggelling SCL signal (high to low or low to high)
 *
 * @return  status    status of probe
 *                    LLD_HSI2C_BADARGS - if arguments are incorrect
 *                    LLD_HSI2C_SUCCESS - generate clock pulse successful
 */
LLD_hsi2cErrorCode_t lld_i2c_recoverBus(hsI2cHandle handle, uint32_t i2cDelay);

/**
 * @brief   lld_i2c_resetCtrl() resets the i2c controller
 *
 * @param   handle      i2c handle
 *
 * @return  status    status of contoller reset
 *                    LLD_HSI2C_BADARGS - if arguments are incorrect
 *                    LLD_HSI2C_SUCCESS - ctrl reset successful
 *                    LLD_HSI2C_FAIL    - ctrl reset failed
 */
LLD_hsi2cErrorCode_t lld_i2c_resetCtrl(hsI2cHandle handle);

/**
 * @brief   lld_i2c_deinit() closes all the handles and clears the instance
 *          for the instance number passed
 *
 * @param   numInstance  instance number
 *
 * @return  status    status of de initialization
 *                    LLD_HSI2C_SUCCESS - de-initialization successful
 */
LLD_hsi2cErrorCode_t lld_i2c_deinit(LLD_Hsi2cInstId_t numInstance);

/**
 * @brief   lld_i2c_getInitParams() gets the init Params
 *          of the instance belonging to the handle passed
 *
 * @param   i2chandle  i2c handle
 *
 * @param   initParams  initParams which needs to be updated.
 *
 * @return  none
 */
void lld_i2c_getInitParams(hsI2cHandle            i2chandle,
                           lld_hsi2c_initParam_t *initParams);

/**
 * @brief   lld_i2c_getInstNum() gets the instance Number
 *          of the handle passed
 *
 * @param   i2chandle  i2c handle
 *
 * @param   instNum  instNum which needs to be updated.
 *
 * @return  none
 */
void lld_i2c_getInstNum(hsI2cHandle i2chandle, LLD_Hsi2cInstId_t *instNum);

/**
 * @brief   lld_i2c_getOpenCount() gets the open count
 *          of the instance belonging to the handle passed
 *
 * @param   i2chandle  i2c handle
 *
 * @param   numOpens   Number of open instances
 *
 * @return  none
 */
void lld_i2c_getOpenCount(hsI2cHandle i2chandle, uint8_t *numOpens);

/**
 * @brief   lld_i2c_SetFifoThreshold() sets the threshold value for FIFO for
 *          buffered transfer.
 *
 *          Note : API should be called before calling lld_i2c_open and after
 *                 calling lld_i2c_init
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *         from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   rxThreshold  Threshold value for Rx FIFO
 *
 * @param   txThreshold  Threshold value for Tx FIFO
 *
 *          rxThreshold or txThreshold can take values 0-31 to configure FIFO
 *          from 1 to 32 bytes
 *          value 0 = 1 byte(default)
 *
 *
 * @return  LLD_hsi2cErrorCode_t    Status.
 *                    HAL_HSI2C_SUCCESS - if success
 *                    HAL_HSI2C_BADARGS - if instanceId is wrong
 */
LLD_hsi2cErrorCode_t lld_i2c_SetFifoThreshold(LLD_Hsi2cInstId_t instanceId,
                                              uint8_t           rxThreshold,
                                              uint8_t           txThreshold);

/**
 * @brief   lld_i2c_clockConfig() configures I2C bus clock frequency
 *
 *          Note : API should be called after calling lld_i2c_open and before
 *                 calling lld_i2c_transfer. This API can be used in two case
 *                 1. If functional clock to I2C is other than that of the
 *                    default functional clock used for TDA2XX(96 MHz)
 *                    and TI814x(48 MHz)
 *                 2. Changing frequency on demand without reinitializing the
 *                    driver
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *         from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   functionalClock  Functional input clock to I2C in Hz
 *              e.g. for 24 MHz input clock provide 24000000
 *              Default functional clock used for TDA2XX is 96 MHz
 *               and for TI814x is 48 MHz
 *
 * @param   busFrequency  I2C output frequency in kbps
 *          as defined by lld_i2c_busspeed enum
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_clockConfig(LLD_Hsi2cInstId_t instanceId,
                                         uint32_t          functionalClock,
                                         lld_i2c_busspeed  busFrequency);


/**
 * @brief   lld_i2c_getClockDiv() returns I2C Clock divider values
 *
 *          Note : API should be called after calling lld_i2c_open and before
 *                 calling lld_i2c_transfer.
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *         from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   sCll  pointer to sCll divider value
 *
 * @param   sClh  pointer to sClh divider value
 *
 * @param   pSc  pointer to pSc divider value
 *
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_getClockDiv(LLD_Hsi2cInstId_t instanceId,
                                         uint32_t         *sCll,
                                         uint32_t         *sClh,
                                         uint32_t         *pSc);

/**
 * @brief   lld_i2c_setClockDiv() returns I2C Clock divider values
 *
 *          Note : API should be called after calling lld_i2c_open and before
 *                 calling lld_i2c_transfer.Normal sequence should be
 *                 lld_i2c_clockConfig, lld_i2c_getClockDiv and then
 *                 lld_i2c_setClockDiv
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *         from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   sCll  Actual sCll divider value
 *
 * @param   sClh  Actual sClh divider value
 *
 * @param   pSc  Actual  pSc divider value
 *
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_setClockDiv(LLD_Hsi2cInstId_t instanceId,
                                         uint32_t          sCll,
                                         uint32_t          sClh,
                                         uint32_t          pSc);
/**
 * @brief      lld_i2c_getFclk() gets the I2C bus input clock frequency
 *
 *             Default functional clock used for TDA2XX is 96 MHz
 *                    and TI814x is 48 MHz
 *
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *         from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 *
 *
 * @return  uint32_t   Functional input clock to I2C in Hz
 */
uint32_t lld_i2c_getFclk(LLD_Hsi2cInstId_t instanceId);

/**
 * @brief   lld_i2c_setFclk() sets the I2C bus input clock frequency
 *
 *          Note : API should be called before calling lld_i2c_open and after
 *          calling lld_i2c_init. This API can be used to specify the input
 *          functional clock to I2C. Applicationr Should take care that
 *           appropriate input clock is already provided to the I2C  as
 *          part of system init.
 *           This API does not take care of checking if the input clock
 *           provided is valid
 *
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *         from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   fclk  Functional input clock to I2C in Hz
 *
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_setFclk(LLD_Hsi2cInstId_t instanceId,
                                     uint32_t          fclk);

/**
 * @brief      lld_i2c_getIclk() gets the I2C internal clock frequency
 *
 *             Default Internal clock used for FAST Mode is 12 MHz
 *                    and NORMAL Mode is 4 MHz
 *
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *          from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 *
 *
 * @return  uint32_t   Internal clock to I2C in Hz.
 */
uint32_t lld_i2c_getIclk(LLD_Hsi2cInstId_t instanceId);

/**
 * @brief   lld_i2c_setIclk() sets the I2C Internal clock frequency
 *
 *          Note : API should be called before calling lld_i2c_open and after
 *          calling lld_i2c_init. This API can be used to specify the input
 *          functional clock to I2C. Applicationr Should take care that
 *           appropriate input clock is already provided to the I2C  as
 *          part of system init.
 *           This API does not take care of checking if the input clock
 *           provided is valid
 *
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *          from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   iclk  Internal clock to I2C in Hz
 *
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_setIclk(LLD_Hsi2cInstId_t instanceId,
                                     uint32_t          iclk);

/**
 * @brief   lld_i2c_getBusFreq() gets the current I2C bus frequency in KHz
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *          from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   busFreq      pointer to store the current i2c bus frequency
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_getBusFreq(LLD_Hsi2cInstId_t instanceId,
                                        lld_i2c_busspeed *busFreq);

/**
 * @brief   lld_i2c_setBusFreq() sets the current I2C bus frequency in KHz.
 *
 * @param   instanceId   i2c instance number can be one of the following values
 *          from
 *          enum LLD_Hsi2cInstId_t
 *              HSI2C_INST_0 = 0,
 *              HSI2C_INST_1
 *              HSI2C_INST_2
 *              HSI2C_INST_3
 *              HSI2C_INST_4
 *              HSI2C_INST_5
 *
 * @param   busFreq   i2c bus frequencyto be configured in KHz
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_setBusFreq(LLD_Hsi2cInstId_t instanceId,
                                        lld_i2c_busspeed  busFreq);

/**
 * @brief   lld_i2c_abort_transfer() aborts the ongoing transfer
 *
 * @param   handle   i2c handle
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_abort_transfer(hsI2cHandle handle);

/**
 * @brief   lld_i2c_getProcessedBufLength() get the number of bytes processed
 *          in the current transfer.
 *
 *          Note : API should be called after calling lld_i2c_transfer.
 *          This API returns the number of bytes processed (read/written) after
 *          initiating the transfer with lld_i2c_transfer.
 *          When this API is called from the application callback function,
 *          received at the end of transfer, it will return total number of
 *          bytes sent/received in the previous transfer.
 *
 * @param   handle          i2c handle.
 *
 * @param   validBufLength  pointer to get number of bytes processed.
 *
 *
 * @return  LLD_hsi2cErrorCode_t    LLD_HSI2C_SUCCESS Status.
 */
LLD_hsi2cErrorCode_t lld_i2c_getProcessedBufLength(hsI2cHandle handle,
                                                   uint32_t   *validBufLength);

#ifdef __cplusplus
}
#endif

#endif
/* @} */
