//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//
//---------------------------------------------------------------------------------------------------------------------
/*!
 *  @file       trace.h
 *
 *  @brief      Kernel Trace enabling/disabling/application interface.
 *
 *              This will have the definitions for kernel side traces
 *              statements and also details of variable traces
 *              supported in existing implementation.
 *
 *  @date       09-02-2009
 *
 *  @internal   09-02-2009, Harshit Srivastava, Revision 0001:
 *              [1] Original version. Inspired from CE trace.
 *              07-04-2009, Mugdha Kamoolkar, Revision 0002:
 *              [1] Implemented trace.
 *              09-04-2009, Mugdha Kamoolkar, Revision 0003:
 *              [1] Updated non enter-leave trace to only print given format.
 *              15-04-2009, Mugdha Kamoolkar, Revision 0004:
 *              [1] Moved to outside knl to make generic.
 *              15-07-2009, Sivaraj R, Revision 0005:
 *              [1] Modified for VPS driver.
 */

#ifndef OSALTRACE_H_0xDA50
#define OSALTRACE_H_0xDA50

#include "tistdtypes.h"
#include "bsp_osal.h"


#ifdef __cplusplus
extern "C" {
#endif

extern Int32 curTrace;
#define GT_TRACESTATE_MASK                 0x0000000FU
#define GT_TRACESTATE_SHIFT                0U
#define GT_TRACEENTER_MASK                 0x000000F0U
#define GT_TRACEENTER_SHIFT                4U
#define GT_TRACESETFAILURE_MASK            0x00000F00U
#define GT_TRACESETFAILURE_SHIFT           8U
#define GT_TRACECLASS_MASK                 0x000F0000U
#define GT_TRACECLASS_SHIFT                16U
#define GT_DEFAULT_MASK                    (GT_ERR | GT_TraceState_Enable)

typedef enum {
    GT_TraceState_Disable = 0x00000000U,
    /*!< Disable trace */
    GT_TraceState_Enable = 0x00000001U,
    /*!< Enable trace */
    GT_TraceState_EndValue = 0x00000002U
                             /*!< End delimiter indicating start of invalid
                              *values for this enum */
} GT_TraceState;

typedef enum {
    GT_TraceEnter_Disable = 0x00000000U,
    /*!< Disable GT_ENTER trace prints */
    GT_TraceEnter_Enable = 0x00000010U,
    /*!< Enable GT_ENTER trace prints */
    GT_TraceEnter_EndValue = 0x00000020U
                             /*!< End delimiter indicating start of invalid
                              *values for this enum */
} GT_TraceEnter;

typedef enum {
    GT_TraceSetFailure_Disable = 0x00000000U,
    /*!< Disable Set Failure trace prints */
    GT_TraceSetFailure_Enable = 0x00000100U,
    /*!< Enable Set Failure trace prints */
    GT_TraceSetFailure_EndValue = 0x00000200U
                                  /*!< End delimiter indicating start of invalid
                                   *values for this enum */
} GT_TraceSetFailure;

typedef enum {
    GT_ERR = 0x00010000U,
    /*!< Class 1 trace: Used for errors/warnings */
    GT_CRIT = 0x00020000U,
    /*!< Class 2 trace: Used for critical information */
    GT_INFO = 0x00030000U,
    /*!< Class 3 trace: Used for additional information */
    GT_INFO1 = 0x00040000U,
    /*!< Class 4 trace: Used for additional information (sub level 1) */
    GT_DEBUG = 0x00050000U,
    /*!< Class 5 trace: Used for block level information */
    GT_ENTER = 0x00060000U,
    /*!< Indicates a function entry class of trace */
    GT_LEAVE = 0x00070000U
               /*!< Indicates a function leave class of trace */
} GT_TraceClass;

/** \brief assert function. */
static inline void GT_assertLocal(UInt32      enableMask,
                                  Bool        condition,
                                  const Char *str,
                                  const Char *fileName,
                                  Int32       lineNum);

/** \brief GT_assert */
#define GT_assert(x, y)                                           \
    (GT_assertLocal((UInt32) (x), (Bool) (y), (const Char *) # y, \
                    (const Char *) __FILE__, (Int32) __LINE__))

static inline void GT_assertLocal(UInt32      enableMask,
                                  Bool        condition,
                                  const Char *str,
                                  const Char *fileName,
                                  Int32       lineNum)
{
    return;
}

#define GT_setFailureReason(maskType, classType, func, status, msg) \
    do {                                                            \
        if (maskType) {                                             \
        }                                                           \
        if (classType) {                                            \
        }                                                           \
        if (status) {                                               \
        }                                                           \
        if ((UInt32) (msg)) {                                       \
        }                                                           \
    } while (FALSE)

/**
 *  \brief Log the trace with zero parameters and information string.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *
 */
void GT_trace0(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString);
/**
 *  \brief Function to log the trace with one additional parameter.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *
 */
void GT_trace1(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0);

/**
 *  \brief Function to log the trace with two additional parameters.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *  \param  param1     [IN] Parameter 1
 *
 */
void GT_trace2(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0,
               UInt32        param1);

/**
 *  \brief Function to log the trace with three additional parameters.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *  \param  param1     [IN] Parameter 1
 *  \param  param2     [IN] Parameter 2
 *
 */
void GT_trace3(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0,
               UInt32        param1,
               UInt32        param2);

/**
 *  \brief Function to log the trace with four additional parameters.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *  \param  param1     [IN] Parameter 1
 *  \param  param2     [IN] Parameter 2
 *  \param  param3     [IN] Parameter 3
 *
 */
void GT_trace4(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0,
               UInt32        param1,
               UInt32        param2,
               UInt32        param3);

/**
 *  \brief Function to log the trace with five additional parameters.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *  \param  param1     [IN] Parameter 1
 *  \param  param2     [IN] Parameter 2
 *  \param  param3     [IN] Parameter 3
 *  \param  param4     [IN] Parameter 4
 *
 */
void GT_trace5(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0,
               UInt32        param1,
               UInt32        param2,
               UInt32        param3,
               UInt32        param4);

/**
 *  \brief Function to log the trace with six additional parameters.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *  \param  param1     [IN] Parameter 1
 *  \param  param2     [IN] Parameter 2
 *  \param  param3     [IN] Parameter 3
 *  \param  param4     [IN] Parameter 4
 *  \param  param5     [IN] Parameter 5
 *
 */
void GT_trace6(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0,
               UInt32        param1,
               UInt32        param2,
               UInt32        param3,
               UInt32        param4,
               UInt32        param5);

/**
 *  \brief Function to log the trace with seven additional parameters.
 *
 *  \param  maskType   [IN] Trace mask
 *  \param  classType  [IN] Trace class type
 *  \param  fileName   [IN] File Name
 *  \param  lineNum    [IN] Line Number
 *  \param  infoString [IN] Print string
 *  \param  param0     [IN] Parameter 0
 *  \param  param1     [IN] Parameter 1
 *  \param  param2     [IN] Parameter 2
 *  \param  param3     [IN] Parameter 3
 *  \param  param4     [IN] Parameter 4
 *  \param  param5     [IN] Parameter 5
 *  \param  param6     [IN] Parameter 6
 *
 */
void GT_trace7(UInt32        maskType,
               GT_TraceClass classType,
               const Char   *fileName,
               Int32         lineNum,
               const Char   *infoString,
               UInt32        param0,
               UInt32        param1,
               UInt32        param2,
               UInt32        param3,
               UInt32        param4,
               UInt32        param5,
               UInt32        param6);

/** \brief Log the trace with zero parameters and information string. */
#define GT_0trace(maskType, classType, infoString)
/** \brief Function to log the trace with one additional parameter. */
#define GT_1trace(maskType, classType, infoString, param0)
/** \brief Function to log the trace with two additional parameters. */
#define GT_2trace(maskType, classType, infoString, param0, param1)
/** \brief Function to log the trace with three additional parameters. */
#define GT_3trace(maskType, classType, infoString, param0, param1, param2)
/** \brief Function to log the trace with four additional parameters. */
#define GT_4trace(maskType, classType, infoString, param0, param1, param2, \
                  param3)
/** \brief Function to log the trace with five additional parameters. */
#define GT_5trace(maskType, classType, infoString, param0, param1, param2, \
                  param3, param4)
/** \brief Function to log the trace with six additional parameters. */
#define GT_6trace(maskType, classType, infoString, param0, param1, param2, \
                  param3, param4, param5)
/** \brief Function to log the trace with seven additional parameters. */
#define GT_7trace(maskType, classType, infoString, param0, param1, param2, \
                  param3, param4, param5, param6)

#endif  /* if defined (TRACE_ENABLE) */

#ifdef __cplusplus
}
#endif

