//---------------------------------------------------------------------------------------------------------------------
// Copyright IEOS Ltd. 2010-2018 Moscow, Russian Federation  (www.ieos.ru)
//    Alexander Norokh 2017-2018
//---------------------------------------------------------------------------------------------------------------------
//
//---------------------------------------------------------------------------------------------------------------------
/**
 *  \ingroup BSP_DRV_VPS_API
 *  \defgroup BSP_DRV_VPS_CAPTURE_API Capture API
 *
 *  This modules define APIs to capture video data using video ports in VPS.
 *  This module can be used for single channel capture as well as
 *  multi-channel capture.
 *
 *  Capture driver supports the following FVID2 APIs
 *  (see also \ref BSP_DRV_FVID2_API)
 *
 * - <b> Creating the driver </b> - Fvid2_create()
 *   <table border="1">
 *    <tr>
 *      <th>Parameter</th>
 *      <th>Value</th>
 *    </tr>
 *    <tr>
 *      <td>drvId</td>
 *      <td>\ref FVID2_VPS_CAPT_VID_DRV</td>
 *    </tr>
 *    <tr>
 *      <td>instanceId</td>
 *      <td>
 *          In case of VIP, instance ID is generated by calling below macro
 *          with proper VIP, slice and port ID. <br>
 *          \ref VPS_CAPT_VIP_MAKE_INST_ID<br>
 *          or the global capture handle.<br>
 *          \ref VPS_CAPT_INST_ALL <br>
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>createArgs</td>
 *      <td>
 *      Vps_CaptCreateParams *
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>createStatusArgs</td>
 *      <td>
 *      Vps_CaptCreateStatus *
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>cbParams</td>
 *      <td>
 *      Fvid2_CbParams *
 *
 *      When Fvid2_CbParams.cbFxn is set, Fvid2_CbParams.cbFxn
 *      gets called based on value of Vps_CaptCreateParams.periodicCbEnable
 *
 *      </td>
 *    </tr>
 *  </table>
 *  \ref Fvid2_Handle returned by Fvid2_create() is used in subsequent FVID2
 *  APIs
 *
 * - <b> Deleting the driver </b> - Fvid2_delete()
 *  <table border="1">
 *    <tr>
 *      <th>Parameter</th>
 *      <th>Value</th>
 *    </tr>
 *    <tr>
 *      <td>handle</td>
 *      <td>Capture driver handle</td>
 *    </tr>
 *    <tr>
 *      <td>deleteArgs</td>
 *      <td>NOT USED, set to NULL</td>
 *    </tr>
 *  </table>
 *
 * - <b> Starting the driver </b> - Fvid2_start()
 *  <table border="1">
 *    <tr>
 *      <th>Parameter</th>
 *      <th>Value</th>
 *    </tr>
 *    <tr>
 *      <td>handle</td>
 *      <td>Capture driver handle</td>
 *    </tr>
 *    <tr>
 *      <td>cmdArgs</td>
 *      <td>NOT USED, set to NULL</td>
 *    </tr>
 *  </table>
 *
 * - <b> Stopping the driver </b> - Fvid2_stop()
 *  <table border="1">
 *    <tr>
 *      <th>Parameter</th>
 *      <th>Value</th>
 *    </tr>
 *    <tr>
 *      <td>handle</td>
 *      <td>Capture driver handle</td>
 *    </tr>
 *    <tr>
 *      <td>cmdArgs</td>
 *      <td>NOT USED, set to NULL</td>
 *    </tr>
 *  </table>
 *
 *  - <b> Controlling the driver </b> - Fvid2_control() <br>
 *  See \ref BSP_DRV_IOCTL_VPS_CAPTURE for the list of IOCTLs supported by the
 *  driver.
 *
 *  - <b> Getting captured frames from the driver </b> - Fvid2_dequeue()
 *  <table border="1">
 *    <tr>
 *      <th>Parameter</th>
 *      <th>Value</th>
 *    </tr>
 *    <tr>
 *      <td>handle</td>
 *      <td>Capture driver handle</td>
 *    </tr>
 *    <tr>
 *      <td>frameList</td>
 *      <td>
 *      [OUT] Fvid2_FrameList.numFrames returns the number of captured frames
 *  returned in this Fvid2_dequeue() invocation. <br>
 *      [OUT] Fvid2_FrameList.frames[0..Fvid2_FrameList.numFrames-1] are the
 *  captured Fvid2_Frame pointers to the captured frames. <br>
 *      [OUT] For each Fvid2_Frame, Fvid2_Frame.perFrameCfg points to
 *  Vps_CaptRtParams that was set during Fvid2_queue()
 *  This structure gets filled with the captured width and height information.
 *  When sub-frame mode is enabled this structure is filled for every
 *  sub-frame. The parameter capturedOutHeight shall be invalid for the first
 *  field/frame. From the second field/frame onwards this value shall hold the
 *  captured height of the previous field/frame. For the last sub-frame in the
 *  EOF ISR the parameter numOutLines is updated with the capturedHeight. <br>
 *      [OUT] FID for every field/frame is filled when Fvid2_dequeue() is
 *  called.
 *  When sub-frame mode is enabled the FID is filled after the first
 *  sub-frame is captured. For the first field/frame this value shall be
 *  invalid. <br>
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>streamId</td>
 *      <td>
 *      Value can be from 0 .. Vps_CaptCreateParams.numStream-1
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>timeout</td>
 *      <td>
 *      Must be BIOS_NO_WAIT
 *      </td>
 *    </tr>
 *  </table>
 *
 *  - <b> Releasing used frames back to the driver </b> - Fvid2_queue() <br>
 *  Also used to queue initial frame buffers to the driver, before calling
 *  Fvid2_start().
 *  <table border="1">
 *    <tr>
 *      <th>Parameter</th>
 *      <th>Value</th>
 *    </tr>
 *    <tr>
 *      <td>handle</td>
 *      <td>Capture driver handle</td>
 *    </tr>
 *    <tr>
 *      <td>frameList</td>
 *      <td>
 *      [IN] Fvid2_FrameList.numFrames sets the number of captured frames given
 *  back to driver in this Fvid2_queue() invocation. <br>
 *      [IN] Fvid2_FrameList.frames[0..Fvid2_FrameList.numFrames-1] are the
 *  captured Fvid2_Frame pointers that are being returned. <br>
 *      [IN] For each Fvid2_Frame, Fvid2_Frame.perFrameCfg points to
 *  Vps_CaptRtParams.
 *            This structure is filled and returned when Fvid2_dequeue() is
 *  called. <br>
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>streamId</td>
 *      <td>
 *      Value can be from 0 .. Vps_CaptCreateParams.numStream-1 OR \ref
 *  VPS_CAPT_STREAM_ID_ANY
 *      </td>
 *    </tr>
 *  </table>
 *
 *  Capture driver application flow is as follows
 *
 *  - <b> Init all required modules by calling respective module init functions
 *  </b>
 *  - <b> Create driver using Fvid2_create() </b>
 *  - <b> Set driver specific parameters using Fvid2_control() </b> <br>
 *        In case of VIP capture driver, set the VIP parameters using
 *        \ref IOCTL_VPS_CAPT_SET_VIP_PARAMS
 *  - <b> Prime buffers using Fvid2_queue() </b>
 *  - <b> Create, configure and start external video decoder or sensor
 *  driver</b>
 *  - <b> Start capture using Fvid2_start() </b>
 *  - <b> Dequeue captured frame and queue back used frames using
 *  Fvid2_dequeue() and Fvid2_queue() </b>
 *  - <b> Stop capture using Fvid2_stop() </b>
 *  - <b> Stop and delete external video decoder or sensor driver </b>
 *  - <b> Delete capture driver using Fvid2_delete() </b>
 *  - <b> De-Init modules by calling respective module de-init functions </b>
 *
 *  @{
 */

/**
 *  \file vps_capture.h
 *
 *  \brief Video Capture API.
 */

#ifndef VPS_CAPTURE_H_
#define VPS_CAPTURE_H_

#include "vps.h"
#include "bsp_cfgMem.h"

#ifdef VPS_VIP_BUILD
#include "vps_captureVip.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name Capture Instance ID's
 *
 *  @{
 */

/** \brief Capture instance - ALL. */
#define VPS_CAPT_INST_ALL               (0xFFU)

/** \brief DSS WB Capture instance . */
#define VPS_CAPT_INST_DSS_WB1            (VPS_CAPT_INST_DSS_WB_BASE)

/** \brief ISS Capture instance. Writes captured CSI2 stream to memory,
 *      processing limited to capabilities of CAL sub-module
 *
 *  \warning The instance identifier macro could change. */
#define VPS_CAPT_INST_ISS_CAL_A         (VPS_CAPT_INST_ISS_BASE)

/** \brief Do not use this as instance identifier for CAL Driver.
 *
 *  \warning The instance identifier macro could change. */
#define VPS_CAPT_INST_ISS_CAL_A_CPI     (VPS_CAPT_INST_ISS_CAL_A + 1U)

/** \brief Not Supported for now.
 *          ISS Capture instance. Writes captured CSI2 streams / from parallel
 *          interface to memory. Complete ISP could be used to process captured
 *          streams.
 *          \warning The instance identifier macro could change. */
#define VPS_CAPT_INST_ISS_CAL_A_OTF     (VPS_CAPT_INST_ISS_CAL_A_CPI + 1U)

/** \brief Maximum number of instances of CAL A based capture possible */
#define VPS_CAPT_INST_ISS_CAL_A_MAX     (VPS_CAPT_INST_ISS_CAL_A_OTF + 1U)
/* @} */

/**
 *  \name Capture max limits or defaults.
 *
 *  @{
 */

/** \brief Capture instance - MAX instances. */
#define VPS_CAPT_INST_MAX               (VPS_CFG_CAPT_NUM_INST_OBJS)

/**
 *  \brief Maximum channels that can be captured per port. The number of
 *  channels varies for a given interface. Please refer the interface
 *  specific interface headers
 */
#define VPS_CAPT_CH_PER_PORT_MAX        (16U)

/**
 *  \brief Maximum stream's that can be captured per channel.
 *
 *  Stream's corresponds to different types of output that are possible
 *  for a given capture input source.
 *
 *  The stream output type is setup by user during Fvid2_create()
 *
 *  Example streams are,
 *
 *  For a 1080P input source,
 *  Stream 0: YUV420 non-scaled output
 *  Stream 1: YUV422 scaled output
 *  Stream 2: RAW VBI output
 *
 *  In case of ISS on-the-fly capture, we could possibly have output from
 *  ReSizer A, ReSizer B, H3A stat (x2), CAL writes (upto 6), etc...
 *  In current usecase scenarios, we could have upto 6 streams at a time.
 *
 *  Refer to user guide for more details about what valid stream
 *  combinations are possible.
 *
 */
#define VPS_CAPT_STREAM_ID_MAX          (8U)

/** \brief Default max number of frames that can be queued per channel. */
#define VPS_CAPT_DEF_QUEUE_LEN_PER_CH   (BSP_CFG_CAPT_QUEUE_LEN_PER_CH)

/* @} */

/**
 *  \brief Value to use for 'streamId' when required to do Fvid2_queue
 *  to any stream
 *
 *  Note, this value cannot be used as 'streamId' with Fvid2_dequeue
 */
#define VPS_CAPT_STREAM_ID_ANY          (FVID2_STREAM_ID_ANY)

/**
 *  \brief Scaler ID to be used in the set scaler coefficient and set/get
 *  advance scaler params IOCTLS.
 *
 *  Since this driver supports only one scaler per driver instance, the scaler
 *  could be identified by the instance handle. The below macro is used for
 *  initializing the scaler ID to a default value.
 *
 *  _ISS_CHANGE_REQ_ - Move this to VIP specific header
 */
#define VPS_CAPT_SCALER_ID_DEFAULT      (0U)

/* Capture IOCTL's */

/**
 *  \addtogroup BSP_DRV_IOCTL_VPS_CAPTURE
 *  @{
 */

/**
 *  \brief Set frame skip info.
 *
 *  This IOCTL can be used to control capture frame drop sequence,
 *  Example, this IOCTL can be used to convert 60fps input source to 30fps
 *  by dropping frames.
 *
 *  This control can be done independently for each stream, channel and is
 *  supported only when VPS_CAPT_BCM_FRM_DROP is selected as buffer
 *  capture mode (bufCaptMode).
 *
 *  \param cmdArgs       [IN]  const Vps_CaptFrameSkip *
 *  \param cmdArgsStatus [OUT] NULL
 *
 *  \return FVID2_SOK on success, else failure.
 */
#define IOCTL_VPS_CAPT_SET_FRAME_SKIP   (VPS_CAPT_IOCTL_BASE + 0x0000U)

/**
 *  \brief Get capture status IOCTL.
 *
 *  This IOCTL can be used to get the capture status like number of frames
 *  queued/dequeued.
 *  Note: These counters will be reset either at the time of driver create or
 *  while starting the capture operation. See respective counter comments for
 *  details.
 *
 *  \param cmdArgs       [OUT] Vps_CaptStatus *
 *  \param cmdArgsStatus [OUT] NULL
 *
 *  \return FVID2_SOK on success, else failure
 *
 */
#define IOCTL_VPS_CAPT_GET_STATUS       (VPS_CAPT_IOCTL_BASE + 0x0001U)

/**
 *  \brief Get channel status
 *
 *  This IOCTL can be used to get status about a specific channel, stream.
 *
 *  This control can be done independently for each stream, channel.
 *
 *  \param cmdArgs       [IN]  const Vps_CaptChStatusArgs *
 *  \param cmdArgsStatus [OUT] Vps_CaptChStatus *
 *
 *  \return FVID2_SOK on success, else failure.
 */
#define IOCTL_VPS_CAPT_GET_CH_STATUS    (VPS_CAPT_IOCTL_BASE + 0x0002U)

/**
 *  \brief Flush capture driver and dequeue all frames including those that
 *  are not captured.
 *
 *  Note: This IOCTL is not yet implemented!!
 *
 *  This API can be called only when driver is in stopped state.
 *  Driver is in stopped state when,
 *  - FIVD2_stop() is called
 *  - Fvid2_create() is called and Fvid2_start() is not called.
 *
 *  This will return frames from all streams for a given capture handle.
 *  If this IOCTL is called with global handle then this will return
 *  frames from all capture handles.
 *
 *  Since maximum FVID2_MAX_FRAME_PTR frames can be returned at a time.
 *  This function should be called in a loop until Fvid2_FrameList.numFrames = 0
 *  is returned or the return value is FVID2_ENO_MORE_BUFFERS in order to get
 *  back all the frames from the capture driver.
 *
 *  In case capture is in running state this function will return error.
 *
 *  This IOCTL will return capture frames, non-captured frames, as well as
 *  all frames held inside the hardware.
 *
 *  \param cmdArgs       [IN]  NULL
 *  \param cmdArgsStatus [OUT] Fvid2_FrameList
 *
 *  \return FVID2_SOK on success, FVID2_ENO_MORE_BUFFERS if all buffers are
 *  returned, other error values on failure.
 */
#define IOCTL_VPS_CAPT_FLUSH            (VPS_CAPT_IOCTL_BASE + 0x0003U)

/**
 *  \brief Enable overflow interrupt
 *
 *  This IOCTL can be used to enable the overflow interrupts from the hardware
 *  The overflow params structure contains the application callback function
 *  when the interrupt occurs. The other parameter is the application instance
 *  object of the calling VIP instance
 *
 *  \param cmdArgs       [IN]  const Vps_CaptOverflowParams *
 *  \param cmdArgsStatus NULL
 *
 *  \return FVID2_SOK on success, else failure.
 */
#define IOCTL_VPS_CAPT_REGISTER_OVF_INTR    (VPS_CAPT_IOCTL_BASE + 0x0004U)

/**
 *  \brief Disable overflow interrupt. This can be used to un-register from the
 *  overflow interrupt. Once this is used, there will be no interrupts for
 *  overflow.
 *
 *
 *  \param cmdArgs       [IN]  NULL
 *  \param cmdArgsStatus       NULL
 *
 *  \return FVID2_SOK on success, else failure.
 */
#define IOCTL_VPS_CAPT_UNREGISTER_OVF_INTR  (VPS_CAPT_IOCTL_BASE + 0x0005U)

/**
 *  \brief Check overflow
 *
 *  This interrupt can be used to check the overflow in a particular VIP
 *  instance Structure passed will be filled with appropriate values
 *
 *
 *  \param cmdArgs              NULL
 *  \param cmdArgsStatus [OUT]  Vps_CaptOverflowCheckParams
 *
 *  \return FVID2_SOK on success, else failure.
 */
#define IOCTL_VPS_CAPT_CHECK_OVERFLOW   (VPS_CAPT_IOCTL_BASE + 0x0006U)

/* @} */

/** \brief VIP capture driver IOCTL base. */
#define VPS_CAPT_VIP_IOCTL_BASE         (VPS_CAPT_IOCTL_BASE + 0x0100U)

/** \brief DSSWB capture driver IOCTL base. */
#define VPS_CAPT_DSSWB_IOCTL_BASE       (VPS_CAPT_IOCTL_BASE + 0x0200U)

/** \brief ISS based capture driver IOCTL base. */
#define VPS_CAPT_ISS_IOCTL_BASE         (VPS_CAPT_IOCTL_BASE + 0x0400U)
/**
 *  \brief Buffer capture mode.
 *          These modes depends on the underlying hardware. Not all capture
 *          interfaces can support all mode. Please refer interface specific
 *          header for expections.
 */
typedef enum
{
    VPS_CAPT_BCM_FRM_DROP = 0,
    /**< In this mode the driver will stop capturing data when there are
     *   no more buffers at the input queue.
     *   The driver will not hold any buffer with it and the last buffer
     *   will be returned to the application through dequeue call.
     *   For this mode, the driver makes use of the VPDMA drop data feature. */
    VPS_CAPT_BCM_LAST_FRM_REPEAT,
    /**< In this mode the driver will keep capturing the data to the last
     *   queued buffer when there are no more buffers at the input queue.
     *   The driver will hold the last buffer with it till the application
     *   queues any new buffer or the capture is stopped. */
    VPS_CAPT_BCM_CIRCULAR_FRM_REPEAT,
    /**< In this mode the driver will keep reusing all the sets of buffer
     *   with it in a circular fashion.
     *   Application cannot get back any buffer from the driver when streaming
     *   is on and dequeue call will result in error. */
    VPS_CAPT_BCM_MAX
    /**< Should be the last value of this enumeration.
     *   Will be used by driver for validating the input parameters. */
} Vps_CaptBufferCaptMode;

/**
 *  \brief Application callback for overflow.
 */
typedef void (*Vps_CaptVipOverflowFxn)(Ptr pObj);

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/**
 *  \brief Capture driver create arguments, used when calling Fvid2_create().
 *          All features may not be supported by all the interfaces.
 *          The exceptions will be called out in the inteface specific headers.
 */
typedef struct
{
    UInt32 videoIfMode;
    /**< [IN] Video capture mode. For valid values see #Fvid2_VideoIfMode. */
    UInt32 videoIfWidth;
    /**< [IN] Video interface mode. For valid values see #Fvid2_VideoIfWidth. */
    UInt32 bufCaptMode;
    /**< [IN] Buffer capture mode.
     *   For valid values see #Vps_CaptBufferCaptMode. */
    UInt32 periodicCbEnable;
    /**< [IN] TRUE: User callback passed during FVID2 create is called
     *        periodically at a fixed duration of about 8msecs
     *        FALSE: User callback passed during FVID2 create is called only
     *        if one or more frames are captured in any of the streams,
     *        channels belonging to this handle. */
    UInt32 chInQueueLength;
    /**< [IN] Maximum number of request/frame per channel that can be submitted
     *   for capture without having to dequeue the captured requests. */
    UInt32 numCh;
    /**< [IN] Number of channel for multi-channel modes,
     *   Must be 1 for FVID2_VIFM_SCH_* capture modes. */
    UInt32 numStream;
    /**< [IN] Number of streams to capture,
     *   MUST be <= VPS_CAPT_STREAM_ID_MAX. */

    UInt32 chNumMap[VPS_CAPT_STREAM_ID_MAX][VPS_CAPT_CH_PER_PORT_MAX];
    /**< [IN] Channel Number to assign to each channel and stream of this
     *   handle.
     *   This is used during Fvid2_queue(), Fvid2_dequeue().
     *   Channel number must be unique across the whole system.
     *   Users can use Vps_captMakeChNum() to generate a system unique
     *   channel number. */
    UInt32 muxModeStartChId;
    /**< [IN] Start channel ID in pixel or line mux mode. Used to add an
     *   offset to start channel mapping. This will be used when the decoder
     *   start channel ID is other than 0.
     *   For example some decoder's channel 0 CHID starts from 4 instead of 0.
     *   This is valid only in multi-channel mode and is ignored in single
     *   channel or other modes. */
    void  *pAdditionalArgs;
    /**< [IN] Used by some driver to hold additional information. Please refer
     *          the device specific include, on expected use. */
} Vps_CaptCreateParams;

/**
 *  \brief Capture driver create status. Returned after calling Fvid2_create().
 */
typedef struct
{
    Int32 retVal;
    /**< [OUT] Create status, FVID2_SOK on success, else failure. */
} Vps_CaptCreateStatus;

/**
 *  \brief Capture driver run-time parameters.
 *
 *  - This structure is returned by capture driver when Fvid2_dequeue()
 *    is called by application
 *  - NOTE:this structure is NOT filled by driver when TILED mode is used
 *  - This structure is returned as part of Fvid2_Frame.perFrameCfg
 *  - Application should make sure Fvid2_Frame.perFrameCfg is set to a valid
 *    Vps_CaptRtParams pointer when queue-ing the frame back to capture driver
 *    \code
 *    Vps_CaptRtParams rtParams;
 *
 *    pFrame->perFrameCfg = &rtParams;
 *    \endcode
 *  - Alternatively, user can pass back the same Fvid2_Frame pointer without
 *    modifying Fvid2_Frame.perFrameCfg
 */
typedef struct
{
    UInt32 capturedOutWidth;
    /**< [OUT] Captured data width in pixels. */
    UInt32 capturedOutHeight;
    /**< [OUT] Captured data height in lines. */
} Vps_CaptRtParams;

/**
 *  struct Vps_CaptStatus
 *  \brief Capture status structure used to get the current status.
 */
typedef struct
{
    UInt32 queueCount;
    /**< [OUT] Counter to keep track of how many requests are queued to the
     *   driver.
     *   Note: This counter will be reset at the time of driver create. */
    UInt32 dequeueCount;
    /**< [OUT] Counter to keep track of how many requests are dequeued from the
     *   driver.
     *   Note: This counter will be reset at the time of driver create. */
    UInt32 overflowCount;
    /**< [OUT] Counter to keep track of the occurrence of overflow error.
     *   Note: This counter will be reset at the time of driver create and
     *   during driver start. */
} Vps_CaptStatus;

/**
 *  \brief Capture Channel Get Status IOCTL input arguments
 */
typedef struct
{
    UInt32 chNum;
    /**< [IN] Channel number for which status is requested. */
    UInt32 frameInterval;
    /**< [IN] Expected interval in units of timer ticks between frames.
     *   Vps_CaptChStatus.isVideoDetected is FALSE if no frame is captured
     *   for a duration of 'frameInterval x 2', else
     *   Vps_CaptChStatus.isVideoDetected is TRUE. */
} Vps_CaptChStatusArgs;

/**
 *  \brief Capture Channel Get Status IOCTL result
 */
typedef struct
{
    UInt32 isVideoDetected;
    /**< [OUT] TRUE: Video detected at this channel,
     *         FALSE: Video not detected at this channel.
     *   Note: This flag is updated properly only when the capture is in
     *   progress as the time stamp comparision makes sense only then.
     *   Otherwise when the capture is stopped, this will always return as
     *   TRUE. */

    UInt32 queueCount;
    /**< [OUT] Counter to keep track of how many requests are queued
     *   to the driver. */
    UInt32 dequeueCount;
    /**< [OUT] Counter to keep track of how many requests are dequeued from the
     *   driver. */

    UInt32 captFrmCount;
    /**< [OUT] Number of frame's captured by driver into the buffer
     *   provided by application. This will also get incremented in case of
     *   frame drop mode. */
    UInt32 fldCount[FVID2_MAX_FIELDS];
    /**< [OUT] Field count for each field. */

    UInt32 maxRecvFrmWidth;
    /**< [OUT] Maximum frame/field width received from start to now. */
    UInt32 minRecvFrmWidth;
    /**< [OUT] Minimum frame/field width received from start to now. */
    UInt32 maxRecvFrmHeight;
    /**< [OUT] Maximum frame/field height received from start to now. */
    UInt32 minRecvFrmHeight;
    /**< [OUT] Minimum frame/field height received from start to now. */

    UInt32 droppedFrmCount;
    /**< [OUT] Number of frame's dropped by driver due to unavailability
     *   of buffer from application - used in frame drop mode. */
    UInt32 repeatFrmCount;
    /**< [OUT] Repeated frame/field count - used in frame repeat mode. */
    UInt32 fidRepeatCount;
    /**< [OUT] Counter to keep track of getting two consecutive top or
     *   bottom fields. */

    UInt32 lastFrmWidth;
    /**< [OUT] Last frame/field captured width in pixels. */
    UInt32 lastFrmHeight;
    /**< [OUT] Last frame/field captured height in lines. */
    UInt32 lastFrmTimeStamp;
    /**< [OUT] Last frame/field captured timestamp in OS ticks. */
    UInt32 lastFid;
    /**< [OUT] Last FID received. */

    /*
     * Error Counts.
     */
    UInt32 descErrCount;
    /**< [OUT] Capture descriptor not written by hardware error count.
     *          Not all interfaces support this, please refer interface
     *          specific header for details. */
} Vps_CaptChStatus;

/**
 *  \brief Frame Skip parameters. Not all interface support this, please refer
 *          interface specific header.
 */
typedef struct
{
    UInt32 chNum;
    /**< [IN] Channel number of which frame skip will be applied. */
    UInt32 frmSkipMask;
    /**< [IN] Frame Skip Mask (bit0..bit29) bitN = 1 Skip frame,
     *   bitN = 0 DO NOT skip frame
     *   Example, 0x0 for no frame skip, 0x2AAAAAAA for skipping
     *   alternate frames. */
} Vps_CaptFrameSkip;

/**
 *  \brief Capture overflow parameter structure
 */
typedef struct
{
    Vps_CaptVipOverflowFxn overflowCbFxn;
    /**< [IN] Application callback function to be called when overflow occurs
     *  _ISS_CHANGE_REQ_ Rename this function type to Vps_CaptOverflowFxn */
    void                  *retParams;
    /**< [OUT] Application handle returned to the callback function as return
     *   parameter */
} Vps_CaptOverflowParams;

/**
 *  \brief Capture overflow parameters, will be filled by the driver during
 *   check for overflow. This structure is required to send to driver by
 *   application to get the status of overflow
 */
typedef struct
{
    UInt32 isOverflowOccured;
    /**< [IN] Flag to check if overflow has occurred or not. */
} Vps_CaptOverflowCheckParams;

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief Get capture instance ID from channel number.
 *
 *  chNum is value which is a combination of
 *      - instance ID
 *      - stream ID for that instance
 *      - channel ID for that stream, instance
 *
 *  For details refer to video capture section in User Guide.
 *
 *  \param chNum        [IN] Channel number
 *
 *  \return instance ID (0 .. VPS_CAPT_INST_MAX - 1)
 */
static inline UInt32 Vps_captGetInstId(UInt32 chNum);

/**
 *  \brief Get capture stream ID from channel number.
 *
 *  chNum is value which is a combination of
 *      - instance ID
 *      - stream ID for that instance
 *      - channel ID for that stream, instance
 *
 *  For details refer to video capture section in User Guide.
 *
 *  \param chNum        [IN] Channel number
 *
 *  \return stream ID (0 .. VPS_CAPT_STREAM_ID_MAX - 1)
 */
static inline UInt32 Vps_captGetStreamId(UInt32 chNum);

/**
 *  \brief Get capture channel ID from channel number.
 *
 *  chNum is value which is a combination of
 *      - instance ID
 *      - stream ID for that instance
 *      - channel ID for that stream, instance
 *
 *  For details refer to video capture section in User Guide.
 *
 *  \param chNum        [IN] Channel number
 *
 *  \return Channel ID (0 .. VPS_CAPT_CH_PER_PORT_MAX - 1)
 */
static inline UInt32 Vps_captGetChId(UInt32 chNum);

/**
 *  \brief Make a system unique channel number.
 *
 *  chNum is value which is a combination of
 *      - instance ID
 *      - stream ID for that instance
 *      - channel ID for that stream, instance
 *
 *  For details refer to video capture section in User Guide.
 *
 *  \param instId       [IN] Instance ID.
 *  \param streamId     [IN] Stream ID.
 *  \param chId         [IN] Channel ID within an instance.
 *
 *  \return Channel number
 */
static inline UInt32 Vps_captMakeChNum(UInt32 instId,
                                       UInt32 streamId,
                                       UInt32 chId);

/**
 *  \brief Vps_CaptCreateParams structure init function.
 *
 *  \param createPrms   [IN] Pointer to #Vps_CaptCreateParams structure.
 *
 */
static inline void VpsCaptCreateParams_init(Vps_CaptCreateParams *createPrms);

/**
 *  \brief Vps_CaptChStatusArgs structure init function.
 *
 *  \param chStatusArgs [IN] Pointer to #Vps_CaptChStatusArgs structure.
 *
 */
static inline void VpsCaptChStatusArgs_init(Vps_CaptChStatusArgs *chStatusArgs);

/**
 *  \brief Vps_CaptFrameSkip structure init function.
 *
 *  \param frmSkip      [IN] Pointer to #Vps_CaptFrameSkip structure.
 *
 */
static inline void VpsCaptFrameSkip_init(Vps_CaptFrameSkip *frmSkip);

/* ========================================================================== */
/*                       Static Function Definitions                          */
/* ========================================================================== */

static inline UInt32 Vps_captGetInstId(UInt32 chNum)
{
    return (chNum / (VPS_CAPT_CH_PER_PORT_MAX * VPS_CAPT_STREAM_ID_MAX));
}

static inline UInt32 Vps_captGetStreamId(UInt32 chNum)
{
    UInt32 value;

    value  = chNum % (VPS_CAPT_CH_PER_PORT_MAX * VPS_CAPT_STREAM_ID_MAX);
    value /= VPS_CAPT_CH_PER_PORT_MAX;

    return (value);
}

static inline UInt32 Vps_captGetChId(UInt32 chNum)
{
    return (chNum % VPS_CAPT_CH_PER_PORT_MAX);
}

static inline UInt32 Vps_captMakeChNum(UInt32 instId,
                                       UInt32 streamId,
                                       UInt32 chId)
{
    return ((instId * VPS_CAPT_CH_PER_PORT_MAX * VPS_CAPT_STREAM_ID_MAX)
            + (streamId * VPS_CAPT_CH_PER_PORT_MAX) + chId);
}

static inline void VpsCaptCreateParams_init(Vps_CaptCreateParams *createPrms)
{
    UInt32 streamId, chId;

    if (NULL != createPrms)
    {
        createPrms->videoIfMode      = FVID2_VIFM_SCH_ES;
        createPrms->videoIfWidth     = FVID2_VIFW_16BIT;
        createPrms->bufCaptMode      = VPS_CAPT_BCM_FRM_DROP;
        createPrms->periodicCbEnable = (UInt32) TRUE;
        createPrms->chInQueueLength  = VPS_CAPT_DEF_QUEUE_LEN_PER_CH;
        createPrms->numCh     = 1U;
        createPrms->numStream = 1U;
        for (streamId = 0U; streamId < VPS_CAPT_STREAM_ID_MAX; streamId++)
        {
            for (chId = 0U; chId < VPS_CAPT_CH_PER_PORT_MAX; chId++)
            {
                createPrms->chNumMap[streamId][chId] = 0U;
            }
        }
        createPrms->muxModeStartChId = 0U;
        createPrms->pAdditionalArgs  = NULL;
    }

    return;
}

static inline void VpsCaptChStatusArgs_init(Vps_CaptChStatusArgs *chStatusArgs)
{
    if (NULL != chStatusArgs)
    {
        chStatusArgs->chNum         = 0U;
        chStatusArgs->frameInterval = 16U;      /* Default to 1/(60FPS) */
    }

    return;
}

static inline void VpsCaptFrameSkip_init(Vps_CaptFrameSkip *frmSkip)
{
    if (NULL != frmSkip)
    {
        frmSkip->chNum       = 0U;
        frmSkip->frmSkipMask = 0U;      /* Default to NO SKIP */
    }

    return;
}

#ifdef __cplusplus
}
#endif

#endif /* #ifndef VPS_CAPTURE_H_ */

/* @} */
