#include <stdio.h>
#include <unistd.h>
#include <VX/vx.h>
#include <VX/vx_khr_pipelining.h>
#include <TI/tivx.h>
#include <tivx_openvx_core_kernels.h>
#include <utils/app_init/include/app_init.h>
#include <tivx_task.h>


#define IN0_IMG_IDX (0u)
#define OUT0_IMG_IDX (1u)
#define INTR_IMG_IDX (2u)
#define BUF_SIZE (4u)

typedef struct {
    vx_context context;
    vx_graph   graph;
    vx_uint32  node_a_graph_parameter_index;
    vx_uint32  node_b_graph_parameter_index;
    vx_uint32  node_c_graph_parameter_index;
    tivx_task task;
    uint8_t stop_task;

} AppObj;

AppObj gAppObj;

static vx_enum kernel_a_id = (vx_status)VX_ERROR_INVALID_PARAMETERS;
static vx_enum kernel_b_id = (vx_status)VX_ERROR_INVALID_PARAMETERS;
static vx_enum kernel_c_id = (vx_status)VX_ERROR_INVALID_PARAMETERS;
static vx_enum kernel_d_id = (vx_status)VX_ERROR_INVALID_PARAMETERS;
static vx_kernel g_kernel[4] = {NULL};


// the kernel functions are defined with minimal code to just demonstrate the feedback graph usecase 
static vx_status VX_CALLBACK kernel_a_init(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_a_run(vx_node node, const vx_reference parameters[], vx_uint32 num){
    vx_status status;
    vx_image in_img, out_img;
    in_img = (vx_image)parameters[IN0_IMG_IDX];
    out_img = (vx_image)parameters[OUT0_IMG_IDX];
    uint8_t *data_ptr_in = NULL;
    vx_rectangle_t rect = { 0, 0, 1, 1};
    vx_map_id map_id_in;
    vx_imagepatch_addressing_t image_addr_in;
    status = vxMapImagePatch(in_img,
                &rect,
                0,
                &map_id_in,
                &image_addr_in,
                (void**)&data_ptr_in,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    printf("kernel_a In_img[0] %d\n", *data_ptr_in);

    uint8_t *data_ptr_out = NULL;
    vx_map_id map_id_out;
    vx_imagepatch_addressing_t image_addr_out;

    status = vxMapImagePatch(out_img,
                &rect,
                0,
                &map_id_out,
                &image_addr_out,
                (void**)&data_ptr_out,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    *data_ptr_out = (*data_ptr_in) + 1;
    printf("kernel_a out_img[0] %d\n", *data_ptr_out);  

    status = vxUnmapImagePatch(in_img, map_id_in);
    status = vxUnmapImagePatch(out_img, map_id_out);

    return status;
}
static vx_status VX_CALLBACK kernel_a_deinit(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_a_validate(vx_node node, const vx_reference parameters[ ], vx_uint32 num, vx_meta_format metas[]){
    return VX_SUCCESS;
}

static vx_status VX_CALLBACK kernel_b_init(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_b_run(vx_node node, const vx_reference parameters[], vx_uint32 num){
    vx_status status;
    vx_image in_img, out_img, feedback_in_img;
    in_img = (vx_image)parameters[IN0_IMG_IDX];
    out_img = (vx_image)parameters[OUT0_IMG_IDX];
    feedback_in_img = (vx_image)parameters[INTR_IMG_IDX];

    uint8_t *data_ptr_in = NULL;
    vx_rectangle_t rect = { 0, 0, 1, 1};
    vx_map_id map_id_in;
    vx_imagepatch_addressing_t image_addr_in;
    status = vxMapImagePatch(in_img,
                &rect,
                0,
                &map_id_in,
                &image_addr_in,
                (void**)&data_ptr_in,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    printf("kernel_b In_img[0] %d\n", *data_ptr_in);

    uint8_t *data_ptr_out = NULL;
    vx_map_id map_id_out;
    vx_imagepatch_addressing_t image_addr_out;
    status = vxMapImagePatch(out_img,
                &rect,
                0,
                &map_id_out,
                &image_addr_out,
                (void**)&data_ptr_out,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    *data_ptr_out = (*data_ptr_in) + 1;
    printf("kernel_b out_img[0] %d\n", *data_ptr_out); 

    uint8_t *data_ptr_intr = NULL;
    vx_map_id map_id_intr;
    vx_imagepatch_addressing_t image_addr_intr;
    status = vxMapImagePatch(feedback_in_img,
                &rect,
                0,
                &map_id_intr,
                &image_addr_intr,
                (void**)&data_ptr_intr,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    printf("kernel_b feedback_input_image[0] %d\n", *data_ptr_intr); 

    status = vxUnmapImagePatch(in_img, map_id_in);
    status = vxUnmapImagePatch(out_img, map_id_out);
    status = vxUnmapImagePatch(feedback_in_img, map_id_intr);

    return status;
}
static vx_status VX_CALLBACK kernel_b_deinit(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_b_validate(vx_node node, const vx_reference parameters[ ], vx_uint32 num, vx_meta_format metas[]){
    return VX_SUCCESS;
}

static vx_status VX_CALLBACK kernel_c_init(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_c_run(vx_node node, const vx_reference parameters[], vx_uint32 num){
    vx_status status;
    vx_image in_img, out_img, feedback_out_img;
    in_img = (vx_image)parameters[IN0_IMG_IDX];
    out_img = (vx_image)parameters[OUT0_IMG_IDX];
    feedback_out_img = (vx_image)parameters[INTR_IMG_IDX];

    uint8_t *data_ptr_in = NULL;
    vx_rectangle_t rect = { 0, 0, 1, 1};
    vx_map_id map_id_in;
    vx_imagepatch_addressing_t image_addr_in;
    status = vxMapImagePatch(in_img,
                &rect,
                0,
                &map_id_in,
                &image_addr_in,
                (void**)&data_ptr_in,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    printf("kernel_c In_img[0] %d\n", *data_ptr_in);

    uint8_t *data_ptr_out = NULL;
    vx_map_id map_id_out;
    vx_imagepatch_addressing_t image_addr_out;
    status = vxMapImagePatch(out_img,
                &rect,
                0,
                &map_id_out,
                &image_addr_out,
                (void**)&data_ptr_out,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    *data_ptr_out = (*data_ptr_in) + 1;
    printf("kernel_c out_img[0] %d\n", *data_ptr_out); 

    uint8_t *data_ptr_intr = NULL;
    vx_map_id map_id_intr;
    vx_imagepatch_addressing_t image_addr_intr;
    status = vxMapImagePatch(feedback_out_img,
                &rect,
                0,
                &map_id_intr,
                &image_addr_intr,
                (void**)&data_ptr_intr,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    *data_ptr_intr = (*data_ptr_in) - 1;
    printf("kernel_c feedback_output_image[0] %d\n", *data_ptr_intr); 

    status = vxUnmapImagePatch(in_img, map_id_in);
    status = vxUnmapImagePatch(out_img, map_id_out);
    status = vxUnmapImagePatch(feedback_out_img, map_id_intr);

    return status;
}
static vx_status VX_CALLBACK kernel_c_deinit(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_c_validate(vx_node node, const vx_reference parameters[ ], vx_uint32 num, vx_meta_format metas[]){
    return VX_SUCCESS;
}

static vx_status VX_CALLBACK kernel_d_init(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_d_run(vx_node node, const vx_reference parameters[], vx_uint32 num){
    vx_status status;
    vx_image in_img, out_img;
    in_img = (vx_image)parameters[IN0_IMG_IDX];
    out_img = (vx_image)parameters[OUT0_IMG_IDX];
    uint8_t *data_ptr_in = NULL;
    vx_rectangle_t rect = { 0, 0, 1, 1};
    vx_map_id map_id_in;
    vx_imagepatch_addressing_t image_addr_in;
    status = vxMapImagePatch(in_img,
                &rect,
                0,
                &map_id_in,
                &image_addr_in,
                (void**)&data_ptr_in,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    printf("kernel_d In_img[0] %d\n", *data_ptr_in);

    uint8_t *data_ptr_out = NULL;
    vx_map_id map_id_out;
    vx_imagepatch_addressing_t image_addr_out;

    status = vxMapImagePatch(out_img,
                &rect,
                0,
                &map_id_out,
                &image_addr_out,
                (void**)&data_ptr_out,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
    *data_ptr_out = (*data_ptr_in) + 1;
    printf("kernel_d out_img[0] %d\n", *data_ptr_out);  

    status = vxUnmapImagePatch(in_img, map_id_in);
    status = vxUnmapImagePatch(out_img, map_id_out);

    return status;
}
static vx_status VX_CALLBACK kernel_d_deinit(vx_node node, const vx_reference parameters[], vx_uint32 num){
    return VX_SUCCESS;
}
static vx_status VX_CALLBACK kernel_d_validate(vx_node node, const vx_reference parameters[ ], vx_uint32 num, vx_meta_format metas[]){
    return VX_SUCCESS;
}

static vx_status load_kernels(vx_context context){
    vx_kernel kernel = NULL;
    vx_status status;
    int index=0;

    { // adding kernel_a
        status = vxAllocateUserKernelId(context, &kernel_a_id);
        kernel = vxAddUserKernel(
            context,
            "kernel_a",
            kernel_a_id,
            kernel_a_run,
            2,
            kernel_a_validate,
            kernel_a_init,
            kernel_a_deinit
        );
        tivxAddKernelTarget(kernel, TIVX_TARGET_MPU_0);
        status = vxGetStatus((vx_reference)kernel);

        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_INPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_OUTPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status == VX_SUCCESS){
            status = vxFinalizeKernel(kernel);
        }
        if(status != VX_SUCCESS){
            vxReleaseKernel(&kernel);
            kernel = NULL;
        }else{
            g_kernel[0] = kernel;
        }
    }
    index=0;
    { // adding kernel_b
        if(status == VX_SUCCESS)
        {
            status = vxAllocateUserKernelId(context, &kernel_b_id);
            kernel = vxAddUserKernel(
                context,
                "kernel_b",
                kernel_b_id,
                kernel_b_run,
                3,
                kernel_b_validate,
                kernel_b_init,
                kernel_b_deinit
            );
        }
        tivxAddKernelTarget(kernel, TIVX_TARGET_MPU_0);

        status = vxGetStatus((vx_reference)kernel);

        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_INPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_OUTPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_INPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status == VX_SUCCESS){
            status = vxFinalizeKernel(kernel);
        }
        if(status != VX_SUCCESS){
            vxReleaseKernel(&kernel);
            kernel = NULL;
        }else{
            g_kernel[1] = kernel;
        }
    }
    index=0;
    { // adding kernel_c
        if(status == VX_SUCCESS)
        {
            status = vxAllocateUserKernelId(context, &kernel_c_id);
            kernel = vxAddUserKernel(
                context,
                "kernel_c",
                kernel_c_id,
                kernel_c_run,
                3,
                kernel_c_validate,
                kernel_c_init,
                kernel_c_deinit
            );
        }
        tivxAddKernelTarget(kernel, TIVX_TARGET_MPU_0);
        status = vxGetStatus((vx_reference)kernel);

        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_INPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_OUTPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_OUTPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status == VX_SUCCESS){
            status = vxFinalizeKernel(kernel);
        }
        if(status != VX_SUCCESS){
            vxReleaseKernel(&kernel);
            kernel = NULL;
        }else{
            g_kernel[2] = kernel;
        }
    }
    index=0;
    { // adding kernel_d
        status = vxAllocateUserKernelId(context, &kernel_d_id);
        kernel = vxAddUserKernel(
            context,
            "kernel_d",
            kernel_d_id,
            kernel_d_run,
            2,
            kernel_d_validate,
            kernel_d_init,
            kernel_d_deinit
        );
        tivxAddKernelTarget(kernel, TIVX_TARGET_MPU_0);

        status = vxGetStatus((vx_reference)kernel);

        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_INPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status==VX_SUCCESS){
            status = vxAddParameterToKernel(
                kernel,
                index++,
                (vx_enum)VX_OUTPUT,
                (vx_enum)VX_TYPE_IMAGE,
                (vx_enum)VX_PARAMETER_STATE_REQUIRED
            );
        }
        if(status == VX_SUCCESS){
            status = vxFinalizeKernel(kernel);
        }
        if(status != VX_SUCCESS){
            vxReleaseKernel(&kernel);
            kernel = NULL;
        }else{
            g_kernel[3] = kernel;
        }
    }

    return status;
}

static vx_status unload_kernels(vx_context context){
    vx_status status;
    status = vxRemoveKernel(g_kernel[0]);
    status = vxRemoveKernel(g_kernel[1]);
    status = vxRemoveKernel(g_kernel[2]);
    status = vxRemoveKernel(g_kernel[3]);
    g_kernel[0] = NULL;
    g_kernel[1] = NULL;
    g_kernel[2] = NULL;
    g_kernel[3] = NULL;

    return status;
}

vx_node node;
static vx_node get_kernel_a_node(vx_graph graph, vx_image in, vx_image out){
    vx_reference refs[] = {(vx_reference)in, (vx_reference)out};
    
    node = tivxCreateNodeByKernelEnum(graph,kernel_a_id,refs,2);
    return node;
}
static vx_node get_kernel_b_node(vx_graph graph, vx_image in, vx_image out, vx_image intr_img){
    vx_node node;
    vx_reference refs[] = {(vx_reference)in, (vx_reference)out, (vx_reference)intr_img};
    
    node = tivxCreateNodeByKernelEnum(graph,kernel_b_id,refs,3);
    return node;
}
static vx_node get_kernel_c_node(vx_graph graph, vx_image in, vx_image out, vx_image intr_img){
    vx_node node;
    vx_reference refs[] = {(vx_reference)in, (vx_reference)out, (vx_reference)intr_img};
    
    node = tivxCreateNodeByKernelEnum(graph,kernel_c_id,refs,3);
    return node;
}
static vx_node get_kernel_d_node(vx_graph graph, vx_image in, vx_image out){
    vx_reference refs[] = {(vx_reference)in, (vx_reference)out};
    
    node = tivxCreateNodeByKernelEnum(graph,kernel_d_id,refs,2);
    return node;
}

static void add_graph_parameter_by_node_index(vx_graph graph, vx_node node, vx_uint32 node_parameter_index)
{
    vx_parameter parameter = vxGetParameterByIndex(node, node_parameter_index);

    vxAddParameterToGraph(graph, parameter);
    vxReleaseParameter(&parameter);
}

static void swapReference(vx_reference ref1, vx_reference ref2)
{
    void *addr[2] = {NULL}, *addr_tmp[2];
    uint32_t size[2], size_tmp[2];
    uint32_t num_entries, num_entries_tmp;

    if(tivxIsReferenceMetaFormatEqual(ref1, ref2))
    {
        tivxReferenceExportHandle((vx_reference)ref1, addr_tmp, size_tmp, 2, &num_entries_tmp);
        tivxReferenceExportHandle((vx_reference)ref2, addr, size, 2, &num_entries);
        tivxReferenceImportHandle((vx_reference)ref1, (const void **)addr, (const uint32_t *)size, num_entries);
        tivxReferenceImportHandle((vx_reference)ref2, (const void **)addr_tmp, (const uint32_t *)size_tmp, num_entries_tmp);
    }
    else
    {
        printf("ERROR: Reference meta data not equal\n");
    }

}

static void app_run_task(void *app_var){
    vx_reference node_b_img, node_c_img;
    uint32_t num_refs;
    AppObj *obj = (AppObj *)app_var;

    while(!obj->stop_task){
        vxGraphParameterDequeueDoneRef(obj->graph, obj->node_b_graph_parameter_index, &node_b_img, 1, &num_refs);
        vxGraphParameterDequeueDoneRef(obj->graph, obj->node_c_graph_parameter_index, &node_c_img, 1, &num_refs);

        // swap the reference
        swapReference(node_b_img, node_c_img);

        vxGraphParameterEnqueueReadyRef(obj->graph, obj->node_b_graph_parameter_index, &node_b_img, 1);
        vxGraphParameterEnqueueReadyRef(obj->graph, obj->node_c_graph_parameter_index, &node_c_img, 1);
    }
}

static int32_t app_run_task_create(AppObj *obj)
{
    tivx_task_create_params_t params;
    vx_status status;

    tivxTaskSetDefaultCreateParams(&params);
    params.task_main = app_run_task;
    params.app_var = obj;
    obj->stop_task = 0;
    status = tivxTaskCreate(&obj->task, &params);
    return status;
}

int main(){

    AppObj *obj = &gAppObj;
    vx_image img1[BUF_SIZE], img2, img3, img4, img5, node_b_img[BUF_SIZE],node_c_img[BUF_SIZE];
    vx_node node_a, node_b, node_c, node_d;
    vx_uint32 width=1, height=1;
    vx_status status;

    status = appInit();

    vx_uint32 pipeline_depth=4, buf_id, loop_id, loop_cnt=10;
    vx_graph_parameter_queue_params_t graph_params_list[3];

    printf(" App started !!! \n");

    obj->context = vxCreateContext();

    status = load_kernels(obj->context);
    if(status == VX_SUCCESS){
        printf("Successfully loaded kernels \n" );
    }
    
    for(int i=0;i<BUF_SIZE;i++){
        img1[i] = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
    }  
    img2 = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
    img3 = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
    img4 = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
    img5 = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
    for(int i=0;i<BUF_SIZE;i++){
        node_b_img[i] = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
        node_c_img[i] = vxCreateImage(obj->context, width, height, (vx_df_image)VX_DF_IMAGE_U8);
    }
    
    obj->graph = vxCreateGraph(obj->context);

    node_a = get_kernel_a_node(obj->graph, img1[0], img2);
    vxSetReferenceName((vx_reference)node_a, "NODE_A");
    node_b = get_kernel_b_node(obj->graph, img2, img3, node_b_img[0]);
    vxSetReferenceName((vx_reference)node_b, "NODE_B");
    node_c = get_kernel_c_node(obj->graph, img3, img4, node_c_img[0]);
    vxSetReferenceName((vx_reference)node_c, "NODE_C");
    node_d = get_kernel_d_node(obj->graph, img4, img5);
    vxSetReferenceName((vx_reference)node_d, "NODE_D");

    obj->node_a_graph_parameter_index = 0;
    add_graph_parameter_by_node_index(obj->graph, node_a, 0);
    graph_params_list[0].graph_parameter_index = obj->node_a_graph_parameter_index;
    graph_params_list[0].refs_list_size = BUF_SIZE;
    graph_params_list[0].refs_list = (vx_reference *)&img1[0];

    obj->node_b_graph_parameter_index = 1;
    add_graph_parameter_by_node_index(obj->graph, node_b, 2);
    graph_params_list[1].graph_parameter_index = obj->node_b_graph_parameter_index;
    graph_params_list[1].refs_list_size = 1; // buffer_depth = 1; since only one buffer is swapped across 2 parameters
    graph_params_list[1].refs_list = (vx_reference *)&node_b_img[0];

    obj->node_c_graph_parameter_index = 2;
    add_graph_parameter_by_node_index(obj->graph, node_c, 2);
    graph_params_list[2].graph_parameter_index = obj->node_c_graph_parameter_index;
    graph_params_list[2].refs_list_size = 1; // buffer_depth = 1; since only one buffer is swapped across 2 parameters
    graph_params_list[2].refs_list = (vx_reference *)&node_c_img[0];

    vxSetGraphScheduleConfig(
        obj->graph,
        (vx_enum)VX_GRAPH_SCHEDULE_MODE_QUEUE_AUTO,
        3,
        graph_params_list
    );

    tivxSetGraphPipelineDepth(obj->graph, pipeline_depth);
    tivxSetNodeParameterNumBufByIndex(node_a, 1, BUF_SIZE);
    tivxSetNodeParameterNumBufByIndex(node_b, 1, BUF_SIZE);
    tivxSetNodeParameterNumBufByIndex(node_c, 1, BUF_SIZE);
    tivxSetNodeParameterNumBufByIndex(node_d, 1, BUF_SIZE);

    status = vxVerifyGraph(obj->graph);

    vx_image cur_in_img;
    uint32_t num_refs;

    uint8_t *data_ptr = NULL;
    vx_rectangle_t rect = { 0, 0, 1, 1};
    vx_map_id map_id;
    vx_imagepatch_addressing_t image_addr;
    int input_count = 0;
    
    for(buf_id=0;buf_id<BUF_SIZE;buf_id++){
        status = vxMapImagePatch(img1[buf_id],
                &rect,
                0,
                &map_id,
                &image_addr,
                (void**)&data_ptr,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
        *data_ptr = input_count;
        input_count += 10;

        vxGraphParameterEnqueueReadyRef(obj->graph, obj->node_a_graph_parameter_index, (vx_reference *)&img1[buf_id], 1);
    }
    // start the node_B giving the first invalid buffer, kernel should have implementation to avoid the first buffer
    vxGraphParameterEnqueueReadyRef(obj->graph, obj->node_b_graph_parameter_index, (vx_reference *)&node_b_img[0], 1);
    // enqueue one buffer to node_c to unblock the node
    vxGraphParameterEnqueueReadyRef(obj->graph, obj->node_c_graph_parameter_index, (vx_reference *)&node_c_img[0], 1);
    app_run_task_create(obj);  // start the enqueue/dequeue task 

    // run the loop
    for(loop_id=0;loop_id<loop_cnt;loop_id++){
        vxGraphParameterDequeueDoneRef(obj->graph, obj->node_a_graph_parameter_index, (vx_reference *)&cur_in_img, 1, &num_refs);

        status = vxMapImagePatch(cur_in_img,
                &rect,
                0,
                &map_id,
                &image_addr,
                (void**)&data_ptr,
                (vx_enum)VX_WRITE_ONLY, (vx_enum)VX_MEMORY_TYPE_HOST, (vx_enum)VX_NOGAP_X);
        *data_ptr = input_count;
        input_count += 10;
        vxGraphParameterEnqueueReadyRef(obj->graph, obj->node_a_graph_parameter_index, (vx_reference *)&cur_in_img, 1);
    }

    vxWaitGraph(obj->graph);
    obj->stop_task = 1;
    tivxTaskWaitMsecs(100);
    tivxTaskDelete(&obj->task);

    for(int i=0;i<BUF_SIZE;i++){
        vxReleaseImage(&img1[i]);
        vxReleaseImage(&node_b_img[i]);
        vxReleaseImage(&node_c_img[i]);
    }
    vxReleaseImage(&img2);
    vxReleaseImage(&img3);
    vxReleaseImage(&img4);
    vxReleaseImage(&img5);

    vxReleaseNode(&node_a);
    vxReleaseNode(&node_b);
    vxReleaseNode(&node_c);
    vxReleaseNode(&node_d);
    vxReleaseGraph(&obj->graph);
    unload_kernels(obj->context);
    vxReleaseContext(&obj->context);

    printf(" Application End !!! \n");

    appDeInit();

    return status;
}