/*
 *
 * Copyright (c) 2020 Texas Instruments Incorporated
 *
 * All rights reserved not granted herein.
 *
 * Limited License.
 *
 * Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
 * license under copyrights and patents it now or hereafter owns or controls to make,
 * have made, use, import, offer to sell and sell ("Utilize") this software subject to the
 * terms herein.  With respect to the foregoing patent license, such license is granted
 * solely to the extent that any such patent is necessary to Utilize the software alone.
 * The patent license shall not apply to any combinations which include this software,
 * other than combinations with devices manufactured by or for TI ("TI Devices").
 * No hardware patent is licensed hereunder.
 *
 * Redistributions must preserve existing copyright notices and reproduce this license
 * (including the above copyright notice and the disclaimer and (if applicable) source
 * code license limitations below) in the documentation and/or other materials provided
 * with the distribution
 *
 * Redistribution and use in binary form, without modification, are permitted provided
 * that the following conditions are met:
 *
 * *       No reverse engineering, decompilation, or disassembly of this software is
 * permitted with respect to any software provided in binary form.
 *
 * *       any redistribution and use are licensed by TI for use only with TI Devices.
 *
 * *       Nothing shall obligate TI to provide you with source code for the software
 * licensed and provided to you in object code.
 *
 * If software source code is provided to you, modification and redistribution of the
 * source code are permitted provided that the following conditions are met:
 *
 * *       any redistribution and use of the source code, including any resulting derivative
 * works, are licensed by TI for use only with TI Devices.
 *
 * *       any redistribution and use of any object code compiled from the source code
 * and any resulting derivative works, are licensed by TI for use only with TI Devices.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of its suppliers
 *
 * may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * DISCLAIMER.
 *
 * THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#ifndef _APP_SENSOR_MODULE_H_
#define _APP_SENSOR_MODULE_H_

#include <TI/tivx.h>
#include <iss_sensors.h>

#define SENSOR_CSI_INSTANCE_BLACK 0     // for HPC2.0 and HCP2.5
#define SENSOR_CSI_INSTANCE_WHITE 1     // for HPC2.0 and HPC2.5
#define SENSOR_CSI_INSTANCE_PURPLE 2    // for HPC2.5

#define SENSOR_NUM_PER_CSI 4

#ifdef __cplusplus
extern "C" {
#endif 

typedef struct {
    IssSensor_CreateParams i_sensorParams;
    vx_char* r_sensor_name; /* see iss_sensors.h, line 188 */
    vx_uint8 i_num_sensors_found;
    vx_uint32 o_features_enabled;   /* see `IssSensor_Features`. default is 'i_features_supported' */
    vx_uint32 i_features_supported;
    uint32_t i_num_cameras_enabled;
    vx_int32 i_image_width;
    vx_int32 i_image_height;
    vx_uint32 i_image_format;
    vx_uint32 i_is_wdr_mode;
    /**
     * bit indicator of camera channels.
     * if bit at position 'n' of `r_channel_mask` is 1,
     * then it means camera at channel 'n' is enabled
     * for example, if r_channel_mask is 0b00101101,
     * then camera at channel 0, 2, 3, 5 is enabled.
     */
    vx_uint32 r_channel_mask;
    vx_bool i_sensor_inited;
    vx_bool i_sensor_started;
} SensorMod;


/**
 * simply setup mod->r_channel_mask, with a csi index
 * \param mod: the sensor module
 * \param num_channels: total number of cameras to be enabled
 * \param csi_idx: the startup csi index, available values are
 *  - SENSOR_CSI_INSTANCE_BLACK
 *  - SENSOR_CSI_INSTANCE_WHITE
 *  - SENSOR_CSI_INSTANCE_PURPLE
 */ 
inline void app_sensor_setup_simple_mask_with_csi(SensorMod *mod, uint32_t num_channels, uint32_t csi_idx)
{
    uint32_t ch;
    for (ch = 0; ch < num_channels; ch++){
        uint32_t tmp = 1 << (ch + (csi_idx * SENSOR_NUM_PER_CSI));
        mod->r_channel_mask |= tmp;
    }
}

/**
 * simply setup mod->r_channel_mask
 */
inline void app_sensor_setup_simple_mask(SensorMod *mod, uint32_t num_channels)
{
    app_sensor_setup_simple_mask_with_csi(mod, num_channels, 0);
}


/**
 * list all available sensors
 */ 
vx_status app_enumerate_sensor();


/**
 * configure serdes and camera according to `r_sensor_name`
 */
vx_status app_config_sensor(SensorMod *mod);

/**
 * \brief start sensor according to `r_sensor_name`
*/
vx_status app_start_sensor(SensorMod *sensor_mod);


/**
 * query parameters of `r_sensor_name`, 
 * and initialize sensro_mod accordingly
 */ 
vx_status app_initiate_sensor(SensorMod *sensor_mod);

/**
 * stop the sensor, and deinit it
 */
void app_stop_sensor(SensorMod *sensor_mod);

#ifdef __cplusplus
}
#endif // __cplusplus

#endif //__APP_SENSOR_MODULE_H_
