/*
 *
 * Copyright (c) 2017 Texas Instruments Incorporated
 *
 * All rights reserved not granted herein.
 *
 * Limited License.
 *
 * Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
 * license under copyrights and patents it now or hereafter owns or controls to make,
 * have made, use, import, offer to sell and sell ("Utilize") this software subject to the
 * terms herein.  With respect to the foregoing patent license, such license is granted
 * solely to the extent that any such patent is necessary to Utilize the software alone.
 * The patent license shall not apply to any combinations which include this software,
 * other than combinations with devices manufactured by or for TI ("TI Devices").
 * No hardware patent is licensed hereunder.
 *
 * Redistributions must preserve existing copyright notices and reproduce this license
 * (including the above copyright notice and the disclaimer and (if applicable) source
 * code license limitations below) in the documentation and/or other materials provided
 * with the distribution
 *
 * Redistribution and use in binary form, without modification, are permitted provided
 * that the following conditions are met:
 *
 * *       No reverse engineering, decompilation, or disassembly of this software is
 * permitted with respect to any software provided in binary form.
 *
 * *       any redistribution and use are licensed by TI for use only with TI Devices.
 *
 * *       Nothing shall obligate TI to provide you with source code for the software
 * licensed and provided to you in object code.
 *
 * If software source code is provided to you, modification and redistribution of the
 * source code are permitted provided that the following conditions are met:
 *
 * *       any redistribution and use of the source code, including any resulting derivative
 * works, are licensed by TI for use only with TI Devices.
 *
 * *       any redistribution and use of any object code compiled from the source code
 * and any resulting derivative works, are licensed by TI for use only with TI Devices.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of its suppliers
 *
 * may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * DISCLAIMER.
 *
 * THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
// #include <TI/j7_imaging_aewb.h>
#include <tivx_utils_file_rd_wr.h>
#include <TI/tivx_soc_j784s4.h>

#include "app_capture_module.h"
#include "app_viss_module.h"
#include "app_aewb_module.h"
#include "app_display_module.h"
#include "app_img_mosaic_module.h"


struct App
{
    vx_context context;
    vx_graph graph;
    const char *cap_target;

    // input
    SensorMod sensor_mod_0;
    SensorMod sensor_mod_1;
    CaptureMod cap_mod_0;
    CaptureMod cap_mod_1;

    AewbMod aewb_mod_0;
    AewbMod aewb_mod_1;
    VissMod viss_mod_0;
    VissMod viss_mod_1;
    ImgMosaicObj mosaic_mod;
    DisplayObj display_mod;

    vx_uint32 num_graph_prms;
    vx_graph_parameter_queue_params_t graph_prms_list[5];
} apps[2];


uint32_t runtime = 30;
static int do_enumerate_only = 0;
static uint32_t num_cameras_enabled = 1;
static uint32_t csi_instance = 0;
static uint32_t write_nframes = 60;
static vx_bool enable_display = 1;
static vx_bool enable_write;
static vx_bool enable_scaler;
static vx_bool enable_cap_error_detection = 1;
static vx_bool do_exit_app;


#define NUM_BUF 4
#define WITH_VISS
#define WITH_DISPLAY


static void app_create_graph(struct App *app)
{
    vx_status status;
    vx_context context = vxCreateContext();

    app->context = context;
    app->graph = vxCreateGraph(context);

    app->cap_mod_0.o_enable_error_detection = enable_cap_error_detection;
    app_add_node_capture(app->graph, &app->cap_mod_0, &app->sensor_mod_0);
    vxSetNodeTarget(app->cap_mod_0.i_node, VX_TARGET_STRING, TIVX_TARGET_CAPTURE1);

    app->cap_mod_1.o_enable_error_detection = enable_cap_error_detection;
    app_add_node_capture(app->graph, &app->cap_mod_1, &app->sensor_mod_1);
    vxSetNodeTarget(app->cap_mod_1.i_node, VX_TARGET_STRING, TIVX_TARGET_CAPTURE2);

    #ifdef WITH_VISS
    app->viss_mod_0.r_input_arr = app->cap_mod_0.i_output_arr;
    app->viss_mod_0.o_enable_replicate = 1;
    app_add_node_viss(app->graph, &app->viss_mod_0, &app->sensor_mod_0);
    vxSetNodeTarget(app->viss_mod_0.i_node, VX_TARGET_STRING, TIVX_TARGET_VPAC1_VISS1);

    app->viss_mod_1.r_input_arr = app->cap_mod_1.i_output_arr;
    app->viss_mod_1.o_enable_replicate = 1;
    app_add_node_viss(app->graph, &app->viss_mod_1, &app->sensor_mod_1);
    vxSetNodeTarget(app->viss_mod_1.i_node, VX_TARGET_STRING, TIVX_TARGET_VPAC1_VISS1);

    app->aewb_mod_0.r_input_h3a_stats_arr = app->viss_mod_0.i_h3a_stats_arr;
    app->aewb_mod_0.o_enable_replicate = 1;
    app_add_node_aewb(app->graph, &app->aewb_mod_0, &app->sensor_mod_0);
    vxSetNodeTarget(app->aewb_mod_0.i_node, VX_TARGET_STRING, TIVX_TARGET_MCU4_0);

    app->aewb_mod_1.r_input_h3a_stats_arr = app->viss_mod_1.i_h3a_stats_arr;
    app->aewb_mod_1.o_enable_replicate = 1;
    app_add_node_aewb(app->graph, &app->aewb_mod_1, &app->sensor_mod_1);
    vxSetNodeTarget(app->aewb_mod_1.i_node, VX_TARGET_STRING, TIVX_TARGET_IPU1_0);

    #ifdef WITH_DISPLAY
    vx_object_array display_input;
    if (1){
        // make 8 cam and 4 cam compatible
        uint32_t scaler_img_width;
        uint32_t scaler_img_height;
        uint32_t num_splits_h;
        uint32_t num_splits_v;
        // if (app->sensor_mod_0.i_num_cameras_enabled > 9){
        //     LOGI("mosaic split into 12 sub images!");
        //     num_splits_h = 4;
        //     num_splits_v = 3;
        // } else if (app->sensor_mod_0.i_num_cameras_enabled > 4){
        //     LOGI("mosaic split into 9 sub images!");
        //     num_splits_h = num_splits_v = 3;
        // } else {
            LOGI("mosaic split into 4 sub images!");
            num_splits_h = num_splits_v = 2;
        // }
        scaler_img_width = 1920 / num_splits_h;
        scaler_img_height = 1080 / num_splits_v;

        app->mosaic_mod.r_input_arr[0] = app->viss_mod_0.i_output_arr;
        app->mosaic_mod.r_input_arr[1] = app->viss_mod_1.i_output_arr;
        app->mosaic_mod.r_num_inputs = 2;
        app->mosaic_mod.r_out_width = 1920;
        app->mosaic_mod.r_out_height = 1080;
        tivxImgMosaicParamsSetDefaults(&app->mosaic_mod.r_params);

        for (int i = 0; i < 4; i++){
            app->mosaic_mod.r_params.windows[i].channel_select = 0;
            app->mosaic_mod.r_params.windows[i].startX = (i % num_splits_h) * scaler_img_width;
            app->mosaic_mod.r_params.windows[i].startY = (i / num_splits_h) * scaler_img_height;
            app->mosaic_mod.r_params.windows[i].width = scaler_img_width;
            app->mosaic_mod.r_params.windows[i].height = scaler_img_height;
            app->mosaic_mod.r_params.windows[i].input_select = 1;
            app->mosaic_mod.r_params.windows[i].channel_select = i;
        }

        app->mosaic_mod.r_params.windows[3].startX = 960;
        app->mosaic_mod.r_params.windows[3].startY = 540;
        app->mosaic_mod.r_params.windows[3].width = scaler_img_width;
        app->mosaic_mod.r_params.windows[3].height = scaler_img_height;
        app->mosaic_mod.r_params.windows[3].input_select = 0;
        app->mosaic_mod.r_params.windows[3].channel_select = 0;

        app->mosaic_mod.r_params.num_windows = 4;
        app->mosaic_mod.r_params.clear_count  = 4;
        app_add_node_img_mosaic(app->graph, &app->mosaic_mod);

        display_input = app->mosaic_mod.i_output_arr;
    } else {
    }

    if (enable_display) {
        vx_uint32 width, height;
        vx_image image = (vx_image)vxGetObjectArrayItem(display_input, 0);
        status = vxQueryImage(image, VX_IMAGE_HEIGHT, &height, sizeof(height));
        status = vxQueryImage(image, VX_IMAGE_WIDTH, &width, sizeof(width));
        vxReleaseImage(&image);
        if (width > 1920 || height > 1080){
            enable_scaler = 1;
            LOGW("display input exceeds 1920x1080!");
        }
        app->display_mod.r_input_arr = display_input;
        app->display_mod.o_screen_height = 1080;
        app->display_mod.o_screen_width = 1920;
        LOGI("display module enabled!");
        app_add_node_display(app->graph, &app->display_mod);
    }

    #endif
    #endif

    #if 0
    app->num_graph_prms = 0;
    #ifdef WITH_VISS
    add_graph_parameter_by_node_index(app->graph, app->cap_mod_0.i_node, 1);
    #else
    add_graph_parameter_by_node_index(app->graph, app->cap_mod_0.i_node, 1);
    #endif

    app->graph_prms_list[app->num_graph_prms].graph_parameter_index = app->num_graph_prms;
    app->graph_prms_list[app->num_graph_prms].refs_list_size = NUM_BUF;
    app->graph_prms_list[app->num_graph_prms].refs_list = 
    #ifdef WITH_VISS
        app_duplicate_vx_refs((vx_reference)(app->cap_mod_0.i_output_arr), NUM_BUF);
    #else
        app_duplicate_vx_refs((vx_reference)(app->cap_mod_0.i_output_arr), NUM_BUF);
    #endif
    app->num_graph_prms++;

    #ifdef WITH_VISS
    add_graph_parameter_by_node_index(app->graph, app->cap_mod_1.i_node, 1);
    #else
    add_graph_parameter_by_node_index(app->graph, app->cap_mod_1.i_node, 1);
    #endif

    app->graph_prms_list[app->num_graph_prms].graph_parameter_index = app->num_graph_prms;
    app->graph_prms_list[app->num_graph_prms].refs_list_size = NUM_BUF;
    app->graph_prms_list[app->num_graph_prms].refs_list = 
    #ifdef WITH_VISS
        app_duplicate_vx_refs((vx_reference)(app->cap_mod_1.i_output_arr), NUM_BUF);
    #else
        app_duplicate_vx_refs((vx_reference)(app->cap_mod_1.i_output_arr), NUM_BUF);
    #endif
    app->num_graph_prms++;

    status = vxSetGraphScheduleConfig(app->graph,
                        VX_GRAPH_SCHEDULE_MODE_QUEUE_AUTO,
                        app->num_graph_prms,
                        app->graph_prms_list
                        );
    LOG_IF_FAIL(status);
    #else
    // vxEnableGraphStreaming(app->graph, app->cap_mod.i_node);
    #endif

    status = vxVerifyGraph(app->graph);
    LOG_IF_FAIL(status);

    if (enable_cap_error_detection){
        app_send_error_frame(&app->cap_mod_0);
        app_send_error_frame(&app->cap_mod_1);
    }

    tivxExportGraphToDot(app->graph, "./", "app_show");
}


static void app_delete_graph(struct App *app)
{
    app_release_duplicated_refs();

    vxReleaseGraph(&app->graph);

    app_delete_node_capture(&app->cap_mod_0);
    app_delete_node_capture(&app->cap_mod_1);

    #ifdef WITH_VISS
    app_delete_node_viss(&app->viss_mod_0);
    app_delete_node_aewb(&app->aewb_mod_0);
    app_delete_node_viss(&app->viss_mod_1);
    app_delete_node_aewb(&app->aewb_mod_1);

    #ifdef WITH_DISPLAY
    // if (app->sensor_mod_0.i_num_cameras_enabled > 1){
        app_delete_node_img_mosaic(&app->mosaic_mod);
    // }
    if (enable_display){
        app_delete_node_display(&app->display_mod);
    }
    #endif
    #endif
    vxReleaseContext(&app->context);
}


#if 1
static void app_run_graph(struct App *app)
{
    vx_status status = 0;

    #if 0
    vx_uint32 num_refs_capture;

    for (int i = 0; i < app->num_graph_prms; i++){
        for (int j = 0; j < NUM_BUF; j++){
            LOGD("pipeup eq. graph param %d, buffer %d", i, j);
            status = vxGraphParameterEnqueueReadyRef(app->graph, i, &app->graph_prms_list[i].refs_list[j], 1);
            LOG_IF_FAIL(status);
        }
    }

    for (uint i = 0; i < -1U && !do_exit_app && status == 0; i++){
        vx_object_array arr0;
        vx_object_array arr1;
        // for (int index = 0; index < app->num_graph_prms; index++){
            LOGD("dq. graph prm index %d", 0);
            status = vxGraphParameterDequeueDoneRef(app->graph, 0, (vx_reference*)&arr0, 1, &num_refs_capture);
            LOG_IF_FAIL(status);

            LOGD("eq. graph prm index %d", 0);
            status = vxGraphParameterEnqueueReadyRef(app->graph, 0, (vx_reference*)&arr0, 1);
            LOG_IF_FAIL(status);

            LOGD("dq. graph prm index %d", 1);
            status = vxGraphParameterDequeueDoneRef(app->graph, 1, (vx_reference*)&arr1, 1, &num_refs_capture);
            LOG_IF_FAIL(status);

            // vx_size nitems;
            // vxQueryObjectArray(arr, VX_OBJECT_ARRAY_NUMITEMS, &nitems, sizeof(nitems));

            // #ifdef WITH_VISS
            // for (vx_size ch = 0; ch < nitems; ch++){
            //     // vx_image img = (vx_image)vxGetObjectArrayItem(arr, ch);
            //     // char buf[64];
            //     // snprintf(buf, sizeof(buf) - 1, "./cam_test_port%d_%d_Ch%ld", index, i, ch);
            //     // write_vximage_to_file(buf, img);
            // }
            // #else
            // for (vx_size ch = 0; ch < nitems; ch++){
            //     tivx_raw_image img = (tivx_raw_image)vxGetObjectArrayItem(arr, ch);
            //     char buf[64];
            //     snprintf(buf, sizeof(buf) - 1, "./cam_test_%d_Ch%ld", i, ch);
            //     write_raw_image_to_file(buf, img);
            // }
            // #endif

            LOGD("eq. graph prm index %d", 1);
            status = vxGraphParameterEnqueueReadyRef(app->graph, 1, (vx_reference*)&arr1, 1);
            LOG_IF_FAIL(status);

        // }
    }

    vxWaitGraph(app->graph);

    #else
    // status = vxStartGraphStreaming(app->graph);
    // LOG_IF_FAIL(status);

    while (!do_exit_app){
        LOGI("processing...");
        status = vxProcessGraph(app->graph);
        LOG_IF_FAIL(status);
    }

    // sleep could be interrupted by signal
    // sleep(runtime);
    // if (do_exit_app){
    //     LOGI("exit app due to signal");
    //     // return;
    // }
    // LOGD("sleep finished, waiting graph...");

    // vxStopGraphStreaming(app->graph);
    #endif
}
#endif


const char *useage = "          \n\n\n\
app camera test program             \n\
                                    \n\
USAGE : ./app_camtest.out [OPTIONS] \n\
                                    \n\
OPTIONS:                            \n\
    -s [sensor_name]: specify sensor name, default: OVX3CF-MAX96717_MARS_MULTI \n\
    -c [csi]: csi select option, default: 0                                    \n\
    -n [num]: number of cameras enabled, default: 1                            \n\
    -t [time]: time to run. unit: s. default: 30                               \n\
    -f [frames]: max frames to save when -w is enabeld, default is 60          \n\
    -w enable save frames to file. save 1 image every 10 images                \n\
    -d enable output frames to DP(Display Port)                                \n\
    -q query all available sensor names only.                                  \n\
    -e enable error detection for capture node, black image will be provided if no valid camera image is available\n\
    -v enable VX_ZONE_INFO                                                     \n\
    \n\n\
"
;

static void parse_option(int argc, char *argv[])
{
    int opt;
    struct App *app = &apps[0];

    while ((opt = getopt(argc, argv, "c:s:n:t:f:wdehqv")) != -1) {
        switch (opt) {
        case 'c':
            csi_instance = atoi(optarg);
            break;

        case 'n':
            num_cameras_enabled = atoi(optarg);
            break;

        case 's':
            app->sensor_mod_0.r_sensor_name = strdup(optarg);
            break;

        case 'd':
            enable_display = 1;
            break;

        case 'w':
            enable_write = 1;
            break;

        case 't':
            runtime = atoi(optarg);
            break;

        case 'q':
            do_enumerate_only = 1;
            break;

        case 'e':
            enable_cap_error_detection = 1;
            break;

        case 'f':
            write_nframes = atoi(optarg);
            break;

        case 'v':
            tivx_set_debug_zone(VX_ZONE_INFO);
            break;

        case 'h':
            printf(useage);
            exit(0);
            break;

        default: /* '?' */
            fprintf(stderr, useage);
            exit(EXIT_FAILURE);
        }
    }
}


static void sig_handler(int signal)
{
    LOGI("handling signal %d", signal);
    do_exit_app = 1;
}

int main(int argc, char *argv[])
{
    int index = 0;

    struct App *app = &apps[index];

    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);

    apps[0].sensor_mod_0.r_sensor_name = "OS_AR0820_MAX9295_LI_MULTI";
    // apps[0].sensor_mod_0.r_sensor_name = "OVX3CF-MAX96717_MARS_MULTI";
    apps[0].sensor_mod_1.r_sensor_name = "OVX3CF-MAX96717_MARS_MULTI";
    apps[0].sensor_mod_0.r_channel_mask = 0x1;
    apps[0].sensor_mod_1.r_channel_mask = 0x70;

    // parse_option(argc, argv);

    if (do_enumerate_only){
        return app_enumerate_sensor();
    }

    // app_sensor_setup_simple_mask_with_csi(&apps[0].sensor_mod_0, num_cameras_enabled, csi_instance);
    // LOGD("sensor name is %s", app->sensor_mod_0.r_sensor_name);

    if (strcmp(app->sensor_mod_0.r_sensor_name, "OS_AR0820_MAX9295_LI_MULTI") == 0){
        apps[0].sensor_mod_0.o_features_enabled = 0x158;
    }
    app_initiate_sensor(&app->sensor_mod_0);
    app_initiate_sensor(&app->sensor_mod_1);

    app_config_sensor(&app->sensor_mod_0);
    app_config_sensor(&app->sensor_mod_1);

    app_create_graph(app);

    app_start_sensor(&app->sensor_mod_0);
    app_start_sensor(&app->sensor_mod_1);

    app_run_graph(app);

    app_delete_graph(app);

    app_stop_sensor(&app->sensor_mod_0);
    app_stop_sensor(&app->sensor_mod_1);

    return 0;
}
