/*
 *
 * Copyright (c) 2018 Texas Instruments Incorporated
 *
 * All rights reserved not granted herein.
 *
 * Limited License.
 *
 * Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
 * license under copyrights and patents it now or hereafter owns or controls to make,
 * have made, use, import, offer to sell and sell ("Utilize") this software subject to the
 * terms herein.  With respect to the foregoing patent license, such license is granted
 * solely to the extent that any such patent is necessary to Utilize the software alone.
 * The patent license shall not apply to any combinations which include this software,
 * other than combinations with devices manufactured by or for TI ("TI Devices").
 * No hardware patent is licensed hereunder.
 *
 * Redistributions must preserve existing copyright notices and reproduce this license
 * (including the above copyright notice and the disclaimer and (if applicable) source
 * code license limitations below) in the documentation and/or other materials provided
 * with the distribution
 *
 * Redistribution and use in binary form, without modification, are permitted provided
 * that the following conditions are met:
 *
 * *       No reverse engineering, decompilation, or disassembly of this software is
 * permitted with respect to any software provided in binary form.
 *
 * *       any redistribution and use are licensed by TI for use only with TI Devices.
 *
 * *       Nothing shall obligate TI to provide you with source code for the software
 * licensed and provided to you in object code.
 *
 * If software source code is provided to you, modification and redistribution of the
 * source code are permitted provided that the following conditions are met:
 *
 * *       any redistribution and use of the source code, including any resulting derivative
 * works, are licensed by TI for use only with TI Devices.
 *
 * *       any redistribution and use of any object code compiled from the source code
 * and any resulting derivative works, are licensed by TI for use only with TI Devices.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of its suppliers
 *
 * may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * DISCLAIMER.
 *
 * THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef _APP_UTILS_MODULE_H_
#define _APP_UTILS_MODULE_H_

#include <stdio.h>
#include <TI/tivx_ext_raw_image.h>
#include <TI/tivx_tensor.h>

// pipeline parameters
#define APP_MAX_TENSOR_DIMS (4u)

// LOG macros
#define filename(s) strrchr(s, '/') ? strrchr(s, '/')+1 : s


#ifndef LOGE
#define LOGE(fmt, ...) app_common_module_default_log_func("%s[%d](error): " fmt "\n", filename(__FILE__), __LINE__, ##__VA_ARGS__);
#endif

#ifndef LOGW
#define LOGW(fmt, ...) app_common_module_default_log_func("%s[%d](warn): " fmt "\n", filename(__FILE__), __LINE__, ##__VA_ARGS__);
#endif

#ifndef LOGI
#define LOGI(fmt, ...) app_common_module_default_log_func("%s[%d](info): " fmt "\n", filename(__FILE__), __LINE__, ##__VA_ARGS__);
#endif

#ifndef LOGD
#define LOGD(fmt, ...) app_common_module_default_log_func("%s[%d](dbg): " fmt "\n", filename(__FILE__), __LINE__, ##__VA_ARGS__);
#endif

#ifndef LOGV
#if 0
#define LOGV(fmt, ...) app_common_module_default_log_func("%s[%d](ver): " fmt "\n", filename(__FILE__), __LINE__, ##__VA_ARGS__);
#else
#define LOGV(fmt, ...)
#endif
#endif

#define LOG_IF_FAIL(status)                     \
    if (status != VX_SUCCESS){                  \
        LOGE("[Error] status: %d", status);             \
    }

#define VX_CHECK(ref)   \
    LOG_IF_FAIL(vxGetStatus((vx_reference)(ref)));  \


#ifdef __cplusplus
extern "C" {
#endif 

/**
 * app common module default log function
 * 
 * this is marked as a weak symbol and can be 
 * overwrited by user app
 */
__attribute__((weak))
__attribute__((format(printf, 1, 2)))
void app_common_module_default_log_func(const char *fmt, ...);

vx_uint32 get_num_planes(vx_df_image format);

/**
 * write a vx_image to a file
 * @param fpath: file path, absolute path or relative path
 * @param image: a vx_image
 * NOTE: only images of format NV12 and uyvy are tested so far,
 * other format is to be tested
 */ 
vx_status write_vximage_to_file(const char *fpath_prefix, vx_image image);

/**
 * load a vx_image from file 
 * see: `write_vximage_to_file`
 * NOTE: only images of format NV12 and uyvy are tested so far,
 * other format is to be tested
 */
vx_status load_vximage_from_file(const char* fpath, vx_image image);

/**
 * load a vx_image from a buf pointed by `ptr` 
 * see: `load_vximage_from_file`
 * NOTE: only images of format NV12 and uyvy are tested so far,
 * other format is to be tested
 */
vx_status load_vximage_from_ptr(const void *ptr, vx_image image);

/**
 * write a tivx_raw_image to file
 * NOTE: this function has not been tested
 */ 
vx_status write_raw_image_to_file(const char *fpath_prefix, tivx_raw_image img_raw);

/**
 * load a tivx_raw_image from file
 * NOTE: this function has not been tested
 */
vx_status load_raw_image_from_file(const char *fpath, tivx_raw_image img_raw);

/**
 * utils functions to show attributes of tiovx reference.
 */ 
void app_show_reference_attributes(vx_reference ref);

/**
 * utils functions to show attributes of tiovx parameter.
 */ 
void app_show_param_attributes(vx_parameter p);

/**
 * utils functions to show attributes of tiovx graph.
 */ 
void app_show_graph_attributes(vx_graph graph);

/**
 * utils functions to show attributes of tiovx node.
 */ 
void app_show_node_attributes(vx_node node);


/**
 * a frequently used function to add graph parameters
 */ 
vx_status add_graph_parameter_by_node_index(vx_graph graph, vx_node node, vx_uint32 node_parameter_index);


/**
 * a convenient way to map a vx_image
 * @param img: a vx_image
 * @param id: an address to save map id, which is to be used in `unmap_img`
 * @param plane_idx: plane index
 * @param usage: VX_READ_ONLY, VX_WRITE_ONLY or VX_READ_AND_WRITE
 * @return NULL on failure
 */ 
void *map_vximg(vx_image img, vx_map_id *id, int plane_idx, vx_enum usage);

/**
 * use this function to unmap a vx_image.
 * 
 * this is expected to be used along with `map_vximg`
 */ 
void unmap_img(vx_image img, vx_map_id id);

/**
 * a convenient way to map a vx_tensor
 * @param img: a vx_tensor
 * @param id: an address to save map id, which is to be used in `unmap_tensor`
 * @param usage: VX_READ_ONLY, VX_WRITE_ONLY or VX_READ_AND_WRITE
 * @return NULL on failure
 */ 
void *map_tensor(vx_tensor tensor, vx_map_id *id, vx_enum usage);

/**
 * use this function to unmap a vx_tensor.
 * 
 * this is expected to be used along with `map_tensor`
 */ 
void unmap_tensor(vx_tensor tensor, vx_map_id map_id);

/**
 * read lut files to shared memory
 * @return memory physical address
 */ 
void *read_lut_file(const char *fpath, size_t lut_size);

/**
 * duplicate vx references
 * this function is used when adding graph parameters to graph,
 * where an array, whose size is BUF_Q_DEPTH, is needed.
 * call `app_release_duplicated_refs` when deinit app
 * @return a vx_reference *, which can be used as graph parameters
 */ 
vx_reference *app_duplicate_vx_refs(vx_reference ref, vx_uint32 num);

/**
 * used to release vx references created by `app_duplicate_vx_refs`
 */ 
void app_release_duplicated_refs();

static inline vx_size app_get_object_array_size(vx_object_array arr){
    vx_size nitems;
    vxQueryObjectArray(arr, VX_OBJECT_ARRAY_NUMITEMS, &nitems, sizeof(nitems));

    return nitems;
}

#ifdef __cplusplus
}
#endif 

#endif // _APP_UTILS_MODULE_H_