/*
 * board.c
 *
 * Board functions for linkiq
 *
 * Based on:
 * TI AM335X based boards
 *
 * Copyright (C) 2011, Texas Instruments, Incorporated - http://www.ti.com/
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <errno.h>
#include <spl.h>
#include <asm/arch/cpu.h>
#include <asm/arch/hardware.h>
#include <asm/arch/omap.h>
#include <asm/arch/ddr_defs.h>
#include <asm/arch/clock.h>
#include <asm/arch/gpio.h>
#include <asm/arch/mmc_host_def.h>
#include <asm/arch/sys_proto.h>
#include <asm/arch/mem.h>
#include <asm/io.h>
#include <asm/emif.h>
#include <asm/gpio.h>
#include <i2c.h>
#include <power/tps65217.h>
#include <environment.h>
#include <watchdog.h>
#include <environment.h>
#include <pcf2127_wdt.h>
#include "board.h"

DECLARE_GLOBAL_DATA_PTR;

/* GPIO that controls power to DDR on EVM-SK */
#define GPIO_TO_PIN(bank, gpio)		(32 * (bank) + (gpio))
#define GPIO_DDR_VTT_EN		GPIO_TO_PIN(0, 7)
#define GPIO_ETH_RESET		GPIO_TO_PIN(3, 18)

#ifndef CONFIG_SKIP_LOWLEVEL_INIT
static const struct ddr_data ddr3_beagleblack_data = {
	.datardsratio0 = 0x38,
	.datawdsratio0 = 0x35,
	.datafwsratio0 = 0x9d,
	.datawrsratio0 = 0x6f,
};

static const struct cmd_control ddr3_beagleblack_cmd_ctrl_data = {
	.cmd0csratio = 0x80,
	.cmd0iclkout = 0,

	.cmd1csratio = 0x80,
	.cmd1iclkout = 0,

	.cmd2csratio = 0x80,
	.cmd2iclkout = 0,
};

static struct emif_regs ddr3_beagleblack_emif_reg_data = {
	.sdram_config		= 0x61C052B2,
	.ref_ctrl		= 0x00000C30,
	.sdram_tim1		= 0x0AAAE4DB,
	.sdram_tim2		= 0x46437FDA,
	.sdram_tim3		= 0x501F83FF,
	.zq_config		= 0x50074BE4,
	.emif_ddr_phy_ctlr_1	= 0x00100307,
};

#ifdef CONFIG_SPL_OS_BOOT

int spl_start_uboot(void)
{
	/* break into full u-boot on 'c' */
	if (serial_tstc() && serial_getc() == 'c')
		return 1;

#ifdef CONFIG_SPL_ENV_SUPPORT
	env_init();
	env_relocate_spec();
	if (getenv_yesno("boot_os") != 1)
		return 1;
#endif

	return 0;
}
#endif

#define OSC	(V_OSCK/1000000)
const struct dpll_params dpll_ddr_bone_black = {
		400, OSC-1, 1, -1, -1, -1, -1};

static struct ctrl_dev *cdev = (struct ctrl_dev *)CTRL_DEVICE_BASE;

void am33xx_spl_board_init(void)
{
	enable_i2c0_pin_mux();
	i2c_init(CONFIG_SYS_OMAP24_I2C_SPEED, CONFIG_SYS_OMAP24_I2C_SLAVE);

	/* Get the frequency */
	dpll_mpu_opp100.m = am335x_get_efuse_mpu_max_freq(cdev);

        /*
         * Override what we have detected since we know if we have
         * a Beaglebone Black it supports 1GHz.
         */
        dpll_mpu_opp100.m = MPUPLL_M_1000;

        /* Set CORE Frequencies to OPP100 */
        do_setup_dpll(&dpll_core_regs, &dpll_core_opp100);

        /* Set MPU Frequency to what we detected now that voltages are set */
	do_setup_dpll(&dpll_mpu_regs, &dpll_mpu_opp100);

#ifdef CONFIG_PCF2127_WATCHDOG
	if (i2c_probe(PCF2127_I2C_ADDR))
		return;
	pcf2127_wdt_enable();
	hw_watchdog_init();
#endif
}

const struct dpll_params *get_dpll_ddr_params(void)
{
        return &dpll_ddr_bone_black;
}

void set_uart_mux_conf(void)
{
#if CONFIG_CONS_INDEX == 1
	enable_uart0_pin_mux();
#elif CONFIG_CONS_INDEX == 5
	enable_uart4_pin_mux();
#endif
}

void set_mux_conf_regs(void)
{
	enable_board_pin_mux();
}

const struct ctrl_ioregs ioregs_bonelt = {
	.cm0ioctl		= MT41K256M16HA125E_IOCTRL_VALUE,
	.cm1ioctl		= MT41K256M16HA125E_IOCTRL_VALUE,
	.cm2ioctl		= MT41K256M16HA125E_IOCTRL_VALUE,
	.dt0ioctl		= MT41K256M16HA125E_IOCTRL_VALUE,
	.dt1ioctl		= MT41K256M16HA125E_IOCTRL_VALUE,
};

void sdram_init(void)
{
	config_ddr(400, &ioregs_bonelt,
		   &ddr3_beagleblack_data,
		   &ddr3_beagleblack_cmd_ctrl_data,
		   &ddr3_beagleblack_emif_reg_data, 0);
}
#endif


#if !defined(CONFIG_SPL_BUILD)
static void request_and_set_gpio(int gpio, char *name, int val)
{
	int ret;

	ret = gpio_request(gpio, name);
	if (ret < 0) {
		printf("%s: Unable to request %s\n", __func__, name);
		return;
	}

	ret = gpio_direction_output(gpio, 0);
	if (ret < 0) {
		printf("%s: Unable to set %s  as output\n", __func__, name);
		goto err_free_gpio;
	}

	gpio_set_value(gpio, val);

	return;

err_free_gpio:
	gpio_free(gpio);
}
#endif


/*
 * Basic board specific setup.  Pinmux has been handled already.
 */
int board_init(void)
{
	i2c_init(CONFIG_SYS_OMAP24_I2C_SPEED, CONFIG_SYS_OMAP24_I2C_SLAVE);
	if (i2c_probe(PCF2127_I2C_ADDR))
		return 0;

	pcf2127_wdt_enable();
	hw_watchdog_init();
#if !defined(CONFIG_SPL_BUILD)
	request_and_set_gpio(GPIO_ETH_RESET, "gpio_eth_reset", 0);
#endif

	gd->bd->bi_boot_params = CONFIG_SYS_SDRAM_BASE + 0x100;
	return 0;
}

#ifdef CONFIG_BOARD_LATE_INIT
int board_late_init(void)
{
        //TBD what todo in here
       return 0;
}
#endif
