#!/bin/sh

# Wireshark-style hex dump using BusyBox devmem with decimal addresses.
# This script dumps memory from 0x30090000 to 0x30092000 (hex range)
# Convert hex addresses to decimal for BusyBox devmem.

START_HEX=0x30090000
END_HEX=0x30092000

# Convert hex to decimal
START_DEC=$((START_HEX))
END_DEC=$((END_HEX))

# We will read 4 bytes (a 32-bit word) at a time.
# We want 16 bytes per line.
BYTES_PER_LINE=16
WORDS_PER_LINE=$((BYTES_PER_LINE / 4))

line_offset=0      # Will hold the starting address of the current line.
hex_line=""
ascii_line=""
byte_count=0       # Number of bytes in the current line

ADDR=$START_DEC

while [ "$ADDR" -lt "$END_DEC" ]; do
    # If starting a new line, record the line's starting offset.
    if [ $byte_count -eq 0 ]; then
        line_offset=$ADDR
    fi

    # Read a 32-bit word from memory.
    word=$(devmem "$ADDR" 32)
    # Convert the read value (assumed in decimal) to an integer.
    # Extract 4 bytes in little-endian order:
    b0=$(( word        & 0xFF ))
    b1=$(( (word >> 8)  & 0xFF ))
    b2=$(( (word >> 16) & 0xFF ))
    b3=$(( (word >> 24) & 0xFF ))

    # Process each byte in order.
    for byte in $b0 $b1 $b2 $b3; do
        # Append the hex representation.
        hex_byte=$(printf "%02X " "$byte")
        hex_line="${hex_line}${hex_byte}"
        # Check if byte is printable ASCII (32-126)
        if [ "$byte" -ge 32 ] && [ "$byte" -le 126 ]; then
            # Convert to ASCII character.
            ascii_char=$(printf "\\$(printf '%03o' "$byte")")
        else
            ascii_char="."
        fi
        ascii_line="${ascii_line}${ascii_char}"
        byte_count=$((byte_count + 1))

        # If we have reached a full line, print it.
        if [ $byte_count -eq $BYTES_PER_LINE ]; then
            printf "%08X:  %s  %s\n" "$line_offset" "$hex_line" "$ascii_line"
            # Reset the line buffers and counter.
            hex_line=""
            ascii_line=""
            byte_count=0
        fi
    done

    # Increment address by 4 bytes.
    ADDR=$((ADDR + 4))
done

# If there are any leftover bytes, pad the hex output and print the final line.
if [ $byte_count -gt 0 ]; then
    missing=$((BYTES_PER_LINE - byte_count))
    pad=""
    i=0
    while [ $i -lt $missing ]; do
        pad="${pad}   "
        i=$((i + 1))
    done
    hex_line="${hex_line}${pad}"
    printf "%08X:  %s  %s\n" "$line_offset" "$hex_line" "$ascii_line"
fi

