/*  ============================================================================
 *   Copyright (c) Texas Instruments Incorporated 2010-2016
 *   Copyright (c) 2017 Appear TV AS - http://appeartv.com
 *
 *  This code is a derivative work of the PCIe driver example
 *  from Texas Instruments Processor SDK available at:
 *  http://www.ti.com/tool/processor-sdk-c667x
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/


#include "atv_pcie_init.h"
#include "atv_uart.h"
#include <stdio.h>
#include <string.h>
#include <ti/csl/csl_bootcfgAux.h>
#include <ti/csl/csl_cacheAux.h>
#include <ti/csl/csl_chip.h>
#include <ti/csl/csl_pscAux.h>
#include <ti/csl/csl_serdes_pcie.h>
#include <ti/csl/csl_xmcAux.h>
#include <ti/csl/cslr_device.h>
#include <ti/drv/pcie/soc/pcie_soc.h>
#include <ti/platform/platform.h>


/**
* Read cycle count
*/
static uint32_t readTime32(void)
{
	return TSCL;
}


/**
* Delay by minimum number of cycles
*/
static void cycleDelay (uint32_t count)
{
	uint32_t start = (uint32_t)readTime32();

	while (((uint32_t)readTime32() - start) < count);
}


/**
* Control access to DBI
*/
static pcieRet_e pcieCfgDbi(Pcie_Handle handle, uint8_t enable)
{
	pcieRegisters_t        regs;
	pcieRet_e              retVal;
	pcieCmdStatusReg_t     cmdStatus;

	memset (&cmdStatus, 0, sizeof(cmdStatus));
	memset (&regs, 0, sizeof(regs));

	regs.cmdStatus = &cmdStatus;
	if ((retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK) {
		atv_write_uart("Read CMD STATUS register failed!\n");
		return retVal;
	}
	cmdStatus.dbi = enable;

	if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK) {
		atv_write_uart("SET CMD STATUS register failed!\n");
		return retVal;
	}

	return pcie_RET_OK;
}


/**
* Configure Power Domain for PCIe
*/
static pcieRet_e pciePowerCfg(void)
{
	/* Turn on the PCIe power domain */
	if (CSL_PSC_getPowerDomainState(CSL_PSC_PD_PCIEX) != PSC_PDSTATE_ON) {
		/* Enable the domain */
		CSL_PSC_enablePowerDomain(CSL_PSC_PD_PCIEX);
		/* Enable MDCTL */
		CSL_PSC_setModuleNextState(CSL_PSC_LPSC_PCIEX, PSC_MODSTATE_ENABLE);
		/* Apply the domain */
		CSL_PSC_startStateTransition(CSL_PSC_PD_PCIEX);
		/* Wait for it to finish */
		while (!CSL_PSC_isStateTransitionDone(CSL_PSC_PD_PCIEX));
	} else {
		atv_write_uart("Power domain is already enabled.  You probably re-ran without device reset (which is OK)\n");
	}

	return pcie_RET_OK;
}


/**
* Configure SERDES
*/
static pcieRet_e pcieSerdesCfg(void)
{
	uint16_t cfg;

	/* Provide PLL reference clock to SERDES inside PCIESS
	Program PLL settings and enable PLL from PCIe SERDES.*/
	cfg = 0x01C9; /* value based on PCIe userguide */

	CSL_BootCfgSetPCIEConfigPLL(cfg);

	/*Wait for PLL to lock (3000 CLKIN1 cycles) */
	cycleDelay(10000);

	return pcie_RET_OK;
}


/**
* Control Link Training
*/
static pcieRet_e pcieLtssmCtrl(Pcie_Handle handle, uint8_t enable)
{
	pcieCmdStatusReg_t       cmdStatus;
	pcieTiConfDeviceCmdReg_t deviceCmd;
	pcieRegisters_t          regs;
	pcieRet_e retVal;

	memset (&cmdStatus,    0, sizeof(cmdStatus));
	memset (&deviceCmd,    0, sizeof(deviceCmd));
	memset (&regs,         0, sizeof(regs));

	regs.cmdStatus = &cmdStatus;
	if ((retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK) {
		if (retVal == pcie_RET_INV_REG) {
			/* The cmdStatus register doesn't exist; try the deviceCmd instead */
			regs.cmdStatus       = NULL;
			regs.tiConfDeviceCmd = &deviceCmd;

			if ((retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK) {
				atv_write_uart("Read CMD STATUS and DEVICE CMD registers failed!\n");
				return retVal;
			}
		} else {
			atv_write_uart("Read CMD STATUS register failed!\n");
			return retVal;
		}
	}

	if (enable) {
		deviceCmd.ltssmEn = cmdStatus.ltssmEn = 1;
	} else {
		deviceCmd.ltssmEn = cmdStatus.ltssmEn = 0;
	}

	if ((retVal = Pcie_writeRegs (handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK) {
		atv_write_uart("SET CMD STATUS register failed!\n");
		return retVal;
	}

	return pcie_RET_OK;
}


/**
* Set generation 2 PCIe mode
*/
static pcieRet_e pcieSetGen2(Pcie_Handle handle)
{
	pcieRet_e              retVal;

	pcieRegisters_t        regs;
	pcieLinkCapReg_t       linkCap;
	pcieGen2Reg_t          gen2;

	uint8_t                targetGen, dirSpd;
	char string_temp[80] = "";

	targetGen = 2;
	dirSpd    = 1;

	memset (&gen2,             0, sizeof(gen2));
	memset (&linkCap,          0, sizeof(linkCap));
	memset (&regs,             0, sizeof(regs));

	/* Set gen1/gen2 in link cap */
	regs.linkCap = &linkCap;
	if ((retVal = Pcie_readRegs (handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK)
	{
		atv_write_uart("GET linkCap register failed!\n");
		return retVal;
	}

	if (linkCap.maxLinkSpeed != targetGen)
	{
		atv_sprintf_uart(string_temp, "PowerUP linkCap gen=%d change to %d\n", linkCap.maxLinkSpeed, targetGen);
		linkCap.maxLinkSpeed = targetGen;
	}
	else
	{
		regs.linkCap = NULL; /* Nothing to write back */
	}

	/* Setting PL_GEN2 */
	gen2.numFts = 0xF;
	gen2.dirSpd = dirSpd;
	/* TODO: Determine what is correct setting here */
	gen2.lnEn   = 2;
	regs.gen2 = &gen2;

	if ((retVal = Pcie_writeRegs (handle, pcie_LOCATION_LOCAL, &regs)) != pcie_RET_OK)
	{
		atv_write_uart("SET GEN2/link cap register failed!\n");
		return retVal;
	}

	return retVal;
}


/**
 * Configure and Enable Inbound Address Translation
 */
static pcieRet_e pcieEnableIbTrans(Pcie_Handle handle)
{
  pcieRet_e retVal;

  pcieRegisters_t      setRegs;
  pcieRegisters_t      getRegs;

  pcieCmdStatusReg_t   cmdStatus;

  memset (&setRegs, 0, sizeof(setRegs));
  memset (&getRegs, 0, sizeof(getRegs));
  memset (&cmdStatus, 0, sizeof(cmdStatus));

  getRegs.cmdStatus = &cmdStatus;
  if ((retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &getRegs)) != pcie_RET_OK) {
    atv_write_uart("Read CMD STATUS register failed!\n");
    return retVal;
  }
  cmdStatus.ibXltEn = 1;
  setRegs.cmdStatus = &cmdStatus;

  if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK) {
    atv_write_uart("SET CMD STATUS register failed!\n");
    return retVal;
  }

  return pcie_RET_OK;
}


/**
* Configure End Point
*/
static pcieRet_e pcieCfgEP(Pcie_Handle handle)
{
	pcieRet_e retVal;

	pcieObSizeReg_t        obSize;
	pcieGen2Reg_t          gen2;
	pcieType0Bar32bitIdx_t type0Bar32bitIdx;
	pcieStatusCmdReg_t     statusCmd;
	pcieDevStatCtrlReg_t   devStatCtrl;
	pcieAccrReg_t          accr;
	uint8_t bar;
	pcieVndDevIdReg_t vndDevId;
	pcieRevIdReg_t revId;

	pcieRegisters_t        setRegs;
	pcieRegisters_t        getRegs;

	memset(&obSize,           0, sizeof(obSize));
	memset(&gen2,             0, sizeof(gen2));
	memset(&type0Bar32bitIdx, 0, sizeof(type0Bar32bitIdx));
	memset(&statusCmd,        0, sizeof(statusCmd));
	memset(&devStatCtrl,      0, sizeof(devStatCtrl));
	memset(&accr,             0, sizeof(accr));

	/*Disable link training*/
	if ((retVal = pcieLtssmCtrl(handle, FALSE)) != pcie_RET_OK) {
		atv_write_uart("Failed to disable Link Training!\n");
		return retVal;
	}

	/* Configure vendor & device ID, class code & revision ID */
	memset(&setRegs, 0, sizeof(setRegs));

	/* Same values as with PCIe ROM boot */
	vndDevId.devId = PCIE_DEVICE_ID;
	vndDevId.vndId = PCIE_VENDOR_ID;
	revId.classCode = PCIE_CLASSCODE;
	revId.revId = PCIE_REVISION;

	setRegs.vndDevId = &vndDevId;
	setRegs.revId = &revId;

	if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK) {
		atv_write_uart("SET VENDOR_DEVICE_ID and CLASSCODE_REVID registers failed!\n");
		return retVal;
	}

	/* Configure the size of the translation regions */
	memset(&setRegs, 0, sizeof(setRegs));

	/* Only required for rev 0 */
	obSize.size = pcie_OB_SIZE_8MB;
	setRegs.obSize = &obSize;

	if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK) {
		atv_write_uart("SET OB_SIZE register failed!\n");
		return retVal;
	}

	/* Set gen2/link cap */
	if ((retVal = pcieSetGen2(handle)) != pcie_RET_OK) {
		atv_write_uart("pcieSetGen2 failed!\n");
		return retVal;
	}

	/* Configure BAR Masks */
	/* First need to enable writing on BAR mask registers */
	if ((retVal = pcieCfgDbi(handle, 1)) != pcie_RET_OK) {
		return retVal;
	}

	/* Configure Masks*/
	memset(&setRegs, 0, sizeof(setRegs));
	setRegs.type0BarMask32bitIdx = &type0Bar32bitIdx;

	/* Configure _all_ BARs, not just the ones we need. */
	for (bar = 0; bar < 6; bar++) {
		type0Bar32bitIdx.idx = bar;
		switch (bar) {
			case 0: type0Bar32bitIdx.reg.reg32 = PCIE_BAR0_BAR_MASK; break;
			case 1: type0Bar32bitIdx.reg.reg32 = PCIE_BAR1_BAR_MASK; break;
			case 2: type0Bar32bitIdx.reg.reg32 = PCIE_BAR2_BAR_MASK; break;
			case 3: type0Bar32bitIdx.reg.reg32 = PCIE_BAR3_BAR_MASK; break;
			case 4: type0Bar32bitIdx.reg.reg32 = PCIE_BAR4_BAR_MASK; break;
			case 5: type0Bar32bitIdx.reg.reg32 = PCIE_BAR5_BAR_MASK; break;
			default:
				type0Bar32bitIdx.reg.reg32 = 0;
				atv_write_uart("Invalid BAR number!\n");
				return pcie_RET_UNSUPPORTED;
		}

		if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK){
			atv_write_uart("SET BAR MASK register failed!\n");
			return retVal;
		}
	}

	/* Disable DBI writes */
	if ((retVal = pcieCfgDbi(handle, 0)) != pcie_RET_OK) {
		return retVal;
	}

	/* Enable memory access and mastership of the bus */
	memset (&setRegs, 0, sizeof(setRegs));
	memset (&getRegs, 0, sizeof(getRegs));

	getRegs.statusCmd = &statusCmd;
	if ((retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &getRegs)) != pcie_RET_OK) {
		atv_write_uart("Read Status Comand register failed!\n");
		return retVal;
	}
	statusCmd.memSp  = 1;
	statusCmd.busMs  = 1;
	statusCmd.resp   = 1;
	statusCmd.serrEn = 1;
	setRegs.statusCmd = &statusCmd;

	if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK) {
		atv_write_uart("SET Status Command register failed!\n");
		return retVal;
	}

	/* Enable Error Reporting */
	memset (&setRegs, 0, sizeof(setRegs));
	memset (&getRegs, 0, sizeof(getRegs));

	getRegs.devStatCtrl = &devStatCtrl;
	if ((retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &getRegs)) != pcie_RET_OK) {
		atv_write_uart("Read Device Status Control register failed!\n");
		return retVal;
	}

	devStatCtrl.reqRp = 1;
	devStatCtrl.fatalErRp = 1;
	devStatCtrl.nFatalErRp = 1;
	devStatCtrl.corErRp = 1;
	setRegs.devStatCtrl = &devStatCtrl;

	if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK) {
		atv_write_uart("SET Device Status Control register failed!\n");
		return retVal;
	}

	/* Enable ECRC -- only required for REV0 */
	memset (&setRegs, 0, sizeof(setRegs));

	accr.chkEn=1;
	accr.chkCap=1;
	accr.genEn=1;
	accr.genCap=1;
	setRegs.accr = &accr;

	if ((retVal = Pcie_writeRegs(handle, pcie_LOCATION_LOCAL, &setRegs)) != pcie_RET_OK) {
		atv_write_uart("SET ACCR register failed!\n");
		return retVal;
	}

	return pcie_RET_OK;
}


/**
 * Wait for Link to come Up
 */
static void pcieWaitLinkUp(Pcie_Handle handle)
{
	pcieRegisters_t  getRegs;

	memset (&getRegs, 0, sizeof(getRegs));

	pcieDebug0Reg_t            ltssmStateReg;
	getRegs.debug0 =          &ltssmStateReg;

	memset (&ltssmStateReg,  0, sizeof(ltssmStateReg));

	uint8_t ltssmState = 0;

	while(ltssmState != pcie_LTSSM_L0) {
		cycleDelay(100);
		if (Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &getRegs) != pcie_RET_OK) {
			atv_write_uart("Read LTSSM state failed!\n");
			return;
		}
		ltssmState = ltssmStateReg.ltssmState;
	}
}


/**
* Check Link Parameters
*/
static pcieRet_e pcieCheckLinkParams(Pcie_Handle handle)
{
	pcieRet_e retVal = pcie_RET_OK;
	pcieRegisters_t regs;
	pcieLinkStatCtrlReg_t linkStatCtrl;
	int32_t expLanes, expSpeed;
	const char *pass = "PASS", *fail = "FAIL";
	const char *result = pass;
	char string_temp[80] = "";

	expSpeed = 2;
	/* TODO: Determine if this is correct setting */
	expLanes = 2;

	/* Get link status */
	memset (&regs, 0, sizeof(regs));
	regs.linkStatCtrl = &linkStatCtrl;

	atv_write_uart("Checking link speed and # of lanes\n");
	retVal = Pcie_readRegs(handle, pcie_LOCATION_LOCAL, &regs);
	if (retVal != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Failed to read linkStatCtrl: %d\n", retVal);
	} else {
		/* Check number of lanes */
		if (expLanes != linkStatCtrl.negotiatedLinkWd) {
			result = fail;
			retVal = pcie_RET_UNSUPPORTED;
		} else {
			result = pass;
		}
		atv_sprintf_uart(string_temp, "Expect %d lanes, found %d lanes (%s)\n",
		(int)expLanes, (int)linkStatCtrl.negotiatedLinkWd, result);

		/* Check speed */
		if (expSpeed != linkStatCtrl.linkSpeed) {
			result = fail;
			retVal = pcie_RET_UNSUPPORTED;
		} else {
			result = pass;
		}
		atv_sprintf_uart(string_temp, "Expect gen %d speed, found gen %d speed (%s)\n",
		(int)expSpeed, (int)linkStatCtrl.linkSpeed, result);
	}

	return retVal;
}


int atv_pcie_init(void)
{
	pcieBarCfg_t barCfg;
	Pcie_Handle handle = NULL;
	pcieIbTransCfg_t ibCfg;
	uint8_t ib_region;
	uint16_t lock = 0;
	void *pcieBase;
	pcieRet_e ret_val;
	char string_temp[80] = "";

	TSCL = 1;

	/* Unlock kicker once, and don't relock, because its not multicore safe */
	CSL_BootCfgUnlockKicker();

	/* Pass device config to LLD
	* (located in ti/drv/pcie/soc/c6678/src/pcie_soc.c)
	*/
	if ((ret_val = Pcie_init (&pcieInitCfg)) != pcie_RET_OK) {
		atv_write_uart("LLD device configuration failed\n");
		return -1;
	}

	/* Power up PCIe Module */
	if ((ret_val = pciePowerCfg()) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "PCIe Power Up failed (%d)\n", (int)ret_val);
		return -1;
	}

	atv_write_uart ("PCIe Power Up.\n");

	if ((ret_val = Pcie_open(0, &handle)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Open failed (%d)\n", (int)ret_val);
		return -1;
	}

	atv_write_uart ("PCIe opened.\n");

	/* Configure SERDES*/
	if ((ret_val = pcieSerdesCfg()) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "PCIe Serdes config failed (%d)\n", (int)ret_val);
		return -1;
	}

	atv_write_uart ("PCIe Serdes configured.\n");

	/* Set the PCIe mode*/
	if ((ret_val = Pcie_setInterfaceMode(handle, pcie_EP_MODE)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Set PCIe Mode failed (%d)\n", (int)ret_val);
		return -1;
	}

	atv_write_uart ("PCIe Mode set to EP.\n");

	/* Wait until the PCIe SERDES PLL locks */
	while (!lock) {
		CSL_BootCfgGetPCIEPLLLock(&lock);
	}

	atv_write_uart("PLL configured.\n");

	/* Configure application registers for End Point*/
	if ((ret_val = pcieCfgEP(handle)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Failed to configure PCIe in EP mode (%d)\n", (int)ret_val);
		return -1;
	}

	atv_write_uart("Application registers configured.\n");

	/* Configure Address Translation */
	barCfg.location = pcie_LOCATION_LOCAL;
	barCfg.mode     = pcie_EP_MODE;
	barCfg.prefetch = pcie_BAR_NON_PREF;
	barCfg.type     = pcie_BAR_TYPE32;
	barCfg.memSpace = pcie_BAR_MEM_MEM;

	ibCfg.ibStartAddrHi = 0;

	for (ib_region = 0; ib_region < PCIE_IB_NUM; ib_region++) {
		barCfg.idx = ib_region + 1;

		switch (ib_region) {
			case 0: barCfg.base = PCIE_IB0_START_ADDR; break;
			case 1: barCfg.base = PCIE_IB1_START_ADDR; break;
			case 2: barCfg.base = PCIE_IB2_START_ADDR; break;
			case 3:
				barCfg.base = PCIE_IB3_START_ADDR;
				barCfg.type = pcie_BAR_TYPE64;
				break;
			default:
				atv_write_uart("Invalid IB region number!\n");
				return -1;
		}

		ibCfg.region        = ib_region;
		ibCfg.ibStartAddrLo = barCfg.base;
		ibCfg.ibOffsetAddr  = barCfg.base;
		ibCfg.ibBar         = barCfg.idx;

		if ((ret_val = Pcie_cfgBar(handle, &barCfg)) != pcie_RET_OK) {
			atv_write_uart("Failed to configure BAR!\n");
			return -1;
		}

		if ((ret_val = Pcie_cfgIbTrans(handle, &ibCfg)) != pcie_RET_OK) {
			atv_write_uart("Failed to configure IB Translation registers!\n");
			return -1;
		}
	}

	atv_write_uart("IB translation configured.\n");

	if ((ret_val = pcieEnableIbTrans(handle)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Failed to enable IB Translation (%d)!\n", (int)ret_val);
		return -1;
	} else {
		atv_write_uart("IB translation enabled.\n");
	}

	atv_write_uart("Starting link training...\n");

	/*Enable link training*/
	if ((ret_val = pcieLtssmCtrl(handle, TRUE)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Failed to Enable Link Training! (%d)\n", (int)ret_val);
		return -1;
	}

	/* Wait for link to be up */
	atv_write_uart("Waiting for link to go up...\n");
	pcieWaitLinkUp(handle);

	atv_write_uart("Link is up.\n");
	if ((ret_val = pcieCheckLinkParams(handle)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "Link width/speed verification FAILed: %d\n", ret_val);
		/* This exit() can be removed if this example is being used as
		* template with non TI card that supports slower or narrower connections
		*/
		return -1;
	}

	atv_write_uart("Link parameters verified.\n");

	if ((ret_val = Pcie_getMemSpaceRange(handle, &pcieBase, NULL)) != pcie_RET_OK) {
		atv_sprintf_uart(string_temp, "getMemSpaceRange failed (%d)\n", (int)ret_val);
		return -1;
	}

	atv_write_uart("Got memory space range of PCIe.\n");

	return 0;
}
